/*
 This file is part of GNU Taler
 (C) 2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  j2s,
  LibtoolVersion,
  Logger,
  succeedOrThrow,
  TalerCorebankApiClient,
  TalerErrorCode,
  TalerExchangeHttpClient,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  useSharedTestkudosEnvironment,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState, harnessHttpLib } from "../harness/harness.js";

const logger = new Logger("test-wallet-devexp-fakeprotover.ts");

export async function runWalletDevexpFakeprotoverTest(t: GlobalTestState) {
  const { walletClient, exchange, merchant, bank } =
    await useSharedTestkudosEnvironment(t);

  const exchangeClient = new TalerExchangeHttpClient(exchange.baseUrl, {
    httpClient: harnessHttpLib,
  });

  const exchangeConfRes = succeedOrThrow(await exchangeClient.getConfig());
  const exchVer = LibtoolVersion.parseVersionOrThrow(exchangeConfRes.version);

  const newerVer = `${exchVer.current + 1}:0:0`;

  const bankClient = new TalerCorebankApiClient(bank.baseUrl);

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    exchange,
    amount: "TESTKUDOS:10",
    bankClient,
  });
  await wres.withdrawalFinishedCond;

  const devExpUri = new URL("taler://dev-experiment/start-fakeprotover");
  devExpUri.searchParams.set("base_url", exchange.baseUrl);
  devExpUri.searchParams.set("fake_ver", newerVer);

  await walletClient.call(WalletApiOperation.InitWallet, {
    config: {
      testing: {
        devModeActive: true,
        skipDefaults: true,
      },
    },
  });

  await walletClient.call(WalletApiOperation.ApplyDevExperiment, {
    devExperimentUri: devExpUri.href,
  });

  logger.info("updating exchange entry after dev experiment");

  const err1 = await t.assertThrowsTalerErrorAsyncLegacy(
    walletClient.call(WalletApiOperation.UpdateExchangeEntry, {
      exchangeBaseUrl: exchange.baseUrl,
      force: true,
    }),
  );

  t.assertTrue(
    err1.errorDetail.code === TalerErrorCode.WALLET_EXCHANGE_UNAVAILABLE,
  );

  t.assertTrue(
    err1.errorDetail.innerError.code ===
      TalerErrorCode.WALLET_EXCHANGE_PROTOCOL_VERSION_INCOMPATIBLE,
  );

  logger.info("done updating exchange entry after dev experiment");

  const err2 = await t.assertThrowsTalerErrorAsyncLegacy(
    walletClient.call(WalletApiOperation.GetWithdrawalDetailsForAmount, {
      amount: "TESTKUDOS:10",
      exchangeBaseUrl: exchange.baseUrl,
    }),
  );

  t.assertTrue(
    err2.errorDetail.code === TalerErrorCode.WALLET_EXCHANGE_UNAVAILABLE,
  );

  t.assertTrue(
    err2.errorDetail.innerError.code ===
      TalerErrorCode.WALLET_EXCHANGE_PROTOCOL_VERSION_INCOMPATIBLE,
  );
}

runWalletDevexpFakeprotoverTest.suites = ["wallet"];
