/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { TalerMerchantApi } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { h, VNode } from "preact";
import { WithId } from "../../../../declaration.js";
import {
  datetimeFormatForSettings,
  usePreference,
} from "../../../../hooks/preference.js";

const TALER_SCREEN_ID = 72;

interface TablePropsIncoming {
  transfers: (TalerMerchantApi.IncomingTransferDetails & WithId)[];
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
  onSelectedToConfirm: (d: TalerMerchantApi.IncomingTransferDetails) => void;
}

export function CardTableIncoming({
  transfers,
  onLoadMoreAfter,
  onLoadMoreBefore,
  onSelectedToConfirm,
}: TablePropsIncoming): VNode {
  const { i18n } = useTranslationContext();
  const [settings] = usePreference();
  return (
    <div class="card has-table">
      <header class="card-header">
        <p class="card-header-title">
          <span class="icon">
            <i class="mdi mdi-arrow-left-right" />
          </span>
          <i18n.Translate>Incoming wire transfers</i18n.Translate>
        </p>
      </header>
      <div class="card-content">
        <div class="b-table has-pagination">
          <div class="table-wrapper has-mobile-cards">
            {transfers.length > 0 ? (
              <div class="table-container">
                {onLoadMoreBefore && (
                  <button
                    type="button"
                    class="button is-fullwidth"
                    data-tooltip={i18n.str`Load more wire transfers preceding the first one`}
                    onClick={onLoadMoreBefore}
                  >
                    <i18n.Translate>Load first page</i18n.Translate>
                  </button>
                )}
                <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
                  <thead>
                    <tr>
                      <th>
                        <i18n.Translate>ID</i18n.Translate>
                      </th>
                      <th>
                        <i18n.Translate>Expected credit</i18n.Translate>
                      </th>
                      <th>
                        <i18n.Translate>Confirmed</i18n.Translate>
                      </th>
                      <th>
                        <i18n.Translate>Validated</i18n.Translate>
                      </th>
                      <th>
                        <i18n.Translate>Executed on</i18n.Translate>
                      </th>
                      {/* <th /> */}
                    </tr>
                  </thead>
                  <tbody>
                    {transfers.map((i) => {
                      return (
                        <tr
                          key={i.id}
                          style={{
                            cursor: !i.confirmed ? "pointer" : undefined
                          }}
                          onClick={
                            !i.confirmed ? () => onSelectedToConfirm(i) : undefined
                          }
                        >
                          <td title={i.wtid}>{i.wtid.substring(0, 16)}...</td>
                          <td>{i.expected_credit_amount}</td>
                          <td>{i.confirmed ? i18n.str`yes` : i18n.str`no`}</td>
                          <td>{i.validated ? i18n.str`yes` : i18n.str`no`}</td>
                          <td>
                            {i.execution_time
                              ? i.execution_time.t_s == "never"
                                ? i18n.str`never`
                                : format(
                                    i.execution_time.t_s * 1000,
                                    datetimeFormatForSettings(settings),
                                  )
                              : i18n.str`unknown`}
                          </td>
                        </tr>
                      );
                    })}
                  </tbody>
                </table>
                {onLoadMoreAfter && (
                  <button
                    type="button"
                    class="button is-fullwidth"
                    data-tooltip={i18n.str`Load more transfers after the last one`}
                    onClick={onLoadMoreAfter}
                  >
                    <i18n.Translate>Load next page</i18n.Translate>
                  </button>
                )}
              </div>
            ) : (
              <EmptyTable />
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

interface TablePropsVerified {
  transfers: (TalerMerchantApi.TransferDetails & WithId)[];
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

export function CardTableVerified({
  transfers,
  onLoadMoreAfter,
  onLoadMoreBefore,
}: TablePropsVerified): VNode {
  const { i18n } = useTranslationContext();
  const [settings] = usePreference();
  return (
    <div class="card has-table">
      <header class="card-header">
        <p class="card-header-title">
          <span class="icon">
            <i class="mdi mdi-arrow-left-right" />
          </span>
          <i18n.Translate>Verified wire transfers</i18n.Translate>
        </p>
      </header>
      <div class="card-content">
        <div class="b-table has-pagination">
          <div class="table-wrapper has-mobile-cards">
            {transfers.length > 0 ? (
              <div class="table-container">
                {onLoadMoreBefore && (
                  <button
                    type="button"
                    class="button is-fullwidth"
                    data-tooltip={i18n.str`Load more wire transfers preceding the first one`}
                    onClick={onLoadMoreBefore}
                  >
                    <i18n.Translate>Load first page</i18n.Translate>
                  </button>
                )}
                <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
                  <thead>
                    <tr>
                      <th>
                        <i18n.Translate>ID</i18n.Translate>
                      </th>
                      <th>
                        <i18n.Translate>Amount</i18n.Translate>
                      </th>
                      <th>
                        <i18n.Translate>Confirmed</i18n.Translate>
                      </th>
                      <th>
                        <i18n.Translate>Verified</i18n.Translate>
                      </th>
                      <th>
                        <i18n.Translate>Executed on</i18n.Translate>
                      </th>
                      {/* <th /> */}
                    </tr>
                  </thead>
                  <tbody>
                    {transfers.map((i) => {
                      return (
                        <tr key={i.id}>
                          <td title={i.wtid}>{i.wtid.substring(0, 16)}...</td>
                          <td>{i.credit_amount}</td>
                          <td>{i.confirmed ? i18n.str`yes` : i18n.str`no`}</td>
                          <td>{i.verified ? i18n.str`yes` : i18n.str`no`}</td>
                          <td>
                            {i.execution_time
                              ? i.execution_time.t_s == "never"
                                ? i18n.str`never`
                                : format(
                                    i.execution_time.t_s * 1000,
                                    datetimeFormatForSettings(settings),
                                  )
                              : i18n.str`unknown`}
                          </td>
                        </tr>
                      );
                    })}
                  </tbody>
                </table>
                {onLoadMoreAfter && (
                  <button
                    type="button"
                    class="button is-fullwidth"
                    data-tooltip={i18n.str`Load more transfers after the last one`}
                    onClick={onLoadMoreAfter}
                  >
                    <i18n.Translate>Load next page</i18n.Translate>
                  </button>
                )}
              </div>
            ) : (
              <EmptyTable />
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

function EmptyTable(): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="content has-text-grey has-text-centered">
      <p>
        <span class="icon is-large">
          <i class="mdi mdi-magnify mdi-48px" />
        </span>
      </p>
      <p>
        <i18n.Translate>
          There are no transfers to list yet, add more by pressing the + sign
        </i18n.Translate>
      </p>
    </div>
  );
}
