/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  HttpStatusCode,
  TalerError,
  TalerErrorCode,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import {
  useTranslationContext
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../../components/exception/loading.js";
import { useOrderDetails } from "../../../../hooks/order.js";
import { LoginPage } from "../../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";
import { DetailPage } from "./DetailPage.js";

const TALER_SCREEN_ID = 45;

export interface Props {
  oid: string;
  onBack: () => void;
}

export default function Update({ oid, onBack }: Props): VNode {
  const result = useOrderDetails(oid);

  const { i18n } = useTranslationContext();

  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case TalerErrorCode.MERCHANT_GENERIC_INSTANCE_UNKNOWN: {
        return <NotFoundPageOrAdminCreate />;
      }
      case TalerErrorCode.MERCHANT_GENERIC_ORDER_UNKNOWN: {
        return <i18n.Translate>Order unknown</i18n.Translate>;
      }
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  return (
    <Fragment>

      <DetailPage
        onBack={onBack}
        id={oid}
        onRefunded={() => {}}
        selected={result.body}
      />
    </Fragment>
  );
}
