/*
 * This file is part of LibEuFin.
 * Copyright (C) 2023-2024 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

package tech.libeufin.nexus.cli

import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.parameters.groups.provideDelegate
import kotlinx.coroutines.delay
import tech.libeufin.common.CommonOption
import tech.libeufin.common.Payto
import tech.libeufin.common.cliCmd
import tech.libeufin.common.fmt
import tech.libeufin.nexus.*
import tech.libeufin.nexus.db.InitiatedPayment
import tech.libeufin.nexus.ebics.EbicsClient
import java.time.Duration
import java.time.Instant
import kotlin.time.toKotlinDuration

/** 
 * Submit an initiated [payment] using [client].
 * 
 * Parse creditor IBAN account metadata then perform an EBICS direct credit.
 * 
 * Returns the orderID
 */
private suspend fun submitInitiatedPayment(
    client: EbicsClient,
    payment: InitiatedPayment
): String {
    val creditAccount = try {
        val payto = Payto.parse(payment.creditPaytoUri).expectIban()
        IbanAccountMetadata(
            iban = payto.iban.value,
            bic = payto.bic,
            name = payto.receiverName!!
        )
    } catch (e: Exception) {
        throw e // TODO handle payto error
    }
    
    val xml = createPain001(
        requestUid = payment.requestUid,
        initiationTimestamp = payment.initiationTime,
        amount = payment.amount,
        creditAccount = creditAccount,
        debitAccount = client.cfg.account,
        wireTransferSubject = payment.wireTransferSubject,
        dialect = client.cfg.dialect
    )
    return client.upload(
        client.cfg.dialect.directDebit(),
        xml
    )
}

/** Submit all pending initiated payments using [client] */
private suspend fun submitBatch(client: EbicsClient) {
    client.db.initiated.submittable(client.cfg.currency).forEach {
        logger.debug("Submitting payment '${it.requestUid}'")
        runCatching { submitInitiatedPayment(client, it) }.fold(
            onSuccess = { orderId -> 
                client.db.initiated.submissionSuccess(it.id, Instant.now(), orderId)
                logger.info("Payment '${it.requestUid}' submitted")
            },
            onFailure = { e ->
                client.db.initiated.submissionFailure(it.id, Instant.now(), e.message)
                logger.error("Payment '${it.requestUid}' submission failure: ${e.fmt()}")
                throw e
            }
        )
    }
}

class EbicsSubmit : CliktCommand("Submits pending initiated payments found in the database") {
    private val common by CommonOption()
    private val transient by transientOption()
    private val ebicsLog by ebicsLogOption()
    
    override fun run() = cliCmd(logger, common.log) {
        nexusConfig(common.config).withDb { db, nexusCfg ->
            val cfg = nexusCfg.ebics
            val (clientKeys, bankKeys) = expectFullKeys(cfg)
            val client = EbicsClient(
                cfg,
                httpClient(),
                db,
                EbicsLogger(ebicsLog),
                clientKeys,
                bankKeys
            )
            val frequency: Duration = if (transient) {
                logger.info("Transient mode: submitting what found and returning.")
                Duration.ZERO
            } else {
                logger.debug("Running with a frequency of ${cfg.submit.frequencyRaw}")
                if (cfg.submit.frequency == Duration.ZERO) {
                    logger.warn("Long-polling not implemented, running therefore in transient mode")
                }
                cfg.submit.frequency
            }
            do {
                try {
                    submitBatch(client)
                } catch (e: Exception) {
                    throw Exception("Failed to submit payments", e)
                }
                // TODO take submitBatch taken time in the delay
                delay(frequency.toKotlinDuration())
            } while (frequency != Duration.ZERO)
        }
    }
}