/*
 * Copyright (C) 2000, 2001  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <fstream>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "fileutil.h"
#include "my_string.h"

using std::cerr;
using std::endl;
using std::ifstream;

static void file_error(const char *error, char *file);
static void internal_error(const char *error);

char *
read_file(char *fileName)
{
  FILE *file;
  char *buffer = 0;
  long int char_count;

  // we open it as binary otherwise we may experience problems under
  // Windows system: when we fread, the number of char read can be
  // less then char_count, and thus we'd get an error...
  if ( (file = fopen(fileName,"rb") ) == 0 )	// The file does not exist :(
    file_error ("Error operning", fileName);
  else
    {
      // let's go to the end of the file...
      if (fseek (file, 0, SEEK_END) != 0)
        file_error ("Error positioning", fileName);

      // ...to read the dimension
      char_count = ftell (file);
      if (char_count < 0)
        file_error ("Error reading position", fileName);

      buffer = (char *) malloc (char_count +1);
      if (! buffer)
        internal_error ("Memory allocation failed");

      // let's go back to the start
      rewind (file);

      if (fread ((void *) buffer, 1, char_count, file) < (size_t) char_count)
        file_error ("read error", fileName);
      buffer[char_count] = '\0';

      fclose (file);
    }

  return buffer;
}

void
file_error(const char *error, char *file)
{
  cerr << PACKAGE << ": " << error << ", file " << file << endl;
  exit (1);
}

void
internal_error(const char *error)
{
  cerr << PACKAGE << ": Internal error: " << error << endl;
  exit (1);
}

// output file name = <outputDir> + input file name + ext
char *
createOutputFileName( char *inputFileName, char *outputDir, const char *ext )
{
  char *input_file_name;
  char path_separator = '/';

  if (! outputDir)
    input_file_name = inputFileName;
  else
    {
      char *file_name; // without the path
      file_name = strrchr (inputFileName, '/');
      
      if (! file_name) // try with DOS separator
        {
          file_name = strrchr (inputFileName, '\\');
          if (file_name)
            path_separator = '\\';
        }


      if (file_name)
        input_file_name = &file_name[1];
      else
        input_file_name = inputFileName;
    }

  int lenght =  strlen(input_file_name) + 
    strlen(ext) + 1 ;
  int output_dir_lenght = 0;
  
  if (outputDir)
    {
      output_dir_lenght = strlen (outputDir);
      lenght += output_dir_lenght + 1; // one is for the additional /
    }

  char *outputFileName = new char[lenght] ;
  outputFileName[0] = '\0';

  if (outputDir)
    {
      strcat (outputFileName, outputDir);
      outputFileName[output_dir_lenght] = path_separator;
      outputFileName[output_dir_lenght + 1] = '\0';
    }
  strcat (outputFileName, input_file_name);
  strcat (outputFileName, ext);

  return outputFileName ;
}

unsigned int
get_line_count(char *file_name)
{
  ifstream input (file_name);

  if (! input)
    file_error ("Error opening", file_name);

  unsigned int count = get_line_count (input);

  input.close ();

  return count;
}

unsigned int
get_line_count(istream &input)
{
  unsigned int count = 0;
  string line;

  while (true)
    {
      getline (input, line);
      if (! input.eof ())
        ++count;
      else
        break;
    }

  return count;
}
