/* archimedes.c -- This code is a simulator for Submicron 2D III-V
   semiconductor Devices (along with SiO2). It implements the Monte Carlo method 
   and a  simplified version of MEP model for the simulation of the 
   semiclassical Boltzmann equation for both electrons and holes.
   It also includes the quantum effects by means of effective
   potential method. It is now able to simulate applied
   magnetic fields along with self consistent Faraday equation.

   Copyright (C) 2004, 2005, 2006, 2007 Jean Michel Sellier <sellier@dmi.unict.it>
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


// =================================================================
// File Name : archimedes.c
// Version   : release 0.1.1
// Date of Creation : 07 Dec.2003, Paris, France, Jean Michel Sellier
// Last Revision : 11 September 2007, Siracusa, Italy, Jean Michel Sellier.
// System of Measure : M.K.S.C. System
// =================================================================

#include<getopt.h>
#include<stdio.h>
#include<unistd.h>
#include<stdlib.h>
#include<math.h>
#include<memory.h>
#include<time.h>
#ifdef	HAVE_STRING_H
#include<string.h>
#else
#include<strings.h>
#endif


// Preprocessor Definitions
#define real double
#define MN3 4
#define NXM 308
#define NYM 308
#define DIME 2003
#define ITMAX 10000000
#define POISSONITMAX 1500
#define SMALL 1.e-5
#define VMAX 1000000
#define NPMAX 10000000 // maximum number of super-particles
#define MCE 0  // MCE stands for MC for electrons only
#define MCH 1  // MCH stands for MC for holes only
#define MCEH 2 // MCEH stands for MC both for electrons and holes
#define MEPE 3 // MEPE stands for MEP model for electrons only
#define MEPH 4 // MEPH stands for MEP model for holes only
#define MEPEH 5 // MEPEH stands for MEP model for electrons and holes
#define GNUPLOTFORMAT 0 // output file in GNUPLOT format
#define MESHFORMAT 1 // output file in Mesh format

// definition of the material reference table
#define NOAMTIA 4   // Number Of All Material Taken Into Account (excluding SiO2)
// ********************
#define iSIO2 -1      // reference number to Silicon Oxide
#define SILICON 0    // reference number to Silicon
#define GAAS 1       // reference number to GaAs
#define GERMANIUM 2  // reference number to Germanium
#define INSB 3       // reference number to InSb
// ********************

#define NUMSIO2 2 // maximum number of SiO2 interfaces

// ===============================
#include "constants.h"
#include "extrema.h"
#include "sign.h"
#include "mm.h"
#include "mm2.h"
#include "definitionoffunctions.h"
// ===============================

// All integers here...
int NUM_VERT,NUM_EXAHEDRA,MEDIA,MAXIMINI;
int SAVEALWAYS;
int nx,ny;
int ISEED,NP1,INUM,IV;
int c,Model_Number,File_Format,Quantum_Flag;
int leid_flag;
int SIO2_UP_FLAG;
int SIO2_DOWN_FLAG;
int FARADAYFLAG;
int i_dom[NXM+1][NYM+1];
int NOVALLEY[NOAMTIA+1];

// All "real"'s here...
real u2d[NXM+1][NYM+1][MN3+1];
real h2d[NXM+1][NYM+1][MN3+1];
real PSI[NXM+1][NYM+1],E[NXM+1][NYM+1][2];
real N_D[NXM+1][NYM+1],N_H[NXM+1][NYM+1];
real dx,dy;
real TEMPO=0.,TF;
real LX,LY;
real TL,DT;
real mstar2,msh;
real BKTQ,QH;
real SMH[NOAMTIA+1][3];
real HHM[NOAMTIA+1][3];
real HM[NOAMTIA+1][3];
real GM[NOAMTIA+1];
real SWK[NOAMTIA+1][3][14][DIME+1];
real P[NPMAX+1][7],EC[3];
real KX,KY,KZ,X,Y;
real TS,EPP,DDmax;
real EDGE[4][NXM+NYM+1][4];
real CIMP,QD2,TAUW;
real EPSRSIO2;
real DG[NXM+1][NYM+1],DL[NXM+1][NYM+1];
real bufx2d[NXM+1][NYM+1];
real bufy2d[NXM+1][NYM+1];
real ux2d[NXM+1][NYM+1][MN3+1];
real uy2d[NXM+1][NYM+1][MN3+1];
real f2d[NXM+1][NYM+1][MN3+1];
real g2d[NXM+1][NYM+1][MN3+1];
real fx2d[NXM+1][NYM+1][MN3+1];
real gy2d[NXM+1][NYM+1][MN3+1];
real c11[7],c12[7],c21[7],c22[7];
real u[7],f[7],g[7],cw[7];
real SIO2_INI[NUMSIO2],SIO2_FIN[NUMSIO2];
real SIO2_POT[NUMSIO2];
real SIO2_THICKNESS[NUMSIO2];
real SIO2[NUMSIO2][NXM+1][NYM+1];
real B[NXM+1][NYM+1];
real EPSR[NOAMTIA+1];
real EPF[NOAMTIA+1];
real MSTAR[NOAMTIA+1][6];
real alphaK[NOAMTIA+1][4];
real EG[NOAMTIA+1];
real HWO[NOAMTIA+1][6];
real DTK[NOAMTIA+1][6];
real ZF[NOAMTIA+1][6];
real RHO[NOAMTIA+1];
real DA[NOAMTIA+1];
real UL[NOAMTIA+1];
real EMIN[NOAMTIA+1][4];

// All structures here...
time_t binarytime;
struct tm *nowtm;
struct option longopts[] =
{
  { "version", no_argument, NULL, 'v' },
  { "help", no_argument, NULL, 'h' }
};
// All files here...
FILE *fp;
// All strings here...
static char *progname;

int 
main(int argc,char* argv[])
{
  int optc;
  int h = 0, v = 0, lose = 0, z = 0;

  progname = argv[0];

  while ((optc = getopt_long (argc, argv, "hv", longopts, (int *) 0))
         != EOF)
    switch (optc)
      {
      case 'v':
        v = 1;
        break;
      case 'h':
        h = 1;
        break;
      default:
        lose = 1;
        break;
      }
  
  if (optind == argc - 1)
    z = 1;
  else if (lose || optind < argc)
    {
      /* Print error message and exit.  */
      if (optind < argc)
        printf("Too many arguments\n");
        printf("Try `%s --help' for more information.\n",progname);
      exit (1);
    }

  /* `help' should come first.  If `help' is requested, ignore the other
     options. */
  if (h)
    {
      /* Print help info and exit.  */
      /* TRANSLATORS: --help output 1
         no-wrap */
      printf("\
GNU archimedes, a simulator for submicron silicon devices.\n");
      printf ("\n");
      /* TRANSLATORS: --help output 2
         no-wrap */
      printf ("\
Usage: %s [OPTION] file...\n",progname);

      printf ("\n");
      /* TRANSLATORS: --help output 3 : options 1/2
         no-wrap */
      printf("\
  -h, --help          display this help and exit\n\
  -v, --version       display version information and exit\n");

      printf ("\n");
      /* TRANSLATORS: --help output 5 (end)
         TRANSLATORS, please don't forget to add the contact address for
         your translation!
         no-wrap */
      printf ("\
Report bugs to sellier@dmi.unict.it or archimedes@southnovel.eu\n");

      exit (0);
    }

  if (v)
    {
      /* Print version number.  */
      printf("archimedes - GNU archimedes 0.1.1\n");
      /* xgettext: no-wrap */
      printf("\n");
      printf("\
Copyright (C) %s Sellier Jean Michel.\n\
There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n\
PARTICULAR PURPOSE.\n\
You may redistribute copies of GNU %s under the terms\n\
of the GNU General Public License.\n\
For more information about these matters, see the file named COPYING.\n",
              "2004, 2005, 2006, 2007","Archimedes");
      exit (0);
    }
  else if (z){
// In case of filename specified
// =============================
     fp=fopen(argv[1],"r"); // here we open th input file...
// File Control, just in case the file does not exist...
     if(fp==NULL){
      printf("%s: fatal error in opening the input file %s\n",
             progname,argv[1]);
      exit(EXIT_FAILURE);
     }
// ========================
// Material constants here!
// ========================
// Silicon electrons in the X-valley
// Germanium electrons in the X-valley
// All the other semiconductor compounds have electrons in the Gamma- and L-valley
// definition of the number of valley for every material
     NOVALLEY[SILICON]=1;
     NOVALLEY[GERMANIUM]=1;
     NOVALLEY[GAAS]=2;
     NOVALLEY[INSB]=2;
// Dielectric constant for Silicon Oxide SiO2
     EPSRSIO2=17.0*EPS0;
// Dielectric constant for Semiconducting materials
     EPSR[SILICON]=11.7*EPS0;
     EPSR[GERMANIUM]=16.2*EPS0;
     EPSR[GAAS]=12.90*EPS0;
     EPSR[INSB]=17.65*EPS0;
// III-V semiconductor compounds high frequency dieletric constant
     EPF[GAAS]=10.92;
     EPF[INSB]=15.68;
     int ii;
     for(ii=0;ii<NOAMTIA;ii++){
       int i;
       for(i=0;i<6;i++){
         HWO[ii][i]=0.;
         DTK[ii][i]=0.;
         ZF[ii][i]=0.;
       }
     }
// Silicon optical phonon scattering energy (eV)
     HWO[SILICON][0]=0.0120;
     HWO[SILICON][1]=0.0185;
     HWO[SILICON][2]=0.0190;
     HWO[SILICON][3]=0.0474;
     HWO[SILICON][4]=0.0612;
     HWO[SILICON][5]=0.0590;
// Germanium optical phonon scattering energy (eV)
     HWO[GERMANIUM][0]=0.09;
// III-V semiconductor compounds optical phonon scattering energy (eV)
     HWO[GAAS][0]=0.03536;
     HWO[INSB][0]=0.0282;
// Silicon optical coupling constants (eV/m)
     DTK[SILICON][0]=0.05e11;
     DTK[SILICON][1]=0.08e11;
     DTK[SILICON][2]=0.03e11;
     DTK[SILICON][3]=0.20e11;
     DTK[SILICON][4]=1.14e11;
     DTK[SILICON][5]=0.20e11;
// Germanium optical coupling constants (eV/m)
     DTK[GERMANIUM][0]=1.889e11;
// III-V semiconductor compounds optical coupling constant (eV/m)
     DTK[GAAS][0]=1.11e11;
     DTK[INSB][0]=0.47e11;
// Silicon optical phonon Z-factor
     ZF[SILICON][0]=1.;
     ZF[SILICON][1]=1.;
     ZF[SILICON][2]=4.;
     ZF[SILICON][3]=4.;
     ZF[SILICON][4]=1.;
     ZF[SILICON][5]=4.;
// Germanium optical phonon Z-factor
     ZF[GERMANIUM][0]=1.;
// III-V semiconductor compounds optical phonon Z-factor
     ZF[GAAS][0]=1.;
     ZF[INSB][0]=1.;
// Silicon Crystal Density (Kg/m^3)
     RHO[SILICON]=2330.;
// Germanium Crystal Density (Kg/m^3)
     RHO[GERMANIUM]=5320.;
// III-V semiconductor compounds Crystal Density (Kg/m^3)
     RHO[GAAS]=5360.;
     RHO[INSB]=5790.;
// Silicon acoustic deformation potential (Joule)
     DA[SILICON]=9.*1.60217733e-19;
// Germanium acoustic deformation potential (Joule)
     DA[GERMANIUM]=9.*1.60217733e-19;
// III-V semiconductor compounds acoustic deformation potential (Joule)
     DA[GAAS]=7.*1.60217733e-19;
     DA[INSB]=5.96*1.60217733e-19;
// Silicon longitudinal sound velocity (m/sec)
     UL[SILICON]=9040.;
// Germanium longitudinal sound velocity (m/sec)
     UL[GERMANIUM]=5400.;
// III-V semiconductor compounds longitudinal sound velocity (m/sec)
     UL[GAAS]=5240.;
     UL[INSB]=4060.;
// III-V Semiconductor materials energy gap
     EG[SILICON]=1.21-3.333e-4*TL;
     printf("EG[SILICON]      = %g\n",EG[SILICON]);
     EG[GERMANIUM]=0.747-3.587e-4*TL;
     printf("EG[GERMANIUM]    = %g\n",EG[GERMANIUM]);
     EG[GAAS]=1.54-4.036e-4*TL;//0.32;???
     printf("EG[GAAS]         = %g\n",EG[GAAS]);
     EG[INSB]=0.2446-2.153e-4*TL;//0.174;???
     printf("EG[INSB]         = %g\n",EG[INSB]);
     printf("\n");
// Silicon minimum energy
     EMIN[SILICON][1]=0.0;
// Germanium minimum energy
     EMIN[GERMANIUM][1]=0.173;
// III-V semiconductor compounds minimum energy of GAMMA-valley
     EMIN[GAAS][1]=0.0;
     EMIN[INSB][1]=0.0;
// III-V semiconductor compounds minimum energy of L-valley (eV)
     EMIN[GAAS][2]=0.323;
     EMIN[INSB][2]=1.038;
// Definition of effective mass for all materials in all valleys
     MSTAR[SILICON][1]=0.32;
     MSTAR[GAAS][1]=0.067; // Gamma-valley
     MSTAR[GAAS][2]=0.350; // L-valley
     MSTAR[GERMANIUM][1]=0.22;
     MSTAR[INSB][1]=0.013;
     MSTAR[INSB][2]=0.442;
// non-parabolicity coefficient for Silicon
     alphaK[SILICON][1]=0.5;
// non-parabolicity coefficient for Germanium
     alphaK[GERMANIUM][1]=0.65;
// non-parabolicity coefficient for GaAs in the GAMMA-valley
     alphaK[GAAS][1]=pow(1.-MSTAR[GAAS][1],2.)/EG[GAAS];//0.611;
     printf("alphaK_gamma[GaAs] = %g\n",alphaK[GAAS][1]);
// non-parabolicity coefficient for GaAs in the L-valley
     alphaK[GAAS][2]=pow(1.-MSTAR[GAAS][2],2.)/EG[GAAS];//0.242;
     printf("alphaK_L[GaAs]     = %g\n",alphaK[GAAS][2]);
// non-parabolicity coefficient for InSb in the GAMMA-valley
     alphaK[INSB][1]=pow(1.-MSTAR[INSB][1],2.)/EG[INSB];//5.59;
     printf("alphaK_gamma[InSb] = %g\n",alphaK[INSB][1]);
// non-parabolicity coefficient for InSb in the L-valley
     alphaK[INSB][2]=pow(1.-MSTAR[INSB][2],2.)/EG[INSB];//1.789;
     printf("alphaK_L[InSb]     = %g\n",alphaK[INSB][2]);
     printf("\n");
// ======================================================
// ======================================================

// We reset the some arrays
// ========================
     memset(&u2d,0,sizeof(u2d));
     memset(&E,0,sizeof(E));
     memset(&EDGE,0,sizeof(EDGE));
     memset(&SIO2,0,sizeof(SIO2));

// Read the geometrical and physical description of the MESFET
// ===========================================================
     Read_Input_File();

// Closure of the input file
// =========================
     fclose(fp);

// Read all the coefficients for MEP simulation
// ============================================
     MEP_coefficients();

// Loading of the initial runtime
// ==============================
     binarytime=time(NULL);
     nowtm=localtime(&binarytime);

     printf("\n\nComputation Started at %s\n",asctime(nowtm));

// Boundary conditions for the model simulated
// ===========================================
     PoissonBCs();
     if(FARADAYFLAG) FaradayBCs();

// Initialisation for Monte Carlo
// ==============================
     if(Model_Number==MCE || Model_Number==MCEH){
      int i;
      for(i=0;i<NOAMTIA;i++) MCparameters(i);
      MCdevice_config();
     }
// HERE IS THE SIMULATION
// ======================
     for(c=1;c<=ITMAX;c++) updating(Model_Number);
// Here we save the outputs
// ========================
     SaveOutputFiles(File_Format,0);
     printf("Final Output has been saved\n");
    }
  else{
// No filename has been specified
// ==============================
   printf("%s: no input file\n",progname);
   exit(0);
  }
    /* Print greeting message and exit. */
   binarytime=time(NULL);
   nowtm=localtime(&binarytime);
   printf("Computation Finished at %s\n",asctime(nowtm));
   return(EXIT_SUCCESS); // Successfull exit
}

/* ------------------- Inclusione funzioni --------------------- */

// ****************************************************

// =======================================================
// Modif. 06 ago.2004 - 18 Aug.2007, Siracusa, J.M.Sellier
// =======================================================
#include "poissonbcs.h"
#include "faradaybcs.h"
#include "media.h"
#include "saveoutput2dmeshformat.h"
#include "saveoutput2dgnuplot.h"
#include "saveoutput2dholegnuplot.h"
#include "saveoutput2dholemeshformat.h"
#include "saveoutputfiles.h"
#include "electric_field.h"
#include "faraday.h"
#include "quantumeffectivepotential.h"
#include "random.h"
#include "mcparameters.h"
#include "deviceconfig.h"
#include "particlecreation.h"
#include "ensemblemontecarlo.h"
#include "drift.h"
#include "scattering.h"
#include "charge.h"
#include "updating.h"
#include "readinputfile.h"
#include "computecurrents.h"
#include "electron_relaxation.h"
#include "ParabMEP2D.h"
#include "HMEPbcs.h"
#include "MEP_interpolation.h"
#include "holemep2d.h"
#include "hole_relaxation.h"
#include "Hole_bcs.h"
// =========================================

/* archimedes.c ends here */

// ***********************************************************************
// ************************************************************************
