/* systems.c, systems-specific routines                                 */

/* Copyright (C) 1996, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>

#include "mpeg-config.h"
#include "global.h"

/* initialize buffer, call once before first getbits or showbits */

/* parse system layer, ignore everything we don't need */
void _mpeg2Next_Packet(struct mpeg2obj *m)
{
  unsigned int code;
  int l;

  for(;;)
  {
    code = _mpeg2Get_Long(m);

    /* remove system layer byte stuffing */
    while ((code & 0xffffff00) != 0x100)
      code = (code<<8) | _mpeg2Get_Byte(m);

    switch(code)
    {
    case PACK_START_CODE: /* pack header */
      /* skip pack header (system_clock_reference and mux_rate) */
      m->ld->Rdptr += 8;
      break;
    case VIDEO_ELEMENTARY_STREAM:   
      code = _mpeg2Get_Word(m);             /* packet_length */
      m->ld->Rdmax = m->ld->Rdptr + code;

      code = _mpeg2Get_Byte(m);

      if((code>>6)==0x02)
      {
        m->ld->Rdptr++;
        code=_mpeg2Get_Byte(m);  /* parse PES_header_data_length */
        m->ld->Rdptr+=code;    /* advance pointer by PES_header_data_length */
        printf("MPEG-2 PES packet\n");
        return;
      }
      else if(code==0xff)
      {
        /* parse MPEG-1 packet header */
        while((code=_mpeg2Get_Byte(m))== 0xFF);
      }
       
      /* stuffing bytes */
      if(code>=0x40)
      {
        if(code>=0x80)
        {
          fprintf(stderr,"Error in packet header\n");
          exit(1);
        }
        /* skip STD_buffer_scale */
        m->ld->Rdptr++;
        code = _mpeg2Get_Byte(m);
      }

      if(code>=0x30)
      {
        if(code>=0x40)
        {
          fprintf(stderr,"Error in packet header\n");
          exit(1);
        }
        /* skip presentation and decoding time stamps */
        m->ld->Rdptr += 9;
      }
      else if(code>=0x20)
      {
        /* skip presentation time stamps */
        m->ld->Rdptr += 4;
      }
      else if(code!=0x0f)
      {
        fprintf(stderr,"Error in packet header\n");
        exit(1);
      }
      return;
    case ISO_END_CODE: /* end */
      /* simulate a buffer full of sequence end codes */
      l = 0;
      while (l<2048)
      {
        m->ld->Rdbfr[l++] = SEQUENCE_END_CODE>>24;
        m->ld->Rdbfr[l++] = SEQUENCE_END_CODE>>16;
        m->ld->Rdbfr[l++] = SEQUENCE_END_CODE>>8;
        m->ld->Rdbfr[l++] = SEQUENCE_END_CODE&0xff;
      }
      m->ld->Rdptr = m->ld->Rdbfr;
      m->ld->Rdmax = m->ld->Rdbfr + 2048;
      return;
    default:
      if(code>=SYSTEM_START_CODE)
      {
        /* skip system headers and non-video packets*/
        code = _mpeg2Get_Word(m);
        m->ld->Rdptr += code;
      }
      else
      {
        fprintf(stderr,"Unexpected startcode %08x in system layer\n",code);
        exit(1);
      }
      break;
    }
  }
}



void _mpeg2Flush_Buffer32(struct mpeg2obj *m)
{
  int Incnt;

  m->ld->Bfr = 0;

  Incnt = m->ld->Incnt;
  Incnt -= 32;

  if (m->System_Stream_Flag && (m->ld->Rdptr >= m->ld->Rdmax-4))
  {
    while (Incnt <= 24)
    {
      if (m->ld->Rdptr >= m->ld->Rdmax)
        _mpeg2Next_Packet(m);
      m->ld->Bfr |= _mpeg2Get_Byte(m) << (24 - Incnt);
      Incnt += 8;
    }
  }
  else
  {
    while (Incnt <= 24)
    {
      if (m->ld->Rdptr >= m->ld->Rdbfr+2048)
        _mpeg2Fill_Buffer(m);
      m->ld->Bfr |= *m->ld->Rdptr++ << (24 - Incnt);
      Incnt += 8;
    }
  }
  m->ld->Incnt = Incnt;

#ifdef VERIFY 
  m->ld->Bitcnt += 32;
#endif /* VERIFY */
}


unsigned int _mpeg2Get_Bits32(struct mpeg2obj *m)
{
  unsigned int l;

  l = _mpeg2Show_Bits(m, 32);
  _mpeg2Flush_Buffer32(m);

  return l;
}


int _mpeg2Get_Long(struct mpeg2obj *m)
{
  int i;

  i = _mpeg2Get_Word(m);
  return (i<<16) | _mpeg2Get_Word(m);
}


