#include <iostream>

#include <seqan/arg_parse.h>

struct ModifyStringOptions
{
    unsigned period;
    unsigned rangeBegin, rangeEnd;
    bool toUppercase;
    bool toLowercase;
    seqan2::CharString text;

    ModifyStringOptions() :
        period(1), rangeBegin(0), rangeEnd(0),toUppercase(false),
        toLowercase(false)
    {}
};

seqan2::ArgumentParser::ParseResult
parseCommandLine(ModifyStringOptions & options, int argc, char const ** argv)
{
	// Setup ArgumentParser.
	seqan2::ArgumentParser parser("modify_string");

	// We require one argument.
	addArgument(parser, seqan2::ArgParseArgument(
	    seqan2::ArgParseArgument::STRING, "TEXT"));

	// Define Options
	addOption(parser, seqan2::ArgParseOption(
	    "i", "period", "Period to use for the index.",
	    seqan2::ArgParseArgument::INTEGER, "INT"));
	setMinValue(parser, "period", "1");
	setDefaultValue(parser, "period", "1");
	addOption(parser, seqan2::ArgParseOption(
	    "r", "range", "Range of the text to modify.",
	    seqan2::ArgParseArgument::INTEGER, "INT", false, 2));
	addOption(parser, seqan2::ArgParseOption(
	    "U", "uppercase", "Select to-uppercase as operation."));
	addOption(parser, seqan2::ArgParseOption(
	    "L", "lowercase", "Select to-lowercase as operation."));

	// Parse command line.
	seqan2::ArgumentParser::ParseResult res = seqan2::parse(parser, argc, argv);

	// Only extract  options if the program will continue after parseCommandLine()
	if (res != seqan2::ArgumentParser::PARSE_OK)
	    return res;

	// Extract option values.
	getOptionValue(options.period, parser, "period");
	getOptionValue(options.rangeBegin, parser, "range", 0);
	getOptionValue(options.rangeEnd, parser, "range", 1);
	options.toUppercase = isSet(parser, "uppercase");
	options.toLowercase = isSet(parser, "lowercase");
	seqan2::getArgumentValue(options.text, parser, 0);

	// If both to-uppercase and to-lowercase were selected then this is an error.
	if (options.toUppercase && options.toLowercase)
	{
	    std::cerr << "ERROR: You cannot specify both to-uppercase and to-lowercase!\n";
	    return seqan2::ArgumentParser::PARSE_ERROR;
	}

	return seqan2::ArgumentParser::PARSE_OK;
}

seqan2::CharString modifyString(seqan2::CharString const & text,
				   ModifyStringOptions const & options)
{
	seqan2::CharString result;

	if (options.toLowercase)
	{
	    for (unsigned i = 0; i < length(text); ++i)
	    {
		if (i >= options.rangeBegin && i < options.rangeEnd &&
		    (i % options.period == 0u))
		    appendValue(result, tolower(text[i]));
		else
		    appendValue(result, text[i]);
	    }
	}
	else
	{
	    for (unsigned i = 0; i < length(text); ++i)
	    {
		if (i >= options.rangeBegin && i < options.rangeEnd &&
		    (i % options.period == 0u))
		    appendValue(result, toupper(text[i]));
		else
		    appendValue(result, text[i]);
	    }
	}

	return result;
}

int main(int argc, char const ** argv)
{
	// Parse the command line.
	ModifyStringOptions options;
	seqan2::ArgumentParser::ParseResult res = parseCommandLine(options, argc, argv);

	// If parsing was not successful then exit with code 1 if there were errors.
	// Otherwise, exit with code 0 (e.g. help was printed).
	if (res != seqan2::ArgumentParser::PARSE_OK)
	    return res == seqan2::ArgumentParser::PARSE_ERROR;

	std::cout << modifyString(options.text, options) << '\n';

	return 0;
}

