# vgp_openssh_ext samba group policy
# Copyright (C) David Mulder <dmulder@suse.com> 2020
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
from samba.gp.gpclass import gp_xml_ext
from base64 import b64encode
from tempfile import NamedTemporaryFile
from samba.common import get_bytes, get_string

intro = b'''
### autogenerated by samba
#
# This file is generated by the vgp_openssh_ext Group Policy
# Client Side Extension. To modify the contents of this file,
# modify the appropriate Group Policy objects which apply
# to this machine. DO NOT MODIFY THIS FILE DIRECTLY.
#

'''

class vgp_openssh_ext(gp_xml_ext):
    def __str__(self):
        return 'VGP/Unix Settings/OpenSSH'

    def process_group_policy(self, deleted_gpo_list, changed_gpo_list,
            cfg_dir='/etc/ssh/sshd_config.d'):
        for guid, settings in deleted_gpo_list:
            self.gp_db.set_guid(guid)
            if str(self) in settings:
                for attribute, sshd_config in settings[str(self)].items():
                    if os.path.exists(sshd_config):
                        os.unlink(sshd_config)
                    self.gp_db.delete(str(self), attribute)
            self.gp_db.commit()

        for gpo in changed_gpo_list:
            if gpo.file_sys_path:
                self.gp_db.set_guid(gpo.name)
                xml = 'MACHINE/VGP/VTLA/SshCfg/SshD/manifest.xml'
                path = os.path.join(gpo.file_sys_path, xml)
                xml_conf = self.parse(path)
                if not xml_conf:
                    continue
                policy = xml_conf.find('policysetting')
                data = policy.find('data')
                configfile = data.find('configfile')
                for configsection in configfile.findall('configsection'):
                    if configsection.find('sectionname').text:
                        continue
                    settings = {}
                    for kv in configsection.findall('keyvaluepair'):
                        settings[kv.find('key')] = kv.find('value')
                    attribute = get_string(b64encode(get_bytes(gpo.name) +
                        get_bytes(cfg_dir)))
                    fname = self.gp_db.retrieve(str(self), attribute)
                    if not os.path.isdir(cfg_dir):
                        os.mkdir(cfg_dir, 0o640)
                    if fname and os.path.exists(fname):
                        f = open(fname, 'wb')
                    else:
                        f = NamedTemporaryFile(prefix='gp_',
                                               delete=False,
                                               dir=cfg_dir)
                    f.write(intro)
                    for k, v in settings.items():
                        f.write(b'%s %s\n' % \
                            (get_bytes(k.text), get_bytes(v.text)))
                    os.chmod(f.name, 0o640)
                    self.gp_db.store(str(self), attribute, f.name)
                    self.gp_db.commit()
                    f.close()

    def rsop(self, gpo):
        output = {}
        if gpo.file_sys_path:
            xml = 'MACHINE/VGP/VTLA/SshCfg/SshD/manifest.xml'
            path = os.path.join(gpo.file_sys_path, xml)
            xml_conf = self.parse(path)
            if not xml_conf:
                return output
            policy = xml_conf.find('policysetting')
            data = policy.find('data')
            configfile = data.find('configfile')
            for configsection in configfile.findall('configsection'):
                if configsection.find('sectionname').text:
                    continue
                for kv in configsection.findall('keyvaluepair'):
                    if str(self) not in output.keys():
                        output[str(self)] = {}
                    output[str(self)][kv.find('key').text] = \
                        kv.find('value').text
        return output
