% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/deriv.R
\name{deriv}
\alias{deriv}
\alias{deriv.bSpline2}
\alias{deriv.dbs}
\alias{deriv.ibs}
\alias{deriv.mSpline}
\alias{deriv.iSpline}
\alias{deriv.cSpline}
\alias{deriv.bernsteinPoly}
\alias{deriv.naturalSpline}
\title{Derivatives of Spline Bases}
\usage{
\method{deriv}{bSpline2}(expr, derivs = 1L, ...)

\method{deriv}{dbs}(expr, derivs = 1L, ...)

\method{deriv}{ibs}(expr, derivs = 1L, ...)

\method{deriv}{mSpline}(expr, derivs = 1L, ...)

\method{deriv}{iSpline}(expr, derivs = 1L, ...)

\method{deriv}{cSpline}(expr, derivs = 1L, ...)

\method{deriv}{bernsteinPoly}(expr, derivs = 1L, ...)

\method{deriv}{naturalSpline}(expr, derivs = 1L, ...)
}
\arguments{
\item{expr}{Objects of class \code{bSpline2}, \code{ibs}, \code{mSpline},
\code{iSpline}, \code{cSpline}, \code{bernsteinPoly} or
\code{naturalSpline} with attributes describing \code{knots},
\code{degree}, etc.}

\item{derivs}{A positive integer specifying the order of derivatives. By
default, it is \code{1L} for the first derivatives.}

\item{...}{Optional argument that are not used.}
}
\value{
A numeric matrix of the same dimension with the input \code{expr}.
}
\description{
Returns derivatives of given order for the given spline bases.
}
\details{
At knots, the derivative is defined to be the right derivative except at the
right boundary knot. By default, the function returns the first derivatives.
For derivatives of order greater than one, nested function calls such as
\code{deriv(deriv(expr))} are supported but not recommended.  For a better
performance, argument \code{derivs} should be specified instead.

This function is designed for objects produced by this package.  It
internally extracts necessary specification about the spline/polynomial
basis matrix from its attributes. Therefore, the function will not work if
the key attributes are not available after some operations.
}
\examples{
library(splines2)
x <- c(seq.int(0, 1, 0.1), NA)  # NA's will be kept.
knots <- c(0.3, 0.5, 0.6)

## integal of B-splines and the corresponding B-splines integrated
ibsMat <- ibs(x, knots = knots)
bsMat <- bSpline(x, knots = knots)

## the first derivative
d1Mat <- deriv(ibsMat)
stopifnot(all.equal(bsMat, d1Mat, check.attributes = FALSE))

## the second derivative
d2Mat1 <- deriv(bsMat)
d2Mat2 <- deriv(ibsMat, derivs = 2L)
## nested calls are supported but not recommended
d2Mat3 <- deriv(deriv(ibsMat))
stopifnot(all.equal(d2Mat1, d2Mat2, check.attributes = FALSE))
stopifnot(all.equal(d2Mat2, d2Mat3, check.attributes = FALSE))

## C-splines, I-splines, M-splines and the derivatives
csMat <- cSpline(x, knots = knots, intercept = TRUE, scale = FALSE)
isMat <- iSpline(x, knots = knots, intercept = TRUE)
stopifnot(all.equal(isMat, deriv(csMat), check.attributes = FALSE))

msMat <- mSpline(x, knots = knots, intercept = TRUE)
stopifnot(all.equal(msMat, deriv(isMat), check.attributes = FALSE))
stopifnot(all.equal(msMat, deriv(csMat, 2), check.attributes = FALSE))
stopifnot(all.equal(msMat, deriv(deriv(csMat)), check.attributes = FALSE))

dmsMat <- mSpline(x, knots = knots, intercept = TRUE, derivs = 1)
stopifnot(all.equal(dmsMat, deriv(msMat), check.attributes = FALSE))
stopifnot(all.equal(dmsMat, deriv(isMat, 2), check.attributes = FALSE))
stopifnot(all.equal(dmsMat, deriv(deriv(isMat)), check.attributes = FALSE))
stopifnot(all.equal(dmsMat, deriv(csMat, 3), check.attributes = FALSE))
stopifnot(all.equal(dmsMat, deriv(deriv(deriv(csMat))), check.attributes = FALSE))
}
