#  holidays
#  --------
#  A fast, efficient Python library for generating country, province and state
#  specific sets of holidays on the fly. It aims to make determining whether a
#  specific date is a holiday as fast and flexible as possible.
#
#  Authors: Vacanza Team and individual contributors (see CONTRIBUTORS file)
#           dr-prodigy <dr.prodigy.github@gmail.com> (c) 2017-2023
#           ryanss <ryanssdev@icloud.com> (c) 2014-2017
#  Website: https://github.com/vacanza/holidays
#  License: MIT (see LICENSE file)

from unittest import TestCase

from holidays.countries.kyrgyzstan import Kyrgyzstan
from tests.common import CommonCountryTests


class TestKyrgyzstan(CommonCountryTests, TestCase):
    @classmethod
    def setUpClass(cls):
        super().setUpClass(Kyrgyzstan)

    def test_new_years_day(self):
        self.assertHolidayName("New Year's Day", (f"{year}-01-01" for year in self.full_range))

    def test_christmas_day(self):
        self.assertHolidayName("Christmas Day", (f"{year}-01-07" for year in self.full_range))

    def test_international_womens_day(self):
        self.assertHolidayName(
            "International Women's Day", (f"{year}-03-08" for year in self.full_range)
        )

    def test_nooruz_mairamy(self):
        self.assertHolidayName("Nooruz Mairamy", (f"{year}-03-21" for year in self.full_range))

    def test_day_of_the_peoples_april_revolution(self):
        name = "Day of the People's April Revolution"
        self.assertHolidayName(name, (f"{year}-04-07" for year in range(2016, self.end_year)))
        self.assertNoHolidayName(name, range(self.start_year, 2016))

    def test_international_workers_day(self):
        self.assertHolidayName(
            "International Workers' Day", (f"{year}-05-01" for year in self.full_range)
        )

    def test_constitution_day(self):
        self.assertHolidayName("Constitution Day", (f"{year}-05-05" for year in self.full_range))

    def test_victory_day(self):
        self.assertHolidayName("Victory Day", (f"{year}-05-09" for year in self.full_range))

    def test_independence_day(self):
        self.assertHolidayName("Independence Day", (f"{year}-08-31" for year in self.full_range))

    def test_days_of_history_and_commemoration_of_ancestors(self):
        self.assertHolidayName(
            "Days of History and Commemoration of Ancestors",
            (f"{year}-11-07" for year in self.full_range),
            (f"{year}-11-08" for year in self.full_range),
        )

    def test_new_years_eve(self):
        self.assertHolidayName("New Year's Eve", (f"{year}-12-31" for year in self.full_range))

    def test_orozo_ait(self):
        name = "Orozo Ait"
        self.assertIslamicNoEstimatedHolidayName(
            name,
            "2020-05-24",
            "2020-05-25",
            "2021-05-13",
            "2021-05-14",
            "2022-05-02",
            "2022-05-03",
            "2023-04-21",
            "2023-04-22",
            "2024-04-10",
            "2024-04-11",
            "2025-03-30",
            "2025-03-31",
        )
        self.assertIslamicNoEstimatedHolidayName(name, self.full_range)

    def test_kurman_ait(self):
        name = "Kurman Ait"
        self.assertIslamicNoEstimatedHolidayName(
            name,
            "2020-07-31",
            "2021-07-20",
            "2022-07-09",
            "2023-06-28",
            "2024-06-16",
            "2025-06-06",
        )
        self.assertIslamicNoEstimatedHolidayName(name, self.full_range)

    def test_2022(self):
        self.assertHolidaysInYear(
            2022,
            ("2022-01-01", "New Year's Day"),
            ("2022-01-07", "Christmas Day"),
            ("2022-02-23", "Fatherland Defender's Day"),
            ("2022-03-08", "International Women's Day"),
            ("2022-03-21", "Nooruz Mairamy"),
            ("2022-04-07", "Day of the People's April Revolution"),
            ("2022-05-01", "International Workers' Day"),
            ("2022-05-02", "Orozo Ait (estimated)"),
            ("2022-05-03", "Orozo Ait (estimated)"),
            ("2022-05-05", "Constitution Day"),
            ("2022-05-09", "Victory Day"),
            ("2022-07-09", "Kurman Ait (estimated)"),
            ("2022-08-31", "Independence Day"),
            ("2022-11-07", "Days of History and Commemoration of Ancestors"),
            ("2022-11-08", "Days of History and Commemoration of Ancestors"),
            ("2022-12-31", "New Year's Eve"),
        )

    def test_2023(self):
        self.assertHolidaysInYear(
            2023,
            ("2023-01-01", "New Year's Day"),
            ("2023-01-07", "Christmas Day"),
            ("2023-02-23", "Fatherland Defender's Day"),
            ("2023-03-08", "International Women's Day"),
            ("2023-03-21", "Nooruz Mairamy"),
            ("2023-04-07", "Day of the People's April Revolution"),
            ("2023-04-21", "Orozo Ait (estimated)"),
            ("2023-04-22", "Orozo Ait (estimated)"),
            ("2023-05-01", "International Workers' Day"),
            ("2023-05-05", "Constitution Day"),
            ("2023-05-09", "Victory Day"),
            ("2023-06-28", "Kurman Ait (estimated)"),
            ("2023-08-31", "Independence Day"),
            ("2023-11-07", "Days of History and Commemoration of Ancestors"),
            ("2023-11-08", "Days of History and Commemoration of Ancestors"),
            ("2023-12-31", "New Year's Eve"),
        )
