/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/**
 * \file cannot-bind-when-active.c
 *
 * The ARB_transform_feedback2 spec says:
 *
 *     "The error INVALID_OPERATION is generated by BindTransformFeedback if
 *     the transform feedback operation is active on the currently bound
 *     transform feedback object, and that operation is not paused (as
 *     described below)."
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 10;
	config.window_visual = PIGLIT_GL_VISUAL_RGB;
	config.khr_no_error_support = PIGLIT_NO_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

enum piglit_result
piglit_display(void)
{
	return PIGLIT_FAIL;
}

static const char vstext[] =
	"varying vec4 x; void main() { gl_Position = vec4(0); x = vec4(0); }";

void piglit_init(int argc, char **argv)
{
	GLuint ids[2];
	GLuint prog;
	GLuint vs;
	GLuint buf;
	bool pass = true;
	static const char *varyings[] = {"x"};

	piglit_require_transform_feedback();
	piglit_require_GLSL();
	piglit_require_extension("GL_ARB_transform_feedback2");

	/* This is all just the boot-strap work for the test.
	 */
	glGenBuffers(1, &buf);
	glBindBuffer(GL_TRANSFORM_FEEDBACK_BUFFER, buf);
	glBufferData(GL_TRANSFORM_FEEDBACK_BUFFER, 1024, NULL, GL_STREAM_READ);

	vs = piglit_compile_shader_text(GL_VERTEX_SHADER, vstext);
	prog = glCreateProgram();
	glAttachShader(prog, vs);

	glTransformFeedbackVaryings(prog, 1, varyings,
				    GL_INTERLEAVED_ATTRIBS);
	glLinkProgram(prog);
	if (!piglit_link_check_status(prog)) {
		pass = false;
		goto done;
	}

	glUseProgram(prog);

	glGenTransformFeedbacks(2, ids);

	glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, ids[0]);
	glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, 0, buf);

	glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, ids[1]);
	glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, 0, buf);

	glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, 0);

	/* Bind the first object and make it active.  Try to bind the other
	 * object.  This should fail.
	 */
	glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, ids[0]);
	glBeginTransformFeedback(GL_TRIANGLES);

	pass = piglit_check_gl_error(0) && pass;

	if (!piglit_khr_no_error) {
		glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, ids[1]);
		pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;
	}

	/* Make the transform feedback object inactive by calling
	 * EndTransformFeedback.  Then try (again) to bind the other object.
	 * This should just work.
	 */
	glEndTransformFeedback();
	glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, ids[1]);
	pass = piglit_check_gl_error(0) && pass;

	/* Make the second object active, and pause it.  Try rebinding the
	 * first object.  This should also just work.
	 */
	glBeginTransformFeedback(GL_TRIANGLES);
	glPauseTransformFeedback();

	glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, ids[0]);
	pass = piglit_check_gl_error(0) && pass;

	/* Rebind the second object, and resume it.  This will make it active.
	 * Re-rebind the first object.  This should fail.
	 */
	glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, ids[1]);
	pass = piglit_check_gl_error(0) && pass;

	glResumeTransformFeedback();
	pass = piglit_check_gl_error(0) && pass;

	if (!piglit_khr_no_error) {
		glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, ids[0]);
		pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;
	}

	/* Make the second object non-active, and restore the default object.
	 * This should work.
	 */
	glEndTransformFeedback();
	glBindTransformFeedback(GL_TRANSFORM_FEEDBACK, 0);
	pass = piglit_check_gl_error(0) && pass;

done:
	glBindBuffer(GL_TRANSFORM_FEEDBACK_BUFFER, 0);
	glDeleteBuffers(1, &buf);

	glDeleteTransformFeedbacks(2, ids);

	glUseProgram(0);
	glDeleteShader(vs);
	glDeleteProgram(prog);

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}
