/* ====================================================================
   Licensed to the Apache Software Foundation (ASF) under one or more
   contributor license agreements.  See the NOTICE file distributed with
   this work for additional information regarding copyright ownership.
   The ASF licenses this file to You under the Apache License, Version 2.0
   (the "License"); you may not use this file except in compliance with
   the License.  You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
==================================================================== */

package org.apache.poi.hssf.record;

import java.io.IOException;

import junit.framework.TestCase;

import org.apache.poi.hssf.usermodel.HSSFName;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.formula.ptg.Area3DPtg;
import org.apache.poi.ss.formula.ptg.ArrayPtg;
import org.apache.poi.ss.formula.ptg.NamePtg;
import org.apache.poi.ss.formula.ptg.Ptg;
import org.apache.poi.ss.formula.ptg.Ref3DPtg;
import org.apache.poi.util.HexRead;

/**
 * Tests the NameRecord serializes/deserializes correctly
 * 
 * @author Danny Mui (dmui at apache dot org)
 */
public final class TestNameRecord extends TestCase {

	/**
	 * Makes sure that additional name information is parsed properly such as menu/description
	 */
	public void testFillExtras() {

		byte[] examples = HexRead.readFromString(""
				+ "88 03 67 06 07 00 00 00 00 00 00 23 00 00 00 4D "
				+ "61 63 72 6F 31 3A 01 00 00 00 11 00 00 4D 61 63 "
				+ "72 6F 20 72 65 63 6F 72 64 65 64 20 32 37 2D 53 "
				+ "65 70 2D 39 33 20 62 79 20 41 4C 4C 57 4F 52");

		NameRecord name = new NameRecord(TestcaseRecordInputStream.create(NameRecord.sid, examples));
		String description = name.getDescriptionText();
		assertNotNull(description);
		assertTrue(description.endsWith("Macro recorded 27-Sep-93 by ALLWOR"));
	}

	public void testReserialize() {
		byte[] data = HexRead
				.readFromString(""
						+ "20 00 00 01 0B 00 00 00 01 00 00 00 00 00 00 06 3B 00 00 00 00 02 00 00 00 09 00]");
		RecordInputStream in = TestcaseRecordInputStream.create(NameRecord.sid, data);
		NameRecord nr = new NameRecord(in);
		assertEquals(0x0020, nr.getOptionFlag());
		byte[] data2 = nr.serialize();
		TestcaseRecordInputStream.confirmRecordEncoding(NameRecord.sid, data, data2);
	}

	public void testFormulaRelAbs_bug46174() throws IOException {
		// perhaps this testcase belongs on TestHSSFName
		HSSFWorkbook wb = new HSSFWorkbook();
		HSSFName name = wb.createName();
		wb.createSheet("Sheet1");
		name.setNameName("test");
		name.setRefersToFormula("Sheet1!$B$3");
		if (name.getRefersToFormula().equals("Sheet1!B3")) {
			fail("Identified bug 46174");
		}
		assertEquals("Sheet1!$B$3", name.getRefersToFormula());
	    wb.close();
	}

	public void testFormulaGeneral() throws IOException {
		// perhaps this testcase belongs on TestHSSFName
		HSSFWorkbook wb = new HSSFWorkbook();
		HSSFName name = wb.createName();
		wb.createSheet("Sheet1");
		name.setNameName("test");
		name.setRefersToFormula("Sheet1!A1+Sheet1!A2");
		assertEquals("Sheet1!A1+Sheet1!A2", name.getRefersToFormula());
		name.setRefersToFormula("5*6");
		assertEquals("5*6", name.getRefersToFormula());
		
		wb.close();
	}

    /**
     *  A NameRecord followed by a ContinueRecord. See Bugzilla 50244
     */
    public void test50244(){
        String record_0x0018 =
                "                                          18 00 " +
                "21 1E 00 00 00 11 08 00 00 00 03 00 00 00 00 00 " +
                "00 77 44 61 74 61 47 72 75 70 70 65 53 65 72 69 " +
                "65 2E 60 00 B7 02 0C 00 57 03 00 B7 02 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 30 31 02 09 00 00 31 2E " +
                "54 32 30 2E 30 30 31 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 30 31 02 09 00 00 31 2E 54 32 30 2E 30 30 " +
                "32 02 09 00 00 31 2E 54 32 30 2E 30 30 32 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 30 32 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 30 33 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 30 33 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "30 33 02 09 00 00 31 2E 54 32 30 2E 30 31 33 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 31 33 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 31 33 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 31 34 02 09 00 00 31 2E 54 32 30 2E " +
                "30 31 34 02 09 00 00 31 2E 54 32 30 2E 30 31 34 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 31 35 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 31 35 02 09 00 00 31 2E " +
                "54 32 30 2E 30 31 35 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 32 35 02 09 00 00 31 2E 54 32 30 2E 30 32 " +
                "35 02 09 00 00 31 2E 54 32 30 2E 30 32 35 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 32 36 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 32 36 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 32 36 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "32 37 02 09 00 00 31 2E 54 32 30 2E 30 32 37 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 32 37 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 33 37 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 33 37 02 09 00 00 31 2E 54 32 30 2E " +
                "30 33 37 02 09 00 00 31 2E 54 32 30 2E 30 33 38 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 33 38 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 33 38 02 09 00 00 31 2E " +
                "54 32 30 2E 30 33 39 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 33 39 02 09 00 00 31 2E 54 32 30 2E 30 33 " +
                "39 02 09 00 00 31 2E 54 32 30 2E 30 34 39 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 34 39 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 34 39 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 35 30 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "35 30 02 09 00 00 31 2E 54 32 30 2E 30 35 30 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 35 31 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 35 31 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 35 31 02 09 00 00 31 2E 54 32 30 2E " +
                "30 36 31 02 09 00 00 31 2E 54 32 30 2E 30 36 31 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 36 31 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 36 32 02 09 00 00 31 2E " +
                "54 32 30 2E 30 36 32 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 36 32 02 09 00 00 31 2E 54 32 30 2E 30 36 " +
                "33 02 09 00 00 31 2E 54 32 30 2E 30 36 33 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 36 33 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 37 33 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 37 33 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "37 33 02 09 00 00 31 2E 54 32 30 2E 30 37 34 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 37 34 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 37 34 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 37 35 02 09 00 00 31 2E 54 32 30 2E " +
                "30 37 35 02 09 00 00 31 2E 54 32 30 2E 30 37 35 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 38 35 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 38 35 02 09 00 00 31 2E " +
                "54 32 30 2E 30 38 35 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 38 36 02 09 00 00 31 2E 54 32 30 2E 30 38 " +
                "36 02 09 00 00 31 2E 54 32 30 2E 30 38 36 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 38 37 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 38 37 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 38 37 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "39 37 02 09 00 00 31 2E 54 32 30 2E 30 39 37 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 39 37 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 39 38 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 39 38 02 09 00 00 31 2E 54 32 30 2E " +
                "30 39 38 02 09 00 00 31 2E 54 32 30 2E 30 39 39 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 39 39 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 39 39 02 09 00 00 31 2E " +
                "54 32 30 2E 31 30 39 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 30 39 02 09 00 00 31 2E 54 32 30 2E 31 30 " +
                "39 02 09 00 00 31 2E 54 32 30 2E 31 31 30 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 31 30 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 31 30 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 31 31 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "31 31 02 09 00 00 31 2E 54 32 30 2E 31 31 31 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 32 31 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 32 31 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 32 31 02 09 00 00 31 2E 54 32 30 2E " +
                "31 32 32 02 09 00 00 31 2E 54 32 30 2E 31 32 32 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 32 32 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 32 33 02 09 00 00 31 2E " +
                "54 32 30 2E 31 32 33 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 32 33 02 09 00 00 31 2E 54 32 30 2E 31 33 " +
                "33 02 09 00 00 31 2E 54 32 30 2E 31 33 33 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 33 33 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 33 34 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 33 34 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "33 34 02 09 00 00 31 2E 54 32 30 2E 31 33 35 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 33 35 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 33 35 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 34 35 02 09 00 00 31 2E 54 32 30 2E " +
                "31 34 35 02 09 00 00 31 2E 54 32 30 2E 31 34 35 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 34 36 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 34 36 02 09 00 00 31 2E " +
                "54 32 30 2E 31 34 36 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 34 37 02 09 00 00 31 2E 54 32 30 2E 31 34 " +
                "37 02 09 00 00 31 2E 54 32 30 2E 31 34 37 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 35 37 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 35 37 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 35 37 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "35 38 02 09 00 00 31 2E 54 32 30 2E 31 35 38 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 35 38 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 35 39 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 35 39 02 09 00 00 31 2E 54 32 30 2E " +
                "31 35 39 02 09 00 00 31 2E 54 32 30 2E 31 36 39 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 36 39 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 36 39 02 09 00 00 31 2E " +
                "54 32 30 2E 31 37 30 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 37 30 02 09 00 00 31 2E 54 32 30 2E 31 37 " +
                "30 02 09 00 00 31 2E 54 32 30 2E 31 37 31 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 37 31 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 37 31 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 30 39 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "30 39 02 09 00 00 31 2E 54 36 30 2E 30 30 39 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 31 30 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 31 30 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 31 30 02 09 00 00 31 2E 54 36 30 2E " +
                "30 31 31 02 09 00 00 31 2E 54 36 30 2E 30 31 31 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 31 31 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 32 31 02 09 00 00 31 2E " +
                "54 36 30 2E 30 32 31 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 32 31 02 09 00 00 31 2E 54 36 30 2E 30 32 " +
                "32 02 09 00 00 31 2E 54 36 30 2E 30 32 32 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 32 32 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 32 33 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 32 33 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "32 33 02 09 00 00 31 2E 54 36 30 2E 30 33 33 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 33 33 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 33 33 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 33 34 02 09 00 00 31 2E 54 36 30 2E " +
                "30 33 34 02 09 00 00 31 2E 54 36 30 2E 30 33 34 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 33 35 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 33 35 02 09 00 00 31 2E " +
                "54 36 30 2E 30 33 35 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 34 35 02 09 00 00 31 2E 54 36 30 2E 30 34 " +
                "35 02 09 00 00 31 2E 54 36 30 2E 30 34 35 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 34 36 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 34 36 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 34 36 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "34 37 02 09 00 00 31 2E 54 36 30 2E 30 34 37 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 34 37 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 35 37 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 35 37 02 09 00 00 31 2E 54 36 30 2E " +
                "30 35 37 02 09 00 00 31 2E 54 36 30 2E 30 35 38 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 35 38 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 35 38 02 09 00 00 31 2E " +
                "54 36 30 2E 30 35 39 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 35 39 02 09 00 00 31 2E 54 36 30 2E 30 35 " +
                "39 02 09 00 00 31 2E 54 36 30 2E 30 36 30 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 36 30 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 36 30 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 35 30 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "35 30 02 09 00 00 31 2E 54 36 30 2E 30 35 30 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 34 39 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 34 39 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 34 39 02 09 00 00 31 2E 54 36 30 2E " +
                "30 34 38 02 09 00 00 31 2E 54 36 30 2E 30 34 38 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 34 38 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 33 38 02 09 00 00 31 2E " +
                "54 36 30 2E 30 33 38 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 33 38 02 09 00 00 31 2E 54 36 30 2E 30 33 " +
                "37 02 09 00 00 31 2E 54 36 30 2E 30 33 37 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 33 37 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 33 36 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 33 36 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "33 36 02 09 00 00 31 2E 54 36 30 2E 30 32 36 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 32 36 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 32 36 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 32 35 02 09 00 00 31 2E 54 36 30 2E " +
                "30 32 35 02 09 00 00 31 2E 54 36 30 2E 30 32 35 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 32 34 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 32 34 02 09 00 00 31 2E " +
                "54 36 30 2E 30 32 34 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 31 34 02 09 00 00 31 2E 54 36 30 2E 30 31 " +
                "34 02 09 00 00 31 2E 54 36 30 2E 30 31 34 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 31 33 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 31 33 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 31 33 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "31 32 02 09 00 00 31 2E 54 36 30 2E 30 31 32 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 31 32 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 30 32 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 30 32 02 09 00 00 31 2E 54 36 30 2E " +
                "30 30 32 02 09 00 00 31 2E 54 36 30 2E 30 30 31 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 30 31 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 30 31 02 09 00 00 31 2E " +
                "54 32 30 2E 31 37 32 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 37 32 02 09 00 00 31 2E 54 32 30 2E 31 37 " +
                "32 02 09 00 00 31 2E 54 32 30 2E 31 36 32 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 36 32 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 36 32 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 36 31 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "36 31 02 09 00 00 31 2E 54 32 30 2E 31 36 31 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 36 30 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 36 30 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 36 30 02 09 00 00 31 2E 54 32 30 2E " +
                "31 35 30 02 09 00 00 31 2E 54 32 30 2E 31 35 30 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 35 30 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 34 39 02 09 00 00 31 2E " +
                "54 32 30 2E 31 34 39 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 34 39 02 09 00 00 31 2E 54 32 30 2E 31 34 " +
                "38 02 09 00 00 31 2E 54 32 30 2E 31 34 38 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 34 38 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 33 38 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 33 38 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "33 38 02 09 00 00 31 2E 54 32 30 2E 31 33 37 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 33 37 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 33 37 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 33 36 02 09 00 00 31 2E 54 32 30 2E " +
                "31 33 36 02 09 00 00 31 2E 54 32 30 2E 31 33 36 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 32 36 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 32 36 02 09 00 00 31 2E " +
                "54 32 30 2E 31 32 36 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 32 35 02 09 00 00 31 2E 54 32 30 2E 31 32 " +
                "35 02 09 00 00 31 2E 54 32 30 2E 31 32 35 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 32 34 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 32 34 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 32 34 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "31 34 02 09 00 00 31 2E 54 32 30 2E 31 31 34 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 31 34 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 31 33 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 31 33 02 09 00 00 31 2E 54 32 30 2E " +
                "31 31 33 02 09 00 00 31 2E 54 32 30 2E 31 31 32 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 31 32 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 31 32 02 09 00 00 31 2E " +
                "54 32 30 2E 31 30 32 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 30 32 02 09 00 00 31 2E 54 32 30 2E 31 30 " +
                "32 02 09 00 00 31 2E 54 32 30 2E 31 30 31 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 30 31 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 30 31 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 30 30 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "30 30 02 09 00 00 31 2E 54 32 30 2E 31 30 30 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 39 30 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 39 30 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 39 30 02 09 00 00 31 2E 54 32 30 2E " +
                "30 38 39 02 09 00 00 31 2E 54 32 30 2E 30 38 39 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 38 39 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 38 38 02 09 00 00 31 2E " +
                "54 32 30 2E 30 38 38 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 38 38 02 09 00 00 31 2E 54 32 30 2E 30 37 " +
                "38 02 09 00 00 31 2E 54 32 30 2E 30 37 38 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 37 38 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 37 37 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 37 37 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "37 37 02 09 00 00 31 2E 54 32 30 2E 30 37 36 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 37 36 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 37 36 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 36 36 02 09 00 00 31 2E 54 32 30 2E " +
                "30 36 36 02 09 00 00 31 2E 54 32 30 2E 30 36 36 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 36 35 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 36 35 02 09 00 00 31 2E " +
                "54 32 30 2E 30 36 35 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 36 34 02 09 00 00 31 2E 54 32 30 2E 30 36 " +
                "34 02 09 00 00 31 2E 54 32 30 2E 30 36 34 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 35 34 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 35 34 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 35 34 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "35 33 02 09 00 00 31 2E 54 32 30 2E 30 35 33 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 35 33 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 35 32 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 35 32 02 09 00 00 31 2E 54 32 30 2E " +
                "30 35 32 02 09 00 00 31 2E 54 32 30 2E 30 34 32 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 34 32 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 34 32 02 09 00 00 31 2E " +
                "54 32 30 2E 30 34 31 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 34 31 02 09 00 00 31 2E 54 32 30 2E 30 34 " +
                "31 02 09 00 00 31 2E 54 32 30 2E 30 34 30 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 34 30 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 34 30 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 33 30 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "33 30 02 09 00 00 31 2E 54 32 30 2E 30 33 30 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 32 39 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 32 39 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 32 39 02 09 00 00 31 2E 54 32 30 2E " +
                "30 32 38 02 09 00 00 31 2E 54 32 30 2E 30 32 38 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 32 38 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 31 38 02 09 00 00 31 2E " +
                "54 32 30 2E 30 31 38 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 31 38 02 09 00 00 31 2E 54 32 30 2E 30 31 " +
                "37 02 09 00 00 31 2E 54 32 30 2E 30 31 37 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 31 37 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 31 36 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 31 36 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "31 36 02 09 00 00 31 2E 54 32 30 2E 30 30 36 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 30 36 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 30 36 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 30 35 02 09 00 00 31 2E 54 32 30 2E " +
                "30 30 35 02 09 00 00 31 2E 54 32 30 2E 30 30 35 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 30 34 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 30 34 02 09 00 00 31 2E " +
                "54 32 30 2E 30 30 34 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 30 37 02 09 00 00 31 2E 54 32 30 2E 30 30 " +
                "37 02 09 00 00 31 2E 54 32 30 2E 30 30 37 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 30 38 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 30 38 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 30 38 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "30 39 02 09 00 00 31 2E 54 32 30 2E 30 30 39 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 30 39 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 31 39 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 31 39 02 09 00 00 31 2E 54 32 30 2E " +
                "30 31 39 02 09 00 00 31 2E 54 32 30 2E 30 32 30 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 32 30 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 32 30 02 09 00 00 31 2E " +
                "54 32 30 2E 30 32 31 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 32 31 02 09 00 00 31 2E 54 32 30 2E 30 32 " +
                "31 02 09 00 00 31 2E 54 32 30 2E 30 33 31 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 33 31 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 33 31 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 33 32 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "33 32 02 09 00 00 31 2E 54 32 30 2E 30 33 32 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 33 33 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 33 33 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 33 33 02 09 00 00 31 2E 54 32 30 2E " +
                "30 34 33 02 09 00 00 31 2E 54 32 30 2E 30 34 33 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 34 33 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 34 34 02 09 00 00 31 2E " +
                "54 32 30 2E 30 34 34 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 34 34 02 09 00 00 31 2E 54 32 30 2E 30 34 " +
                "35 02 09 00 00 31 2E 54 32 30 2E 30 34 35 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 34 35 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 35 35 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 35 35 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "35 35 02 09 00 00 31 2E 54 32 30 2E 30 35 36 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 35 36 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 35 36 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 35 37 02 09 00 00 31 2E 54 32 30 2E " +
                "30 35 37 02 09 00 00 31 2E 54 32 30 2E 30 35 37 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 36 37 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 36 37 02 09 00 00 31 2E " +
                "54 32 30 2E 30 36 37 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 36 38 02 09 00 00 31 2E 54 32 30 2E 30 36 " +
                "38 02 09 00 00 31 2E 54 32 30 2E 30 36 38 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 36 39 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 36 39 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 36 39 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "37 39 02 09 00 00 31 2E 54 32 30 2E 30 37 39 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 37 39 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 38 30 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 38 30 02 09 00 00 31 2E 54 32 30 2E " +
                "30 38 30 02 09 00 00 31 2E 54 32 30 2E 30 38 31 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 38 31 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 38 31 02 09 00 00 31 2E " +
                "54 32 30 2E 30 39 31 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 39 31 02 09 00 00 31 2E 54 32 30 2E 30 39 " +
                "31 02 09 00 00 31 2E 54 32 30 2E 30 39 32 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 39 32 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 39 32 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 39 33 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "39 33 02 09 00 00 31 2E 54 32 30 2E 30 39 33 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 30 33 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 30 33 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 30 33 02 09 00 00 31 2E 54 32 30 2E " +
                "31 30 34 02 09 00 00 31 2E 54 32 30 2E 31 30 34 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 30 34 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 30 35 02 09 00 00 31 2E " +
                "54 32 30 2E 31 30 35 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 30 35 02 09 00 00 31 2E 54 32 30 2E 31 31 " +
                "35 02 09 00 00 31 2E 54 32 30 2E 31 31 35 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 31 35 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 31 36 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 31 36 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "31 36 02 09 00 00 31 2E 54 32 30 2E 31 31 37 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 31 37 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 31 37 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 32 37 02 09 00 00 31 2E 54 32 30 2E " +
                "31 32 37 02 09 00 00 31 2E 54 32 30 2E 31 32 37 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 32 38 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 32 38 02 09 00 00 31 2E " +
                "54 32 30 2E 31 32 38 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 32 39 02 09 00 00 31 2E 54 32 30 2E 31 32 " +
                "39 02 09 00 00 31 2E 54 32 30 2E 31 32 39 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 33 39 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 33 39 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 33 39 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "34 30 02 09 00 00 31 2E 54 32 30 2E 31 34 30 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 34 30 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 34 31 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 34 31 02 09 00 00 31 2E 54 32 30 2E " +
                "31 34 31 02 09 00 00 31 2E 54 32 30 2E 31 35 31 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 35 31 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 35 31 02 09 00 00 31 2E " +
                "54 32 30 2E 31 35 32 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 35 32 02 09 00 00 31 2E 54 32 30 2E 31 35 " +
                "32 02 09 00 00 31 2E 54 32 30 2E 31 35 33 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 35 33 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 35 33 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 36 33 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "36 33 02 09 00 00 31 2E 54 32 30 2E 31 36 33 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 36 34 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 36 34 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 36 34 02 09 00 00 31 2E 54 32 30 2E " +
                "31 36 35 02 09 00 00 31 2E 54 32 30 2E 31 36 35 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 36 35 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 30 33 02 09 00 00 31 2E " +
                "54 36 30 2E 30 30 33 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 30 33 02 09 00 00 31 2E 54 36 30 2E 30 30 " +
                "34 02 09 00 00 31 2E 54 36 30 2E 30 30 34 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 30 34 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 30 35 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 30 35 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "30 35 02 09 00 00 31 2E 54 36 30 2E 30 31 35 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 31 35 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 31 35 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 31 36 02 09 00 00 31 2E 54 36 30 2E " +
                "30 31 36 02 09 00 00 31 2E 54 36 30 2E 30 31 36 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 31 37 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 31 37 02 09 00 00 31 2E " +
                "54 36 30 2E 30 31 37 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 32 37 02 09 00 00 31 2E 54 36 30 2E 30 32 " +
                "37 02 09 00 00 31 2E 54 36 30 2E 30 32 37 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 32 38 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 32 38 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 32 38 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "32 39 02 09 00 00 31 2E 54 36 30 2E 30 32 39 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 32 39 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 33 39 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 33 39 02 09 00 00 31 2E 54 36 30 2E " +
                "30 33 39 02 09 00 00 31 2E 54 36 30 2E 30 34 30 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 34 30 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 34 30 02 09 00 00 31 2E " +
                "54 36 30 2E 30 34 31 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 34 31 02 09 00 00 31 2E 54 36 30 2E 30 34 " +
                "31 02 09 00 00 31 2E 54 36 30 2E 30 35 31 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 35 31 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 35 31 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 35 32 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "35 32 02 09 00 00 31 2E 54 36 30 2E 30 35 32 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 35 33 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 35 33 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 35 33 02 09 00 00 31 2E 54 36 30 2E " +
                "30 35 36 02 09 00 00 31 2E 54 36 30 2E 30 35 36 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 35 36 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 35 35 02 09 00 00 31 2E " +
                "54 36 30 2E 30 35 35 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 35 35 02 09 00 00 31 2E 54 36 30 2E 30 35 " +
                "34 02 09 00 00 31 2E 54 36 30 2E 30 35 34 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 35 34 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 34 34 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 34 34 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "34 34 02 09 00 00 31 2E 54 36 30 2E 30 34 33 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 34 33 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 34 33 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 34 32 02 09 00 00 31 2E 54 36 30 2E " +
                "30 34 32 02 09 00 00 31 2E 54 36 30 2E 30 34 32 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 33 32 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 33 32 02 09 00 00 31 2E " +
                "54 36 30 2E 30 33 32 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 33 31 02 09 00 00 31 2E 54 36 30 2E 30 33 " +
                "31 02 09 00 00 31 2E 54 36 30 2E 30 33 31 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 33 30 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 33 30 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 33 30 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "32 30 02 09 00 00 31 2E 54 36 30 2E 30 32 30 02 " +
                "09 00 00 31 2E 54 36 30 2E 30 32 30 02 09 00 00 " +
                "31 2E 54 36 30 2E 30 31 39 02 09 00 00 31 2E 54 " +
                "36 30 2E 30 31 39 02 09 00 00 31 2E 54 36 30 2E " +
                "30 31 39 02 09 00 00 31 2E 54 36 30 2E 30 31 38 " +
                "02 09 00 00 31 2E 54 36 30 2E 30 31 38 02 09 00 " +
                "00 31 2E 54 36 30 2E 30 31 38 02 09 00 00 31 2E " +
                "54 36 30 2E 30 30 38 02 09 00 00 31 2E 54 36 30 " +
                "2E 30 30 38 02 09 00 00 31 2E 54 36 30 2E 30 30 " +
                "38 02 09 00 00 31 2E 54 36 30 2E 30 30 37 02 09 " +
                "00 00 31 2E 54 36 30 2E 30 30 37 02 09 00 00 31 " +
                "2E 54 36 30 2E 30 30 37 02 09 00 00 31 2E 54 36 " +
                "30 2E 30 30 36 02 09 00 00 31 2E 54 36 30 2E 30 " +
                "30 36 02 09 00 00 31 2E 54 36 30 2E 30 30 36 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 36 38 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 36 38 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 36 38 02 09 00 00 31 2E 54 32 30 2E " +
                "31 36 37 02 09 00 00 31 2E 54 32 30 2E 31 36 37 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 36 37 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 36 36 02 09 00 00 31 2E " +
                "54 32 30 2E 31 36 36 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 36 36 02 09 00 00 31 2E 54 32 30 2E 31 35 " +
                "36 02 09 00 00 31 2E 54 32 30 2E 31 35 36 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 35 36 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 35 35 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 35 35 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "35 35 02 09 00 00 31 2E 54 32 30 2E 31 35 34 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 35 34 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 35 34 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 34 34 02 09 00 00 31 2E 54 32 30 2E " +
                "31 34 34                                       ";

        String record_0x03C =
                "         3C 00 62 05 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 34 34 02 09 00 00 31 2E 54 32 30 2E 31 34 " +
                "33 02 09 00 00 31 2E 54 32 30 2E 31 34 33 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 34 33 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 34 32 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 34 32 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "34 32 02 09 00 00 31 2E 54 32 30 2E 31 33 32 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 33 32 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 33 32 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 33 31 02 09 00 00 31 2E 54 32 30 2E " +
                "31 33 31 02 09 00 00 31 2E 54 32 30 2E 31 33 31 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 33 30 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 33 30 02 09 00 00 31 2E " +
                "54 32 30 2E 31 33 30 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 32 30 02 09 00 00 31 2E 54 32 30 2E 31 32 " +
                "30 02 09 00 00 31 2E 54 32 30 2E 31 32 30 02 09 " +
                "00 00 31 2E 54 32 30 2E 31 31 39 02 09 00 00 31 " +
                "2E 54 32 30 2E 31 31 39 02 09 00 00 31 2E 54 32 " +
                "30 2E 31 31 39 02 09 00 00 31 2E 54 32 30 2E 31 " +
                "31 38 02 09 00 00 31 2E 54 32 30 2E 31 31 38 02 " +
                "09 00 00 31 2E 54 32 30 2E 31 31 38 02 09 00 00 " +
                "31 2E 54 32 30 2E 31 30 38 02 09 00 00 31 2E 54 " +
                "32 30 2E 31 30 38 02 09 00 00 31 2E 54 32 30 2E " +
                "31 30 38 02 09 00 00 31 2E 54 32 30 2E 31 30 37 " +
                "02 09 00 00 31 2E 54 32 30 2E 31 30 37 02 09 00 " +
                "00 31 2E 54 32 30 2E 31 30 37 02 09 00 00 31 2E " +
                "54 32 30 2E 31 30 36 02 09 00 00 31 2E 54 32 30 " +
                "2E 31 30 36 02 09 00 00 31 2E 54 32 30 2E 31 30 " +
                "36 02 09 00 00 31 2E 54 32 30 2E 30 39 36 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 39 36 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 39 36 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 39 35 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "39 35 02 09 00 00 31 2E 54 32 30 2E 30 39 35 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 39 34 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 39 34 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 39 34 02 09 00 00 31 2E 54 32 30 2E " +
                "30 38 34 02 09 00 00 31 2E 54 32 30 2E 30 38 34 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 38 34 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 38 33 02 09 00 00 31 2E " +
                "54 32 30 2E 30 38 33 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 38 33 02 09 00 00 31 2E 54 32 30 2E 30 38 " +
                "32 02 09 00 00 31 2E 54 32 30 2E 30 38 32 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 38 32 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 37 32 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 37 32 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "37 32 02 09 00 00 31 2E 54 32 30 2E 30 37 31 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 37 31 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 37 31 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 37 30 02 09 00 00 31 2E 54 32 30 2E " +
                "30 37 30 02 09 00 00 31 2E 54 32 30 2E 30 37 30 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 36 30 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 36 30 02 09 00 00 31 2E " +
                "54 32 30 2E 30 36 30 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 35 39 02 09 00 00 31 2E 54 32 30 2E 30 35 " +
                "39 02 09 00 00 31 2E 54 32 30 2E 30 35 39 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 35 38 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 35 38 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 35 38 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "34 38 02 09 00 00 31 2E 54 32 30 2E 30 34 38 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 34 38 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 34 37 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 34 37 02 09 00 00 31 2E 54 32 30 2E " +
                "30 34 37 02 09 00 00 31 2E 54 32 30 2E 30 34 36 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 34 36 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 34 36 02 09 00 00 31 2E " +
                "54 32 30 2E 30 33 36 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 33 36 02 09 00 00 31 2E 54 32 30 2E 30 33 " +
                "36 02 09 00 00 31 2E 54 32 30 2E 30 33 35 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 33 35 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 33 35 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 33 34 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "33 34 02 09 00 00 31 2E 54 32 30 2E 30 33 34 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 32 34 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 32 34 02 09 00 00 31 2E 54 " +
                "32 30 2E 30 32 34 02 09 00 00 31 2E 54 32 30 2E " +
                "30 32 33 02 09 00 00 31 2E 54 32 30 2E 30 32 33 " +
                "02 09 00 00 31 2E 54 32 30 2E 30 32 33 02 09 00 " +
                "00 31 2E 54 32 30 2E 30 32 32 02 09 00 00 31 2E " +
                "54 32 30 2E 30 32 32 02 09 00 00 31 2E 54 32 30 " +
                "2E 30 32 32 02 09 00 00 31 2E 54 32 30 2E 30 31 " +
                "32 02 09 00 00 31 2E 54 32 30 2E 30 31 32 02 09 " +
                "00 00 31 2E 54 32 30 2E 30 31 32 02 09 00 00 31 " +
                "2E 54 32 30 2E 30 31 31 02 09 00 00 31 2E 54 32 " +
                "30 2E 30 31 31 02 09 00 00 31 2E 54 32 30 2E 30 " +
                "31 31 02 09 00 00 31 2E 54 32 30 2E 30 31 30 02 " +
                "09 00 00 31 2E 54 32 30 2E 30 31 30 02 09 00 00 " +
                "31 2E 54 32 30 2E 30 31 30                     ";

        byte[] data1 = HexRead.readFromString(record_0x0018 + record_0x03C);
        RecordInputStream in1 = TestcaseRecordInputStream.create(data1);
        NameRecord nr1 = new NameRecord(in1);
        assert_bug50244(nr1);

        byte[] data2 = nr1.serialize();

        assertEquals(data1.length, data2.length);
        RecordInputStream in2 = TestcaseRecordInputStream.create(data2);
        NameRecord nr2 = new NameRecord(in2);
        assert_bug50244(nr2);
    }

    private void assert_bug50244(NameRecord nr){
        assertEquals("wDataGruppeSerie.", nr.getNameText());
        Ptg[] ptg = nr.getNameDefinition();
        assertEquals(1, ptg.length);
        ArrayPtg arr = (ArrayPtg)ptg[0];
        assertEquals(696, arr.getRowCount());
        assertEquals(1, arr.getColumnCount());
        Object[][] vals = arr.getTokenArrayValues();
        assertEquals("1.T20.001", vals[0][0]);
        assertEquals("1.T20.010", vals[vals.length - 1][0]);
    }

    public void testBug57923() {
        NameRecord record = new NameRecord();
        assertEquals(0, record.getExternSheetNumber());
        
        record.setNameDefinition(new Ptg[] {});
        assertEquals(0, record.getExternSheetNumber());
        
        record.setNameDefinition(new Ptg[] {new NamePtg(1)});
        assertEquals(0, record.getExternSheetNumber());

        record.setNameDefinition(new Ptg[] {new Area3DPtg("area", 1)});
        assertEquals(1, record.getExternSheetNumber());

        record.setNameDefinition(new Ptg[] {new Ref3DPtg("A1", 1)});
        assertEquals(1, record.getExternSheetNumber());
    }
    
}
