//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <string>

// template <class T>
//    basic_string& assign(const T& t, size_type pos, size_type n=npos); // C++17, constexpr since C++20

#include <string>
#include <stdexcept>
#include <cassert>

#include "test_macros.h"
#include "min_allocator.h"
#include "asan_testing.h"

template <class S, class SV>
TEST_CONSTEXPR_CXX20 void test(S s, SV sv, typename S::size_type pos, typename S::size_type n, S expected) {
  if (pos <= sv.size()) {
    s.assign(sv, pos, n);
    LIBCPP_ASSERT(s.__invariants());
    assert(s == expected);
    LIBCPP_ASSERT(is_string_asan_correct(s));
  }
#ifndef TEST_HAS_NO_EXCEPTIONS
  else if (!TEST_IS_CONSTANT_EVALUATED) {
    try {
      s.assign(sv, pos, n);
      assert(false);
    } catch (std::out_of_range&) {
      assert(pos > sv.size());
    }
  }
#endif
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 void test_npos(S s, SV sv, typename S::size_type pos, S expected) {
  if (pos <= sv.size()) {
    s.assign(sv, pos);
    LIBCPP_ASSERT(s.__invariants());
    assert(s == expected);
  }
#ifndef TEST_HAS_NO_EXCEPTIONS
  else if (!TEST_IS_CONSTANT_EVALUATED) {
    try {
      s.assign(sv, pos);
      assert(false);
    } catch (std::out_of_range&) {
      assert(pos > sv.size());
    }
  }
#endif
}

template <class S>
TEST_CONSTEXPR_CXX20 void test_string() {
  typedef std::string_view SV;
  test(S(), SV(), 0, 0, S());
  test(S(), SV(), 1, 0, S());
  test(S(), SV("12345"), 0, 3, S("123"));
  test(S(), SV("12345"), 1, 4, S("2345"));
  test(S(), SV("12345"), 3, 15, S("45"));
  test(S(), SV("12345"), 5, 15, S(""));
  test(S(), SV("12345"), 6, 15, S("not happening"));
  test(S(), SV("12345678901234567890"), 0, 0, S());
  test(S(), SV("12345678901234567890"), 1, 1, S("2"));
  test(S(), SV("12345678901234567890"), 2, 3, S("345"));
  test(S(), SV("12345678901234567890"), 12, 13, S("34567890"));
  test(S(), SV("12345678901234567890"), 21, 13, S("not happening"));

  test(S("12345"), SV(), 0, 0, S());
  test(S("12345"), SV("12345"), 2, 2, S("34"));
  test(S("12345"), SV("1234567890"), 0, 100, S("1234567890"));

  test(S("12345678901234567890"), SV(), 0, 0, S());
  test(S("12345678901234567890"), SV("12345"), 1, 3, S("234"));
  test(S("12345678901234567890"), SV("12345678901234567890"), 5, 10, S("6789012345"));
}

TEST_CONSTEXPR_CXX20 bool test() {
  test_string<std::string>();
#if TEST_STD_VER >= 11
  test_string<std::basic_string<char, std::char_traits<char>, min_allocator<char>>>();
  test_string<std::basic_string<char, std::char_traits<char>, safe_allocator<char>>>();
#endif

  {
    typedef std::string S;
    typedef std::string_view SV;
    test_npos(S(), SV(), 0, S());
    test_npos(S(), SV(), 1, S());
    test_npos(S(), SV("12345"), 0, S("12345"));
    test_npos(S(), SV("12345"), 1, S("2345"));
    test_npos(S(), SV("12345"), 3, S("45"));
    test_npos(S(), SV("12345"), 5, S(""));
    test_npos(S(), SV("12345"), 6, S("not happening"));
  }

  {
    std::string s       = "ABCD";
    std::string_view sv = "EFGH";
    char arr[]          = "IJKL";

    s.assign("CDEF", 0); // calls assign(const char *, len)
    assert(s == "");
    s.clear();

    s.assign("QRST", 0, std::string::npos); // calls assign(string("QRST", pos, len)
    assert(s == "QRST");
    s.clear();

    s.assign(sv, 0); // calls assign(T, pos, npos)
    assert(s == sv);
    s.clear();

    s.assign(sv, 0, std::string::npos); // calls assign(T, pos, npos)
    assert(s == sv);
    s.clear();

    s.assign(arr, 0); // calls assign(const char *, len)
    assert(s == "");
    s.clear();

    s.assign(arr, 0, std::string::npos); // calls assign(string("IJKL"), pos, npos)
    assert(s == "IJKL");
    s.clear();

    s.assign(arr, 0); // calls assign(const char *, len)
    assert(s == "");
    s.clear();
  }

  {
    std::string s       = "ABCD";
    std::string_view sv = s;
    s.assign(sv);
    assert(s == "ABCD");

    sv = s;
    s.assign(sv, 0, std::string::npos);
    assert(s == "ABCD");
  }

  {
    std::string s       = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    std::string_view sv = s;
    s.assign(sv);
    assert(s == "ABCDEFGHIJKLMNOPQRSTUVWXYZ");

    sv = s;
    s.assign(sv, 0, std::string::npos);
    assert(s == "ABCDEFGHIJKLMNOPQRSTUVWXYZ");
  }
  return true;
}

int main(int, char**) {
  test();
#if TEST_STD_VER > 17
  static_assert(test());
#endif

  return 0;
}
