#include "alutInternal.h"
#include <math.h>
#include <string.h>

#if defined(_WIN32)
#define random() rand()
#endif

static const double sampleFrequency = 44100;

/*
 * The following waveformFoo functions expect the phase of the previous call and
 * the current phase, both in the range [0..+1). They return an amplitude in the
 * range [-1..+1].
 */

typedef double (*waveformFunction) (double lastPhase, double phase);

static double waveformSine(double UNUSED(lastPhase), double phase)
{
  static const double pi = 3.14159265358979323846;

  return sin(phase * pi);
}

static double waveformSquare(double UNUSED(lastPhase), double phase)
{
  return (phase >= 0.5) ? -1 : 1;
}

static double waveformSawtooth(double UNUSED(lastPhase), double phase)
{
  return 2 * phase - 1;
}

static double waveformWhitenoise(double UNUSED(lastPhase), double UNUSED(phase))
{
  static const long prime = 67867967L;

  return 2 * (double)(random() % prime) / prime - 1;
}

static double waveformImpulse(double lastPhase, double phase)
{
  return (lastPhase > phase) ? 1 : 0;
}

static waveformFunction getWaveformFunction(ALenum waveshape)
{
  switch (waveshape)
  {
  case ALUT_WAVEFORM_SINE:
    return &waveformSine;
  case ALUT_WAVEFORM_SQUARE:
    return &waveformSquare;
  case ALUT_WAVEFORM_SAWTOOTH:
    return &waveformSawtooth;
  case ALUT_WAVEFORM_WHITENOISE:
    return &waveformWhitenoise;
  case ALUT_WAVEFORM_IMPULSE:
    return &waveformImpulse;
  }
  _alutSetError(ALUT_ERROR_INVALID_ENUM);
  return NULL;
}

static OutputStream *generateWaveform(ALenum waveshape, ALfloat frequency, ALfloat phase, ALfloat duration)
{
  waveformFunction func;
  double sampleDuration, lastPhase, numSamplesD;
  size_t numBytes, numSamples, i;
  OutputStream *stream;

  func = getWaveformFunction(waveshape);
  if (func == NULL)
  {
    return NULL;
  }

  /* ToDo: Shall we test phase for [-180 .. +180]? */
  if (frequency <= 0 || duration < 0)
  {
    _alutSetError(ALUT_ERROR_INVALID_VALUE);
    return NULL;
  }

  /* allocate stream to hold AU header and sample data */
  sampleDuration = floor((frequency * duration) + 0.5) / frequency;
  /* GCC is a bit picky about casting function calls, so we do it in two
   * steps... */
  numSamplesD = floor(sampleDuration * sampleFrequency);
  numSamples = (size_t) numSamplesD;
  numBytes = numSamples * sizeof(int16_t);
  stream = _alutOutputStreamConstruct(AU_HEADER_SIZE + numBytes);
  if (stream == NULL)
  {
    return NULL;
  }

  /* write AU header for our 16bit mono data */
  if (!_alutOutputStreamWriteInt32BE(stream, 0x2e736e64) || /* ".snd" */
      !_alutOutputStreamWriteInt32BE(stream, AU_HEADER_SIZE) ||
      !_alutOutputStreamWriteInt32BE(stream, (Int32BigEndian) numBytes) ||
      !_alutOutputStreamWriteInt32BE(stream, AU_PCM_16) ||
      !_alutOutputStreamWriteInt32BE(stream, (Int32BigEndian) sampleFrequency) || !_alutOutputStreamWriteInt32BE(stream, 1))
  {
    _alutOutputStreamDestroy(stream);
    return NULL;
  }

  /* normalize phase from degrees */
  phase /= 180;

  /* the value corresponding to i = -1 below */
  lastPhase = phase - frequency / sampleFrequency;
  lastPhase -= floor(lastPhase);

  /* calculate samples */
  for (i = 0; i < numSamples; i++)
  {
    double p = phase + frequency * (double)i / sampleFrequency;
    double currentPhase = p - floor(p);
    double amplitude = func(lastPhase, currentPhase);

    if (!_alutOutputStreamWriteInt16BE(stream, (Int16BigEndian) (amplitude * 32767)))
    {
      _alutOutputStreamDestroy(stream);
      return NULL;
    }
    lastPhase = currentPhase;
  }

  return stream;
}

ALvoid *alutLoadMemoryWaveform(ALenum waveshape, ALfloat frequency, ALfloat phase, ALfloat duration, ALenum * format, ALsizei * size, ALfloat * freq)
{
  OutputStream *outputStream;
  InputStream *inputStream;
  ALvoid *data;

  if (!_alutSanityCheck())
  {
    return NULL;
  }

  outputStream = generateWaveform(waveshape, frequency, phase, duration);
  if (outputStream == NULL)
  {
    return NULL;
  }

  /* We could do something more efficient here if the internal stream
   * structures were known, but this would break the abstraction. */
  inputStream = _alutInputStreamConstructFromMemory(_alutOutputStreamGetData(outputStream), _alutOutputStreamGetLength(outputStream));
  if (inputStream == NULL)
  {
    _alutOutputStreamDestroy(outputStream);
    return NULL;
  }

  data = _alutLoadMemoryFromInputStream(inputStream, format, size, freq);
  _alutOutputStreamDestroy(outputStream);
  return data;
}

ALuint alutCreateBufferWaveform(ALenum waveshape, ALfloat frequency, ALfloat phase, ALfloat duration)
{
  OutputStream *outputStream;
  InputStream *inputStream;
  ALuint buffer;

  if (!_alutSanityCheck())
  {
    return AL_NONE;
  }

  outputStream = generateWaveform(waveshape, frequency, phase, duration);
  if (outputStream == NULL)
  {
    return AL_NONE;
  }

  /* We could do something more efficient here if the internal stream
   * structures were known, but this would break the abstraction. */
  inputStream = _alutInputStreamConstructFromMemory(_alutOutputStreamGetData(outputStream), _alutOutputStreamGetLength(outputStream));
  if (inputStream == NULL)
  {
    _alutOutputStreamDestroy(outputStream);
    return AL_NONE;
  }

  buffer = _alutCreateBufferFromInputStream(inputStream);
  _alutOutputStreamDestroy(outputStream);
  return buffer;
}

/* converted from helloworld.wav */
static uint8_t helloWorldSample[] = {
  0x2e, 0x73, 0x6e, 0x64,       /* ".snd" */
  0x00, 0x00, 0x00, 0x18,       /* byte offset to data part (24) */
  0x00, 0x00, 0x1f, 0xf2,       /* number of bytes in the data part (8178) */
  0x00, 0x00, 0x00, 0x01,       /* encoding of the data part (8-bit ISDN u-law) */
  0x00, 0x00, 0x2b, 0x11,       /* number of samples per second (11025) */
  0x00, 0x00, 0x00, 0x01,       /* number of interleaved channels (1) */

  0x7e, 0x7c, 0x7d, 0x7a, 0x79, 0xfc, 0xf8, 0x7e, 0x73, 0x6f, 0x7c, 0x7d,
  0x6f, 0x71, 0x78, 0x78, 0x77, 0x6f, 0x6b, 0x6e, 0x71, 0x6c, 0x6b, 0x75,
  0xff, 0x7b, 0x70, 0x6d, 0x6d, 0x6a, 0x6c, 0x74, 0x70, 0x73, 0x71, 0x63,
  0x68, 0xfb, 0xf4, 0x78, 0x6f, 0xfd, 0xef, 0x74, 0x62, 0x70, 0xf6, 0x6a,
  0x60, 0x6b, 0xf9, 0x7e, 0x6a, 0x7b, 0xea, 0x78, 0x62, 0x75, 0xe3, 0xe7,
  0x71, 0x66, 0x78, 0xf9, 0x6d, 0x6d, 0x77, 0x74, 0x6f, 0x68, 0x6a, 0xfe,
  0xf8, 0x7e, 0x6f, 0x6b, 0x72, 0xfe, 0xea, 0xe7, 0x72, 0x60, 0x6a, 0xed,
  0xe8, 0x6f, 0x62, 0x62, 0x6a, 0xf3, 0xef, 0x7e, 0xfd, 0x70, 0x64, 0x6e,
  0xed, 0xe6, 0xec, 0x6b, 0x5e, 0x7b, 0xe0, 0xe7, 0x7e, 0x62, 0x5c, 0x72,
  0xee, 0xf5, 0x74, 0x65, 0x69, 0x6d, 0xfd, 0xe6, 0xef, 0x6c, 0x6a, 0x7e,
  0xf6, 0x7a, 0x67, 0x64, 0x7b, 0x7c, 0x6b, 0xf0, 0xde, 0xfe, 0x5c, 0x5f,
  0x7c, 0xe5, 0xe4, 0xf6, 0x6e, 0x69, 0x7e, 0xe2, 0xe0, 0x77, 0x5d, 0x6a,
  0xf3, 0x6e, 0x6a, 0x79, 0xfa, 0xed, 0xeb, 0xeb, 0xec, 0x7e, 0x6f, 0x78,
  0x69, 0x64, 0x7d, 0xff, 0x74, 0x67, 0x6c, 0xe5, 0xee, 0x77, 0xf2, 0xeb,
  0xf2, 0x68, 0x67, 0x73, 0x6f, 0x74, 0xf2, 0xeb, 0x78, 0x66, 0x66, 0x71,
  0xe7, 0xe2, 0x6d, 0x65, 0xf1, 0x7a, 0x5f, 0x64, 0x73, 0x79, 0x75, 0xeb,
  0xe8, 0x6c, 0x6e, 0xee, 0xe5, 0xee, 0x66, 0x6c, 0xf5, 0x6b, 0x6f, 0xe5,
  0xec, 0x7b, 0x7d, 0x79, 0x76, 0x68, 0x67, 0xf6, 0xef, 0x6e, 0x73, 0xfd,
  0x7c, 0xf6, 0x75, 0x68, 0x72, 0x6a, 0x5e, 0x68, 0x77, 0x77, 0x68, 0x5e,
  0x6e, 0xea, 0xfb, 0x6a, 0xf5, 0xf1, 0x70, 0x6a, 0x6b, 0xde, 0xdf, 0x66,
  0x6e, 0x7d, 0x64, 0x5f, 0xf5, 0xe6, 0x7a, 0x7a, 0xee, 0xeb, 0x74, 0x60,
  0x76, 0xde, 0xf2, 0x58, 0x66, 0xdb, 0xdd, 0x6d, 0x5b, 0xfa, 0xe3, 0x68,
  0x6e, 0xe1, 0xe1, 0x73, 0x55, 0x5a, 0xe5, 0xda, 0xee, 0x7e, 0xef, 0x7c,
  0x6b, 0x7a, 0xe9, 0xe7, 0x72, 0x5e, 0x79, 0xe5, 0xfa, 0xff, 0x78, 0x64,
  0x6b, 0x7a, 0xf6, 0xf7, 0x78, 0xef, 0xe8, 0xf8, 0x7a, 0xef, 0xde, 0xe7,
  0x58, 0x58, 0xde, 0xdf, 0x6c, 0x74, 0xfa, 0xef, 0xed, 0xfd, 0xf8, 0xf4,
  0x66, 0x5f, 0xe7, 0xe0, 0x6e, 0x7c, 0xee, 0x71, 0x67, 0x69, 0xfb, 0xe7,
  0xed, 0x72, 0x6c, 0x6f, 0x67, 0x6b, 0xe8, 0xe2, 0x70, 0x6b, 0xf0, 0x64,
  0x58, 0xee, 0xe3, 0x66, 0x74, 0xe7, 0xed, 0x6d, 0x58, 0x56, 0x62, 0x6e,
  0x6d, 0xe8, 0xd8, 0xe2, 0x6f, 0x5e, 0x63, 0xfd, 0xfa, 0x6f, 0x64, 0x59,
  0x5b, 0x6e, 0x71, 0x69, 0x6c, 0x6c, 0x74, 0xe3, 0xdd, 0xf5, 0x69, 0x71,
  0x7e, 0x62, 0x65, 0xe6, 0xda, 0xee, 0x64, 0xed, 0xcd, 0xcd, 0xd8, 0xd6,
  0xcd, 0xd2, 0xea, 0xe0, 0xcd, 0xcf, 0xe9, 0x74, 0xdd, 0xd0, 0xd4, 0xdc,
  0xef, 0xe2, 0xdd, 0x69, 0x66, 0x76, 0x6c, 0x6d, 0x5c, 0x5a, 0x67, 0x6b,
  0x5c, 0x4c, 0x48, 0x4d, 0x5d, 0x5a, 0x4c, 0x51, 0x4f, 0x4f, 0x58, 0x4d,
  0x4e, 0x4d, 0x40, 0x40, 0x49, 0x4d, 0x4f, 0x4d, 0x3e, 0x38, 0x4c, 0xbe,
  0xb1, 0xb9, 0xc9, 0xcb, 0xcd, 0xf2, 0xe8, 0xd1, 0xd4, 0xeb, 0x4d, 0x54,
  0xca, 0xc2, 0xbf, 0xb7, 0xb2, 0xb7, 0xc6, 0xc4, 0xb8, 0xbe, 0xde, 0x77,
  0xd0, 0xcd, 0xe1, 0xdf, 0xcf, 0xc7, 0xda, 0x54, 0x59, 0x5d, 0x4f, 0x49,
  0x49, 0x51, 0x4d, 0x45, 0x46, 0x4a, 0x46, 0x3b, 0x38, 0x38, 0x38, 0x35,
  0x32, 0x33, 0x38, 0xe7, 0xb0, 0xb0, 0xbc, 0xce, 0xd8, 0x74, 0x47, 0x44,
  0x4b, 0x50, 0x43, 0x3e, 0x57, 0xca, 0xbb, 0xbc, 0xc0, 0xc1, 0xca, 0xde,
  0xee, 0xe0, 0xd3, 0x6c, 0x4d, 0xcf, 0xb7, 0xb9, 0xc7, 0xd1, 0xc5, 0xc7,
  0xe2, 0xf7, 0xd1, 0xcc, 0xfe, 0x5d, 0xe7, 0xce, 0xcd, 0xe1, 0x74, 0x67,
  0x4d, 0x3e, 0x3b, 0x3d, 0x3b, 0x39, 0x34, 0x33, 0x3c, 0x4b, 0x52, 0x4f,
  0xce, 0xaf, 0xae, 0xc3, 0x5c, 0x51, 0x51, 0x3d, 0x36, 0x3c, 0x4b, 0x4d,
  0x47, 0x5c, 0xc7, 0xb9, 0xbb, 0xbe, 0xbe, 0xc6, 0xe7, 0x57, 0x5a, 0x6b,
  0x60, 0x51, 0x54, 0xd0, 0xb7, 0xb4, 0xbc, 0xc0, 0xc5, 0xdc, 0x6b, 0xee,
  0xde, 0xe5, 0x63, 0x5c, 0x7d, 0xe1, 0xd1, 0xcf, 0xe8, 0x51, 0x3d, 0x32,
  0x2d, 0x2c, 0x30, 0x3e, 0x4d, 0x52, 0x62, 0xbe, 0xac, 0xad, 0xbc, 0xdc,
  0x58, 0x43, 0x38, 0x36, 0x3e, 0x48, 0x47, 0x46, 0x6a, 0xbf, 0xb4, 0xb4,
  0xb9, 0xc0, 0xcf, 0x71, 0x57, 0x5b, 0x5d, 0x50, 0x47, 0x4b, 0x63, 0xdd,
  0xc4, 0xb3, 0xad, 0xb2, 0xc9, 0xfc, 0xf9, 0xf5, 0x58, 0x4d, 0x53, 0x5a,
  0x56, 0x53, 0xf1, 0xcd, 0xda, 0x52, 0x3d, 0x38, 0x2f, 0x27, 0x2b, 0x43,
  0xd5, 0xbd, 0xb0, 0xae, 0xb6, 0xc4, 0xe9, 0x6f, 0x71, 0x4a, 0x38, 0x34,
  0x36, 0x3a, 0x4a, 0xde, 0xbe, 0xb6, 0xb9, 0xbf, 0xc3, 0xc9, 0xcf, 0xe3,
  0x5d, 0x4e, 0x44, 0x3f, 0x45, 0x52, 0xf6, 0xd2, 0xc3, 0xb6, 0xb2, 0xb5,
  0xb8, 0xbd, 0xd2, 0x4e, 0x41, 0x43, 0x47, 0x4a, 0x4a, 0x53, 0x5e, 0x51,
  0x48, 0x41, 0x3e, 0x2e, 0x28, 0x37, 0xc1, 0xac, 0xac, 0xb4, 0xbf, 0xd5,
  0x51, 0x46, 0x6e, 0xde, 0x4d, 0x38, 0x2f, 0x33, 0x4a, 0xce, 0xba, 0xb1,
  0xb3, 0xc0, 0xde, 0x71, 0xdc, 0xce, 0xde, 0x59, 0x47, 0x3e, 0x3e, 0x49,
  0x7a, 0xc8, 0xc0, 0xc3, 0xbc, 0xb7, 0xb8, 0xbb, 0xbf, 0xd3, 0x52, 0x3f,
  0x3d, 0x44, 0x4e, 0x4f, 0x4f, 0x4b, 0x45, 0x3f, 0x3c, 0x38, 0x2e, 0x2f,
  0xe1, 0xab, 0xaa, 0xb6, 0xc1, 0xca, 0xf7, 0x47, 0x44, 0xfa, 0xdd, 0x3e,
  0x2d, 0x2f, 0x41, 0xec, 0xc6, 0xb8, 0xaf, 0xb4, 0xcd, 0x66, 0xe8, 0xcb,
  0xcc, 0x6e, 0x4a, 0x44, 0x3e, 0x3c, 0x47, 0xdc, 0xc0, 0xc2, 0xc9, 0xc0,
  0xb7, 0xb4, 0xbd, 0xc9, 0xca, 0xde, 0x48, 0x3b, 0x3f, 0x4e, 0x51, 0x49,
  0x42, 0x42, 0x43, 0x39, 0x32, 0x2f, 0x31, 0xde, 0xaa, 0xaa, 0xb6, 0xbb,
  0xc4, 0x6c, 0x4b, 0x49, 0x6c, 0xd5, 0x43, 0x2e, 0x31, 0x3c, 0x51, 0xce,
  0xbc, 0xb1, 0xb1, 0xc2, 0xee, 0xd8, 0xc8, 0xc7, 0xd6, 0x5e, 0x55, 0x42,
  0x37, 0x3c, 0x50, 0xd9, 0xcb, 0xc4, 0xb9, 0xb5, 0xb6, 0xbd, 0xc5, 0xbe,
  0xc6, 0x5e, 0x42, 0x3e, 0x3f, 0x3d, 0x3a, 0x3a, 0x3d, 0x3b, 0x33, 0x2d,
  0x2a, 0x36, 0xb7, 0xa5, 0xab, 0xb1, 0xb4, 0xc9, 0x52, 0x44, 0x4b, 0xcf,
  0xd7, 0x37, 0x2d, 0x2f, 0x34, 0x3f, 0xeb, 0xbb, 0xad, 0xaf, 0xc5, 0xd4,
  0xc7, 0xc2, 0xc1, 0xc5, 0xd2, 0xfb, 0x43, 0x31, 0x33, 0x3f, 0x53, 0xcf,
  0xba, 0xb6, 0xb5, 0xba, 0xbe, 0xbb, 0xba, 0xc4, 0xdc, 0x5c, 0x43, 0x3a,
  0x35, 0x35, 0x37, 0x35, 0x30, 0x2c, 0x27, 0x2c, 0xcf, 0xaa, 0xab, 0xac,
  0xab, 0xb3, 0xc9, 0x5c, 0x58, 0xc6, 0xc3, 0x47, 0x33, 0x2f, 0x2d, 0x2f,
  0x38, 0x58, 0xb9, 0xb2, 0xbf, 0xc6, 0xbf, 0xbe, 0xbb, 0xba, 0xba, 0xb9,
  0xcc, 0x40, 0x35, 0x38, 0x43, 0x5e, 0xde, 0xc5, 0xb9, 0xbd, 0xd9, 0xeb,
  0xc9, 0xbe, 0xc6, 0xe4, 0x5b, 0x48, 0x39, 0x2f, 0x2c, 0x2b, 0x29, 0x28,
  0x32, 0xd3, 0xb7, 0xbc, 0xba, 0xb2, 0xb3, 0xba, 0xc4, 0xbf, 0xb5, 0xbe,
  0x4c, 0x38, 0x37, 0x35, 0x32, 0x35, 0x46, 0xd6, 0xd2, 0x66, 0xe6, 0xc0,
  0xb9, 0xb8, 0xb6, 0xb2, 0xb2, 0xbf, 0x65, 0x64, 0xdc, 0xea, 0x7c, 0xe2,
  0xdc, 0xfc, 0x4f, 0x49, 0x5a, 0xe5, 0xf0, 0x5e, 0x55, 0x48, 0x3a, 0x2e,
  0x2a, 0x27, 0x2b, 0x51, 0xb8, 0xb9, 0xbe, 0xba, 0xbd, 0xcf, 0xda, 0xcc,
  0xb8, 0xb0, 0xc3, 0x4e, 0x40, 0x39, 0x33, 0x37, 0x45, 0xd9, 0xc1, 0xd9,
  0x58, 0xf9, 0xd8, 0xcf, 0xbf, 0xb6, 0xb0, 0xb3, 0xc9, 0x57, 0x69, 0xd7,
  0xd4, 0xcb, 0xc2, 0xc7, 0xf5, 0x48, 0x42, 0x4d, 0x6c, 0x6f, 0x69, 0x62,
  0x4b, 0x3a, 0x2f, 0x2a, 0x28, 0x29, 0x30, 0xed, 0xb5, 0xb7, 0xbd, 0xbb,
  0xbe, 0xcb, 0xd0, 0xc8, 0xb8, 0xb4, 0xc8, 0x4a, 0x3d, 0x39, 0x35, 0x38,
  0x46, 0xd9, 0xc3, 0xd6, 0x5f, 0xf2, 0xcf, 0xc9, 0xbf, 0xb7, 0xb1, 0xb4,
  0xc4, 0xe1, 0xd0, 0xca, 0xcf, 0xd0, 0xcd, 0xd0, 0xfd, 0x49, 0x41, 0x4a,
  0x55, 0x59, 0x56, 0x53, 0x4a, 0x3a, 0x2e, 0x28, 0x26, 0x2a, 0x38, 0xda,
  0xb6, 0xb5, 0xb9, 0xb8, 0xc1, 0xde, 0xd2, 0xc0, 0xba, 0xb8, 0xc7, 0x50,
  0x41, 0x3b, 0x35, 0x3a, 0x4f, 0xe0, 0xca, 0xcd, 0xec, 0xdd, 0xd1, 0xd5,
  0xc3, 0xb7, 0xb5, 0xb8, 0xbf, 0xc8, 0xc9, 0xcb, 0xd9, 0xd9, 0xce, 0xd5,
  0x69, 0x4e, 0x4c, 0x4c, 0x4f, 0x51, 0x4e, 0x4f, 0x4a, 0x3c, 0x30, 0x2a,
  0x27, 0x2a, 0x2d, 0x38, 0xc6, 0xaf, 0xba, 0xbf, 0xbb, 0xc4, 0xd0, 0xcd,
  0xc9, 0xb9, 0xb3, 0xce, 0x48, 0x45, 0x3e, 0x3a, 0x3e, 0x4b, 0xde, 0xc2,
  0xd4, 0x6e, 0xd6, 0xcc, 0xce, 0xc4, 0xbb, 0xb6, 0xb5, 0xbc, 0xc7, 0xc3,
  0xc4, 0xd3, 0xdc, 0xd5, 0xdd, 0x64, 0x4e, 0x4a, 0x4f, 0x61, 0x5d, 0x55,
  0x5a, 0x4f, 0x3c, 0x31, 0x2b, 0x28, 0x2b, 0x2c, 0x2c, 0x44, 0xb7, 0xb5,
  0xc6, 0xbb, 0xb6, 0xc3, 0xcd, 0xd2, 0xc9, 0xb5, 0xb9, 0x73, 0x58, 0x64,
  0x40, 0x39, 0x3e, 0x4b, 0xda, 0xc7, 0xe2, 0xdd, 0xc3, 0xcc, 0xd6, 0xc5,
  0xbf, 0xba, 0xb6, 0xbe, 0xc2, 0xbd, 0xc5, 0xde, 0xdf, 0xdd, 0xf5, 0x61,
  0x55, 0x51, 0x64, 0xf7, 0x63, 0x66, 0xfd, 0x57, 0x43, 0x38, 0x2e, 0x2b,
  0x2b, 0x2a, 0x29, 0x2b, 0x3f, 0xbf, 0xba, 0xc6, 0xba, 0xb1, 0xbc, 0xce,
  0xd0, 0xc5, 0xb8, 0xba, 0xd8, 0xee, 0xea, 0x46, 0x39, 0x3b, 0x42, 0x66,
  0xd3, 0xe7, 0xda, 0xc3, 0xca, 0xd7, 0xc8, 0xc0, 0xbd, 0xb9, 0xbc, 0xbd,
  0xb9, 0xc1, 0xdd, 0xdf, 0xdf, 0x65, 0x5b, 0x61, 0x64, 0x77, 0xf4, 0x6f,
  0xf1, 0xd9, 0xe7, 0x5b, 0x49, 0x3b, 0x30, 0x2c, 0x2a, 0x29, 0x2a, 0x2a,
  0x2c, 0x4c, 0xbd, 0xc2, 0xc9, 0xb4, 0xb0, 0xbb, 0xc3, 0xc6, 0xbe, 0xb5,
  0xbe, 0xfe, 0xe5, 0xe8, 0x43, 0x3a, 0x3b, 0x3f, 0x56, 0xfc, 0x5d, 0xd3,
  0xbb, 0xc3, 0xcc, 0xc0, 0xc0, 0xc0, 0xbc, 0xc4, 0xc2, 0xb8, 0xbf, 0xd8,
  0xd5, 0xe1, 0x5f, 0x6d, 0x6d, 0x66, 0xda, 0xd2, 0xe8, 0xdc, 0xd1, 0xdc,
  0xe8, 0x6a, 0x47, 0x3c, 0x37, 0x2e, 0x2a, 0x2a, 0x29, 0x29, 0x29, 0x2d,
  0x4d, 0xbf, 0xc5, 0xbf, 0xae, 0xae, 0xbb, 0xc2, 0xc6, 0xc4, 0xbd, 0xcd,
  0x63, 0xdf, 0xf4, 0x3f, 0x3b, 0x3e, 0x40, 0x5a, 0xf1, 0x78, 0xc6, 0xb9,
  0xc4, 0xc7, 0xbe, 0xc4, 0xc3, 0xbe, 0xc9, 0xc8, 0xbb, 0xc2, 0xd3, 0xc8,
  0xcf, 0x7d, 0xe5, 0x7a, 0x5f, 0xd6, 0xd1, 0xf0, 0xd2, 0xc9, 0xd8, 0xda,
  0xd9, 0x68, 0x4f, 0x42, 0x37, 0x33, 0x30, 0x2c, 0x29, 0x2a, 0x29, 0x26,
  0x2b, 0x47, 0xc8, 0xcb, 0xbf, 0xae, 0xae, 0xba, 0xbf, 0xcc, 0xc7, 0xbb,
  0xce, 0x5e, 0xce, 0xd5, 0x46, 0x40, 0x41, 0x3f, 0x5f, 0xf8, 0x5a, 0xc6,
  0xb6, 0xc3, 0xc4, 0xbb, 0xc7, 0xc5, 0xbe, 0xd6, 0xcf, 0xbb, 0xc7, 0xd4,
  0xc1, 0xcd, 0xf7, 0xd5, 0xed, 0x5e, 0xd5, 0xd2, 0xec, 0xcd, 0xca, 0xdc,
  0xd0, 0xce, 0xe9, 0xe3, 0xe0, 0x54, 0x41, 0x3a, 0x32, 0x2e, 0x2b, 0x27,
  0x26, 0x28, 0x26, 0x23, 0x2a, 0xff, 0xb2, 0xbe, 0xb9, 0xa5, 0xa8, 0xbb,
  0xc6, 0xe0, 0xeb, 0xc7, 0x55, 0x3b, 0xea, 0xd5, 0x3e, 0x3c, 0x47, 0x48,
  0xdc, 0xc7, 0xce, 0xb1, 0xa8, 0xb1, 0xb8, 0xb2, 0xbd, 0xc9, 0xcf, 0x53,
  0x63, 0xd8, 0x48, 0x41, 0xec, 0xde, 0xd8, 0xc2, 0xca, 0xc2, 0xb4, 0xc0,
  0xd8, 0xbf, 0xc6, 0xea, 0xdd, 0x70, 0x60, 0xda, 0x6a, 0x4e, 0xfb, 0x75,
  0x47, 0x3e, 0x37, 0x30, 0x2f, 0x2c, 0x29, 0x2a, 0x2b, 0x28, 0x27, 0x27,
  0x25, 0x3c, 0xac, 0xa8, 0xb0, 0xa0, 0x9d, 0xb3, 0xe2, 0x57, 0x35, 0x40,
  0x54, 0x2f, 0x3a, 0xc2, 0x52, 0x36, 0x76, 0xd3, 0xcb, 0xae, 0xae, 0xad,
  0xa0, 0xa7, 0xc1, 0xbd, 0xc7, 0x42, 0x43, 0x3e, 0x35, 0x4d, 0x4e, 0x34,
  0x40, 0xf9, 0x5a, 0xcd, 0xb7, 0xb1, 0xa8, 0xa9, 0xbc, 0xbd, 0xbe, 0x4d,
  0x44, 0x4d, 0x44, 0x68, 0xe4, 0x4f, 0xe7, 0xca, 0xec, 0xf0, 0xe6, 0x64,
  0x5b, 0x45, 0x38, 0x3a, 0x3a, 0x31, 0x30, 0x32, 0x2e, 0x2d, 0x2a, 0x23,
  0x25, 0x24, 0x1f, 0x3e, 0xa1, 0xa3, 0xab, 0x9b, 0x9c, 0xba, 0xe5, 0x3b,
  0x2b, 0x4e, 0x50, 0x2a, 0x4a, 0xb8, 0x4c, 0x43, 0xcf, 0xe3, 0xb8, 0xa7,
  0xb0, 0xab, 0x9e, 0xad, 0xd5, 0xcd, 0x47, 0x35, 0x3d, 0x31, 0x32, 0x66,
  0x4f, 0x3b, 0x5e, 0xdc, 0xe8, 0xc1, 0xbc, 0xba, 0xad, 0xad, 0xbd, 0xc1,
  0xc7, 0x58, 0x4b, 0x4c, 0x49, 0xf9, 0xcf, 0xdd, 0xcf, 0xc7, 0xce, 0xdc,
  0xe7, 0xdd, 0xe2, 0x6e, 0x6a, 0x5a, 0x50, 0x52, 0x42, 0x3d, 0x47, 0x3e,
  0x37, 0x3a, 0x34, 0x2d, 0x2c, 0x28, 0x26, 0x27, 0x25, 0x26, 0x2b, 0x27,
  0x2e, 0xaa, 0x9c, 0xa8, 0x9f, 0x99, 0xa8, 0xca, 0x45, 0x29, 0x35, 0x54,
  0x2b, 0x32, 0xbc, 0xc6, 0xfb, 0xbf, 0xc1, 0xaf, 0xa3, 0xac, 0xaf, 0xa1,
  0xaa, 0xd6, 0x66, 0x3d, 0x33, 0x3b, 0x30, 0x2f, 0x62, 0xe1, 0x49, 0xef,
  0xcb, 0xcc, 0xbc, 0xbf, 0xcb, 0xb8, 0xb8, 0xe0, 0x6c, 0x6f, 0x4f, 0x51,
  0x4e, 0x4f, 0xd0, 0xbf, 0xc7, 0xc4, 0xc0, 0xc5, 0xcf, 0x6a, 0x59, 0x6f,
  0x60, 0x55, 0x5c, 0x7e, 0xdc, 0xe7, 0x65, 0xee, 0xe0, 0x60, 0x4f, 0x4b,
  0x43, 0x3e, 0x38, 0x35, 0x39, 0x3a, 0x39, 0x3b, 0x3d, 0x39, 0x37, 0x34,
  0x2e, 0x2c, 0x2c, 0x28, 0x29, 0x28, 0x29, 0xca, 0x9e, 0xa2, 0xa4, 0x99,
  0xa0, 0xc0, 0x68, 0x2d, 0x2b, 0x58, 0x39, 0x2e, 0xc7, 0xbc, 0x77, 0xc0,
  0xbf, 0xbd, 0xa7, 0xaa, 0xb5, 0xa6, 0xaa, 0xcc, 0xe8, 0x42, 0x33, 0x41,
  0x3a, 0x34, 0xf0, 0xcc, 0x69, 0xce, 0xcb, 0xce, 0xbb, 0xbf, 0xcd, 0xbb,
  0xbd, 0xec, 0x5e, 0x4a, 0x40, 0x48, 0x45, 0x46, 0xe5, 0xc8, 0xc6, 0xc2,
  0xc4, 0xc2, 0xc4, 0xdf, 0x77, 0xe0, 0x7a, 0x5c, 0x5a, 0x5d, 0x7d, 0x75,
  0x5e, 0xfa, 0xda, 0xde, 0xf7, 0x6f, 0x6a, 0x56, 0x46, 0x3f, 0x40, 0x41,
  0x41, 0x43, 0x49, 0x53, 0x52, 0x4b, 0x49, 0x48, 0x43, 0x3c, 0x37, 0x38,
  0x39, 0x36, 0x35, 0x32, 0x30, 0x31, 0x2f, 0x2d, 0x2e, 0x30, 0x33, 0xc3,
  0x9e, 0x9f, 0xa4, 0x9c, 0xa2, 0xc8, 0x4d, 0x30, 0x2b, 0x49, 0x46, 0x37,
  0xc7, 0xb7, 0xd1, 0xbe, 0xbe, 0xc0, 0xab, 0xac, 0xb6, 0xaa, 0xad, 0xd4,
  0xea, 0x41, 0x32, 0x41, 0x3d, 0x38, 0xe2, 0xcc, 0xee, 0xc9, 0xcb, 0xd1,
  0xbe, 0xc6, 0xcf, 0xbe, 0xc5, 0xee, 0x5f, 0x48, 0x41, 0x43, 0x3f, 0x46,
  0x66, 0xdb, 0xcc, 0xcd, 0xd0, 0xcb, 0xd3, 0x7c, 0xec, 0xd7, 0xd5, 0xdc,
  0xee, 0x73, 0x65, 0x59, 0x51, 0x53, 0x68, 0xe1, 0xdd, 0xdc, 0xd9, 0xec,
  0x60, 0x59, 0x51, 0x4f, 0x53, 0x54, 0x5c, 0x61, 0x58, 0x57, 0x59, 0x51,
  0x4d, 0x49, 0x44, 0x44, 0x42, 0x3e, 0x3e, 0x3e, 0x3e, 0x3e, 0x3c, 0x3b,
  0x3b, 0x37, 0x34, 0x32, 0x2f, 0x2d, 0x2f, 0x33, 0x36, 0xbf, 0x9f, 0xa1,
  0xa7, 0x9f, 0xa6, 0xcb, 0x55, 0x34, 0x30, 0x62, 0x4c, 0x3d, 0xc6, 0xbd,
  0xd6, 0xc1, 0xcb, 0xc0, 0xaa, 0xaf, 0xb7, 0xaa, 0xb3, 0xd7, 0xf5, 0x3b,
  0x36, 0x49, 0x3a, 0x3b, 0xde, 0xdb, 0xe5, 0xcc, 0xd5, 0xca, 0xbd, 0xc9,
  0xcb, 0xc3, 0xd0, 0x6f, 0x4e, 0x42, 0x43, 0x44, 0x45, 0x4e, 0x6f, 0xd7,
  0xd0, 0xdb, 0xdf, 0xdd, 0xe3, 0xfb, 0x6c, 0xf8, 0xeb, 0x65, 0x53, 0x4f,
  0x55, 0x5a, 0x57, 0xf2, 0xcb, 0xcc, 0xd3, 0xd6, 0xea, 0x6f, 0x5e, 0x4d,
  0x51, 0x60, 0x59, 0x63, 0xf1, 0xf4, 0xe3, 0xdf, 0xed, 0xe5, 0xee, 0x61,
  0x5e, 0x58, 0x4f, 0x4f, 0x4a, 0x45, 0x45, 0x45, 0x46, 0x47, 0x48, 0x4a,
  0x45, 0x3f, 0x3d, 0x3a, 0x37, 0x35, 0x31, 0x31, 0x33, 0x31, 0x33, 0x44,
  0xb5, 0xa2, 0xa6, 0xa9, 0xa2, 0xad, 0xeb, 0x41, 0x34, 0x3a, 0x59, 0x44,
  0x45, 0xc7, 0xc7, 0xe3, 0xce, 0xc7, 0xb6, 0xac, 0xb1, 0xb0, 0xad, 0xbb,
  0xe4, 0x4b, 0x3a, 0x3e, 0x43, 0x3b, 0x49, 0xe5, 0xee, 0x7b, 0xea, 0xd9,
  0xc4, 0xbe, 0xc1, 0xbe, 0xbd, 0xcb, 0x62, 0x49, 0x43, 0x40, 0x40, 0x46,
  0x50, 0x76, 0xe0, 0xea, 0xf1, 0xdf, 0xd9, 0xdc, 0xde, 0xd9, 0xda, 0xfe,
  0x54, 0x49, 0x47, 0x48, 0x47, 0x46, 0x55, 0xf8, 0xeb, 0xe7, 0xd7, 0xca,
  0xcb, 0xde, 0xed, 0xdf, 0xe1, 0x74, 0x5c, 0x60, 0x7a, 0x6f, 0x61, 0x7c,
  0xda, 0xd2, 0xd6, 0xd9, 0xd5, 0xd9, 0xf2, 0x61, 0x5a, 0x5d, 0x61, 0x59,
  0x58, 0x60, 0x5c, 0x4f, 0x4b, 0x47, 0x42, 0x3e, 0x3c, 0x3b, 0x3b, 0x3a,
  0x39, 0x3a, 0x3a, 0x3a, 0x3a, 0x37, 0x36, 0x3a, 0x6f, 0xae, 0xa8, 0xae,
  0xaa, 0xaa, 0xc9, 0x4a, 0x3f, 0x38, 0x4e, 0xf5, 0x48, 0x77, 0xc6, 0xfd,
  0x63, 0xd5, 0xc4, 0xb3, 0xae, 0xb2, 0xaf, 0xb3, 0xcb, 0x56, 0x45, 0x44,
  0x48, 0x49, 0x4d, 0x60, 0x73, 0x5d, 0x52, 0x69, 0xd0, 0xc5, 0xbe, 0xbc,
  0xbd, 0xc3, 0xea, 0x4c, 0x45, 0x43, 0x44, 0x4a, 0x52, 0x68, 0x7d, 0x65,
  0x5f, 0x67, 0xfb, 0xde, 0xda, 0xd6, 0xd1, 0xe2, 0x5b, 0x4c, 0x47, 0x46,
  0x49, 0x49, 0x51, 0x6d, 0x72, 0x60, 0x7e, 0xde, 0xe0, 0xd6, 0xc9, 0xc7,
  0xca, 0xd6, 0x78, 0x60, 0x5a, 0x4f, 0x54, 0x76, 0xde, 0xd5, 0xd4, 0xdb,
  0xd7, 0xd3, 0xdc, 0xdd, 0xd7, 0xdc, 0xed, 0x6d, 0x59, 0x51, 0x51, 0x50,
  0x54, 0x5f, 0x61, 0x59, 0x52, 0x4c, 0x49, 0x47, 0x42, 0x3f, 0x42, 0x40,
  0x3c, 0x3b, 0x3e, 0x40, 0x3c, 0x38, 0x38, 0x38, 0x3a, 0x48, 0xbf, 0xaa,
  0xab, 0xaf, 0xab, 0xb5, 0xf2, 0x55, 0x46, 0x4c, 0xd9, 0x5f, 0x47, 0x74,
  0x5f, 0x45, 0x54, 0xd7, 0xbb, 0xaf, 0xb0, 0xb3, 0xb3, 0xbe, 0xea, 0x4f,
  0x54, 0x65, 0x5d, 0x55, 0x4f, 0x4d, 0x4a, 0x41, 0x44, 0xfe, 0xca, 0xc1,
  0xbd, 0xbd, 0xc1, 0xd0, 0x5a, 0x50, 0x59, 0x56, 0x51, 0x51, 0x4e, 0x4a,
  0x46, 0x43, 0x49, 0x68, 0xd5, 0xce, 0xc9, 0xc6, 0xce, 0xe9, 0x5f, 0x57,
  0x5c, 0x5d, 0x53, 0x51, 0x52, 0x4d, 0x47, 0x48, 0x54, 0x78, 0xd9, 0xcc,
  0xc7, 0xc3, 0xc2, 0xc8, 0xd6, 0xe2, 0xed, 0x69, 0x5a, 0x5c, 0x5f, 0x5d,
  0x5b, 0x5c, 0x67, 0xeb, 0xdf, 0xda, 0xcf, 0xcb, 0xcc, 0xd9, 0xef, 0x72,
  0x60, 0x55, 0x4f, 0x50, 0x53, 0x4f, 0x4c, 0x4b, 0x4d, 0x4f, 0x4f, 0x51,
  0x59, 0x5a, 0x54, 0x51, 0x4d, 0x4a, 0x47, 0x44, 0x42, 0x41, 0x41, 0x40,
  0x3e, 0x3c, 0x3a, 0x39, 0x4d, 0xb9, 0xad, 0xb3, 0xae, 0xab, 0xbc, 0xde,
  0x72, 0x55, 0xe0, 0xd5, 0x4e, 0x4e, 0x63, 0x45, 0x3d, 0x4c, 0xdc, 0xbe,
  0xb8, 0xb8, 0xb8, 0xb8, 0xc3, 0xee, 0xef, 0xd4, 0xde, 0x7d, 0x5d, 0x4c,
  0x46, 0x3c, 0x39, 0x46, 0x7a, 0xd3, 0xc8, 0xc3, 0xc2, 0xc7, 0xdc, 0xf5,
  0xdc, 0xd3, 0xda, 0xf7, 0x67, 0x55, 0x46, 0x3f, 0x40, 0x4c, 0x67, 0xef,
  0xdf, 0xd5, 0xd5, 0xe3, 0xfd, 0xeb, 0xd9, 0xd6, 0xdf, 0x7c, 0x5f, 0x53,
  0x47, 0x41, 0x47, 0x55, 0x61, 0x6b, 0xf6, 0xdd, 0xd4, 0xd0, 0xce, 0xcc,
  0xca, 0xcd, 0xda, 0xf3, 0x6b, 0x61, 0x5a, 0x56, 0x5e, 0x6c, 0x70, 0x77,
  0xf0, 0xe3, 0xdc, 0xd8, 0xd8, 0xda, 0xde, 0xf2, 0x67, 0x5a, 0x52, 0x4d,
  0x4b, 0x4b, 0x4c, 0x4d, 0x4e, 0x4e, 0x4f, 0x52, 0x54, 0x54, 0x56, 0x58,
  0x54, 0x4e, 0x4c, 0x4a, 0x45, 0x42, 0x45, 0x44, 0x42, 0x42, 0x40, 0x40,
  0x3f, 0x41, 0xcd, 0xb0, 0xb9, 0xba, 0xaf, 0xba, 0xcd, 0xd3, 0xfe, 0xd0,
  0xc2, 0xed, 0x54, 0x5f, 0x4d, 0x3f, 0x41, 0x61, 0xc9, 0xbf, 0xbe, 0xbf,
  0xbe, 0xbe, 0xcf, 0xd6, 0xc3, 0xc1, 0xcb, 0xe0, 0x5b, 0x4e, 0x43, 0x3c,
  0x3f, 0x53, 0xeb, 0xe1, 0xe2, 0xd7, 0xd6, 0xdd, 0xdd, 0xd3, 0xc6, 0xc4,
  0xcf, 0xe9, 0x61, 0x4e, 0x44, 0x42, 0x47, 0x4f, 0x58, 0x56, 0x4f, 0x55,
  0x5c, 0x5d, 0x7b, 0xd5, 0xcb, 0xcb, 0xd4, 0xed, 0x6b, 0x5f, 0x5a, 0x5e,
  0xe5, 0xd3, 0xdc, 0x7c, 0x66, 0x60, 0x5e, 0x62, 0xef, 0xd4, 0xd0, 0xdb,
  0xf1, 0x72, 0x71, 0x77, 0xf1, 0xdc, 0xd1, 0xd5, 0xed, 0x66, 0x5b, 0x5b,
  0x5c, 0x5a, 0x63, 0x6a, 0x57, 0x4d, 0x4c, 0x4c, 0x4d, 0x4f, 0x52, 0x58,
  0x58, 0x50, 0x4a, 0x4a, 0x4e, 0x4d, 0x4a, 0x4c, 0x4d, 0x4b, 0x44, 0x40,
  0x43, 0x46, 0x46, 0x46, 0x4a, 0x48, 0x47, 0xcf, 0xb6, 0xbb, 0xbb, 0xb0,
  0xb8, 0xc5, 0xc7, 0xd8, 0xcd, 0xc1, 0xd9, 0x5d, 0x6a, 0x5b, 0x46, 0x44,
  0x59, 0xe1, 0xce, 0xc8, 0xcc, 0xc8, 0xc0, 0xca, 0xce, 0xc0, 0xbe, 0xc4,
  0xce, 0xea, 0x5c, 0x4e, 0x47, 0x42, 0x4b, 0x61, 0x5a, 0x52, 0x5d, 0x61,
  0x65, 0xf1, 0xdd, 0xcd, 0xc5, 0xc9, 0xd8, 0xe7, 0xf6, 0x5e, 0x56, 0x58,
  0x58, 0x57, 0x51, 0x47, 0x45, 0x4a, 0x4c, 0x4e, 0x62, 0xea, 0xdd, 0xdf,
  0xef, 0xe2, 0xd4, 0xd8, 0xd9, 0xce, 0xcd, 0xd0, 0xd8, 0xed, 0x76, 0xfc,
  0x72, 0x63, 0x70, 0xf3, 0x6b, 0x61, 0x66, 0x64, 0x7b, 0xe1, 0xe0, 0xdc,
  0xd4, 0xdc, 0x7c, 0x68, 0x60, 0x5d, 0x5c, 0x59, 0x54, 0x51, 0x4d, 0x47,
  0x45, 0x48, 0x4a, 0x4e, 0x53, 0x55, 0x54, 0x53, 0x4f, 0x4d, 0x4e, 0x51,
  0x53, 0x53, 0x51, 0x4f, 0x4f, 0x4d, 0x4b, 0x4d, 0x50, 0x4f, 0x4d, 0x48,
  0x45, 0xe3, 0xbc, 0xc6, 0xc5, 0xb5, 0xba, 0xc4, 0xc4, 0xcc, 0xc6, 0xbc,
  0xc6, 0xe0, 0xdd, 0xd9, 0x58, 0x46, 0x58, 0xfc, 0xeb, 0xd7, 0xe5, 0xe6,
  0xcc, 0xd2, 0xe8, 0xd0, 0xc0, 0xc0, 0xca, 0xcc, 0xd4, 0xf3, 0x6c, 0x53,
  0x4d, 0x68, 0x6c, 0x4f, 0x4d, 0x50, 0x4f, 0x4e, 0x54, 0x66, 0xe6, 0xd4,
  0xda, 0xf8, 0xeb, 0xe6, 0x79, 0x6b, 0xf8, 0xe2, 0xee, 0x63, 0x55, 0x4e,
  0x4f, 0x4e, 0x4c, 0x5e, 0xec, 0xfb, 0xf8, 0xdf, 0xd6, 0xd4, 0xd3, 0xcf,
  0xcc, 0xcb, 0xcf, 0xe1, 0xe9, 0xeb, 0x6a, 0x5f, 0x6b, 0x70, 0x6b, 0x66,
  0x5e, 0x5b, 0x5f, 0x61, 0x5b, 0x65, 0xfb, 0x6e, 0x5f, 0x5d, 0x58, 0x55,
  0x54, 0x51, 0x50, 0x54, 0x54, 0x4d, 0x49, 0x4a, 0x49, 0x49, 0x4b, 0x4c,
  0x4d, 0x4e, 0x4b, 0x4b, 0x4e, 0x53, 0x5c, 0x62, 0x6d, 0xf9, 0xfa, 0x69,
  0x5f, 0x66, 0x6e, 0x6c, 0x72, 0xea, 0xcf, 0xc7, 0xcb, 0xcc, 0xc4, 0xc4,
  0xc9, 0xc9, 0xc9, 0xc7, 0xc5, 0xca, 0xd3, 0xd8, 0xd5, 0xe2, 0x78, 0xee,
  0xe6, 0xea, 0xeb, 0xf6, 0xfe, 0xee, 0xe7, 0xee, 0xed, 0xda, 0xd5, 0xdb,
  0xdf, 0xe0, 0xe4, 0xe8, 0xf8, 0x6f, 0x79, 0xf8, 0x6f, 0x5e, 0x5d, 0x5e,
  0x5a, 0x56, 0x57, 0x5b, 0x5f, 0x63, 0x61, 0x64, 0x75, 0x7d, 0x7c, 0xea,
  0xe4, 0xdf, 0xd5, 0xd2, 0xd5, 0xd4, 0xd4, 0xdc, 0xe1, 0xdf, 0xec, 0xf9,
  0xf5, 0x7a, 0x6c, 0x68, 0x63, 0x62, 0x68, 0x71, 0x7e, 0x7c, 0x77, 0x72,
  0x65, 0x5f, 0x61, 0x5e, 0x5c, 0x5e, 0x5e, 0x58, 0x55, 0x51, 0x4d, 0x4c,
  0x4c, 0x4b, 0x4a, 0x4b, 0x4c, 0x4a, 0x4a, 0x4b, 0x4b, 0x4a, 0x4b, 0x4e,
  0x51, 0x55, 0x5b, 0x61, 0x6d, 0xff, 0xf5, 0xf2, 0xec, 0xe2, 0xdf, 0xe5,
  0xe8, 0xe4, 0xe0, 0xe5, 0xe2, 0xd8, 0xd1, 0xcd, 0xcb, 0xca, 0xca, 0xc9,
  0xcb, 0xce, 0xcd, 0xcc, 0xce, 0xcf, 0xd1, 0xd5, 0xd9, 0xdd, 0xe9, 0xf6,
  0xf2, 0xf6, 0x7c, 0xfa, 0xf6, 0xf5, 0xef, 0xf3, 0xfa, 0xf5, 0xee, 0xee,
  0xf1, 0xee, 0xea, 0xea, 0xed, 0xef, 0xf6, 0xfe, 0x77, 0x73, 0x71, 0x6f,
  0x6e, 0x6c, 0x6d, 0x6f, 0x6b, 0x6e, 0xf7, 0xed, 0xe7, 0xdd, 0xd9, 0xd9,
  0xd8, 0xd6, 0xda, 0xde, 0xdf, 0xe6, 0xea, 0xed, 0xfd, 0x6f, 0x6d, 0x68,
  0x5f, 0x5d, 0x5b, 0x59, 0x58, 0x57, 0x56, 0x53, 0x51, 0x4f, 0x4d, 0x4d,
  0x4d, 0x4d, 0x4c, 0x4d, 0x4d, 0x4d, 0x4d, 0x4d, 0x4e, 0x4d, 0x4d, 0x4d,
  0x4d, 0x4f, 0x52, 0x51, 0x54, 0x5b, 0x5e, 0x5e, 0x62, 0x6d, 0xfc, 0xee,
  0xe8, 0xdf, 0xdd, 0xdb, 0xdb, 0xdd, 0xdc, 0xda, 0xdc, 0xdc, 0xd7, 0xd1,
  0xcf, 0xce, 0xcc, 0xcb, 0xcc, 0xcc, 0xce, 0xd0, 0xd2, 0xd4, 0xd7, 0xd8,
  0xd8, 0xd6, 0xd9, 0xdc, 0xdd, 0xe2, 0xe8, 0xe7, 0xea, 0xeb, 0xe6, 0xe4,
  0xe8, 0xec, 0xf0, 0xf7, 0xf9, 0x7a, 0x6f, 0x71, 0x78, 0x77, 0x79, 0xfc,
  0xfc, 0xf9, 0xf3, 0xf2, 0xeb, 0xe0, 0xdd, 0xdd, 0xda, 0xd8, 0xdb, 0xde,
  0xdf, 0xe5, 0xef, 0xf7, 0xf9, 0x7c, 0x79, 0x7a, 0x6e, 0x6b, 0x68, 0x61,
  0x5d, 0x5a, 0x58, 0x57, 0x55, 0x54, 0x53, 0x52, 0x4f, 0x4e, 0x4e, 0x4d,
  0x4c, 0x4c, 0x4d, 0x4e, 0x50, 0x51, 0x50, 0x51, 0x53, 0x53, 0x53, 0x52,
  0x53, 0x54, 0x56, 0x59, 0x5c, 0x60, 0x66, 0x69, 0x6e, 0x75, 0x7c, 0xfc,
  0xf6, 0xee, 0xe9, 0xe2, 0xde, 0xdd, 0xdc, 0xdb, 0xda, 0xdb, 0xd9, 0xd7,
  0xd5, 0xd3, 0xcf, 0xce, 0xcd, 0xcd, 0xcd, 0xcc, 0xce, 0xd1, 0xd3, 0xd4,
  0xd7, 0xda, 0xdb, 0xdb, 0xdb, 0xdc, 0xdd, 0xe1, 0xe4, 0xeb, 0xf5, 0xff,
  0x76, 0x71, 0x6f, 0x6e, 0x72, 0x74, 0x6f, 0x72, 0x72, 0x6d, 0x6d, 0x6e,
  0x72, 0x7c, 0xf9, 0xea, 0xdf, 0xdb, 0xd8, 0xd6, 0xd5, 0xd6, 0xdb, 0xe0,
  0xe6, 0xee, 0xf3, 0xfc, 0x70, 0x6b, 0x6a, 0x68, 0x65, 0x60, 0x5e, 0x5b,
  0x58, 0x58, 0x57, 0x55, 0x54, 0x54, 0x56, 0x56, 0x57, 0x5a, 0x5a, 0x5b,
  0x5b, 0x59, 0x57, 0x56, 0x55, 0x55, 0x56, 0x56, 0x59, 0x5c, 0x5d, 0x5d,
  0x5d, 0x5d, 0x5c, 0x5b, 0x5c, 0x5e, 0x5e, 0x5f, 0x62, 0x64, 0x6a, 0x71,
  0x77, 0xfa, 0xee, 0xeb, 0xe8, 0xe7, 0xe4, 0xe3, 0xe3, 0xe1, 0xdf, 0xde,
  0xdc, 0xda, 0xd8, 0xd7, 0xd7, 0xd4, 0xd2, 0xd2, 0xd2, 0xd0, 0xd1, 0xd3,
  0xd2, 0xd4, 0xd7, 0xd7, 0xd6, 0xd9, 0xdb, 0xda, 0xdc, 0xdf, 0xe1, 0xe6,
  0xe9, 0xed, 0xfb, 0x75, 0x6e, 0x6a, 0x69, 0x67, 0x66, 0x6b, 0x71, 0x7d,
  0xf4, 0xed, 0xe5, 0xe2, 0xe1, 0xde, 0xde, 0xe1, 0xe8, 0xef, 0xf9, 0x79,
  0x70, 0x6d, 0x6c, 0x6d, 0x6f, 0x6e, 0x68, 0x67, 0x68, 0x60, 0x5d, 0x5d,
  0x5c, 0x5a, 0x59, 0x59, 0x5a, 0x5b, 0x5b, 0x5c, 0x5e, 0x5e, 0x5d, 0x5e,
  0x5d, 0x5d, 0x5d, 0x5d, 0x5e, 0x5f, 0x5f, 0x5e, 0x5f, 0x5f, 0x5f, 0x5f,
  0x60, 0x62, 0x64, 0x67, 0x69, 0x6b, 0x6d, 0x6d, 0x6e, 0x74, 0x7b, 0x7c,
  0xfe, 0xf8, 0xf3, 0xed, 0xe9, 0xe6, 0xe3, 0xe3, 0xe1, 0xdf, 0xe1, 0xe1,
  0xdf, 0xe0, 0xe4, 0xe5, 0xe5, 0xe7, 0xe4, 0xe2, 0xe3, 0xe0, 0xdd, 0xdb,
  0xda, 0xd8, 0xd7, 0xd7, 0xd6, 0xd6, 0xd7, 0xd8, 0xda, 0xde, 0xdf, 0xdf,
  0xe3, 0xe7, 0xe5, 0xe2, 0xe0, 0xe3, 0xe7, 0xe7, 0xe9, 0xed, 0xf6, 0x7e,
  0x79, 0x73, 0x6e, 0x6e, 0x6d, 0x6c, 0x6b, 0x67, 0x67, 0x6a, 0x68, 0x67,
  0x67, 0x65, 0x64, 0x64, 0x64, 0x63, 0x64, 0x66, 0x66, 0x68, 0x69, 0x67,
  0x67, 0x67, 0x67, 0x67, 0x65, 0x66, 0x68, 0x69, 0x69, 0x6d, 0x6d, 0x6b,
  0x6d, 0x6d, 0x6c, 0x6b, 0x69, 0x6d, 0x6e, 0x6c, 0x6a, 0x6a, 0x6b, 0x68,
  0x68, 0x6b, 0x6a, 0x6a, 0x6f, 0x73, 0x6f, 0x72, 0x77, 0x75, 0x75, 0x78,
  0x7b, 0x7a, 0xfe, 0xfb, 0xf9, 0xf5, 0xf4, 0xed, 0xe9, 0xea, 0xe9, 0xe6,
  0xe3, 0xe3, 0xe4, 0xe0, 0xdf, 0xe2, 0xe4, 0xe3, 0xe3, 0xe7, 0xe6, 0xe2,
  0xe2, 0xe4, 0xe2, 0xe0, 0xdf, 0xdf, 0xdf, 0xdf, 0xde, 0xde, 0xdf, 0xe3,
  0xe2, 0xe1, 0xe8, 0xed, 0xee, 0xee, 0xf0, 0xf5, 0xf6, 0xf6, 0xf8, 0xfd,
  0xfd, 0xfc, 0xff, 0x78, 0x73, 0x74, 0x74, 0x71, 0x71, 0x72, 0x75, 0x79,
  0x7b, 0x7e, 0xfb, 0xf5, 0xf5, 0xfa, 0xfd, 0x7e, 0x75, 0x70, 0x70, 0x70,
  0x70, 0x70, 0x6f, 0x70, 0x75, 0x71, 0x6f, 0x70, 0x6f, 0x6c, 0x6b, 0x6b,
  0x6b, 0x6a, 0x68, 0x68, 0x67, 0x67, 0x67, 0x66, 0x68, 0x69, 0x67, 0x69,
  0x68, 0x67, 0x6a, 0x6a, 0x6b, 0x6b, 0x6c, 0x6e, 0x6c, 0x6a, 0x6a, 0x6c,
  0x6b, 0x6a, 0x6d, 0x6f, 0x6f, 0x6f, 0x70, 0x72, 0x72, 0x76, 0x7c, 0xfd,
  0xf5, 0xf0, 0xf2, 0xf4, 0xf2, 0xf0, 0xf1, 0xf9, 0xfb, 0xf1, 0xf1, 0xf2,
  0xec, 0xe8, 0xe7, 0xe4, 0xdf, 0xde, 0xdc, 0xd9, 0xd8, 0xd8, 0xd8, 0xd7,
  0xd8, 0xda, 0xdb, 0xde, 0xe2, 0xe3, 0xe4, 0xe4, 0xe6, 0xe8, 0xe7, 0xe7,
  0xe9, 0xec, 0xef, 0xf2, 0xf3, 0xfb, 0x79, 0x76, 0x77, 0x76, 0x70, 0x6f,
  0x6f, 0x6c, 0x6d, 0x6e, 0x6f, 0x71, 0x70, 0x75, 0x7e, 0x7c, 0x77, 0xfe,
  0xf5, 0xf8, 0xf9, 0xf8, 0xfa, 0xfb, 0xfe, 0xfb, 0xfa, 0x7d, 0xfd, 0xfa,
  0xf9, 0xfd, 0x7b, 0x7c, 0x77, 0x72, 0x72, 0x70, 0x6d, 0x6b, 0x69, 0x66,
  0x67, 0x65, 0x63, 0x62, 0x60, 0x62, 0x63, 0x62, 0x62, 0x61, 0x62, 0x61,
  0x60, 0x61, 0x60, 0x61, 0x61, 0x61, 0x63, 0x66, 0x67, 0x68, 0x6b, 0x6d,
  0x6e, 0x74, 0x7b, 0xfd, 0xfa, 0xf7, 0xf2, 0xef, 0xee, 0xee, 0xed, 0xec,
  0xea, 0xe9, 0xe8, 0xe6, 0xe6, 0xe7, 0xe9, 0xeb, 0xea, 0xe9, 0xe6, 0xe4,
  0xe3, 0xe0, 0xdf, 0xde, 0xdd, 0xde, 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xe0,
  0xe2, 0xe4, 0xe9, 0xe8, 0xe6, 0xe9, 0xeb, 0xeb, 0xeb, 0xef, 0xf2, 0xf0,
  0xf4, 0xfa, 0xfe, 0x77, 0x74, 0x74, 0x71, 0x6f, 0x6f, 0x71, 0x6f, 0x6e,
  0x71, 0x73, 0x76, 0x7c, 0xff, 0xfe, 0xfa, 0xfb, 0xfb, 0xf5, 0xf5, 0xfa,
  0xfb, 0xfc, 0xfe, 0x7e, 0xff, 0x79, 0x74, 0x76, 0x79, 0x75, 0x74, 0x7a,
  0x78, 0x75, 0x75, 0x72, 0x6f, 0x6d, 0x6d, 0x6d, 0x69, 0x65, 0x63, 0x63,
  0x61, 0x5e, 0x5f, 0x60, 0x60, 0x60, 0x61, 0x5f, 0x5e, 0x5f, 0x5f, 0x60,
  0x63, 0x65, 0x68, 0x69, 0x6a, 0x6c, 0x6c, 0x6e, 0x70, 0x76, 0x7b, 0xfe,
  0xf5, 0xef, 0xef, 0xef, 0xec, 0xea, 0xeb, 0xe9, 0xe8, 0xec, 0xf0, 0xf4,
  0xef, 0xeb, 0xe9, 0xea, 0xe6, 0xe0, 0xde, 0xdd, 0xdc, 0xda, 0xd9, 0xd9,
  0xdb, 0xdd, 0xdd, 0xde, 0xe2, 0xe2, 0xe1, 0xe3, 0xe6, 0xe5, 0xe4, 0xe6,
  0xe7, 0xe7, 0xe9, 0xed, 0xef, 0xf6, 0xfe, 0xff, 0x7b, 0x72, 0x72, 0x73,
  0x6f, 0x6e, 0x6f, 0x73, 0x6f, 0x6d, 0x6c, 0x6b, 0x6a, 0x6a, 0x6e, 0x73,
  0x75, 0x7a, 0x7e, 0xfa, 0xf7, 0xf2, 0xef, 0xee, 0xee, 0xf0, 0xf2, 0xf4,
  0xf9, 0xf9, 0xfc, 0xfe, 0x7d, 0x79, 0x77, 0x72, 0x73, 0x71, 0x6d, 0x6d,
  0x6d, 0x68, 0x65, 0x67, 0x66, 0x60, 0x5f, 0x5e, 0x5f, 0x5e, 0x5d, 0x5d,
  0x5e, 0x5e, 0x5e, 0x5e, 0x60, 0x61, 0x62, 0x66, 0x66, 0x64, 0x67, 0x69,
  0x66, 0x67, 0x6a, 0x6b, 0x6a, 0x6c, 0x71, 0x72, 0x74, 0x76, 0x75, 0x7a,
  0x79, 0x7a, 0xf8, 0xef, 0xeb, 0xe7, 0xdf, 0xdc, 0xda, 0xd7, 0xd4, 0xd2,
  0xd0, 0xd0, 0xd1, 0xd3, 0xd3, 0xd6, 0xda, 0xdd, 0xde, 0xdf, 0xe5, 0xe5,
  0xe5, 0xe9, 0xeb, 0xeb, 0xef, 0xf2, 0xf5, 0x7d, 0x72, 0x6f, 0x6e, 0x6d,
  0x6c, 0x6e, 0x73, 0x72, 0x72, 0x73, 0x75, 0x78, 0x77, 0x74, 0x74, 0x72,
  0x72, 0x73, 0x6f, 0x6f, 0x72, 0x74, 0x7d, 0xf3, 0xef, 0xec, 0xe7, 0xe5,
  0xe6, 0xe6, 0xe7, 0xe9, 0xea, 0xee, 0xfa, 0xfd, 0xfd, 0x7a, 0x79, 0xfe,
  0xfd, 0x78, 0x74, 0x71, 0x6b, 0x66, 0x65, 0x65, 0x5f, 0x5e, 0x5f, 0x5e,
  0x5d, 0x5d, 0x5e, 0x5e, 0x5d, 0x5e, 0x5e, 0x5f, 0x5f, 0x5e, 0x5d, 0x5d,
  0x5d, 0x5d, 0x5d, 0x5d, 0x5c, 0x5c, 0x5c, 0x5e, 0x5e, 0x5e, 0x60, 0x5f,
  0x62, 0x67, 0x68, 0x6d, 0xfd, 0xee, 0xe7, 0xdd, 0xd6, 0xd2, 0xcf, 0xcd,
  0xcb, 0xcb, 0xcb, 0xca, 0xcb, 0xcd, 0xcf, 0xd0, 0xd4, 0xd8, 0xdb, 0xdf,
  0xe9, 0xea, 0xed, 0xfc, 0x7b, 0x7a, 0x71, 0x6b, 0x68, 0x65, 0x60, 0x5f,
  0x5f, 0x5f, 0x5f, 0x63, 0x67, 0x6b, 0x77, 0xf8, 0xef, 0xec, 0xe8, 0xe5,
  0xe6, 0xe9, 0xed, 0xf4, 0xfe, 0x76, 0x6f, 0x6f, 0x72, 0x79, 0xfb, 0xee,
  0xe7, 0xe2, 0xdf, 0xde, 0xde, 0xe0, 0xe7, 0xef, 0xfd, 0x74, 0x6e, 0x71,
  0x72, 0x75, 0x7c, 0xfc, 0xfa, 0x7e, 0x77, 0x6f, 0x67, 0x60, 0x5d, 0x5a,
  0x58, 0x58, 0x57, 0x57, 0x5a, 0x5b, 0x5b, 0x5c, 0x5c, 0x5c, 0x59, 0x59,
  0x59, 0x57, 0x56, 0x55, 0x55, 0x55, 0x56, 0x56, 0x57, 0x5a, 0x5b, 0x5d,
  0x61, 0x64, 0x64, 0x68, 0x6f, 0x7c, 0xee, 0xe4, 0xdc, 0xd3, 0xce, 0xcc,
  0xc9, 0xc6, 0xc5, 0xc5, 0xc5, 0xc6, 0xc9, 0xcb, 0xce, 0xd4, 0xd8, 0xdc,
  0xe5, 0xed, 0xf4, 0xfd, 0x73, 0x6a, 0x66, 0x64, 0x5f, 0x5d, 0x5c, 0x5b,
  0x5c, 0x5c, 0x5d, 0x64, 0x6f, 0x7b, 0xf7, 0xe6, 0xdf, 0xdf, 0xdc, 0xdb,
  0xdc, 0xdf, 0xe5, 0xe9, 0xf2, 0xfe, 0x74, 0x6b, 0x6a, 0x68, 0x63, 0x61,
  0x65, 0x67, 0x63, 0x63, 0x67, 0x6c, 0x73, 0x7c, 0xef, 0xe3, 0xdf, 0xdc,
  0xd8, 0xd4, 0xd4, 0xd6, 0xd8, 0xdb, 0xe0, 0xe8, 0xf7, 0x6e, 0x66, 0x5e,
  0x5a, 0x59, 0x58, 0x59, 0x57, 0x57, 0x59, 0x58, 0x56, 0x55, 0x54, 0x53,
  0x50, 0x50, 0x51, 0x53, 0x54, 0x55, 0x57, 0x58, 0x57, 0x55, 0x54, 0x53,
  0x51, 0x4f, 0x50, 0x50, 0x50, 0x52, 0x52, 0x57, 0x69, 0x7c, 0xfa, 0xd8,
  0xca, 0xc9, 0xc7, 0xbf, 0xbf, 0xc0, 0xbf, 0xc1, 0xc4, 0xc5, 0xca, 0xce,
  0xcf, 0xd2, 0xd7, 0xe1, 0xe6, 0xe6, 0x74, 0x60, 0x63, 0x5c, 0x53, 0x52,
  0x52, 0x53, 0x58, 0x5c, 0x5f, 0x74, 0xe6, 0xde, 0xdd, 0xd7, 0xd1, 0xd5,
  0xd9, 0xd7, 0xda, 0xe2, 0xe2, 0xe6, 0xee, 0xf6, 0x77, 0x6a, 0x65, 0x5f,
  0x58, 0x51, 0x50, 0x4f, 0x4d, 0x4c, 0x4f, 0x53, 0x56, 0x60, 0x7a, 0xec,
  0xdf, 0xd6, 0xd0, 0xcd, 0xcc, 0xcc, 0xcb, 0xcb, 0xcd, 0xcf, 0xd3, 0xd6,
  0xdd, 0xe9, 0xf1, 0x76, 0x63, 0x5d, 0x58, 0x53, 0x52, 0x50, 0x4e, 0x4f,
  0x51, 0x51, 0x4f, 0x52, 0x57, 0x56, 0x53, 0x55, 0x57, 0x55, 0x54, 0x54,
  0x53, 0x51, 0x4f, 0x4d, 0x4e, 0x4d, 0x4b, 0x4a, 0x4c, 0x4c, 0x4b, 0x4c,
  0x4d, 0x4d, 0x56, 0x7a, 0xf8, 0xde, 0xc6, 0xc0, 0xc1, 0xbb, 0xb8, 0xbb,
  0xbc, 0xbd, 0xc1, 0xc5, 0xca, 0xce, 0xd5, 0xd8, 0xd8, 0xee, 0x75, 0xf0,
  0x62, 0x51, 0x55, 0x52, 0x4d, 0x4f, 0x55, 0x5c, 0x6e, 0xe7, 0xd8, 0xd2,
  0xcb, 0xc9, 0xce, 0xcd, 0xcc, 0xd6, 0xde, 0xdc, 0xe1, 0xf0, 0xfd, 0x75,
  0x69, 0x5d, 0x56, 0x4e, 0x4b, 0x4a, 0x47, 0x44, 0x49, 0x4e, 0x4f, 0x58,
  0x6e, 0xef, 0xe4, 0xdb, 0xd5, 0xd2, 0xcf, 0xce, 0xd0, 0xcd, 0xc8, 0xcc,
  0xcd, 0xc9, 0xcd, 0xd9, 0xdf, 0xf4, 0x61, 0x58, 0x56, 0x51, 0x51, 0x57,
  0x58, 0x59, 0x66, 0x67, 0x5d, 0x5e, 0x61, 0x59, 0x54, 0x56, 0x54, 0x52,
  0x51, 0x51, 0x4f, 0x4f, 0x4d, 0x48, 0x46, 0x45, 0x3f, 0x3e, 0x3f, 0x40,
  0x41, 0x44, 0x49, 0x4c, 0x4e, 0x4f, 0x53, 0xfe, 0xd3, 0xd5, 0xc7, 0xb7,
  0xb7, 0xb9, 0xb4, 0xb5, 0xb9, 0xbd, 0xc7, 0xce, 0xd7, 0xef, 0x66, 0x5f,
  0x7d, 0x64, 0x4e, 0x5c, 0x5e, 0x4c, 0x4e, 0x53, 0x58, 0x6f, 0xed, 0xd7,
  0xc9, 0xc1, 0xbf, 0xc4, 0xbf, 0xc0, 0xcf, 0xd7, 0xd5, 0xe8, 0x76, 0x67,
  0x62, 0x61, 0x58, 0x4f, 0x4b, 0x4a, 0x49, 0x42, 0x44, 0x4d, 0x4f, 0x58,
  0xf7, 0xdb, 0xd3, 0xd1, 0xd2, 0xd3, 0xd4, 0xd6, 0xe0, 0xdd, 0xcf, 0xd3,
  0xdb, 0xcf, 0xcf, 0xdd, 0xed, 0xfb, 0x6d, 0x5f, 0x5a, 0x5c, 0x68, 0xf9,
  0xff, 0xfc, 0xde, 0xe2, 0x6e, 0x69, 0x69, 0x5d, 0x56, 0x54, 0x56, 0x55,
  0x52, 0x4d, 0x49, 0x49, 0x43, 0x3e, 0x3e, 0x3f, 0x3d, 0x3c, 0x3e, 0x41,
  0x41, 0x41, 0x43, 0x48, 0x4a, 0x49, 0x4b, 0x51, 0x72, 0xcd, 0xc6, 0xbf,
  0xb3, 0xb1, 0xb7, 0xb6, 0xb8, 0xc0, 0xcb, 0xd7, 0xe6, 0xea, 0xf8, 0x6e,
  0x66, 0xfc, 0x6a, 0x4b, 0x4f, 0x59, 0x4c, 0x53, 0xfd, 0xdb, 0xca, 0xc3,
  0xbf, 0xbd, 0xbd, 0xc1, 0xcd, 0xce, 0xcf, 0xfb, 0xf9, 0xe5, 0xfe, 0x77,
  0x63, 0x59, 0x53, 0x4c, 0x47, 0x44, 0x4b, 0x55, 0x55, 0x78, 0xd4, 0xd1,
  0xcf, 0xd1, 0xd6, 0xda, 0xea, 0x72, 0x7b, 0xe8, 0xe8, 0xec, 0xdc, 0xdc,
  0xf9, 0x70, 0x6b, 0x60, 0x61, 0x6d, 0xef, 0xdf, 0xd4, 0xd1, 0xd4, 0xd3,
  0xdb, 0x7d, 0x6d, 0x65, 0x59, 0x57, 0x5d, 0x5f, 0x5b, 0x58, 0x50, 0x4a,
  0x46, 0x40, 0x3c, 0x3d, 0x3d, 0x3c, 0x3e, 0x40, 0x3f, 0x3e, 0x3e, 0x3d,
  0x3d, 0x3e, 0x41, 0x45, 0x4b, 0x57, 0x5e, 0xdc, 0xbf, 0xbf, 0xbf, 0xb3,
  0xb2, 0xbb, 0xb9, 0xbc, 0xc5, 0xca, 0xd9, 0xf0, 0xe9, 0xee, 0x5c, 0x52,
  0x6c, 0x5b, 0x4a, 0x5a, 0x6f, 0x6b, 0xdc, 0xce, 0xc8, 0xbf, 0xbe, 0xc2,
  0xc2, 0xc0, 0xca, 0xdb, 0xd2, 0xd9, 0x76, 0x7e, 0x6c, 0x5e, 0x5b, 0x4f,
  0x4b, 0x4e, 0x53, 0x4f, 0x56, 0xeb, 0xdf, 0xe0, 0xd1, 0xcf, 0xd3, 0xd7,
  0xe2, 0xe9, 0xe2, 0xe6, 0x7d, 0xf3, 0xe1, 0xf9, 0x67, 0x6a, 0x64, 0x61,
  0x66, 0x6a, 0xf0, 0xdb, 0xdc, 0xdf, 0xd8, 0xda, 0xf7, 0x70, 0x76, 0x68,
  0x61, 0x61, 0x5d, 0x5e, 0x5c, 0x4f, 0x4b, 0x4b, 0x46, 0x3f, 0x3f, 0x40,
  0x3e, 0x3d, 0x3e, 0x3d, 0x3c, 0x3c, 0x3a, 0x3b, 0x3e, 0x3f, 0x41, 0x49,
  0x4e, 0x4f, 0x51, 0xf0, 0xc2, 0xbe, 0xbd, 0xb1, 0xaf, 0xb6, 0xb9, 0xbd,
  0xc9, 0xcf, 0xec, 0x6c, 0xea, 0xe5, 0x6b, 0x59, 0x6e, 0x62, 0x4a, 0x51,
  0x65, 0x7b, 0xd2, 0xcb, 0xc2, 0xbb, 0xbc, 0xc2, 0xc6, 0xc8, 0xd4, 0xff,
  0xec, 0xeb, 0x72, 0x6e, 0x5d, 0x5a, 0x57, 0x4a, 0x47, 0x4d, 0x57, 0x5c,
  0x79, 0xd3, 0xcc, 0xcc, 0xcd, 0xd4, 0xd5, 0xdb, 0xef, 0xdf, 0xd4, 0xd7,
  0xde, 0xdd, 0xe4, 0x6c, 0x60, 0x5d, 0x5e, 0x73, 0xfe, 0xf1, 0xdc, 0xd9,
  0xeb, 0xfc, 0xf3, 0x6b, 0x5c, 0x5e, 0x5f, 0x5e, 0x5c, 0x54, 0x4e, 0x4a,
  0x43, 0x3e, 0x3d, 0x3e, 0x3c, 0x3c, 0x3d, 0x3c, 0x3a, 0x38, 0x36, 0x37,
  0x3a, 0x3c, 0x3e, 0x49, 0x53, 0x50, 0x5e, 0xcc, 0xbc, 0xbc, 0xb8, 0xae,
  0xaf, 0xb8, 0xba, 0xc1, 0xca, 0xd2, 0x68, 0x68, 0xe5, 0x77, 0x57, 0x59,
  0x72, 0x59, 0x4e, 0x6d, 0xe4, 0xd1, 0xc6, 0xc4, 0xbc, 0xba, 0xc0, 0xc9,
  0xcb, 0xce, 0xef, 0x63, 0x6f, 0x64, 0x5b, 0x52, 0x4b, 0x4f, 0x4d, 0x46,
  0x4c, 0x61, 0xfd, 0xe7, 0xd1, 0xc8, 0xc8, 0xca, 0xcf, 0xd4, 0xd0, 0xd7,
  0xde, 0xd5, 0xd1, 0xdc, 0xe9, 0xf6, 0x71, 0x69, 0x61, 0x65, 0xeb, 0xde,
  0xe6, 0xe2, 0xdb, 0xe4, 0x76, 0x6e, 0x6c, 0x65, 0x61, 0x5a, 0x57, 0x54,
  0x4a, 0x42, 0x3e, 0x3c, 0x39, 0x37, 0x37, 0x37, 0x36, 0x36, 0x34, 0x34,
  0x36, 0x36, 0x39, 0x3d, 0x43, 0x4a, 0x78, 0xbc, 0xb6, 0xb8, 0xae, 0xac,
  0xb5, 0xb9, 0xbe, 0xc9, 0xca, 0xdb, 0x76, 0xe9, 0xec, 0x55, 0x46, 0x4f,
  0x50, 0x48, 0x5e, 0xdb, 0xcb, 0xbd, 0xbd, 0xbd, 0xb8, 0xbd, 0xc8, 0xcb,
  0xca, 0xd3, 0xf4, 0xf1, 0x6b, 0x53, 0x4c, 0x40, 0x3f, 0x45, 0x42, 0x46,
  0x5b, 0xef, 0xde, 0xd4, 0xcd, 0xcb, 0xcb, 0xce, 0xcf, 0xc7, 0xc4, 0xcb,
  0xcb, 0xca, 0xd5, 0xf0, 0x6d, 0x65, 0x6a, 0x6d, 0x69, 0xf1, 0xda, 0xe6,
  0x78, 0xef, 0xf2, 0x68, 0x66, 0x6b, 0x69, 0x65, 0x57, 0x4c, 0x48, 0x3f,
  0x3a, 0x37, 0x36, 0x34, 0x33, 0x33, 0x33, 0x32, 0x30, 0x2f, 0x33, 0x36,
  0x37, 0x41, 0xce, 0xb7, 0xb7, 0xaf, 0xaa, 0xae, 0xb8, 0xbd, 0xc8, 0xc8,
  0xcc, 0xf0, 0xde, 0xd2, 0x64, 0x46, 0x45, 0x49, 0x43, 0x4a, 0xfa, 0xce,
  0xbc, 0xb9, 0xbc, 0xb7, 0xb9, 0xc5, 0xcb, 0xc7, 0xc9, 0xd4, 0xdd, 0xeb,
  0x5f, 0x4e, 0x3f, 0x3c, 0x41, 0x43, 0x43, 0x58, 0xe2, 0xda, 0xd5, 0xcf,
  0xce, 0xcc, 0xcc, 0xcb, 0xc4, 0xbf, 0xc2, 0xca, 0xcb, 0xd6, 0x69, 0x59,
  0x58, 0x58, 0x5f, 0x67, 0xfc, 0xde, 0xe5, 0x73, 0x71, 0xf9, 0x6e, 0x63,
  0x79, 0x7c, 0x62, 0x52, 0x45, 0x3e, 0x3a, 0x34, 0x32, 0x32, 0x32, 0x31,
  0x32, 0x32, 0x31, 0x31, 0x33, 0x35, 0x3f, 0xd7, 0xba, 0xb7, 0xaf, 0xaa,
  0xaf, 0xb9, 0xbf, 0xce, 0xd0, 0xd3, 0xee, 0xdb, 0xd8, 0x5f, 0x48, 0x44,
  0x45, 0x42, 0x4d, 0xe7, 0xc9, 0xbb, 0xb8, 0xba, 0xb8, 0xbb, 0xc8, 0xcb,
  0xca, 0xcd, 0xd6, 0xdd, 0xf8, 0x59, 0x49, 0x3d, 0x3c, 0x40, 0x43, 0x49,
  0x6b, 0xda, 0xd3, 0xce, 0xce, 0xcc, 0xc4, 0xc5, 0xc6, 0xbd, 0xbb, 0xc1,
  0xc8, 0xcf, 0xe7, 0x62, 0x53, 0x4f, 0x5b, 0x6d, 0x62, 0x72, 0xe3, 0x6e,
  0x59, 0x5c, 0x5b, 0x57, 0x54, 0x50, 0x4c, 0x46, 0x3d, 0x37, 0x34, 0x31,
  0x2e, 0x2e, 0x2f, 0x2f, 0x2f, 0x31, 0x34, 0x37, 0x45, 0xcc, 0xbc, 0xb8,
  0xad, 0xac, 0xb1, 0xb6, 0xbf, 0xca, 0xc9, 0xd8, 0xeb, 0xd6, 0xde, 0x56,
  0x49, 0x4a, 0x48, 0x47, 0x5b, 0xdf, 0xc6, 0xba, 0xbc, 0xbb, 0xb8, 0xbe,
  0xca, 0xcb, 0xcd, 0xd3, 0xdf, 0xfc, 0x5e, 0x50, 0x46, 0x3d, 0x3e, 0x43,
  0x45, 0x4e, 0x7e, 0xd8, 0xce, 0xcc, 0xc7, 0xbf, 0xbf, 0xc1, 0xbe, 0xbb,
  0xbd, 0xc6, 0xcd, 0xd2, 0xef, 0x59, 0x51, 0x53, 0x5c, 0x5b, 0x5a, 0x75,
  0x7b, 0x5e, 0x58, 0x59, 0x57, 0x4d, 0x48, 0x44, 0x3f, 0x3a, 0x34, 0x30,
  0x2f, 0x2d, 0x2c, 0x2d, 0x2e, 0x30, 0x31, 0x3a, 0xdd, 0xbf, 0xbf, 0xb3,
  0xad, 0xb2, 0xb8, 0xbd, 0xc7, 0xc4, 0xcb, 0xee, 0xd9, 0xd7, 0x56, 0x46,
  0x4a, 0x4b, 0x4a, 0x5b, 0xdd, 0xc9, 0xbb, 0xbc, 0xbe, 0xb7, 0xbc, 0xc8,
  0xc6, 0xc6, 0xcd, 0xd9, 0xfa, 0x5d, 0x52, 0x46, 0x3d, 0x3f, 0x47, 0x46,
  0x4b, 0x6d, 0xe1, 0xd8, 0xce, 0xca, 0xc2, 0xbd, 0xc0, 0xbf, 0xbb, 0xbe,
  0xc9, 0xcf, 0xda, 0xfb, 0x5c, 0x53, 0x55, 0x5f, 0x5e, 0x59, 0x6e, 0xff,
  0x5c, 0x56, 0x57, 0x50, 0x4a, 0x42, 0x3d, 0x3b, 0x37, 0x30, 0x2e, 0x2f,
  0x2d, 0x2c, 0x2f, 0x30, 0x33, 0x4e, 0xc8, 0xc5, 0xb9, 0xae, 0xb3, 0xb9,
  0xba, 0xca, 0xcc, 0xc8, 0xef, 0xf1, 0xd4, 0x62, 0x49, 0x4c, 0x4c, 0x46,
  0x56, 0xe6, 0xd2, 0xbe, 0xbb, 0xbe, 0xb9, 0xba, 0xc6, 0xc7, 0xc6, 0xcf,
  0xda, 0xeb, 0x5b, 0x4f, 0x49, 0x3d, 0x3d, 0x43, 0x43, 0x49, 0x65, 0xe2,
  0xd1, 0xc8, 0xc6, 0xbf, 0xbb, 0xbd, 0xbe, 0xba, 0xbc, 0xc5, 0xcb, 0xd6,
  0xf8, 0x64, 0x58, 0x55, 0x61, 0x66, 0x5b, 0x6c, 0xf6, 0x5c, 0x54, 0x53,
  0x4a, 0x44, 0x3d, 0x38, 0x35, 0x31, 0x2e, 0x2d, 0x2d, 0x2d, 0x2e, 0x30,
  0x3b, 0xdf, 0xca, 0xc7, 0xb2, 0xb0, 0xbc, 0xbd, 0xc5, 0xcf, 0xca, 0xdc,
  0xfd, 0xd3, 0xe0, 0x4f, 0x4c, 0x54, 0x4c, 0x4d, 0xef, 0xcf, 0xc2, 0xba,
  0xbc, 0xba, 0xb7, 0xc1, 0xcb, 0xc5, 0xcc, 0xdb, 0xe8, 0x6b, 0x56, 0x4b,
  0x3f, 0x3c, 0x41, 0x44, 0x44, 0x54, 0xe2, 0xd5, 0xcd, 0xc8, 0xc3, 0xbf,
  0xbf, 0xc1, 0xbe, 0xbd, 0xc5, 0xce, 0xd3, 0xe9, 0x5c, 0x54, 0x54, 0x5a,
  0x64, 0x63, 0x7c, 0xe5, 0x70, 0x57, 0x51, 0x4c, 0x40, 0x39, 0x35, 0x34,
  0x31, 0x2d, 0x2d, 0x2f, 0x2f, 0x2f, 0x3f, 0xd9, 0xcd, 0xc3, 0xb3, 0xb4,
  0xbb, 0xbe, 0xcf, 0xd4, 0xcc, 0x75, 0x6c, 0xd5, 0x71, 0x4c, 0x4e, 0x4f,
  0x4b, 0x51, 0x78, 0xd4, 0xbf, 0xbb, 0xbd, 0xb8, 0xb7, 0xc3, 0xca, 0xc9,
  0xd3, 0xdf, 0xfa, 0x5d, 0x5a, 0x4e, 0x3f, 0x3f, 0x46, 0x44, 0x46, 0x5b,
  0xe0, 0xcd, 0xc7, 0xc5, 0xbf, 0xbd, 0xc2, 0xc5, 0xbf, 0xc2, 0xca, 0xce,
  0xd5, 0xec, 0x60, 0x54, 0x51, 0x59, 0x5b, 0x58, 0x69, 0xf9, 0x5c, 0x4e,
  0x49, 0x41, 0x3a, 0x33, 0x2f, 0x30, 0x2e, 0x2c, 0x2f, 0x34, 0x39, 0x5a,
  0xd1, 0xce, 0xba, 0xb5, 0xbe, 0xbd, 0xbf, 0xd5, 0xd0, 0xd6, 0x7a, 0xe8,
  0xf6, 0x52, 0x4e, 0x54, 0x4d, 0x4e, 0xf3, 0xd3, 0xc8, 0xbc, 0xba, 0xb9,
  0xb8, 0xbe, 0xc5, 0xc5, 0xd0, 0xee, 0xf9, 0x5f, 0x4f, 0x4b, 0x43, 0x3f,
  0x41, 0x41, 0x43, 0x4e, 0x73, 0xdc, 0xcc, 0xc1, 0xbe, 0xbd, 0xbd, 0xc1,
  0xc1, 0xc3, 0xce, 0xd5, 0xd9, 0x7b, 0x5d, 0x58, 0x51, 0x50, 0x53, 0x56,
  0x5f, 0x6d, 0x5c, 0x50, 0x4d, 0x43, 0x3a, 0x34, 0x2f, 0x2e, 0x2e, 0x2e,
  0x32, 0x3b, 0x5f, 0xd0, 0xcf, 0xbc, 0xb7, 0xc4, 0xc5, 0xc6, 0xd8, 0xcf,
  0xd5, 0xf8, 0xda, 0xf5, 0x4d, 0x4d, 0x4e, 0x48, 0x4b, 0x78, 0xd1, 0xc6,
  0xbc, 0xba, 0xb9, 0xb8, 0xbf, 0xc7, 0xc5, 0xcf, 0xef, 0xee, 0x6c, 0x55,
  0x4c, 0x42, 0x3f, 0x42, 0x3f, 0x43, 0x57, 0xeb, 0xd6, 0xc7, 0xbe, 0xbc,
  0xbb, 0xbe, 0xbf, 0xbe, 0xc6, 0xd1, 0xcf, 0xdb, 0x67, 0x57, 0x50, 0x4e,
  0x4d, 0x4c, 0x4f, 0x57, 0x55, 0x4d, 0x4b, 0x45, 0x3a, 0x32, 0x2f, 0x2e,
  0x2d, 0x2f, 0x34, 0x42, 0xdd, 0xca, 0xc9, 0xb9, 0xba, 0xcb, 0xcb, 0xce,
  0xdf, 0xd4, 0xdf, 0xe7, 0xda, 0x62, 0x4b, 0x4c, 0x4b, 0x48, 0x52, 0xdd,
  0xc7, 0xbe, 0xb9, 0xb7, 0xb6, 0xb9, 0xc3, 0xc8, 0xc8, 0xd8, 0xf0, 0xef,
  0x69, 0x55, 0x4a, 0x41, 0x42, 0x42, 0x40, 0x4b, 0x74, 0xd5, 0xc9, 0xbf,
  0xb9, 0xb9, 0xbd, 0xbe, 0xc0, 0xc9, 0xd2, 0xda, 0xde, 0x7c, 0x55, 0x4e,
  0x4d, 0x47, 0x44, 0x47, 0x4a, 0x4a, 0x47, 0x43, 0x3f, 0x37, 0x2f, 0x2e,
  0x2e, 0x2e, 0x31, 0x3d, 0x78, 0xc5, 0xc5, 0xbb, 0xb5, 0xc0, 0xce, 0xcd,
  0xde, 0xdf, 0xdf, 0xed, 0xd3, 0xde, 0x53, 0x4e, 0x4e, 0x48, 0x49, 0x62,
  0xcf, 0xc1, 0xbb, 0xb7, 0xb5, 0xb7, 0xbf, 0xca, 0xc9, 0xd6, 0x6d, 0xfb,
  0xfb, 0x5e, 0x53, 0x48, 0x43, 0x42, 0x3e, 0x43, 0x5c, 0xe2, 0xcd, 0xbd,
  0xb7, 0xb9, 0xbb, 0xbc, 0xc4, 0xcb, 0xd3, 0xdf, 0xdc, 0xe8, 0x67, 0x68,
  0x5c, 0x4c, 0x4a, 0x48, 0x45, 0x43, 0x42, 0x3e, 0x37, 0x31, 0x2e, 0x2c,
  0x2c, 0x2f, 0x36, 0x4e, 0xc8, 0xc1, 0xba, 0xb0, 0xba, 0xc6, 0xc9, 0xe7,
  0x72, 0xf5, 0x66, 0xeb, 0xdf, 0x5d, 0x52, 0x50, 0x47, 0x44, 0x4f, 0xea,
  0xcb, 0xbe, 0xb7, 0xb3, 0xb3, 0xb8, 0xc1, 0xc8, 0xd8, 0x63, 0x5f, 0x63,
  0x59, 0x53, 0x4d, 0x49, 0x45, 0x40, 0x46, 0x56, 0x73, 0xcb, 0xbb, 0xb8,
  0xb6, 0xb5, 0xbb, 0xc2, 0xcb, 0xdd, 0xf5, 0x7e, 0x6b, 0x69, 0x6b, 0x5c,
  0x51, 0x4c, 0x44, 0x3e, 0x3d, 0x39, 0x31, 0x30, 0x30, 0x2d, 0x2e, 0x33,
  0x35, 0x48, 0xcc, 0xc8, 0xbd, 0xb0, 0xb9, 0xc1, 0xbf, 0xdc, 0x71, 0xf6,
  0x53, 0x5d, 0xf3, 0x55, 0x53, 0x5c, 0x4c, 0x4b, 0x5b, 0xee, 0xce, 0xbf,
  0xb8, 0xb3, 0xb0, 0xb4, 0xba, 0xbd, 0xcc, 0x71, 0x60, 0x55, 0x4b, 0x4c,
  0x49, 0x46, 0x49, 0x4b, 0x4b, 0x59, 0xde, 0xcb, 0xc2, 0xb9, 0xb6, 0xb8,
  0xbb, 0xbf, 0xcc, 0xdb, 0x6c, 0x5c, 0x5f, 0x57, 0x50, 0x53, 0x4e, 0x45,
  0x3e, 0x39, 0x32, 0x2e, 0x2c, 0x2c, 0x2c, 0x2e, 0x35, 0x3c, 0x6e, 0xc0,
  0xc0, 0xb8, 0xaf, 0xb9, 0xbf, 0xc1, 0xdd, 0xef, 0x7b, 0x4e, 0x5c, 0x5e,
  0x47, 0x4c, 0x51, 0x49, 0x4d, 0x69, 0xd9, 0xc7, 0xbc, 0xb6, 0xb2, 0xb1,
  0xb5, 0xbb, 0xbe, 0xcc, 0x7a, 0x66, 0x57, 0x4a, 0x4a, 0x47, 0x47, 0x51,
  0x4e, 0x51, 0xe1, 0xce, 0xcb, 0xbd, 0xb8, 0xb9, 0xb9, 0xbb, 0xc3, 0xcb,
  0xdc, 0x64, 0x69, 0x5e, 0x4d, 0x4f, 0x50, 0x43, 0x3e, 0x37, 0x2e, 0x2c,
  0x2a, 0x29, 0x2a, 0x2e, 0x32, 0x3c, 0xe0, 0xbe, 0xbe, 0xb4, 0xae, 0xb9,
  0xbd, 0xc2, 0xe8, 0xeb, 0x69, 0x4b, 0x60, 0x58, 0x43, 0x4d, 0x4d, 0x45,
  0x4d, 0x64, 0xd9, 0xc4, 0xbc, 0xb5, 0xaf, 0xaf, 0xb5, 0xb9, 0xbd, 0xce,
  0xfd, 0x6a, 0x52, 0x4b, 0x4b, 0x48, 0x52, 0x59, 0x4e, 0x7c, 0xd3, 0xda,
  0xc6, 0xbb, 0xbc, 0xbb, 0xbb, 0xbf, 0xc5, 0xd0, 0xfe, 0x6a, 0x5f, 0x4e,
  0x4c, 0x4f, 0x46, 0x3d, 0x37, 0x2e, 0x2c, 0x2a, 0x27, 0x29, 0x2e, 0x2f,
  0x3c, 0xcf, 0xbd, 0xbc, 0xb0, 0xae, 0xb7, 0xbb, 0xc4, 0xdd, 0xdf, 0x5c,
  0x4c, 0x6b, 0x52, 0x42, 0x4d, 0x4b, 0x42, 0x4b, 0x5d, 0xdd, 0xc4, 0xbc,
  0xb5, 0xae, 0xaf, 0xb6, 0xb7, 0xbc, 0xd3, 0xfe, 0x62, 0x4e, 0x4a, 0x4b,
  0x4d, 0x57, 0x5b, 0x5e, 0xe9, 0xd9, 0xd4, 0xc8, 0xbf, 0xbd, 0xbd, 0xbe,
  0xbf, 0xc8, 0xd8, 0xfe, 0x5d, 0x4f, 0x49, 0x45, 0x42, 0x3d, 0x34, 0x2e,
  0x2d, 0x2a, 0x27, 0x29, 0x2d, 0x2f, 0x3c, 0xce, 0xbd, 0xb9, 0xad, 0xad,
  0xb5, 0xb8, 0xc6, 0xe6, 0xe0, 0x54, 0x4d, 0xfb, 0x52, 0x48, 0x58, 0x4c,
  0x44, 0x4b, 0x54, 0xef, 0xc8, 0xbd, 0xb4, 0xae, 0xae, 0xb3, 0xb5, 0xbc,
  0xd7, 0x6b, 0x57, 0x48, 0x4c, 0x55, 0x4f, 0x6a, 0xe4, 0x6d, 0xe3, 0xd1,
  0xdb, 0xcd, 0xc3, 0xc4, 0xbf, 0xbd, 0xc3, 0xc9, 0xd5, 0x65, 0x51, 0x4a,
  0x3f, 0x3c, 0x39, 0x30, 0x2d, 0x2d, 0x29, 0x28, 0x2b, 0x2c, 0x2f, 0x49,
  0xcc, 0xc0, 0xb2, 0xaa, 0xad, 0xb2, 0xb7, 0xca, 0xde, 0x67, 0x49, 0x57,
  0x5d, 0x49, 0x55, 0x5b, 0x48, 0x4a, 0x4d, 0x52, 0xe6, 0xcd, 0xbf, 0xb3,
  0xaf, 0xb0, 0xb0, 0xb4, 0xbf, 0xd3, 0x72, 0x4e, 0x4b, 0x4f, 0x4e, 0x61,
  0xdb, 0xec, 0xe1, 0xd1, 0xe9, 0xe7, 0xd0, 0xd0, 0xc8, 0xc2, 0xc4, 0xc2,
  0xc8, 0xdf, 0x6d, 0x52, 0x41, 0x3b, 0x34, 0x2e, 0x2d, 0x2b, 0x29, 0x2a,
  0x2c, 0x2c, 0x33, 0x57, 0xd6, 0xc5, 0xaf, 0xac, 0xaf, 0xaf, 0xb7, 0xc6,
  0xcd, 0x5a, 0x48, 0x5b, 0x49, 0x44, 0x5b, 0x4f, 0x4a, 0x53, 0x4f, 0x5e,
  0xe4, 0xd8, 0xc1, 0xb6, 0xb4, 0xb3, 0xaf, 0xb5, 0xbf, 0xcc, 0xf6, 0x5b,
  0x56, 0x4c, 0x59, 0xdb, 0xdf, 0xdc, 0xc9, 0xce, 0xda, 0xd3, 0xd5, 0xcf,
  0xcc, 0xce, 0xc6, 0xc6, 0xdb, 0xf8, 0x61, 0x44, 0x38, 0x2f, 0x2d, 0x2b,
  0x29, 0x28, 0x2c, 0x2e, 0x2e, 0x3d, 0xdc, 0xdb, 0xc2, 0xaf, 0xb1, 0xb4,
  0xb3, 0xbf, 0xc3, 0xce, 0x4e, 0x5e, 0x64, 0x3f, 0x49, 0x58, 0x48, 0x4d,
  0x54, 0x5a, 0xdf, 0xd6, 0xcd, 0xba, 0xb5, 0xb7, 0xb3, 0xb2, 0xba, 0xc4,
  0xce, 0xdf, 0x76, 0x56, 0x54, 0xec, 0xe2, 0xfc, 0xd2, 0xcb, 0xd6, 0xd6,
  0xcf, 0xce, 0xcf, 0xd5, 0xce, 0xca, 0xda, 0x76, 0x68, 0x4c, 0x39, 0x30,
  0x2d, 0x2b, 0x28, 0x27, 0x2b, 0x2d, 0x2f, 0x3f, 0xe3, 0xd2, 0xbe, 0xb1,
  0xb2, 0xb4, 0xb6, 0xbf, 0xc2, 0xce, 0x5f, 0xf8, 0x70, 0x49, 0x50, 0x54,
  0x49, 0x4d, 0x4d, 0x55, 0xeb, 0xde, 0xce, 0xbb, 0xb7, 0xb7, 0xb4, 0xb4,
  0xba, 0xc3, 0xcd, 0xd2, 0xeb, 0x59, 0x68, 0xe2, 0xef, 0xf0, 0xd9, 0xd3,
  0xdb, 0xe2, 0xd9, 0xd2, 0xda, 0xda, 0xce, 0xcf, 0xe3, 0x70, 0x5a, 0x42,
  0x34, 0x2e, 0x2c, 0x29, 0x28, 0x29, 0x2c, 0x2e, 0x3b, 0x6a, 0xd6, 0xc1,
  0xb2, 0xb3, 0xb4, 0xb5, 0xbf, 0xc5, 0xcb, 0x65, 0x76, 0xf4, 0x4d, 0x58,
  0x5e, 0x4c, 0x51, 0x52, 0x4f, 0x6e, 0xe4, 0xd6, 0xc0, 0xba, 0xb9, 0xb5,
  0xb5, 0xba, 0xbf, 0xca, 0xd1, 0xe1, 0x64, 0x6b, 0xe1, 0xe2, 0xde, 0xcf,
  0xce, 0xd7, 0xda, 0xd9, 0xda, 0xdd, 0xdf, 0xd9, 0xd6, 0xe8, 0x66, 0x58,
  0x41, 0x34, 0x2f, 0x2d, 0x29, 0x29, 0x2b, 0x2c, 0x2f, 0x40, 0x6e, 0xdb,
  0xbd, 0xb2, 0xb6, 0xb5, 0xb6, 0xc2, 0xc3, 0xcf, 0x69, 0xe1, 0x6f, 0x4d,
  0x64, 0x59, 0x4a, 0x52, 0x4f, 0x52, 0x7a, 0xec, 0xcf, 0xbe, 0xbc, 0xb9,
  0xb5, 0xb7, 0xbc, 0xc2, 0xc9, 0xd1, 0xf2, 0x6b, 0xeb, 0xe2, 0xea, 0xdc,
  0xd1, 0xd5, 0xdd, 0xdb, 0xd9, 0xdf, 0xe1, 0xdb, 0xd7, 0xde, 0x74, 0x5f,
  0x4a, 0x38, 0x31, 0x2e, 0x2a, 0x29, 0x2a, 0x2b, 0x2e, 0x3a, 0x5d, 0xe0,
  0xc2, 0xb2, 0xb4, 0xb6, 0xb4, 0xbf, 0xc7, 0xc9, 0xfe, 0xed, 0xe3, 0x54,
  0x65, 0x6e, 0x4a, 0x4d, 0x4f, 0x4a, 0x56, 0x69, 0xdf, 0xc6, 0xbe, 0xba,
  0xb6, 0xb5, 0xb9, 0xbf, 0xc5, 0xcb, 0xde, 0xf7, 0xde, 0xd5, 0xd7, 0xd2,
  0xcb, 0xce, 0xd9, 0xdc, 0xdf, 0xeb, 0xf4, 0xef, 0xdf, 0xe2, 0x72, 0x5f,
  0x4b, 0x3a, 0x33, 0x2e, 0x2b, 0x29, 0x2a, 0x2b, 0x2e, 0x3d, 0x5e, 0xe7,
  0xc0, 0xb4, 0xb8, 0xb7, 0xb6, 0xc3, 0xc7, 0xcb, 0x7e, 0xdb, 0xe2, 0x56,
  0xf7, 0x6b, 0x49, 0x4f, 0x4d, 0x48, 0x52, 0x5b, 0xee, 0xca, 0xc2, 0xbc,
  0xb7, 0xb8, 0xbc, 0xbf, 0xc3, 0xcb, 0xda, 0xde, 0xd2, 0xd0, 0xd2, 0xcb,
  0xc9, 0xcf, 0xd9, 0xda, 0xdf, 0xef, 0x7e, 0xef, 0xe3, 0xfa, 0x5e, 0x54,
  0x42, 0x35, 0x2f, 0x2d, 0x2a, 0x29, 0x2b, 0x2c, 0x35, 0x4e, 0x69, 0xcd,
  0xb5, 0xb7, 0xba, 0xb5, 0xbe, 0xca, 0xc8, 0xe5, 0xe7, 0xd6, 0x5e, 0x73,
  0xde, 0x51, 0x4e, 0x57, 0x4a, 0x4b, 0x54, 0x63, 0xd7, 0xc7, 0xc1, 0xba,
  0xb7, 0xbb, 0xbe, 0xbf, 0xc6, 0xd3, 0xd8, 0xcf, 0xce, 0xd0, 0xcb, 0xc6,
  0xcb, 0xd7, 0xdb, 0xe0, 0x72, 0x5f, 0x67, 0x73, 0x64, 0x55, 0x4e, 0x43,
  0x37, 0x30, 0x2e, 0x2b, 0x2a, 0x2b, 0x2c, 0x35, 0x4d, 0x60, 0xcf, 0xb6,
  0xb8, 0xbb, 0xb5, 0xbe, 0xcc, 0xc9, 0xe5, 0xe2, 0xd3, 0x68, 0xee, 0xd6,
  0x5b, 0x54, 0x5b, 0x4c, 0x4c, 0x50, 0x5b, 0xdd, 0xcc, 0xc6, 0xbb, 0xb8,
  0xbb, 0xbe, 0xbe, 0xc3, 0xd0, 0xd6, 0xcc, 0xcc, 0xcf, 0xc8, 0xc4, 0xcb,
  0xd2, 0xd8, 0xe2, 0x7e, 0x5e, 0x5f, 0x6d, 0x5e, 0x50, 0x4b, 0x3f, 0x36,
  0x2f, 0x2d, 0x2b, 0x2a, 0x2a, 0x2d, 0x3b, 0x51, 0x65, 0xc3, 0xb4, 0xba,
  0xb9, 0xb6, 0xc2, 0xc7, 0xcb, 0xec, 0xd4, 0xd7, 0x67, 0xd9, 0xdb, 0x58,
  0x5c, 0x57, 0x4b, 0x4e, 0x4e, 0x5a, 0xdc, 0xd0, 0xca, 0xbd, 0xbb, 0xbe,
  0xc0, 0xbe, 0xc3, 0xd3, 0xce, 0xc4, 0xcc, 0xcd, 0xc2, 0xc5, 0xcd, 0xd1,
  0xdd, 0xe5, 0x6d, 0x57, 0x62, 0x62, 0x50, 0x4c, 0x45, 0x3b, 0x34, 0x2f,
  0x2d, 0x2b, 0x2a, 0x2b, 0x2f, 0x3f, 0x4e, 0x68, 0xbd, 0xb6, 0xbd, 0xb6,
  0xb7, 0xc4, 0xc1, 0xcd, 0xda, 0xc7, 0xdc, 0xe7, 0xc8, 0xe0, 0x62, 0x79,
  0x53, 0x4b, 0x4b, 0x48, 0x56, 0xfb, 0xea, 0xce, 0xbf, 0xbe, 0xc1, 0xbe,
  0xbc, 0xc4, 0xcd, 0xc2, 0xc1, 0xcb, 0xc4, 0xbf, 0xc5, 0xcb, 0xd3, 0xdd,
  0xeb, 0x57, 0x50, 0x5c, 0x51, 0x48, 0x44, 0x3f, 0x39, 0x32, 0x2f, 0x2d,
  0x2c, 0x2b, 0x2d, 0x35, 0x48, 0x4e, 0xe7, 0xb8, 0xba, 0xbd, 0xb3, 0xbb,
  0xc3, 0xbe, 0xd3, 0xcd, 0xc4, 0xe5, 0xd1, 0xc5, 0xed, 0x7a, 0xf5, 0x52,
  0x4d, 0x49, 0x48, 0x54, 0x60, 0x6c, 0xd7, 0xc7, 0xc6, 0xc9, 0xbf, 0xbe,
  0xca, 0xc7, 0xbe, 0xc3, 0xc3, 0xbe, 0xc2, 0xc3, 0xc8, 0xdc, 0xda, 0xee,
  0x52, 0x55, 0x56, 0x4a, 0x43, 0x3d, 0x39, 0x35, 0x2f, 0x2d, 0x2d, 0x2d,
  0x2c, 0x2f, 0x3f, 0x4e, 0x58, 0xc4, 0xb8, 0xbe, 0xb8, 0xb6, 0xc1, 0xbe,
  0xc5, 0xd7, 0xc3, 0xcd, 0xea, 0xc7, 0xce, 0x75, 0xea, 0x68, 0x4f, 0x4e,
  0x49, 0x4a, 0x59, 0x64, 0x70, 0xd7, 0xca, 0xcc, 0xc7, 0xbf, 0xc5, 0xc7,
  0xbf, 0xc0, 0xc3, 0xbd, 0xbf, 0xc3, 0xbf, 0xcb, 0xda, 0xd6, 0x7b, 0x59,
  0x59, 0x4e, 0x47, 0x40, 0x3b, 0x37, 0x32, 0x2f, 0x2d, 0x2d, 0x2d, 0x2d,
  0x34, 0x46, 0x4d, 0x79, 0xbd, 0xbc, 0xbe, 0xb5, 0xba, 0xc1, 0xbd, 0xc9,
  0xce, 0xc3, 0xd0, 0xd6, 0xc8, 0xd8, 0xfd, 0xf3, 0x59, 0x4b, 0x49, 0x48,
  0x48, 0x4f, 0x5f, 0x73, 0xdc, 0xcb, 0xca, 0xc6, 0xc0, 0xc5, 0xc2, 0xbe,
  0xc2, 0xbf, 0xbb, 0xc1, 0xc4, 0xc1, 0xcd, 0xdc, 0xe2, 0x72, 0x5a, 0x52,
  0x4b, 0x41, 0x3c, 0x39, 0x34, 0x30, 0x2f, 0x2e, 0x2e, 0x2f, 0x30, 0x3a,
  0x4c, 0x51, 0xde, 0xbc, 0xbf, 0xbf, 0xb6, 0xbc, 0xc3, 0xbe, 0xc6, 0xca,
  0xc5, 0xcc, 0xce, 0xcc, 0xd6, 0xea, 0x79, 0x5e, 0x4d, 0x49, 0x4c, 0x4b,
  0x4c, 0x63, 0xf8, 0xed, 0xcf, 0xc7, 0xc9, 0xc6, 0xbf, 0xbf, 0xc0, 0xbe,
  0xbb, 0xbd, 0xbf, 0xc0, 0xc6, 0xcd, 0xd7, 0xee, 0x6d, 0x5a, 0x4d, 0x47,
  0x3f, 0x3c, 0x38, 0x34, 0x31, 0x2f, 0x2f, 0x2e, 0x2f, 0x33, 0x3d, 0x4a,
  0x4f, 0xd8, 0xbe, 0xc4, 0xc0, 0xb9, 0xbe, 0xc6, 0xc1, 0xc3, 0xcb, 0xca,
  0xc8, 0xc9, 0xcd, 0xd2, 0xdd, 0x7e, 0x5e, 0x50, 0x4a, 0x4d, 0x4f, 0x4d,
  0x5b, 0xef, 0xee, 0xdb, 0xca, 0xc9, 0xcb, 0xc2, 0xbe, 0xc2, 0xbf, 0xbb,
  0xbd, 0xc1, 0xbf, 0xc3, 0xcf, 0xd8, 0xde, 0x69, 0x54, 0x52, 0x47, 0x3d,
  0x3c, 0x39, 0x35, 0x33, 0x33, 0x31, 0x2f, 0x32, 0x34, 0x3a, 0x47, 0x4f,
  0x7a, 0xca, 0xc4, 0xc6, 0xc0, 0xbd, 0xc4, 0xcc, 0xc4, 0xc3, 0xce, 0xcc,
  0xc4, 0xc7, 0xd2, 0xd4, 0xd3, 0xfb, 0x5c, 0x5a, 0x57, 0x57, 0x56, 0x5f,
  0xfd, 0xee, 0xdb, 0xd1, 0xd3, 0xcd, 0xc7, 0xc7, 0xc8, 0xc3, 0xbe, 0xc0,
  0xc4, 0xc1, 0xc3, 0xcd, 0xd6, 0xda, 0xf5, 0x5b, 0x53, 0x4e, 0x44, 0x3e,
  0x3d, 0x3b, 0x36, 0x34, 0x34, 0x33, 0x32, 0x34, 0x35, 0x3b, 0x46, 0x4d,
  0x5f, 0xd6, 0xca, 0xcb, 0xca, 0xc4, 0xc5, 0xcc, 0xcc, 0xc5, 0xc5, 0xc8,
  0xc6, 0xc2, 0xc6, 0xce, 0xd5, 0xdd, 0x7a, 0x63, 0x5d, 0x5a, 0x60, 0x65,
  0x6e, 0xf1, 0xdc, 0xd0, 0xd4, 0xd6, 0xcc, 0xc8, 0xcb, 0xca, 0xc2, 0xbf,
  0xc3, 0xc7, 0xc5, 0xc7, 0xd3, 0xe2, 0xf2, 0x6a, 0x5c, 0x4c, 0x47, 0x47,
  0x40, 0x3f, 0x3c, 0x39, 0x3c, 0x38, 0x36, 0x36, 0x37, 0x39, 0x3c, 0x47,
  0x4f, 0x62, 0xdd, 0xd3, 0xcf, 0xd0, 0xcc, 0xca, 0xd2, 0xd0, 0xca, 0xc6,
  0xc4, 0xc5, 0xc4, 0xc3, 0xc8, 0xd0, 0xd9, 0xe8, 0x73, 0x6e, 0x68, 0x66,
  0x77, 0xf2, 0xea, 0xe3, 0xde, 0xdc, 0xdc, 0xdb, 0xd6, 0xcf, 0xcd, 0xcb,
  0xc7, 0xc7, 0xc9, 0xcb, 0xd0, 0xd9, 0xe5, 0xf6, 0x6d, 0x63, 0x58, 0x4f,
  0x4d, 0x4b, 0x48, 0x42, 0x3f, 0x3f, 0x3c, 0x3b, 0x3b, 0x3b, 0x3b, 0x3a,
  0x3e, 0x47, 0x4d, 0x54, 0x67, 0xe5, 0xd9, 0xd8, 0xd9, 0xd5, 0xd1, 0xd2,
  0xcf, 0xcd, 0xca, 0xc7, 0xc4, 0xc4, 0xc9, 0xcb, 0xce, 0xd4, 0xdc, 0xeb,
  0xec, 0xe3, 0xe6, 0xe5, 0xe6, 0xe3, 0xde, 0xdd, 0xdc, 0xde, 0xdc, 0xd5,
  0xd4, 0xd2, 0xcf, 0xcf, 0xce, 0xd2, 0xda, 0xdd, 0xe6, 0xf7, 0x7d, 0x6f,
  0x69, 0x68, 0x5d, 0x59, 0x56, 0x4d, 0x4c, 0x4c, 0x4c, 0x4a, 0x44, 0x46,
  0x45, 0x43, 0x47, 0x42, 0x42, 0x45, 0x48, 0x52, 0x55, 0x5c, 0x6a, 0x72,
  0xef, 0xf5, 0xea, 0xe2, 0xe3, 0xdb, 0xd6, 0xcf, 0xcc, 0xcb, 0xca, 0xcc,
  0xcd, 0xd0, 0xd4, 0xd7, 0xda, 0xda, 0xd8, 0xd6, 0xd6, 0xd6, 0xd5, 0xd9,
  0xdd, 0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdd, 0xdd, 0xde, 0xdf, 0xe6, 0xe9,
  0xf2, 0xfe, 0xf3, 0xfc, 0x70, 0x79, 0x7b, 0x6e, 0x67, 0x67, 0x65, 0x5c,
  0x58, 0x56, 0x57, 0x56, 0x54, 0x57, 0x52, 0x4f, 0x4e, 0x4b, 0x4a, 0x49,
  0x47, 0x48, 0x4c, 0x4f, 0x53, 0x5a, 0x5d, 0x60, 0x67, 0x65, 0x69, 0x6b,
  0x70, 0xf9, 0xeb, 0xdf, 0xda, 0xd5, 0xcf, 0xd0, 0xcf, 0xcf, 0xd3, 0xd2,
  0xd2, 0xd1, 0xcf, 0xd0, 0xd0, 0xd1, 0xd4, 0xd7, 0xdc, 0xde, 0xdf, 0xe2,
  0xe0, 0xdf, 0xdf, 0xe6, 0xe9, 0xe7, 0xec, 0xfc, 0x7b, 0x7c, 0x72, 0x6c,
  0x72, 0x7c, 0x75, 0x77, 0xfa, 0x73, 0x6f, 0x6c, 0x64, 0x68, 0x61, 0x62,
  0x64, 0x5c, 0x5f, 0x5e, 0x5b, 0x5d, 0x58, 0x57, 0x59, 0x54, 0x56, 0x56,
  0x53, 0x53, 0x53, 0x54, 0x57, 0x56, 0x59, 0x5f, 0x61, 0x63, 0x68, 0x6d,
  0x71, 0x7e, 0xfc, 0xf5, 0xe7, 0xe4, 0xe2, 0xda, 0xd9, 0xdb, 0xdb, 0xd9,
  0xd9, 0xda, 0xd9, 0xd8, 0xd7, 0xd7, 0xda, 0xda, 0xdc, 0xdf, 0xe3, 0xe9,
  0xea, 0xef, 0xf6, 0xef, 0xf5, 0x78, 0x79, 0xff, 0x73, 0x70, 0x73, 0x6f,
  0x70, 0x6f, 0x74, 0xfc, 0xff, 0xfb, 0xfd, 0x76, 0x79, 0x7b, 0x7a, 0x77,
  0x7c, 0x78, 0x6d, 0xf0, 0xf5, 0x64, 0x74, 0x76, 0x65, 0x6e, 0x6b, 0x6a,
  0x69, 0x5f, 0x69, 0x67, 0x5f, 0x65, 0x61, 0x61, 0x63, 0x5e, 0x5f, 0x60,
  0x5c, 0x5d, 0x5f, 0x61, 0x62, 0x61, 0x68, 0x77, 0x75, 0x6e, 0x7e, 0xfe,
  0x77, 0xf8, 0xf7, 0xf4, 0xeb, 0xec, 0xe8, 0xe3, 0xe6, 0xe1, 0xde, 0xe0,
  0xde, 0xdd, 0xdf, 0xdd, 0xdf, 0xe0, 0xde, 0xe7, 0xe5, 0xe5, 0xf6, 0xf3,
  0xf5, 0xfd, 0x7d, 0x7b, 0xfa, 0x7d, 0x7a, 0x76, 0x76, 0x78, 0x6d, 0xe2,
  0xe8, 0x55, 0x6a, 0xea, 0x61, 0x79, 0xf6, 0x6a, 0xfb, 0xfa, 0x79, 0xf5,
  0x7e, 0xfe, 0xf6, 0x75, 0x79, 0x7d, 0x79, 0xfe, 0x79, 0x71, 0x75, 0x79,
  0x6d, 0x6e, 0x75, 0x6e, 0x6f, 0x69, 0x6d, 0x79, 0x6b, 0x6b, 0x74, 0x70,
  0x6f, 0x6f, 0x6c, 0x70, 0x71, 0x71, 0x73, 0x6d, 0x6d, 0x64, 0xfa, 0xee,
  0x5a, 0x65, 0x6c, 0x64, 0x7d, 0x6f
};

static InputStream *generateHelloWorld(void)
{
  return _alutInputStreamConstructFromMemory(helloWorldSample, sizeof(helloWorldSample));
}

ALvoid *alutLoadMemoryHelloWorld(ALenum * format, ALsizei * size, ALfloat * frequency)
{
  if (!_alutSanityCheck())
  {
    return NULL;
  }
  return _alutLoadMemoryFromInputStream(generateHelloWorld(), format, size, frequency);
}

ALuint alutCreateBufferHelloWorld(void)
{
  if (!_alutSanityCheck())
  {
    return AL_NONE;
  }
  return _alutCreateBufferFromInputStream(generateHelloWorld());
}
