/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public Liccense as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Splitsel   splitsel        Split time selection
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "param_conversion.h"
#include "util_files.h"

void *
Splitsel(void *process)
{
  int nrecs = 0;
  int varID, levelID;
  int tsID;
  size_t nmiss;

  cdoInitialize(process);

  if (processSelf().m_ID != 0) cdoAbort("This operator can't be combined with other operators!");

  const auto lcopy = unchangedRecord();

  cdoOperatorAdd("splitsel", 0, 0, nullptr);

  // operatorInputArg("nsets <noffset <nskip>>");

  const auto nargc = operatorArgc();
  if (nargc < 1) cdoAbort("Too few arguments! Need %d found %d.", 1, nargc);

  const auto nsets = parameter2int(cdoOperatorArgv(0));
  const auto noffset = (nargc > 1) ? parameter2int(cdoOperatorArgv(1)) : 0;
  const auto nskip = (nargc > 2) ? parameter2int(cdoOperatorArgv(2)) : 0;

  if (Options::cdoVerbose) cdoPrint("nsets = %d, noffset = %d, nskip = %d", nsets, noffset, nskip);

  if (nsets < 1) cdoAbort("nsets must be greater than 0!");
  if (noffset < 0) cdoAbort("noffset must be greater or equal 0!");
  if (nskip < 0)
    {
      if (-nskip >= nsets) cdoAbort("Absolute value of negative nskip must be less than nsets!");
      if (cdoAssertFilesOnly() == false) cdoAbort("Negative nskip not allowed in combination with other operators!");
    }

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  // int taxisID2 = cdoTaxisCreate(TAXIS_ABSOLUTE);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  VarList varList1;
  varListInit(varList1, vlistID1);

  char filename[8192];
  strcpy(filename, cdoGetObase());
  const auto nchars = strlen(filename);

  auto refname = cdoGetStreamName(0);
  char filesuffix[32] = { 0 };
  cdoGenFileSuffix(filesuffix, sizeof(filesuffix), cdoInqFiletype(streamID1), vlistID1, refname);

  Varray<double> array;
  //  if ( ! lcopy )
  {
    auto gridsizemax = vlistGridsizeMax(vlistID1);
    if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
    array.resize(gridsizemax);
  }

  const auto nvars = vlistNvars(vlistID1);
  int nconst = 0;
  for (varID = 0; varID < nvars; varID++)
    if (varList1[varID].timetype == TIME_CONSTANT) nconst++;

  FieldVector2D vars;
  if (nconst)
    {
      vars.resize(nvars);

      for (varID = 0; varID < nvars; varID++)
        {
          if (varList1[varID].timetype == TIME_CONSTANT)
            {
              const auto gridID = varList1[varID].gridID;
              const auto gridsize = varList1[varID].gridsize;
              const auto nlevels = varList1[varID].nlevels;

              vars[varID].resize(nlevels);

              for (levelID = 0; levelID < nlevels; levelID++)
                {
                  vars[varID][levelID].grid = gridID;
                  vars[varID][levelID].resize(gridsize);
                }
            }
        }
    }

  int index = 0;

  // offset
  for (tsID = 0; tsID < noffset; tsID++)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0)
        {
          cdoWarning("noffset is larger than number of timesteps!");
          goto LABEL_END;
        }

      if (tsID == 0 && nconst)
        for (int recID = 0; recID < nrecs; recID++)
          {
            cdoInqRecord(streamID1, &varID, &levelID);
            if (varList1[varID].timetype == TIME_CONSTANT)
              {
                cdoReadRecord(streamID1, vars[varID][levelID].vec_d.data(), &nmiss);
                vars[varID][levelID].nmiss = nmiss;
              }
          }
    }

  while (true)
    {
      sprintf(filename + nchars, "%06d", index);
      sprintf(filename + nchars + 6, "%s", filesuffix);

      if (Options::cdoVerbose) cdoPrint("create file %s", filename);

      CdoStreamID streamID2 = CDO_STREAM_UNDEF;
      int tsID2 = 0;

      for (int i = 0; i < nsets; ++i)
        {
          nrecs = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs == 0) break;

          taxisCopyTimestep(taxisID2, taxisID1);

          if (streamID2 == CDO_STREAM_UNDEF)
            {
              streamID2 = cdoOpenWrite(filename);
              cdoDefVlist(streamID2, vlistID2);
            }

          cdoDefTimestep(streamID2, tsID2);

          if (tsID > 0 && tsID2 == 0 && nconst)
            {
              for (varID = 0; varID < nvars; varID++)
                {
                  if (varList1[varID].timetype == TIME_CONSTANT)
                    {
                      const auto nlevels = varList1[varID].nlevels;
                      for (levelID = 0; levelID < nlevels; levelID++)
                        {
                          cdoDefRecord(streamID2, varID, levelID);
                          nmiss = vars[varID][levelID].nmiss;
                          cdoWriteRecord(streamID2, vars[varID][levelID].vec_d.data(), nmiss);
                        }
                    }
                }
            }

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);
              cdoDefRecord(streamID2, varID, levelID);
              if (lcopy && !(tsID == 0 && nconst))
                {
                  cdoCopyRecord(streamID2, streamID1);
                }
              else
                {
                  cdoReadRecord(streamID1, array.data(), &nmiss);
                  cdoWriteRecord(streamID2, array.data(), nmiss);

                  if (tsID == 0 && nconst)
                    {
                      if (varList1[varID].timetype == TIME_CONSTANT)
                        {
                          varrayCopy(varList1[varID].gridsize, array, vars[varID][levelID].vec_d);
                          vars[varID][levelID].nmiss = nmiss;
                        }
                    }
                }
            }

          tsID++;
          tsID2++;
        }

      cdoStreamClose(streamID2);
      if (nrecs == 0) break;

      if (cdoStreamInqTimestep(streamID1, tsID) == 0) break;

      // skip
      for (int i = 0; i < nskip; ++i)
        if (cdoStreamInqTimestep(streamID1, tsID + i) == 0) break;

      tsID += nskip;

      if (cdoStreamInqTimestep(streamID1, tsID) == 0) break;

      index++;
    }

LABEL_END:

  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
