#include <cgx.h>


#define MAX_LABELS   32


// Colormap name
// Possible values: "classic", "jet", "turbo", "viridis", "inferno", "coolwarm", "gray"
extern char cmap_name[];


// Classic GraphiX colormap
void get_cmap_classic(float v, float **r, float **g, float **b)
{
    if ( v < 0.1 )       **r = .5 - v * 1./0.2;
    else if ( v < 0.5 )  **r = 0. ;
    else if ( v < 0.8 )  **r = (v-.5) * 1./0.3 ;
    else if ( v < 0.9 )  **r = 1. ;
    else                 **r = 1. - (v-0.9) * 1./0.4 ;

    if ( v < 0.1 )       **g = 0.;
    else if ( v < 0.3 )  **g = (v-.1) * 1./0.2;
    else if ( v < 0.5 )  **g = 1. - (v-.3) * 1./0.4;
    else if ( v < 0.8 )  **g = .5 + (v-.5) * 1./0.6;
    else                 **g = 1. - (v-.8) * 1./0.2;

    if ( v < 0.5 )  **b = 1. - v * 1./0.5;
    else **b = 0;
}


// Viridis colormap - Berkeley Institute for Data Science
void get_cmap_viridis(float v, float **r, float **g, float **b)
{
  float cmap_data[256][3] = {
    {0.26700401, 0.00487433, 0.32941519},
    {0.26851048, 0.00960483, 0.33542652},
    {0.26994384, 0.01462494, 0.34137895},
    {0.27130489, 0.01994186, 0.34726862},
    {0.27259384, 0.02556309, 0.35309303},
    {0.27380934, 0.03149748, 0.35885256},
    {0.27495242, 0.03775181, 0.36454323},
    {0.27602238, 0.04416723, 0.37016418},
    {0.2770184 , 0.05034437, 0.37571452},
    {0.27794143, 0.05632444, 0.38119074},
    {0.27879067, 0.06214536, 0.38659204},
    {0.2795655 , 0.06783587, 0.39191723},
    {0.28026658, 0.07341724, 0.39716349},
    {0.28089358, 0.07890703, 0.40232944},
    {0.28144581, 0.0843197 , 0.40741404},
    {0.28192358, 0.08966622, 0.41241521},
    {0.28232739, 0.09495545, 0.41733086},
    {0.28265633, 0.10019576, 0.42216032},
    {0.28291049, 0.10539345, 0.42690202},
    {0.28309095, 0.11055307, 0.43155375},
    {0.28319704, 0.11567966, 0.43611482},
    {0.28322882, 0.12077701, 0.44058404},
    {0.28318684, 0.12584799, 0.44496   },
    {0.283072  , 0.13089477, 0.44924127},
    {0.28288389, 0.13592005, 0.45342734},
    {0.28262297, 0.14092556, 0.45751726},
    {0.28229037, 0.14591233, 0.46150995},
    {0.28188676, 0.15088147, 0.46540474},
    {0.28141228, 0.15583425, 0.46920128},
    {0.28086773, 0.16077132, 0.47289909},
    {0.28025468, 0.16569272, 0.47649762},
    {0.27957399, 0.17059884, 0.47999675},
    {0.27882618, 0.1754902 , 0.48339654},
    {0.27801236, 0.18036684, 0.48669702},
    {0.27713437, 0.18522836, 0.48989831},
    {0.27619376, 0.19007447, 0.49300074},
    {0.27519116, 0.1949054 , 0.49600488},
    {0.27412802, 0.19972086, 0.49891131},
    {0.27300596, 0.20452049, 0.50172076},
    {0.27182812, 0.20930306, 0.50443413},
    {0.27059473, 0.21406899, 0.50705243},
    {0.26930756, 0.21881782, 0.50957678},
    {0.26796846, 0.22354911, 0.5120084 },
    {0.26657984, 0.2282621 , 0.5143487 },
    {0.2651445 , 0.23295593, 0.5165993 },
    {0.2636632 , 0.23763078, 0.51876163},
    {0.26213801, 0.24228619, 0.52083736},
    {0.26057103, 0.2469217 , 0.52282822},
    {0.25896451, 0.25153685, 0.52473609},
    {0.25732244, 0.2561304 , 0.52656332},
    {0.25564519, 0.26070284, 0.52831152},
    {0.25393498, 0.26525384, 0.52998273},
    {0.25219404, 0.26978306, 0.53157905},
    {0.25042462, 0.27429024, 0.53310261},
    {0.24862899, 0.27877509, 0.53455561},
    {0.2468114 , 0.28323662, 0.53594093},
    {0.24497208, 0.28767547, 0.53726018},
    {0.24311324, 0.29209154, 0.53851561},
    {0.24123708, 0.29648471, 0.53970946},
    {0.23934575, 0.30085494, 0.54084398},
    {0.23744138, 0.30520222, 0.5419214 },
    {0.23552606, 0.30952657, 0.54294396},
    {0.23360277, 0.31382773, 0.54391424},
    {0.2316735 , 0.3181058 , 0.54483444},
    {0.22973926, 0.32236127, 0.54570633},
    {0.22780192, 0.32659432, 0.546532  },
    {0.2258633 , 0.33080515, 0.54731353},
    {0.22392515, 0.334994  , 0.54805291},
    {0.22198915, 0.33916114, 0.54875211},
    {0.22005691, 0.34330688, 0.54941304},
    {0.21812995, 0.34743154, 0.55003755},
    {0.21620971, 0.35153548, 0.55062743},
    {0.21429757, 0.35561907, 0.5511844 },
    {0.21239477, 0.35968273, 0.55171011},
    {0.2105031 , 0.36372671, 0.55220646},
    {0.20862342, 0.36775151, 0.55267486},
    {0.20675628, 0.37175775, 0.55311653},
    {0.20490257, 0.37574589, 0.55353282},
    {0.20306309, 0.37971644, 0.55392505},
    {0.20123854, 0.38366989, 0.55429441},
    {0.1994295 , 0.38760678, 0.55464205},
    {0.1976365 , 0.39152762, 0.55496905},
    {0.19585993, 0.39543297, 0.55527637},
    {0.19410009, 0.39932336, 0.55556494},
    {0.19235719, 0.40319934, 0.55583559},
    {0.19063135, 0.40706148, 0.55608907},
    {0.18892259, 0.41091033, 0.55632606},
    {0.18723083, 0.41474645, 0.55654717},
    {0.18555593, 0.4185704 , 0.55675292},
    {0.18389763, 0.42238275, 0.55694377},
    {0.18225561, 0.42618405, 0.5571201 },
    {0.18062949, 0.42997486, 0.55728221},
    {0.17901879, 0.43375572, 0.55743035},
    {0.17742298, 0.4375272 , 0.55756466},
    {0.17584148, 0.44128981, 0.55768526},
    {0.17427363, 0.4450441 , 0.55779216},
    {0.17271876, 0.4487906 , 0.55788532},
    {0.17117615, 0.4525298 , 0.55796464},
    {0.16964573, 0.45626209, 0.55803034},
    {0.16812641, 0.45998802, 0.55808199},
    {0.1666171 , 0.46370813, 0.55811913},
    {0.16511703, 0.4674229 , 0.55814141},
    {0.16362543, 0.47113278, 0.55814842},
    {0.16214155, 0.47483821, 0.55813967},
    {0.16066467, 0.47853961, 0.55811466},
    {0.15919413, 0.4822374 , 0.5580728 },
    {0.15772933, 0.48593197, 0.55801347},
    {0.15626973, 0.4896237 , 0.557936  },
    {0.15481488, 0.49331293, 0.55783967},
    {0.15336445, 0.49700003, 0.55772371},
    {0.1519182 , 0.50068529, 0.55758733},
    {0.15047605, 0.50436904, 0.55742968},
    {0.14903918, 0.50805136, 0.5572505 },
    {0.14760731, 0.51173263, 0.55704861},
    {0.14618026, 0.51541316, 0.55682271},
    {0.14475863, 0.51909319, 0.55657181},
    {0.14334327, 0.52277292, 0.55629491},
    {0.14193527, 0.52645254, 0.55599097},
    {0.14053599, 0.53013219, 0.55565893},
    {0.13914708, 0.53381201, 0.55529773},
    {0.13777048, 0.53749213, 0.55490625},
    {0.1364085 , 0.54117264, 0.55448339},
    {0.13506561, 0.54485335, 0.55402906},
    {0.13374299, 0.54853458, 0.55354108},
    {0.13244401, 0.55221637, 0.55301828},
    {0.13117249, 0.55589872, 0.55245948},
    {0.1299327 , 0.55958162, 0.55186354},
    {0.12872938, 0.56326503, 0.55122927},
    {0.12756771, 0.56694891, 0.55055551},
    {0.12645338, 0.57063316, 0.5498411 },
    {0.12539383, 0.57431754, 0.54908564},
    {0.12439474, 0.57800205, 0.5482874 },
    {0.12346281, 0.58168661, 0.54744498},
    {0.12260562, 0.58537105, 0.54655722},
    {0.12183122, 0.58905521, 0.54562298},
    {0.12114807, 0.59273889, 0.54464114},
    {0.12056501, 0.59642187, 0.54361058},
    {0.12009154, 0.60010387, 0.54253043},
    {0.11973756, 0.60378459, 0.54139999},
    {0.11951163, 0.60746388, 0.54021751},
    {0.11942341, 0.61114146, 0.53898192},
    {0.11948255, 0.61481702, 0.53769219},
    {0.11969858, 0.61849025, 0.53634733},
    {0.12008079, 0.62216081, 0.53494633},
    {0.12063824, 0.62582833, 0.53348834},
    {0.12137972, 0.62949242, 0.53197275},
    {0.12231244, 0.63315277, 0.53039808},
    {0.12344358, 0.63680899, 0.52876343},
    {0.12477953, 0.64046069, 0.52706792},
    {0.12632581, 0.64410744, 0.52531069},
    {0.12808703, 0.64774881, 0.52349092},
    {0.13006688, 0.65138436, 0.52160791},
    {0.13226797, 0.65501363, 0.51966086},
    {0.13469183, 0.65863619, 0.5176488 },
    {0.13733921, 0.66225157, 0.51557101},
    {0.14020991, 0.66585927, 0.5134268 },
    {0.14330291, 0.66945881, 0.51121549},
    {0.1466164 , 0.67304968, 0.50893644},
    {0.15014782, 0.67663139, 0.5065889 },
    {0.15389405, 0.68020343, 0.50417217},
    {0.15785146, 0.68376525, 0.50168574},
    {0.16201598, 0.68731632, 0.49912906},
    {0.1663832 , 0.69085611, 0.49650163},
    {0.1709484 , 0.69438405, 0.49380294},
    {0.17570671, 0.6978996 , 0.49103252},
    {0.18065314, 0.70140222, 0.48818938},
    {0.18578266, 0.70489133, 0.48527326},
    {0.19109018, 0.70836635, 0.48228395},
    {0.19657063, 0.71182668, 0.47922108},
    {0.20221902, 0.71527175, 0.47608431},
    {0.20803045, 0.71870095, 0.4728733 },
    {0.21400015, 0.72211371, 0.46958774},
    {0.22012381, 0.72550945, 0.46622638},
    {0.2263969 , 0.72888753, 0.46278934},
    {0.23281498, 0.73224735, 0.45927675},
    {0.2393739 , 0.73558828, 0.45568838},
    {0.24606968, 0.73890972, 0.45202405},
    {0.25289851, 0.74221104, 0.44828355},
    {0.25985676, 0.74549162, 0.44446673},
    {0.26694127, 0.74875084, 0.44057284},
    {0.27414922, 0.75198807, 0.4366009 },
    {0.28147681, 0.75520266, 0.43255207},
    {0.28892102, 0.75839399, 0.42842626},
    {0.29647899, 0.76156142, 0.42422341},
    {0.30414796, 0.76470433, 0.41994346},
    {0.31192534, 0.76782207, 0.41558638},
    {0.3198086 , 0.77091403, 0.41115215},
    {0.3277958 , 0.77397953, 0.40664011},
    {0.33588539, 0.7770179 , 0.40204917},
    {0.34407411, 0.78002855, 0.39738103},
    {0.35235985, 0.78301086, 0.39263579},
    {0.36074053, 0.78596419, 0.38781353},
    {0.3692142 , 0.78888793, 0.38291438},
    {0.37777892, 0.79178146, 0.3779385 },
    {0.38643282, 0.79464415, 0.37288606},
    {0.39517408, 0.79747541, 0.36775726},
    {0.40400101, 0.80027461, 0.36255223},
    {0.4129135 , 0.80304099, 0.35726893},
    {0.42190813, 0.80577412, 0.35191009},
    {0.43098317, 0.80847343, 0.34647607},
    {0.44013691, 0.81113836, 0.3409673 },
    {0.44936763, 0.81376835, 0.33538426},
    {0.45867362, 0.81636288, 0.32972749},
    {0.46805314, 0.81892143, 0.32399761},
    {0.47750446, 0.82144351, 0.31819529},
    {0.4870258 , 0.82392862, 0.31232133},
    {0.49661536, 0.82637633, 0.30637661},
    {0.5062713 , 0.82878621, 0.30036211},
    {0.51599182, 0.83115784, 0.29427888},
    {0.52577622, 0.83349064, 0.2881265 },
    {0.5356211 , 0.83578452, 0.28190832},
    {0.5455244 , 0.83803918, 0.27562602},
    {0.55548397, 0.84025437, 0.26928147},
    {0.5654976 , 0.8424299 , 0.26287683},
    {0.57556297, 0.84456561, 0.25641457},
    {0.58567772, 0.84666139, 0.24989748},
    {0.59583934, 0.84871722, 0.24332878},
    {0.60604528, 0.8507331 , 0.23671214},
    {0.61629283, 0.85270912, 0.23005179},
    {0.62657923, 0.85464543, 0.22335258},
    {0.63690157, 0.85654226, 0.21662012},
    {0.64725685, 0.85839991, 0.20986086},
    {0.65764197, 0.86021878, 0.20308229},
    {0.66805369, 0.86199932, 0.19629307},
    {0.67848868, 0.86374211, 0.18950326},
    {0.68894351, 0.86544779, 0.18272455},
    {0.69941463, 0.86711711, 0.17597055},
    {0.70989842, 0.86875092, 0.16925712},
    {0.72039115, 0.87035015, 0.16260273},
    {0.73088902, 0.87191584, 0.15602894},
    {0.74138803, 0.87344918, 0.14956101},
    {0.75188414, 0.87495143, 0.14322828},
    {0.76237342, 0.87642392, 0.13706449},
    {0.77285183, 0.87786808, 0.13110864},
    {0.78331535, 0.87928545, 0.12540538},
    {0.79375994, 0.88067763, 0.12000532},
    {0.80418159, 0.88204632, 0.11496505},
    {0.81457634, 0.88339329, 0.11034678},
    {0.82494028, 0.88472036, 0.10621724},
    {0.83526959, 0.88602943, 0.1026459 },
    {0.84556056, 0.88732243, 0.09970219},
    {0.8558096 , 0.88860134, 0.09745186},
    {0.86601325, 0.88986815, 0.09595277},
    {0.87616824, 0.89112487, 0.09525046},
    {0.88627146, 0.89237353, 0.09537439},
    {0.89632002, 0.89361614, 0.09633538},
    {0.90631121, 0.89485467, 0.09812496},
    {0.91624212, 0.89609127, 0.1007168 },
    {0.92610579, 0.89732977, 0.10407067},
    {0.93590444, 0.8985704 , 0.10813094},
    {0.94563626, 0.899815  , 0.11283773},
    {0.95529972, 0.90106534, 0.11812832},
    {0.96489353, 0.90232311, 0.12394051},
    {0.97441665, 0.90358991, 0.13021494},
    {0.98386829, 0.90486726, 0.13689671},
    {0.99324789, 0.90615657, 0.1439362 }};
  int i = (int)floor(v * 255);
  **r = cmap_data[i][0];
  **g = cmap_data[i][1];
  **b = cmap_data[i][2];
}


// Turbo colormap - Google
void get_cmap_turbo(float v, float **r, float **g, float **b)
{
  float cmap_data[256][3] = {
    {0.18995, 0.07176, 0.23217},
    {0.19483, 0.08339, 0.26149},
    {0.19956, 0.09498, 0.29024},
    {0.20415, 0.10652, 0.31844},
    {0.20860, 0.11802, 0.34607},
    {0.21291, 0.12947, 0.37314},
    {0.21708, 0.14087, 0.39964},
    {0.22111, 0.15223, 0.42558},
    {0.22500, 0.16354, 0.45096},
    {0.22875, 0.17481, 0.47578},
    {0.23236, 0.18603, 0.50004},
    {0.23582, 0.19720, 0.52373},
    {0.23915, 0.20833, 0.54686},
    {0.24234, 0.21941, 0.56942},
    {0.24539, 0.23044, 0.59142},
    {0.24830, 0.24143, 0.61286},
    {0.25107, 0.25237, 0.63374},
    {0.25369, 0.26327, 0.65406},
    {0.25618, 0.27412, 0.67381},
    {0.25853, 0.28492, 0.69300},
    {0.26074, 0.29568, 0.71162},
    {0.26280, 0.30639, 0.72968},
    {0.26473, 0.31706, 0.74718},
    {0.26652, 0.32768, 0.76412},
    {0.26816, 0.33825, 0.78050},
    {0.26967, 0.34878, 0.79631},
    {0.27103, 0.35926, 0.81156},
    {0.27226, 0.36970, 0.82624},
    {0.27334, 0.38008, 0.84037},
    {0.27429, 0.39043, 0.85393},
    {0.27509, 0.40072, 0.86692},
    {0.27576, 0.41097, 0.87936},
    {0.27628, 0.42118, 0.89123},
    {0.27667, 0.43134, 0.90254},
    {0.27691, 0.44145, 0.91328},
    {0.27701, 0.45152, 0.92347},
    {0.27698, 0.46153, 0.93309},
    {0.27680, 0.47151, 0.94214},
    {0.27648, 0.48144, 0.95064},
    {0.27603, 0.49132, 0.95857},
    {0.27543, 0.50115, 0.96594},
    {0.27469, 0.51094, 0.97275},
    {0.27381, 0.52069, 0.97899},
    {0.27273, 0.53040, 0.98461},
    {0.27106, 0.54015, 0.98930},
    {0.26878, 0.54995, 0.99303},
    {0.26592, 0.55979, 0.99583},
    {0.26252, 0.56967, 0.99773},
    {0.25862, 0.57958, 0.99876},
    {0.25425, 0.58950, 0.99896},
    {0.24946, 0.59943, 0.99835},
    {0.24427, 0.60937, 0.99697},
    {0.23874, 0.61931, 0.99485},
    {0.23288, 0.62923, 0.99202},
    {0.22676, 0.63913, 0.98851},
    {0.22039, 0.64901, 0.98436},
    {0.21382, 0.65886, 0.97959},
    {0.20708, 0.66866, 0.97423},
    {0.20021, 0.67842, 0.96833},
    {0.19326, 0.68812, 0.96190},
    {0.18625, 0.69775, 0.95498},
    {0.17923, 0.70732, 0.94761},
    {0.17223, 0.71680, 0.93981},
    {0.16529, 0.72620, 0.93161},
    {0.15844, 0.73551, 0.92305},
    {0.15173, 0.74472, 0.91416},
    {0.14519, 0.75381, 0.90496},
    {0.13886, 0.76279, 0.89550},
    {0.13278, 0.77165, 0.88580},
    {0.12698, 0.78037, 0.87590},
    {0.12151, 0.78896, 0.86581},
    {0.11639, 0.79740, 0.85559},
    {0.11167, 0.80569, 0.84525},
    {0.10738, 0.81381, 0.83484},
    {0.10357, 0.82177, 0.82437},
    {0.10026, 0.82955, 0.81389},
    {0.09750, 0.83714, 0.80342},
    {0.09532, 0.84455, 0.79299},
    {0.09377, 0.85175, 0.78264},
    {0.09287, 0.85875, 0.77240},
    {0.09267, 0.86554, 0.76230},
    {0.09320, 0.87211, 0.75237},
    {0.09451, 0.87844, 0.74265},
    {0.09662, 0.88454, 0.73316},
    {0.09958, 0.89040, 0.72393},
    {0.10342, 0.89600, 0.71500},
    {0.10815, 0.90142, 0.70599},
    {0.11374, 0.90673, 0.69651},
    {0.12014, 0.91193, 0.68660},
    {0.12733, 0.91701, 0.67627},
    {0.13526, 0.92197, 0.66556},
    {0.14391, 0.92680, 0.65448},
    {0.15323, 0.93151, 0.64308},
    {0.16319, 0.93609, 0.63137},
    {0.17377, 0.94053, 0.61938},
    {0.18491, 0.94484, 0.60713},
    {0.19659, 0.94901, 0.59466},
    {0.20877, 0.95304, 0.58199},
    {0.22142, 0.95692, 0.56914},
    {0.23449, 0.96065, 0.55614},
    {0.24797, 0.96423, 0.54303},
    {0.26180, 0.96765, 0.52981},
    {0.27597, 0.97092, 0.51653},
    {0.29042, 0.97403, 0.50321},
    {0.30513, 0.97697, 0.48987},
    {0.32006, 0.97974, 0.47654},
    {0.33517, 0.98234, 0.46325},
    {0.35043, 0.98477, 0.45002},
    {0.36581, 0.98702, 0.43688},
    {0.38127, 0.98909, 0.42386},
    {0.39678, 0.99098, 0.41098},
    {0.41229, 0.99268, 0.39826},
    {0.42778, 0.99419, 0.38575},
    {0.44321, 0.99551, 0.37345},
    {0.45854, 0.99663, 0.36140},
    {0.47375, 0.99755, 0.34963},
    {0.48879, 0.99828, 0.33816},
    {0.50362, 0.99879, 0.32701},
    {0.51822, 0.99910, 0.31622},
    {0.53255, 0.99919, 0.30581},
    {0.54658, 0.99907, 0.29581},
    {0.56026, 0.99873, 0.28623},
    {0.57357, 0.99817, 0.27712},
    {0.58646, 0.99739, 0.26849},
    {0.59891, 0.99638, 0.26038},
    {0.61088, 0.99514, 0.25280},
    {0.62233, 0.99366, 0.24579},
    {0.63323, 0.99195, 0.23937},
    {0.64362, 0.98999, 0.23356},
    {0.65394, 0.98775, 0.22835},
    {0.66428, 0.98524, 0.22370},
    {0.67462, 0.98246, 0.21960},
    {0.68494, 0.97941, 0.21602},
    {0.69525, 0.97610, 0.21294},
    {0.70553, 0.97255, 0.21032},
    {0.71577, 0.96875, 0.20815},
    {0.72596, 0.96470, 0.20640},
    {0.73610, 0.96043, 0.20504},
    {0.74617, 0.95593, 0.20406},
    {0.75617, 0.95121, 0.20343},
    {0.76608, 0.94627, 0.20311},
    {0.77591, 0.94113, 0.20310},
    {0.78563, 0.93579, 0.20336},
    {0.79524, 0.93025, 0.20386},
    {0.80473, 0.92452, 0.20459},
    {0.81410, 0.91861, 0.20552},
    {0.82333, 0.91253, 0.20663},
    {0.83241, 0.90627, 0.20788},
    {0.84133, 0.89986, 0.20926},
    {0.85010, 0.89328, 0.21074},
    {0.85868, 0.88655, 0.21230},
    {0.86709, 0.87968, 0.21391},
    {0.87530, 0.87267, 0.21555},
    {0.88331, 0.86553, 0.21719},
    {0.89112, 0.85826, 0.21880},
    {0.89870, 0.85087, 0.22038},
    {0.90605, 0.84337, 0.22188},
    {0.91317, 0.83576, 0.22328},
    {0.92004, 0.82806, 0.22456},
    {0.92666, 0.82025, 0.22570},
    {0.93301, 0.81236, 0.22667},
    {0.93909, 0.80439, 0.22744},
    {0.94489, 0.79634, 0.22800},
    {0.95039, 0.78823, 0.22831},
    {0.95560, 0.78005, 0.22836},
    {0.96049, 0.77181, 0.22811},
    {0.96507, 0.76352, 0.22754},
    {0.96931, 0.75519, 0.22663},
    {0.97323, 0.74682, 0.22536},
    {0.97679, 0.73842, 0.22369},
    {0.98000, 0.73000, 0.22161},
    {0.98289, 0.72140, 0.21918},
    {0.98549, 0.71250, 0.21650},
    {0.98781, 0.70330, 0.21358},
    {0.98986, 0.69382, 0.21043},
    {0.99163, 0.68408, 0.20706},
    {0.99314, 0.67408, 0.20348},
    {0.99438, 0.66386, 0.19971},
    {0.99535, 0.65341, 0.19577},
    {0.99607, 0.64277, 0.19165},
    {0.99654, 0.63193, 0.18738},
    {0.99675, 0.62093, 0.18297},
    {0.99672, 0.60977, 0.17842},
    {0.99644, 0.59846, 0.17376},
    {0.99593, 0.58703, 0.16899},
    {0.99517, 0.57549, 0.16412},
    {0.99419, 0.56386, 0.15918},
    {0.99297, 0.55214, 0.15417},
    {0.99153, 0.54036, 0.14910},
    {0.98987, 0.52854, 0.14398},
    {0.98799, 0.51667, 0.13883},
    {0.98590, 0.50479, 0.13367},
    {0.98360, 0.49291, 0.12849},
    {0.98108, 0.48104, 0.12332},
    {0.97837, 0.46920, 0.11817},
    {0.97545, 0.45740, 0.11305},
    {0.97234, 0.44565, 0.10797},
    {0.96904, 0.43399, 0.10294},
    {0.96555, 0.42241, 0.09798},
    {0.96187, 0.41093, 0.09310},
    {0.95801, 0.39958, 0.08831},
    {0.95398, 0.38836, 0.08362},
    {0.94977, 0.37729, 0.07905},
    {0.94538, 0.36638, 0.07461},
    {0.94084, 0.35566, 0.07031},
    {0.93612, 0.34513, 0.06616},
    {0.93125, 0.33482, 0.06218},
    {0.92623, 0.32473, 0.05837},
    {0.92105, 0.31489, 0.05475},
    {0.91572, 0.30530, 0.05134},
    {0.91024, 0.29599, 0.04814},
    {0.90463, 0.28696, 0.04516},
    {0.89888, 0.27824, 0.04243},
    {0.89298, 0.26981, 0.03993},
    {0.88691, 0.26152, 0.03753},
    {0.88066, 0.25334, 0.03521},
    {0.87422, 0.24526, 0.03297},
    {0.86760, 0.23730, 0.03082},
    {0.86079, 0.22945, 0.02875},
    {0.85380, 0.22170, 0.02677},
    {0.84662, 0.21407, 0.02487},
    {0.83926, 0.20654, 0.02305},
    {0.83172, 0.19912, 0.02131},
    {0.82399, 0.19182, 0.01966},
    {0.81608, 0.18462, 0.01809},
    {0.80799, 0.17753, 0.01660},
    {0.79971, 0.17055, 0.01520},
    {0.79125, 0.16368, 0.01387},
    {0.78260, 0.15693, 0.01264},
    {0.77377, 0.15028, 0.01148},
    {0.76476, 0.14374, 0.01041},
    {0.75556, 0.13731, 0.00942},
    {0.74617, 0.13098, 0.00851},
    {0.73661, 0.12477, 0.00769},
    {0.72686, 0.11867, 0.00695},
    {0.71692, 0.11268, 0.00629},
    {0.70680, 0.10680, 0.00571},
    {0.69650, 0.10102, 0.00522},
    {0.68602, 0.09536, 0.00481},
    {0.67535, 0.08980, 0.00449},
    {0.66449, 0.08436, 0.00424},
    {0.65345, 0.07902, 0.00408},
    {0.64223, 0.07380, 0.00401},
    {0.63082, 0.06868, 0.00401},
    {0.61923, 0.06367, 0.00410},
    {0.60746, 0.05878, 0.00427},
    {0.59550, 0.05399, 0.00453},
    {0.58336, 0.04931, 0.00486},
    {0.57103, 0.04474, 0.00529},
    {0.55852, 0.04028, 0.00579},
    {0.54583, 0.03593, 0.00638},
    {0.53295, 0.03169, 0.00705},
    {0.51989, 0.02756, 0.00780},
    {0.50664, 0.02354, 0.00863},
    {0.49321, 0.01963, 0.00955},
    {0.47960, 0.01583, 0.01055}};
  int i = (int)floor(v * 255);
  **r = cmap_data[i][0];
  **g = cmap_data[i][1];
  **b = cmap_data[i][2];
}


// Inferno colormap - Berkeley Institute for Data Science
void get_cmap_inferno(float v, float **r, float **g, float **b)
{
  float cmap_data[256][3] = {
    {1.46159096e-03, 4.66127766e-04, 1.38655200e-02},
    {2.26726368e-03, 1.26992553e-03, 1.85703520e-02},
    {3.29899092e-03, 2.24934863e-03, 2.42390508e-02},
    {4.54690615e-03, 3.39180156e-03, 3.09092475e-02},
    {6.00552565e-03, 4.69194561e-03, 3.85578980e-02},
    {7.67578856e-03, 6.13611626e-03, 4.68360336e-02},
    {9.56051094e-03, 7.71344131e-03, 5.51430756e-02},
    {1.16634769e-02, 9.41675403e-03, 6.34598080e-02},
    {1.39950388e-02, 1.12247138e-02, 7.18616890e-02},
    {1.65605595e-02, 1.31362262e-02, 8.02817951e-02},
    {1.93732295e-02, 1.51325789e-02, 8.87668094e-02},
    {2.24468865e-02, 1.71991484e-02, 9.73274383e-02},
    {2.57927373e-02, 1.93306298e-02, 1.05929835e-01},
    {2.94324251e-02, 2.15030771e-02, 1.14621328e-01},
    {3.33852235e-02, 2.37024271e-02, 1.23397286e-01},
    {3.76684211e-02, 2.59207864e-02, 1.32232108e-01},
    {4.22525554e-02, 2.81385015e-02, 1.41140519e-01},
    {4.69146287e-02, 3.03236129e-02, 1.50163867e-01},
    {5.16437624e-02, 3.24736172e-02, 1.59254277e-01},
    {5.64491009e-02, 3.45691867e-02, 1.68413539e-01},
    {6.13397200e-02, 3.65900213e-02, 1.77642172e-01},
    {6.63312620e-02, 3.85036268e-02, 1.86961588e-01},
    {7.14289181e-02, 4.02939095e-02, 1.96353558e-01},
    {7.66367560e-02, 4.19053329e-02, 2.05798788e-01},
    {8.19620773e-02, 4.33278666e-02, 2.15289113e-01},
    {8.74113897e-02, 4.45561662e-02, 2.24813479e-01},
    {9.29901526e-02, 4.55829503e-02, 2.34357604e-01},
    {9.87024972e-02, 4.64018731e-02, 2.43903700e-01},
    {1.04550936e-01, 4.70080541e-02, 2.53430300e-01},
    {1.10536084e-01, 4.73986708e-02, 2.62912235e-01},
    {1.16656423e-01, 4.75735920e-02, 2.72320803e-01},
    {1.22908126e-01, 4.75360183e-02, 2.81624170e-01},
    {1.29284984e-01, 4.72930838e-02, 2.90788012e-01},
    {1.35778450e-01, 4.68563678e-02, 2.99776404e-01},
    {1.42377819e-01, 4.62422566e-02, 3.08552910e-01},
    {1.49072957e-01, 4.54676444e-02, 3.17085139e-01},
    {1.55849711e-01, 4.45588056e-02, 3.25338414e-01},
    {1.62688939e-01, 4.35542881e-02, 3.33276678e-01},
    {1.69575148e-01, 4.24893149e-02, 3.40874188e-01},
    {1.76493202e-01, 4.14017089e-02, 3.48110606e-01},
    {1.83428775e-01, 4.03288858e-02, 3.54971391e-01},
    {1.90367453e-01, 3.93088888e-02, 3.61446945e-01},
    {1.97297425e-01, 3.84001825e-02, 3.67534629e-01},
    {2.04209298e-01, 3.76322609e-02, 3.73237557e-01},
    {2.11095463e-01, 3.70296488e-02, 3.78563264e-01},
    {2.17948648e-01, 3.66146049e-02, 3.83522415e-01},
    {2.24762908e-01, 3.64049901e-02, 3.88128944e-01},
    {2.31538148e-01, 3.64052511e-02, 3.92400150e-01},
    {2.38272961e-01, 3.66209949e-02, 3.96353388e-01},
    {2.44966911e-01, 3.70545017e-02, 4.00006615e-01},
    {2.51620354e-01, 3.77052832e-02, 4.03377897e-01},
    {2.58234265e-01, 3.85706153e-02, 4.06485031e-01},
    {2.64809649e-01, 3.96468666e-02, 4.09345373e-01},
    {2.71346664e-01, 4.09215821e-02, 4.11976086e-01},
    {2.77849829e-01, 4.23528741e-02, 4.14392106e-01},
    {2.84321318e-01, 4.39325787e-02, 4.16607861e-01},
    {2.90763373e-01, 4.56437598e-02, 4.18636756e-01},
    {2.97178251e-01, 4.74700293e-02, 4.20491164e-01},
    {3.03568182e-01, 4.93958927e-02, 4.22182449e-01},
    {3.09935342e-01, 5.14069729e-02, 4.23720999e-01},
    {3.16281835e-01, 5.34901321e-02, 4.25116277e-01},
    {3.22609671e-01, 5.56335178e-02, 4.26376869e-01},
    {3.28920763e-01, 5.78265505e-02, 4.27510546e-01},
    {3.35216916e-01, 6.00598734e-02, 4.28524320e-01},
    {3.41499828e-01, 6.23252772e-02, 4.29424503e-01},
    {3.47771086e-01, 6.46156100e-02, 4.30216765e-01},
    {3.54032169e-01, 6.69246832e-02, 4.30906186e-01},
    {3.60284449e-01, 6.92471753e-02, 4.31497309e-01},
    {3.66529195e-01, 7.15785403e-02, 4.31994185e-01},
    {3.72767575e-01, 7.39149211e-02, 4.32400419e-01},
    {3.79000659e-01, 7.62530701e-02, 4.32719214e-01},
    {3.85228383e-01, 7.85914864e-02, 4.32954973e-01},
    {3.91452659e-01, 8.09267058e-02, 4.33108763e-01},
    {3.97674379e-01, 8.32568129e-02, 4.33182647e-01},
    {4.03894278e-01, 8.55803445e-02, 4.33178526e-01},
    {4.10113015e-01, 8.78961593e-02, 4.33098056e-01},
    {4.16331169e-01, 9.02033992e-02, 4.32942678e-01},
    {4.22549249e-01, 9.25014543e-02, 4.32713635e-01},
    {4.28767696e-01, 9.47899342e-02, 4.32411996e-01},
    {4.34986885e-01, 9.70686417e-02, 4.32038673e-01},
    {4.41207124e-01, 9.93375510e-02, 4.31594438e-01},
    {4.47428382e-01, 1.01597079e-01, 4.31080497e-01},
    {4.53650614e-01, 1.03847716e-01, 4.30497898e-01},
    {4.59874623e-01, 1.06089165e-01, 4.29845789e-01},
    {4.66100494e-01, 1.08321923e-01, 4.29124507e-01},
    {4.72328255e-01, 1.10546584e-01, 4.28334320e-01},
    {4.78557889e-01, 1.12763831e-01, 4.27475431e-01},
    {4.84789325e-01, 1.14974430e-01, 4.26547991e-01},
    {4.91022448e-01, 1.17179219e-01, 4.25552106e-01},
    {4.97257069e-01, 1.19379132e-01, 4.24487908e-01},
    {5.03492698e-01, 1.21575414e-01, 4.23356110e-01},
    {5.09729541e-01, 1.23768654e-01, 4.22155676e-01},
    {5.15967304e-01, 1.25959947e-01, 4.20886594e-01},
    {5.22205646e-01, 1.28150439e-01, 4.19548848e-01},
    {5.28444192e-01, 1.30341324e-01, 4.18142411e-01},
    {5.34682523e-01, 1.32533845e-01, 4.16667258e-01},
    {5.40920186e-01, 1.34729286e-01, 4.15123366e-01},
    {5.47156706e-01, 1.36928959e-01, 4.13510662e-01},
    {5.53391649e-01, 1.39134147e-01, 4.11828882e-01},
    {5.59624442e-01, 1.41346265e-01, 4.10078028e-01},
    {5.65854477e-01, 1.43566769e-01, 4.08258132e-01},
    {5.72081108e-01, 1.45797150e-01, 4.06369246e-01},
    {5.78303656e-01, 1.48038934e-01, 4.04411444e-01},
    {5.84521407e-01, 1.50293679e-01, 4.02384829e-01},
    {5.90733615e-01, 1.52562977e-01, 4.00289528e-01},
    {5.96939751e-01, 1.54848232e-01, 3.98124897e-01},
    {6.03138930e-01, 1.57151161e-01, 3.95891308e-01},
    {6.09330184e-01, 1.59473549e-01, 3.93589349e-01},
    {6.15512627e-01, 1.61817111e-01, 3.91219295e-01},
    {6.21685340e-01, 1.64183582e-01, 3.88781456e-01},
    {6.27847374e-01, 1.66574724e-01, 3.86276180e-01},
    {6.33997746e-01, 1.68992314e-01, 3.83703854e-01},
    {6.40135447e-01, 1.71438150e-01, 3.81064906e-01},
    {6.46259648e-01, 1.73913876e-01, 3.78358969e-01},
    {6.52369348e-01, 1.76421271e-01, 3.75586209e-01},
    {6.58463166e-01, 1.78962399e-01, 3.72748214e-01},
    {6.64539964e-01, 1.81539111e-01, 3.69845599e-01},
    {6.70598572e-01, 1.84153268e-01, 3.66879025e-01},
    {6.76637795e-01, 1.86806728e-01, 3.63849195e-01},
    {6.82656407e-01, 1.89501352e-01, 3.60756856e-01},
    {6.88653158e-01, 1.92238994e-01, 3.57602797e-01},
    {6.94626769e-01, 1.95021500e-01, 3.54387853e-01},
    {7.00575937e-01, 1.97850703e-01, 3.51112900e-01},
    {7.06499709e-01, 2.00728196e-01, 3.47776863e-01},
    {7.12396345e-01, 2.03656029e-01, 3.44382594e-01},
    {7.18264447e-01, 2.06635993e-01, 3.40931208e-01},
    {7.24102613e-01, 2.09669834e-01, 3.37423766e-01},
    {7.29909422e-01, 2.12759270e-01, 3.33861367e-01},
    {7.35683432e-01, 2.15905976e-01, 3.30245147e-01},
    {7.41423185e-01, 2.19111589e-01, 3.26576275e-01},
    {7.47127207e-01, 2.22377697e-01, 3.22855952e-01},
    {7.52794009e-01, 2.25705837e-01, 3.19085410e-01},
    {7.58422090e-01, 2.29097492e-01, 3.15265910e-01},
    {7.64009940e-01, 2.32554083e-01, 3.11398734e-01},
    {7.69556038e-01, 2.36076967e-01, 3.07485188e-01},
    {7.75058888e-01, 2.39667435e-01, 3.03526312e-01},
    {7.80517023e-01, 2.43326720e-01, 2.99522665e-01},
    {7.85928794e-01, 2.47055968e-01, 2.95476756e-01},
    {7.91292674e-01, 2.50856232e-01, 2.91389943e-01},
    {7.96607144e-01, 2.54728485e-01, 2.87263585e-01},
    {8.01870689e-01, 2.58673610e-01, 2.83099033e-01},
    {8.07081807e-01, 2.62692401e-01, 2.78897629e-01},
    {8.12239008e-01, 2.66785558e-01, 2.74660698e-01},
    {8.17340818e-01, 2.70953688e-01, 2.70389545e-01},
    {8.22385784e-01, 2.75197300e-01, 2.66085445e-01},
    {8.27372474e-01, 2.79516805e-01, 2.61749643e-01},
    {8.32299481e-01, 2.83912516e-01, 2.57383341e-01},
    {8.37165425e-01, 2.88384647e-01, 2.52987700e-01},
    {8.41968959e-01, 2.92933312e-01, 2.48563825e-01},
    {8.46708768e-01, 2.97558528e-01, 2.44112767e-01},
    {8.51383572e-01, 3.02260213e-01, 2.39635512e-01},
    {8.55992130e-01, 3.07038188e-01, 2.35132978e-01},
    {8.60533241e-01, 3.11892183e-01, 2.30606009e-01},
    {8.65005747e-01, 3.16821833e-01, 2.26055368e-01},
    {8.69408534e-01, 3.21826685e-01, 2.21481734e-01},
    {8.73740530e-01, 3.26906201e-01, 2.16885699e-01},
    {8.78000715e-01, 3.32059760e-01, 2.12267762e-01},
    {8.82188112e-01, 3.37286663e-01, 2.07628326e-01},
    {8.86301795e-01, 3.42586137e-01, 2.02967696e-01},
    {8.90340885e-01, 3.47957340e-01, 1.98286080e-01},
    {8.94304553e-01, 3.53399363e-01, 1.93583583e-01},
    {8.98192017e-01, 3.58911240e-01, 1.88860212e-01},
    {9.02002544e-01, 3.64491949e-01, 1.84115876e-01},
    {9.05735448e-01, 3.70140419e-01, 1.79350388e-01},
    {9.09390090e-01, 3.75855533e-01, 1.74563472e-01},
    {9.12965874e-01, 3.81636138e-01, 1.69754764e-01},
    {9.16462251e-01, 3.87481044e-01, 1.64923826e-01},
    {9.19878710e-01, 3.93389034e-01, 1.60070152e-01},
    {9.23214783e-01, 3.99358867e-01, 1.55193185e-01},
    {9.26470039e-01, 4.05389282e-01, 1.50292329e-01},
    {9.29644083e-01, 4.11479007e-01, 1.45366973e-01},
    {9.32736555e-01, 4.17626756e-01, 1.40416519e-01},
    {9.35747126e-01, 4.23831237e-01, 1.35440416e-01},
    {9.38675494e-01, 4.30091162e-01, 1.30438175e-01},
    {9.41521384e-01, 4.36405243e-01, 1.25409440e-01},
    {9.44284543e-01, 4.42772199e-01, 1.20354038e-01},
    {9.46964741e-01, 4.49190757e-01, 1.15272059e-01},
    {9.49561766e-01, 4.55659658e-01, 1.10163947e-01},
    {9.52075421e-01, 4.62177656e-01, 1.05030614e-01},
    {9.54505523e-01, 4.68743522e-01, 9.98735931e-02},
    {9.56851903e-01, 4.75356048e-01, 9.46952268e-02},
    {9.59114397e-01, 4.82014044e-01, 8.94989073e-02},
    {9.61292850e-01, 4.88716345e-01, 8.42893891e-02},
    {9.63387110e-01, 4.95461806e-01, 7.90731907e-02},
    {9.65397031e-01, 5.02249309e-01, 7.38591143e-02},
    {9.67322465e-01, 5.09077761e-01, 6.86589199e-02},
    {9.69163264e-01, 5.15946092e-01, 6.34881971e-02},
    {9.70919277e-01, 5.22853259e-01, 5.83674890e-02},
    {9.72590351e-01, 5.29798246e-01, 5.33237243e-02},
    {9.74176327e-01, 5.36780059e-01, 4.83920090e-02},
    {9.75677038e-01, 5.43797733e-01, 4.36177922e-02},
    {9.77092313e-01, 5.50850323e-01, 3.90500131e-02},
    {9.78421971e-01, 5.57936911e-01, 3.49306227e-02},
    {9.79665824e-01, 5.65056600e-01, 3.14091591e-02},
    {9.80823673e-01, 5.72208516e-01, 2.85075931e-02},
    {9.81895311e-01, 5.79391803e-01, 2.62497353e-02},
    {9.82880522e-01, 5.86605627e-01, 2.46613416e-02},
    {9.83779081e-01, 5.93849168e-01, 2.37702263e-02},
    {9.84590755e-01, 6.01121626e-01, 2.36063833e-02},
    {9.85315301e-01, 6.08422211e-01, 2.42021174e-02},
    {9.85952471e-01, 6.15750147e-01, 2.55921853e-02},
    {9.86502013e-01, 6.23104667e-01, 2.78139496e-02},
    {9.86963670e-01, 6.30485011e-01, 3.09075459e-02},
    {9.87337182e-01, 6.37890424e-01, 3.49160639e-02},
    {9.87622296e-01, 6.45320152e-01, 3.98857472e-02},
    {9.87818759e-01, 6.52773439e-01, 4.55808037e-02},
    {9.87926330e-01, 6.60249526e-01, 5.17503867e-02},
    {9.87944783e-01, 6.67747641e-01, 5.83286889e-02},
    {9.87873910e-01, 6.75267000e-01, 6.52570167e-02},
    {9.87713535e-01, 6.82806802e-01, 7.24892330e-02},
    {9.87463516e-01, 6.90366218e-01, 7.99897176e-02},
    {9.87123759e-01, 6.97944391e-01, 8.77314215e-02},
    {9.86694229e-01, 7.05540424e-01, 9.56941797e-02},
    {9.86174970e-01, 7.13153375e-01, 1.03863324e-01},
    {9.85565739e-01, 7.20782460e-01, 1.12228756e-01},
    {9.84865203e-01, 7.28427497e-01, 1.20784651e-01},
    {9.84075129e-01, 7.36086521e-01, 1.29526579e-01},
    {9.83195992e-01, 7.43758326e-01, 1.38453063e-01},
    {9.82228463e-01, 7.51441596e-01, 1.47564573e-01},
    {9.81173457e-01, 7.59134892e-01, 1.56863224e-01},
    {9.80032178e-01, 7.66836624e-01, 1.66352544e-01},
    {9.78806183e-01, 7.74545028e-01, 1.76037298e-01},
    {9.77497453e-01, 7.82258138e-01, 1.85923357e-01},
    {9.76108474e-01, 7.89973753e-01, 1.96017589e-01},
    {9.74637842e-01, 7.97691563e-01, 2.06331925e-01},
    {9.73087939e-01, 8.05409333e-01, 2.16876839e-01},
    {9.71467822e-01, 8.13121725e-01, 2.27658046e-01},
    {9.69783146e-01, 8.20825143e-01, 2.38685942e-01},
    {9.68040817e-01, 8.28515491e-01, 2.49971582e-01},
    {9.66242589e-01, 8.36190976e-01, 2.61533898e-01},
    {9.64393924e-01, 8.43848069e-01, 2.73391112e-01},
    {9.62516656e-01, 8.51476340e-01, 2.85545675e-01},
    {9.60625545e-01, 8.59068716e-01, 2.98010219e-01},
    {9.58720088e-01, 8.66624355e-01, 3.10820466e-01},
    {9.56834075e-01, 8.74128569e-01, 3.23973947e-01},
    {9.54997177e-01, 8.81568926e-01, 3.37475479e-01},
    {9.53215092e-01, 8.88942277e-01, 3.51368713e-01},
    {9.51546225e-01, 8.96225909e-01, 3.65627005e-01},
    {9.50018481e-01, 9.03409063e-01, 3.80271225e-01},
    {9.48683391e-01, 9.10472964e-01, 3.95289169e-01},
    {9.47594362e-01, 9.17399053e-01, 4.10665194e-01},
    {9.46809163e-01, 9.24168246e-01, 4.26373236e-01},
    {9.46391536e-01, 9.30760752e-01, 4.42367495e-01},
    {9.46402951e-01, 9.37158971e-01, 4.58591507e-01},
    {9.46902568e-01, 9.43347775e-01, 4.74969778e-01},
    {9.47936825e-01, 9.49317522e-01, 4.91426053e-01},
    {9.49544830e-01, 9.55062900e-01, 5.07859649e-01},
    {9.51740304e-01, 9.60586693e-01, 5.24203026e-01},
    {9.54529281e-01, 9.65895868e-01, 5.40360752e-01},
    {9.57896053e-01, 9.71003330e-01, 5.56275090e-01},
    {9.61812020e-01, 9.75924241e-01, 5.71925382e-01},
    {9.66248822e-01, 9.80678193e-01, 5.87205773e-01},
    {9.71161622e-01, 9.85282161e-01, 6.02154330e-01},
    {9.76510983e-01, 9.89753437e-01, 6.16760413e-01},
    {9.82257307e-01, 9.94108844e-01, 6.31017009e-01},
    {9.88362068e-01, 9.98364143e-01, 6.44924005e-01}};
  int i = (int)floor(v * 255);
  **r = cmap_data[i][0];
  **g = cmap_data[i][1];
  **b = cmap_data[i][2];
}


// provided by Rafal Brzegowy
void get_cmap_jet(float v, float **r, float **g, float **b)
{
	if ( v <= 0.375 )       **r = 0.;
	else if ( v <= 0.625 )  **r = (v-.375)/.25;
        else if ( v <= 0.875 ) 	**r = 1.;
        else                  	**r = 1.-((v-.875)/.125)*.5;
	
	if ( v <= 0.125 )       **g = 0.;
        else if ( v <= 0.375 )  **g = (v-.125)/.25;
        else if ( v <= 0.625 )  **g = 1.;
        else if ( v <= 0.875 )  **g = (.875-v)/.25;
        else		        **g = 0.;
	
        if ( v <= 0.125 )      	**b = .5+(v/.125)*.5;
        else if ( v <= 0.375 )	**b = 1.;
        else if ( v <= 0.625 )	**b = (.625-v)/.25;
        else                 	**b =  0.;
}



// CoolWarm colormap - Sandia National Laboratories (Moreland,2009) provided by Suyono Nt.
void get_cmap_coolwarm(float v, float **r, float **g, float **b)
{
  float cmap_data[256][3] = {
    {0.229999503869523, 0.298998934049375, 0.754000138575591},
    {0.234517509186022, 0.305864718251243, 0.760211287847582},
    {0.23905139222321, 0.31271835359723, 0.766361370695118},
    {0.243601736591344, 0.319559616592088, 0.772449408370893},
    {0.24816908043486, 0.326388233402671, 0.77847443329274},
    {0.252753918157713, 0.333203884627665, 0.784435489186598},
    {0.257356702007336, 0.3400062095466, 0.790331631227838},
    {0.261977843534708, 0.346794809915534, 0.796161926180882},
    {0.266617714945733, 0.35356925336688, 0.801925452537047},
    {0.271276650357161, 0.360329076462606, 0.80762130065058},
    {0.275954946968569, 0.367073787443038, 0.813248572872804},
    {0.280652866160487, 0.373802868707694, 0.818806383684336},
    {0.285370634527404, 0.380515779059584, 0.824293859825332},
    {0.290108444853348, 0.387211955740263, 0.829710140423685},
    {0.294866457036705, 0.39389081627931, 0.83505437712116},
    {0.299644798970133, 0.400551760178892, 0.8403257341974},
    {0.304443567380646, 0.407194170451442, 0.845523388691759},
    {0.309262828634354, 0.413817415026243, 0.850646530522937},
    {0.314102619509701, 0.420420848038791, 0.855694362606348},
    {0.318962947942621, 0.42700381101513, 0.86066610096921},
    {0.323843793746534, 0.433565633961902, 0.865560974863305},
    {0.328745109309755, 0.440105636371649, 0.87037822687536},
    {0.333666820272563, 0.446623128151737, 0.875117113035049},
    {0.338608826185831, 0.453117410484401, 0.879776902920533},
    {0.343571001152916, 0.459587776624529, 0.884356879761559},
    {0.348553194456271, 0.466033512641087, 0.888856340540034},
    {0.353555231169988, 0.472453898107474, 0.893274596088089},
    {0.358576912759404, 0.478848206745484, 0.897610971183564},
    {0.363618017668653, 0.48521570702712, 0.901864804642922},
    {0.368678301896987, 0.491555662738006, 0.90603544941153},
    {0.373757499564537, 0.497867333505808, 0.910122272651316},
    {0.378855323468087, 0.504149975296706, 0.914124655825743},
    {0.383971465627369, 0.510402840882649, 0.918041994782103},
    {0.389105597822293, 0.516625180281884, 0.921873699831088},
    {0.394257372121458, 0.522816241174981, 0.925619195823635},
    {0.399426421402254, 0.528975269298376, 0.929277922225006},
    {0.404612359862811, 0.535101508817254, 0.932849333186093},
    {0.409814783525984, 0.541194202679442, 0.936332897611926},
    {0.415033270735587, 0.547252592951789, 0.939728099227369},
    {0.420267382644994, 0.553275921140423, 0.943034436639982},
    {0.425516663698255, 0.559263428496111, 0.946251423400032},
    {0.430780642103825, 0.565214356305846, 0.949378588057647},
    {0.436058830300991, 0.571127946171707, 0.952415474217086},
    {0.4413507254191, 0.577003440277912, 0.955361640588115},
    {0.44665580972962, 0.58284008164693, 0.958216661034486},
    {0.451973551091126, 0.588637114385435, 0.960980124619491},
    {0.457303403387259, 0.594393783920807, 0.963651635648588},
    {0.462644806957693, 0.600109337228853, 0.966230813709096},
    {0.467997189022184, 0.605783023053323, 0.968717293706932},
    {0.473359964097731, 0.611414092117798, 0.97111072590039},
    {0.478732534408903, 0.617001797330432, 0.97341077593097},
    {0.484114290291382, 0.622545393982025, 0.975617124851209},
    {0.489504610588761, 0.628044139937846, 0.977729469149558},
    {0.494902863042672, 0.633497295823597, 0.979747520772257},
    {0.500308404676262, 0.638904125205883, 0.981671007142227},
    {0.505720582171105, 0.644263894767512, 0.983499671174965},
    {0.511138732237593, 0.649575874477933, 0.985233271291444},
    {0.516562181978859, 0.654839337759102, 0.986871581428013},
    {0.521990249248321, 0.660053561647008, 0.988414391043284},
    {0.527422243000888, 0.665217826949137, 0.989861505122032},
    {0.532857463637911, 0.670331418398056, 0.991212744176074},
    {0.538295203345948, 0.675393624801347, 0.99246794424215},
    {0.543734746429415, 0.680403739188069, 0.993626956876802},
    {0.549175369637206, 0.685361058951912, 0.994689649148243},
    {0.55461634248336, 0.690264885991224, 0.995655903625224},
    {0.560056927561847, 0.695114526846042, 0.996525618362914},
    {0.565496380855573, 0.699909292832266, 0.997298706885765},
    {0.570933952039671, 0.70464850017311, 0.997975098167398},
    {0.57636888477919, 0.70933147012794, 0.998554736607497},
    {0.581800417021229, 0.713957529118605, 0.999037582005719},
    {0.58722778128166, 0.718526008853367, 0.999423609532622},
    {0.592650204926479, 0.72303624644852, 0.999712809697627},
    {0.598066910447909, 0.727487584547778, 0.999905188314007},
    {0.603477115735333, 0.731879371439515, 1},
    {0.608880034341146, 0.736210961171929, 0.999999580442504},
    {0.61427487574164, 0.740481713666196, 0.999901681743982},
    {0.619660845592978, 0.744690994827674, 0.999707136984893},
    {0.625037145982386, 0.748838176655215, 0.999416027869349},
    {0.630402975674634, 0.752922637348636, 0.999028451133396},
    {0.635757530353895, 0.756943761414401, 0.998544518489454},
    {0.641100002861094, 0.76090093976955, 0.99796435656787},
    {0.646429583426816, 0.76479356984392, 0.997288106855568},
    {0.651745459899873, 0.768621055680699, 0.996515925631835},
    {0.657046817971618, 0.772382808035335, 0.995647983901228},
    {0.662332841396096, 0.77607824447284, 0.99468446732363},
    {0.667602712206112, 0.77970678946352, 0.993625576141458},
    {0.672855610925311, 0.783267874477135, 0.992471525104042},
    {0.678090716776351, 0.786760938075545, 0.991222543389173},
    {0.683307207885253, 0.790185426003826, 0.989878874521848},
    {0.688504261482004, 0.793540791279909, 0.988440776290216},
    {0.693681054097512, 0.796826494282724, 0.98690852065873},
    {0.698836761756975, 0.800042002838893, 0.985282393678538},
    {0.703970560169744, 0.803186792307961, 0.983562695395095},
    {0.709081624915778, 0.806260345666195, 0.981749739753039},
    {0.714169131628733, 0.809262153588942, 0.979843854498323},
    {0.719232256175789, 0.812191714531561, 0.977845381077618},
    {0.724270174834274, 0.815048534808944, 0.975754674535002},
    {0.729282064465158, 0.817832128673612, 0.973572103405955},
    {0.734267102683483, 0.820542018392398, 0.971298049608644},
    {0.739224468025814, 0.823177734321728, 0.968932908332542},
    {0.744153340114752, 0.825738814981478, 0.966477087924374},
    {0.749052899820601, 0.82822480712743, 0.96393100977139},
    {0.753922329420233, 0.830635265822307, 0.961295108182006},
    {0.758760812753225, 0.832969754505392, 0.958569830263783},
    {0.763567535375327, 0.835227845060731, 0.95575563579879},
    {0.76834168470932, 0.837409117883898, 0.952852997116326},
    {0.773082450193319, 0.839513161947337, 0.949862398963033},
    {0.777789023426588, 0.84153957486426, 0.946784338370386},
    {0.782460598312911, 0.843487962951091, 0.943619324519584},
    {0.787096371201585, 0.84535794128846, 0.940367878603829},
    {0.79169554102608, 0.84714913378072, 0.937030533688002},
    {0.796257309440422, 0.848861173213987, 0.933607834565744},
    {0.800780880953345, 0.850493701312688, 0.930100337613927},
    {0.805265463060274, 0.852046368794602, 0.926508610644532},
    {0.809710266373171, 0.853518835424378, 0.922833232753911},
    {0.814114504748303, 0.854910770065524, 0.91907479416944},
    {0.818477395411976, 0.856221850730831, 0.915233896093559},
    {0.822798159084272, 0.857451764631246, 0.911311150545172},
    {0.827076020100851, 0.85860020822314, 0.907307180198417},
    {0.831310206532838, 0.859666887253979, 0.903222618218773},
    {0.83549995030486, 0.860651516806363, 0.899058108096501},
    {0.83964448731125, 0.861553821340419, 0.894814303477392},
    {0.843743057530485, 0.862373534734519, 0.890491867990793},
    {0.847794905137875, 0.863110400324303, 0.8860914750749},
    {0.851799278616545, 0.863764170939984, 0.881613807799273},
    {0.855755430866766, 0.864334608941901, 0.877059558684542},
    {0.859662619313642, 0.864821486254302, 0.872429429519271},
    {0.863520106013216, 0.86522458439732, 0.86772413117393},
    {0.867643188110251, 0.864514034915036, 0.86261614859855},
    {0.871997300188924, 0.862691278878098, 0.857134411693155},
    {0.876255167957442, 0.860786699659963, 0.851617444979445},
    {0.880416967845337, 0.858800556671119, 0.846066168334393},
    {0.884482864685423, 0.856733118237242, 0.840481500511256},
    {0.888453012702675, 0.854584661498578, 0.834864358985154},
    {0.892327556446231, 0.852355472306594, 0.829215659800157},
    {0.896106631668472, 0.850045845117798, 0.823536317417899},
    {0.899790366154808, 0.847656082884652, 0.817827244567743},
    {0.903378880507531, 0.845186496943479, 0.812089352098517},
    {0.906872288886791, 0.842637406899273, 0.806323548831847},
    {0.910270699711566, 0.840009140507301, 0.800530741417113},
    {0.913574216323204, 0.8373020335514, 0.794711834188034},
    {0.916782937613987, 0.834516429718829, 0.788867729020921},
    {0.919896958622917, 0.83165268047159, 0.782999325194605},
    {0.922916371100816, 0.828711144914049, 0.777107519252075},
    {0.925841264046632, 0.825692189656736, 0.771193204863825},
    {0.928671724216722, 0.822596188676188, 0.765257272692955},
    {0.931407836608762, 0.819423523170646, 0.759300610262019},
    {0.934049684921792, 0.816174581411474, 0.753324101821667},
    {0.936597351993828, 0.812849758590093, 0.747328628221076},
    {0.939050920218321, 0.809449456660255, 0.741315066780201},
    {0.941410471940732, 0.805974084175453, 0.73528429116386},
    {0.943676089836305, 0.802424056121246, 0.729237171257679},
    {0.945847857270144, 0.79879979374227, 0.723174573045899},
    {0.947925858640541, 0.795101724363689, 0.717097358491085},
    {0.949910179706505, 0.791330281206824, 0.711006385415724},
    {0.951800907900325, 0.78748590319867, 0.70490250738577},
    {0.953598132625986, 0.783569034775005, 0.698786573596112},
    {0.95530194554417, 0.779580125676776, 0.692659428758006},
    {0.956912440844559, 0.775519630739391, 0.686521912988491},
    {0.958429715506068, 0.771388009674572, 0.680374861701783},
    {0.959853869545649, 0.767185726844354, 0.67421910550269},
    {0.961185006256221, 0.762913251026804, 0.668055470082052},
    {0.962423232434257, 0.758571055173009, 0.661884776114213},
    {0.963568658597548, 0.754159616154822, 0.65570783915657},
    {0.964621399193606, 0.749679414502856, 0.649525469551186},
    {0.965581572799143, 0.745130934134127, 0.6433384723285},
    {0.966449302311052, 0.740514662068753, 0.637147647113151},
    {0.967224715129268, 0.735831088135034, 0.630953788031928},
    {0.967907943331888, 0.731080704662185, 0.624757683623865},
    {0.968499123842878, 0.726264006159971, 0.618560116752503},
    {0.968998398592707, 0.721381488984381, 0.612361864520328},
    {0.969405914672195, 0.716433650988458, 0.606163698185414},
    {0.969721824479875, 0.711420991157289, 0.599966383080285},
    {0.969946285863132, 0.706344009226102, 0.593770678533009},
    {0.970079462253369, 0.701203205280315, 0.587577337790557},
    {0.970121522795447, 0.695999079336297, 0.581387107944436},
    {0.970072642471613, 0.690732130901479, 0.575200729858619},
    {0.96993300222014, 0.685402858512333, 0.569018938099802},
    {0.969702789048866, 0.68001175924864, 0.562842460869995},
    {0.969382196143828, 0.674559328222277, 0.556672019941486},
    {0.968971422973169, 0.669046058038632, 0.550508330594186},
    {0.968470675386481, 0.663472438228566, 0.5443521015554},
    {0.967880165709737, 0.65783895464866, 0.538204034942027},
    {0.967200112835984, 0.652146088847254, 0.532064826205232},
    {0.966430742311901, 0.646394317393575, 0.525935164077615},
    {0.965572286420388, 0.640584111166963, 0.519815730522903},
    {0.964624984259297, 0.634715934602944, 0.513707200688195},
    {0.963589081816417, 0.628790244892546, 0.507610242858806},
    {0.962464832040844, 0.622807491130921, 0.501525518415717},
    {0.961252494910814, 0.616768113410917, 0.495453681795706},
    {0.959952337498132, 0.610672541856812, 0.489395380454148},
    {0.958564634029262, 0.604521195592905, 0.483351254830581},
    {0.95708966594318, 0.598314481641114, 0.477321938317026},
    {0.955527721946081, 0.592052793741094, 0.471308057229144},
    {0.953879098063002, 0.585736511085672, 0.465310230780258},
    {0.952144097686451, 0.57936599696362, 0.459329071058297},
    {0.950323031622108, 0.57294159730086, 0.453365183005708},
    {0.948416218131665, 0.566463639090201, 0.4474191644024},
    {0.946423982972865, 0.559932428698541, 0.441491605851772},
    {0.944346659436813, 0.553348250039155, 0.435583090769892},
    {0.942184588382596, 0.546711362595224, 0.429694195377886},
    {0.939938118269283, 0.540021999279031, 0.423825488697617},
    {0.93760760518535, 0.533280364109342, 0.417977532550726},
    {0.935193412875573, 0.52648662968724, 0.412150881561099},
    {0.932695912765446, 0.519640934448169, 0.406346083160873},
    {0.930115483983149, 0.512743379664974, 0.400563677600038},
    {0.927452513379131, 0.505794026173367, 0.394804197959753},
    {0.924707395543324, 0.498792890787342, 0.389068170169467},
    {0.921880532820037, 0.491739942367494, 0.38335611302795},
    {0.918972335320559, 0.484635097499965, 0.377668538228348},
    {0.915983220933511, 0.477478215737544, 0.372005950387394},
    {0.912913615332969, 0.470269094347285, 0.366368847078878},
    {0.90976395198439, 0.463007462500508, 0.36075771887154},
    {0.906534672148371, 0.455692974831135, 0.355173049371513},
    {0.90322622488227, 0.448325204276504, 0.349615315269472},
    {0.899839067039705, 0.440903634100853, 0.344084986392662},
    {0.896373663267962, 0.433427648985018, 0.338582525761964},
    {0.892830486003327, 0.425896525046024, 0.333108389654201},
    {0.889210015464384, 0.418309418626304, 0.327663027669866},
    {0.885512739643257, 0.410665353663527, 0.322246882806488},
    {0.881739154294872, 0.402963207417049, 0.316860391537864},
    {0.877889762924204, 0.39520169428451, 0.311503983899383},
    {0.873965076771559, 0.387379347390017, 0.306178083579696},
    {0.8699656147959, 0.379494497561343, 0.300883108019013},
    {0.865891903656227, 0.37154524923423, 0.295619468514294},
    {0.86174447769103, 0.363529452723187, 0.290387570331645},
    {0.857523878895836, 0.355444672174402, 0.285187812826247},
    {0.853230656898852, 0.347288148360209, 0.280020589570149},
    {0.848865368934727, 0.339056755276074, 0.274886288488297},
    {0.844428579816439, 0.330746949246731, 0.26978529200318},
    {0.83992086190533, 0.322354708919708, 0.264717977188514},
    {0.835342795079293, 0.313875464096442, 0.25968471593238},
    {0.830694966699119, 0.30530401078823, 0.254685875110311},
    {0.825977971573034, 0.296634409135975, 0.249721816768777},
    {0.821192411919414, 0.287859859827339, 0.244792898319632},
    {0.816338897327716, 0.278972553277785, 0.239899472746041},
    {0.811418044717604, 0.269963483958878, 0.235041888820489},
    {0.806430478296323, 0.260822219626468, 0.230220491335487},
    {0.801376829514291, 0.251536611469235, 0.225435621347621},
    {0.796257737018953, 0.242092425812954, 0.22068761643565},
    {0.791073846606894, 0.232472870099256, 0.215976810973353},
    {0.785825811174225, 0.22265797397565, 0.211303536417923},
    {0.78051429066526, 0.212623768082272, 0.20666812161469},
    {0.775139952019497, 0.202341174346873, 0.202070893119019},
    {0.769703469116915, 0.191774474878576, 0.197512175536282},
    {0.764205522721598, 0.180879148089055, 0.192992291880808},
    {0.758646800423722, 0.169598723673214, 0.188511563954777},
    {0.753027996579893, 0.157860057746477, 0.184070312748068},
    {0.747349812251885, 0.145565946623769, 0.17966885886006},
    {0.741612955143772, 0.132583003797394, 0.17530752294447},
    {0.735818139537497, 0.118720506828836, 0.170986626178292},
    {0.729966086226886, 0.103690437173123, 0.166706490755933},
    {0.724057522450145, 0.087023398885818, 0.162467440409651},
    {0.718093181820861, 0.0678617415228623, 0.158269800957381},
    {0.712073804257538, 0.0443031972853228, 0.154113900879019},
    {0.706000135911704, 0.0159918240339806, 0.1500000719222},
  };
  int i = (int)floor(v * 255);
  **r = cmap_data[i][0];
  **g = cmap_data[i][1];
  **b = cmap_data[i][2];
}


// RedBlue colormap - Matplotlib provided by Suyono Nt.
void get_cmap_redblue(float v, float **r, float **g, float **b)
{
  float cmap_data[256][3] = {
    {0.403922, 0.000000, 0.121569},
    {0.415456, 0.003691, 0.123414},
    {0.426990, 0.007382, 0.125260},
    {0.438524, 0.011073, 0.127105},
    {0.450058, 0.014764, 0.128950},
    {0.461592, 0.018454, 0.130796},
    {0.473126, 0.022145, 0.132641},
    {0.484660, 0.025836, 0.134487},
    {0.496194, 0.029527, 0.136332},
    {0.507728, 0.033218, 0.138178},
    {0.519262, 0.036909, 0.140023},
    {0.530796, 0.040600, 0.141869},
    {0.542330, 0.044291, 0.143714},
    {0.553864, 0.047982, 0.145559},
    {0.565398, 0.051672, 0.147405},
    {0.576932, 0.055363, 0.149250},
    {0.588466, 0.059054, 0.151096},
    {0.600000, 0.062745, 0.152941},
    {0.611534, 0.066436, 0.154787},
    {0.623068, 0.070127, 0.156632},
    {0.634602, 0.073818, 0.158478},
    {0.646136, 0.077509, 0.160323},
    {0.657670, 0.081200, 0.162168},
    {0.669204, 0.084890, 0.164014},
    {0.680738, 0.088581, 0.165859},
    {0.692272, 0.092272, 0.167705},
    {0.700807, 0.099654, 0.171242},
    {0.706344, 0.110727, 0.176471},
    {0.711880, 0.121799, 0.181699},
    {0.717416, 0.132872, 0.186928},
    {0.722953, 0.143945, 0.192157},
    {0.728489, 0.155017, 0.197386},
    {0.734025, 0.166090, 0.202614},
    {0.739562, 0.177163, 0.207843},
    {0.745098, 0.188235, 0.213072},
    {0.750634, 0.199308, 0.218301},
    {0.756171, 0.210381, 0.223529},
    {0.761707, 0.221453, 0.228758},
    {0.767243, 0.232526, 0.233987},
    {0.772780, 0.243599, 0.239216},
    {0.778316, 0.254671, 0.244444},
    {0.783852, 0.265744, 0.249673},
    {0.789389, 0.276817, 0.254902},
    {0.794925, 0.287889, 0.260131},
    {0.800461, 0.298962, 0.265359},
    {0.805998, 0.310035, 0.270588},
    {0.811534, 0.321107, 0.275817},
    {0.817070, 0.332180, 0.281046},
    {0.822607, 0.343253, 0.286275},
    {0.828143, 0.354325, 0.291503},
    {0.833679, 0.365398, 0.296732},
    {0.839216, 0.376471, 0.301961},
    {0.843829, 0.387082, 0.310112},
    {0.848443, 0.397693, 0.318262},
    {0.853057, 0.408305, 0.326413},
    {0.857670, 0.418916, 0.334564},
    {0.862284, 0.429527, 0.342714},
    {0.866897, 0.440138, 0.350865},
    {0.871511, 0.450750, 0.359016},
    {0.876125, 0.461361, 0.367166},
    {0.880738, 0.471972, 0.375317},
    {0.885352, 0.482584, 0.383468},
    {0.889965, 0.493195, 0.391619},
    {0.894579, 0.503806, 0.399769},
    {0.899193, 0.514418, 0.407920},
    {0.903806, 0.525029, 0.416071},
    {0.908420, 0.535640, 0.424221},
    {0.913033, 0.546251, 0.432372},
    {0.917647, 0.556863, 0.440523},
    {0.922261, 0.567474, 0.448674},
    {0.926874, 0.578085, 0.456824},
    {0.931488, 0.588697, 0.464975},
    {0.936102, 0.599308, 0.473126},
    {0.940715, 0.609919, 0.481276},
    {0.945329, 0.620531, 0.489427},
    {0.949942, 0.631142, 0.497578},
    {0.954556, 0.641753, 0.505729},
    {0.957555, 0.651211, 0.515110},
    {0.958939, 0.659516, 0.525721},
    {0.960323, 0.667820, 0.536332},
    {0.961707, 0.676125, 0.546944},
    {0.963091, 0.684429, 0.557555},
    {0.964475, 0.692734, 0.568166},
    {0.965859, 0.701038, 0.578777},
    {0.967243, 0.709343, 0.589389},
    {0.968627, 0.717647, 0.600000},
    {0.970012, 0.725952, 0.610611},
    {0.971396, 0.734256, 0.621223},
    {0.972780, 0.742561, 0.631834},
    {0.974164, 0.750865, 0.642445},
    {0.975548, 0.759170, 0.653057},
    {0.976932, 0.767474, 0.663668},
    {0.978316, 0.775779, 0.674279},
    {0.979700, 0.784083, 0.684890},
    {0.981084, 0.792388, 0.695502},
    {0.982468, 0.800692, 0.706113},
    {0.983852, 0.808997, 0.716724},
    {0.985236, 0.817301, 0.727336},
    {0.986621, 0.825606, 0.737947},
    {0.988005, 0.833910, 0.748558},
    {0.989389, 0.842215, 0.759170},
    {0.990773, 0.850519, 0.769781},
    {0.992157, 0.858824, 0.780392},
    {0.991234, 0.863130, 0.787774},
    {0.990311, 0.867436, 0.795156},
    {0.989389, 0.871742, 0.802537},
    {0.988466, 0.876048, 0.809919},
    {0.987543, 0.880354, 0.817301},
    {0.986621, 0.884660, 0.824683},
    {0.985698, 0.888966, 0.832065},
    {0.984775, 0.893272, 0.839446},
    {0.983852, 0.897578, 0.846828},
    {0.982930, 0.901884, 0.854210},
    {0.982007, 0.906190, 0.861592},
    {0.981084, 0.910496, 0.868973},
    {0.980161, 0.914802, 0.876355},
    {0.979239, 0.919108, 0.883737},
    {0.978316, 0.923414, 0.891119},
    {0.977393, 0.927720, 0.898501},
    {0.976471, 0.932026, 0.905882},
    {0.975548, 0.936332, 0.913264},
    {0.974625, 0.940638, 0.920646},
    {0.973702, 0.944944, 0.928028},
    {0.972780, 0.949250, 0.935409},
    {0.971857, 0.953556, 0.942791},
    {0.970934, 0.957862, 0.950173},
    {0.970012, 0.962168, 0.957555},
    {0.969089, 0.966474, 0.964937},
    {0.965705, 0.967243, 0.968089},
    {0.959862, 0.964475, 0.967013},
    {0.954018, 0.961707, 0.965936},
    {0.948174, 0.958939, 0.964860},
    {0.942330, 0.956171, 0.963783},
    {0.936486, 0.953403, 0.962707},
    {0.930642, 0.950634, 0.961630},
    {0.924798, 0.947866, 0.960554},
    {0.918954, 0.945098, 0.959477},
    {0.913110, 0.942330, 0.958401},
    {0.907266, 0.939562, 0.957324},
    {0.901423, 0.936794, 0.956248},
    {0.895579, 0.934025, 0.955171},
    {0.889735, 0.931257, 0.954095},
    {0.883891, 0.928489, 0.953018},
    {0.878047, 0.925721, 0.951942},
    {0.872203, 0.922953, 0.950865},
    {0.866359, 0.920185, 0.949789},
    {0.860515, 0.917416, 0.948712},
    {0.854671, 0.914648, 0.947636},
    {0.848827, 0.911880, 0.946559},
    {0.842983, 0.909112, 0.945483},
    {0.837140, 0.906344, 0.944406},
    {0.831296, 0.903576, 0.943329},
    {0.825452, 0.900807, 0.942253},
    {0.819608, 0.898039, 0.941176},
    {0.809919, 0.893118, 0.938408},
    {0.800231, 0.888197, 0.935640},
    {0.790542, 0.883276, 0.932872},
    {0.780854, 0.878354, 0.930104},
    {0.771165, 0.873433, 0.927336},
    {0.761476, 0.868512, 0.924567},
    {0.751788, 0.863591, 0.921799},
    {0.742099, 0.858670, 0.919031},
    {0.732411, 0.853749, 0.916263},
    {0.722722, 0.848827, 0.913495},
    {0.713033, 0.843906, 0.910727},
    {0.703345, 0.838985, 0.907958},
    {0.693656, 0.834064, 0.905190},
    {0.683968, 0.829143, 0.902422},
    {0.674279, 0.824221, 0.899654},
    {0.664591, 0.819300, 0.896886},
    {0.654902, 0.814379, 0.894118},
    {0.645213, 0.809458, 0.891349},
    {0.635525, 0.804537, 0.888581},
    {0.625836, 0.799616, 0.885813},
    {0.616148, 0.794694, 0.883045},
    {0.606459, 0.789773, 0.880277},
    {0.596770, 0.784852, 0.877509},
    {0.587082, 0.779931, 0.874740},
    {0.577393, 0.775010, 0.871972},
    {0.566474, 0.768704, 0.868512},
    {0.554325, 0.761015, 0.864360},
    {0.542176, 0.753326, 0.860208},
    {0.530027, 0.745636, 0.856055},
    {0.517878, 0.737947, 0.851903},
    {0.505729, 0.730258, 0.847751},
    {0.493579, 0.722568, 0.843599},
    {0.481430, 0.714879, 0.839446},
    {0.469281, 0.707190, 0.835294},
    {0.457132, 0.699500, 0.831142},
    {0.444983, 0.691811, 0.826990},
    {0.432834, 0.684122, 0.822837},
    {0.420684, 0.676432, 0.818685},
    {0.408535, 0.668743, 0.814533},
    {0.396386, 0.661053, 0.810381},
    {0.384237, 0.653364, 0.806228},
    {0.372088, 0.645675, 0.802076},
    {0.359939, 0.637985, 0.797924},
    {0.347789, 0.630296, 0.793772},
    {0.335640, 0.622607, 0.789619},
    {0.323491, 0.614917, 0.785467},
    {0.311342, 0.607228, 0.781315},
    {0.299193, 0.599539, 0.777163},
    {0.287043, 0.591849, 0.773010},
    {0.274894, 0.584160, 0.768858},
    {0.262745, 0.576471, 0.764706},
    {0.257516, 0.569550, 0.761169},
    {0.252288, 0.562630, 0.757632},
    {0.247059, 0.555709, 0.754095},
    {0.241830, 0.548789, 0.750557},
    {0.236601, 0.541869, 0.747020},
    {0.231373, 0.534948, 0.743483},
    {0.226144, 0.528028, 0.739946},
    {0.220915, 0.521107, 0.736409},
    {0.215686, 0.514187, 0.732872},
    {0.210458, 0.507266, 0.729335},
    {0.205229, 0.500346, 0.725798},
    {0.200000, 0.493426, 0.722261},
    {0.194771, 0.486505, 0.718724},
    {0.189542, 0.479585, 0.715186},
    {0.184314, 0.472664, 0.711649},
    {0.179085, 0.465744, 0.708112},
    {0.173856, 0.458824, 0.704575},
    {0.168627, 0.451903, 0.701038},
    {0.163399, 0.444983, 0.697501},
    {0.158170, 0.438062, 0.693964},
    {0.152941, 0.431142, 0.690427},
    {0.147712, 0.424221, 0.686890},
    {0.142484, 0.417301, 0.683353},
    {0.137255, 0.410381, 0.679815},
    {0.132026, 0.403460, 0.676278},
    {0.127259, 0.395848, 0.668743},
    {0.122953, 0.387543, 0.657209},
    {0.118647, 0.379239, 0.645675},
    {0.114341, 0.370934, 0.634141},
    {0.110035, 0.362630, 0.622607},
    {0.105729, 0.354325, 0.611073},
    {0.101423, 0.346021, 0.599539},
    {0.097116, 0.337716, 0.588005},
    {0.092810, 0.329412, 0.576471},
    {0.088504, 0.321107, 0.564937},
    {0.084198, 0.312803, 0.553403},
    {0.079892, 0.304498, 0.541869},
    {0.075586, 0.296194, 0.530335},
    {0.071280, 0.287889, 0.518800},
    {0.066974, 0.279585, 0.507266},
    {0.062668, 0.271280, 0.495732},
    {0.058362, 0.262976, 0.484198},
    {0.054056, 0.254671, 0.472664},
    {0.049750, 0.246367, 0.461130},
    {0.045444, 0.238062, 0.449596},
    {0.041138, 0.229758, 0.438062},
    {0.036832, 0.221453, 0.426528},
    {0.032526, 0.213149, 0.414994},
    {0.028220, 0.204844, 0.403460},
    {0.023914, 0.196540, 0.391926},
    {0.019608, 0.188235, 0.380392},
  };
  int i = (int)floor(v * 255);
  **r = cmap_data[i][0];
  **g = cmap_data[i][1];
  **b = cmap_data[i][2];
}


// Gray colormap - Matplotlib provided by Suyono Nt.
void get_cmap_gray(float v, float **r, float **g, float **b)
{
  float cmap_data[256][3] = {
    {0.000000, 0.000000, 0.000000},
    {0.003922, 0.003922, 0.003922},
    {0.007843, 0.007843, 0.007843},
    {0.011765, 0.011765, 0.011765},
    {0.015686, 0.015686, 0.015686},
    {0.019608, 0.019608, 0.019608},
    {0.023529, 0.023529, 0.023529},
    {0.027451, 0.027451, 0.027451},
    {0.031373, 0.031373, 0.031373},
    {0.035294, 0.035294, 0.035294},
    {0.039216, 0.039216, 0.039216},
    {0.043137, 0.043137, 0.043137},
    {0.047059, 0.047059, 0.047059},
    {0.050980, 0.050980, 0.050980},
    {0.054902, 0.054902, 0.054902},
    {0.058824, 0.058824, 0.058824},
    {0.062745, 0.062745, 0.062745},
    {0.066667, 0.066667, 0.066667},
    {0.070588, 0.070588, 0.070588},
    {0.074510, 0.074510, 0.074510},
    {0.078431, 0.078431, 0.078431},
    {0.082353, 0.082353, 0.082353},
    {0.086275, 0.086275, 0.086275},
    {0.090196, 0.090196, 0.090196},
    {0.094118, 0.094118, 0.094118},
    {0.098039, 0.098039, 0.098039},
    {0.101961, 0.101961, 0.101961},
    {0.105882, 0.105882, 0.105882},
    {0.109804, 0.109804, 0.109804},
    {0.113725, 0.113725, 0.113725},
    {0.117647, 0.117647, 0.117647},
    {0.121569, 0.121569, 0.121569},
    {0.125490, 0.125490, 0.125490},
    {0.129412, 0.129412, 0.129412},
    {0.133333, 0.133333, 0.133333},
    {0.137255, 0.137255, 0.137255},
    {0.141176, 0.141176, 0.141176},
    {0.145098, 0.145098, 0.145098},
    {0.149020, 0.149020, 0.149020},
    {0.152941, 0.152941, 0.152941},
    {0.156863, 0.156863, 0.156863},
    {0.160784, 0.160784, 0.160784},
    {0.164706, 0.164706, 0.164706},
    {0.168627, 0.168627, 0.168627},
    {0.172549, 0.172549, 0.172549},
    {0.176471, 0.176471, 0.176471},
    {0.180392, 0.180392, 0.180392},
    {0.184314, 0.184314, 0.184314},
    {0.188235, 0.188235, 0.188235},
    {0.192157, 0.192157, 0.192157},
    {0.196078, 0.196078, 0.196078},
    {0.200000, 0.200000, 0.200000},
    {0.203922, 0.203922, 0.203922},
    {0.207843, 0.207843, 0.207843},
    {0.211765, 0.211765, 0.211765},
    {0.215686, 0.215686, 0.215686},
    {0.219608, 0.219608, 0.219608},
    {0.223529, 0.223529, 0.223529},
    {0.227451, 0.227451, 0.227451},
    {0.231373, 0.231373, 0.231373},
    {0.235294, 0.235294, 0.235294},
    {0.239216, 0.239216, 0.239216},
    {0.243137, 0.243137, 0.243137},
    {0.247059, 0.247059, 0.247059},
    {0.250980, 0.250980, 0.250980},
    {0.254902, 0.254902, 0.254902},
    {0.258824, 0.258824, 0.258824},
    {0.262745, 0.262745, 0.262745},
    {0.266667, 0.266667, 0.266667},
    {0.270588, 0.270588, 0.270588},
    {0.274510, 0.274510, 0.274510},
    {0.278431, 0.278431, 0.278431},
    {0.282353, 0.282353, 0.282353},
    {0.286275, 0.286275, 0.286275},
    {0.290196, 0.290196, 0.290196},
    {0.294118, 0.294118, 0.294118},
    {0.298039, 0.298039, 0.298039},
    {0.301961, 0.301961, 0.301961},
    {0.305882, 0.305882, 0.305882},
    {0.309804, 0.309804, 0.309804},
    {0.313725, 0.313725, 0.313725},
    {0.317647, 0.317647, 0.317647},
    {0.321569, 0.321569, 0.321569},
    {0.325490, 0.325490, 0.325490},
    {0.329412, 0.329412, 0.329412},
    {0.333333, 0.333333, 0.333333},
    {0.337255, 0.337255, 0.337255},
    {0.341176, 0.341176, 0.341176},
    {0.345098, 0.345098, 0.345098},
    {0.349020, 0.349020, 0.349020},
    {0.352941, 0.352941, 0.352941},
    {0.356863, 0.356863, 0.356863},
    {0.360784, 0.360784, 0.360784},
    {0.364706, 0.364706, 0.364706},
    {0.368627, 0.368627, 0.368627},
    {0.372549, 0.372549, 0.372549},
    {0.376471, 0.376471, 0.376471},
    {0.380392, 0.380392, 0.380392},
    {0.384314, 0.384314, 0.384314},
    {0.388235, 0.388235, 0.388235},
    {0.392157, 0.392157, 0.392157},
    {0.396078, 0.396078, 0.396078},
    {0.400000, 0.400000, 0.400000},
    {0.403922, 0.403922, 0.403922},
    {0.407843, 0.407843, 0.407843},
    {0.411765, 0.411765, 0.411765},
    {0.415686, 0.415686, 0.415686},
    {0.419608, 0.419608, 0.419608},
    {0.423529, 0.423529, 0.423529},
    {0.427451, 0.427451, 0.427451},
    {0.431373, 0.431373, 0.431373},
    {0.435294, 0.435294, 0.435294},
    {0.439216, 0.439216, 0.439216},
    {0.443137, 0.443137, 0.443137},
    {0.447059, 0.447059, 0.447059},
    {0.450980, 0.450980, 0.450980},
    {0.454902, 0.454902, 0.454902},
    {0.458824, 0.458824, 0.458824},
    {0.462745, 0.462745, 0.462745},
    {0.466667, 0.466667, 0.466667},
    {0.470588, 0.470588, 0.470588},
    {0.474510, 0.474510, 0.474510},
    {0.478431, 0.478431, 0.478431},
    {0.482353, 0.482353, 0.482353},
    {0.486275, 0.486275, 0.486275},
    {0.490196, 0.490196, 0.490196},
    {0.494118, 0.494118, 0.494118},
    {0.498039, 0.498039, 0.498039},
    {0.501961, 0.501961, 0.501961},
    {0.505882, 0.505882, 0.505882},
    {0.509804, 0.509804, 0.509804},
    {0.513725, 0.513725, 0.513725},
    {0.517647, 0.517647, 0.517647},
    {0.521569, 0.521569, 0.521569},
    {0.525490, 0.525490, 0.525490},
    {0.529412, 0.529412, 0.529412},
    {0.533333, 0.533333, 0.533333},
    {0.537255, 0.537255, 0.537255},
    {0.541176, 0.541176, 0.541176},
    {0.545098, 0.545098, 0.545098},
    {0.549020, 0.549020, 0.549020},
    {0.552941, 0.552941, 0.552941},
    {0.556863, 0.556863, 0.556863},
    {0.560784, 0.560784, 0.560784},
    {0.564706, 0.564706, 0.564706},
    {0.568627, 0.568627, 0.568627},
    {0.572549, 0.572549, 0.572549},
    {0.576471, 0.576471, 0.576471},
    {0.580392, 0.580392, 0.580392},
    {0.584314, 0.584314, 0.584314},
    {0.588235, 0.588235, 0.588235},
    {0.592157, 0.592157, 0.592157},
    {0.596078, 0.596078, 0.596078},
    {0.600000, 0.600000, 0.600000},
    {0.603922, 0.603922, 0.603922},
    {0.607843, 0.607843, 0.607843},
    {0.611765, 0.611765, 0.611765},
    {0.615686, 0.615686, 0.615686},
    {0.619608, 0.619608, 0.619608},
    {0.623529, 0.623529, 0.623529},
    {0.627451, 0.627451, 0.627451},
    {0.631373, 0.631373, 0.631373},
    {0.635294, 0.635294, 0.635294},
    {0.639216, 0.639216, 0.639216},
    {0.643137, 0.643137, 0.643137},
    {0.647059, 0.647059, 0.647059},
    {0.650980, 0.650980, 0.650980},
    {0.654902, 0.654902, 0.654902},
    {0.658824, 0.658824, 0.658824},
    {0.662745, 0.662745, 0.662745},
    {0.666667, 0.666667, 0.666667},
    {0.670588, 0.670588, 0.670588},
    {0.674510, 0.674510, 0.674510},
    {0.678431, 0.678431, 0.678431},
    {0.682353, 0.682353, 0.682353},
    {0.686275, 0.686275, 0.686275},
    {0.690196, 0.690196, 0.690196},
    {0.694118, 0.694118, 0.694118},
    {0.698039, 0.698039, 0.698039},
    {0.701961, 0.701961, 0.701961},
    {0.705882, 0.705882, 0.705882},
    {0.709804, 0.709804, 0.709804},
    {0.713725, 0.713725, 0.713725},
    {0.717647, 0.717647, 0.717647},
    {0.721569, 0.721569, 0.721569},
    {0.725490, 0.725490, 0.725490},
    {0.729412, 0.729412, 0.729412},
    {0.733333, 0.733333, 0.733333},
    {0.737255, 0.737255, 0.737255},
    {0.741176, 0.741176, 0.741176},
    {0.745098, 0.745098, 0.745098},
    {0.749020, 0.749020, 0.749020},
    {0.752941, 0.752941, 0.752941},
    {0.756863, 0.756863, 0.756863},
    {0.760784, 0.760784, 0.760784},
    {0.764706, 0.764706, 0.764706},
    {0.768627, 0.768627, 0.768627},
    {0.772549, 0.772549, 0.772549},
    {0.776471, 0.776471, 0.776471},
    {0.780392, 0.780392, 0.780392},
    {0.784314, 0.784314, 0.784314},
    {0.788235, 0.788235, 0.788235},
    {0.792157, 0.792157, 0.792157},
    {0.796078, 0.796078, 0.796078},
    {0.800000, 0.800000, 0.800000},
    {0.803922, 0.803922, 0.803922},
    {0.807843, 0.807843, 0.807843},
    {0.811765, 0.811765, 0.811765},
    {0.815686, 0.815686, 0.815686},
    {0.819608, 0.819608, 0.819608},
    {0.823529, 0.823529, 0.823529},
    {0.827451, 0.827451, 0.827451},
    {0.831373, 0.831373, 0.831373},
    {0.835294, 0.835294, 0.835294},
    {0.839216, 0.839216, 0.839216},
    {0.843137, 0.843137, 0.843137},
    {0.847059, 0.847059, 0.847059},
    {0.850980, 0.850980, 0.850980},
    {0.854902, 0.854902, 0.854902},
    {0.858824, 0.858824, 0.858824},
    {0.862745, 0.862745, 0.862745},
    {0.866667, 0.866667, 0.866667},
    {0.870588, 0.870588, 0.870588},
    {0.874510, 0.874510, 0.874510},
    {0.878431, 0.878431, 0.878431},
    {0.882353, 0.882353, 0.882353},
    {0.886275, 0.886275, 0.886275},
    {0.890196, 0.890196, 0.890196},
    {0.894118, 0.894118, 0.894118},
    {0.898039, 0.898039, 0.898039},
    {0.901961, 0.901961, 0.901961},
    {0.905882, 0.905882, 0.905882},
    {0.909804, 0.909804, 0.909804},
    {0.913725, 0.913725, 0.913725},
    {0.917647, 0.917647, 0.917647},
    {0.921569, 0.921569, 0.921569},
    {0.925490, 0.925490, 0.925490},
    {0.929412, 0.929412, 0.929412},
    {0.933333, 0.933333, 0.933333},
    {0.937255, 0.937255, 0.937255},
    {0.941176, 0.941176, 0.941176},
    {0.945098, 0.945098, 0.945098},
    {0.949020, 0.949020, 0.949020},
    {0.952941, 0.952941, 0.952941},
    {0.956863, 0.956863, 0.956863},
    {0.960784, 0.960784, 0.960784},
    {0.964706, 0.964706, 0.964706},
    {0.968627, 0.968627, 0.968627},
    {0.972549, 0.972549, 0.972549},
    {0.976471, 0.976471, 0.976471},
    {0.980392, 0.980392, 0.980392},
    {0.984314, 0.984314, 0.984314},
    {0.988235, 0.988235, 0.988235},
    {0.992157, 0.992157, 0.992157},
    {0.996078, 0.996078, 0.996078},
    {1.000000, 1.000000, 1.000000},
  };
  int i = (int)floor(v * 255);
  **r = cmap_data[i][0];
  **g = cmap_data[i][1];
  **b = cmap_data[i][2];
}


// Call appropriate colormap generator
// v,r,g,b ∈ [0;1]
void define_rgb(float v, float *r, float *g, float *b)
{
  if( compare( cmap_name, "viridis", 2)==2)
    get_cmap_viridis(v, &r, &g, &b);
  else if( compare( cmap_name, "turbo", 2)==2)
    get_cmap_turbo(v, &r, &g, &b);
  else if( compare( cmap_name, "inferno", 2)==2)
    get_cmap_inferno(v, &r, &g, &b);
  else if( compare( cmap_name, "coolwarm", 2)==2)
    get_cmap_coolwarm(v, &r, &g, &b);
  else if( compare( cmap_name, "classic", 2)==2)
    get_cmap_classic(v, &r, &g, &b);
  else if( compare( cmap_name, "jet", 2)==2)
    get_cmap_jet(v, &r, &g, &b);
  else if( compare( cmap_name, "redblue", 2)==2)
    get_cmap_redblue(v, &r, &g, &b);
  else if( compare( cmap_name, "gray", 2)==2)
    get_cmap_gray(v, &r, &g, &b);
  else printf("colormap %s not defined\n",cmap_name);
}


/* schreibt einen Text (x,y,z linkes unteres Eck)  */
void text(double x, double y, double z,char *msg, void *glut_font)
{
  glRasterPos3d(x, y, z);
  while (*msg) {
    glutBitmapCharacter(glut_font, *msg);
    msg++;
  }
}


/* scala verbindet Farben mit Zahlenwerten (rgb-mode)*/
/* kb, kh Kaestchenhoehe und Breite                  */
/* dx, dy  Ursprung der Scala (linkes unteres Eck)   */
/* divisions anzahl der Kaestchen                    */
void scala_rgb( double dx, double dy, int divisions, double bmin, double bmax, double *col, void *glut_font, char format)
{
  int  j, i;
  float r, g, b;
  double df, f;
  static char string[13];
  double     kb, kh;
  int flag=0;

  df = (bmax-bmin)/divisions;
  f  = bmin;

  kh= 1.0/divisions*1.5;
  kb= 1.0/20.;


  for (i=0; i<divisions; i++)
  {
    glBegin ( GL_POLYGON );
      define_rgb( (float)(f-bmin)/(bmax-bmin), &r,&g,&b);
      glColor3d ( r, g, b );
      glVertex2d ( dx-kb*0., dy+kh*0. );
      glVertex2d ( dx-kb*1., dy+kh*0. );
      define_rgb( (float)(f+df-bmin)/(bmax-bmin), &r,&g,&b);
      glColor3d ( r, g, b );
      glVertex2d ( dx-kb*1., dy+kh*1. );
      glVertex2d ( dx-kb*0., dy+kh*1. );
    glEnd();

    if(flag==0)
    {
      if(format=='f') sprintf ( string, "%-10f ", f);
      else if(format=='i') sprintf ( string, "%-10d ", (int)f);
      else sprintf ( string, "%-10.2e ", f);
      glColor3dv ( col );
      glRasterPos2d( (dx+kb*0.2), dy-kh*0.1 );
      for ( j=0; j<10; j++) glutBitmapCharacter(glut_font, string[j]);
    }
    flag++;
    if(flag>divisions/MAX_LABELS) flag=0;

    dy+=kh;
    f+=df;
  }
  if(flag==0)
  {
    if(format=='f') sprintf ( string, "%-10f ", bmax);
    else if(format=='i') sprintf ( string, "%-10d ", (int)bmax);
    else sprintf ( string, "%-10.2e ", bmax);
    glColor3dv ( col );
    glRasterPos2d( (dx+kb*0.2), dy-kh*0.1 );
    for ( j=0; j<10; j++) glutBitmapCharacter(glut_font, string[j]);
  }
}


/* scalai verbindet Farben mit Zahlenwerten         */
/* kb, kh Kaestchenhoehe und Breite                */
/* dx, dy  Ursprung der Scala (linkes unteres Eck) */
/* divisions anzahl der Kaestchen                  */
void scala_tex(double ratio, double dx, double dy, int divisions, double bmin, double bmax, double scale, double *col, void *glut_font, char format, char fnr, char lnr)
{
  int j, i;
  double     df, f,ds;
  static char string[13];
  double     kb, kh;
  int flag=0, divisor;

  /* recalculate bmax,bmin if a clipped scale is to be used (fnr,lnr >0) */
  divisor=divisions;
  if(lnr) divisor--;
  if(fnr) divisor--;
  ds=(bmax-bmin)/divisor;
  if(lnr) bmax+=ds;
  if(fnr) bmin-=ds;

  df = (bmax-bmin)/(divisions);
  f  = bmin;

  kh= 1.0/divisions*1.5*ratio;
  kb= 1.0/20.;

  for (i=0; i<divisions; i++)
  {
    glColor3d( 1,1,1);
    glEnable(GL_TEXTURE_1D);
    glTexCoord1d    ( (GLdouble)i/(GLdouble)divisions*scale );
    glBegin ( GL_POLYGON );
      glVertex2d ( dx-kb*0., dy+kh*0. );
      glVertex2d ( dx-kb*1., dy+kh*0. );
      glVertex2d ( dx-kb*1., dy+kh*1. );
      glVertex2d ( dx-kb*0., dy+kh*1. );
    glEnd();
    glDisable(GL_TEXTURE_1D);

    if(flag==0)
    {
      if(format=='f') sprintf ( string, "%-10f ", f);
      else if(format=='i') sprintf ( string, "%-10d ", (int)f);
      else if(format=='l') sprintf ( string, "1e%-.2f ", f);
      else
      {
        if((fnr)&&(i==1) && (abs(f)<df*1e-4)) sprintf ( string, "%-10.2e ", 0.);
        else sprintf ( string, "%-10.2e ", f);
      }
      glColor3dv ( col );
      glRasterPos2d( (dx+kb*0.2), dy-kh*0.1 );
      if((fnr)&&(!i)); else { for ( j=0; j<10; j++) glutBitmapCharacter(glut_font, string[j]); }
    }
    flag++;
    if(flag>divisions/MAX_LABELS) flag=0;

    dy+=kh;
    f+=df;
  }
  if((flag==0)&&(!lnr))
  {
    if(format=='f') sprintf ( string, "%-10f ", bmax);
    else if(format=='i') sprintf ( string, "%-10d ", (int)bmax);
    else if(format=='l') sprintf ( string, "1e%-.2f ", bmax);
    else sprintf ( string, "%-10.2e ", bmax);
    glColor3dv ( col );
    glRasterPos2d( (dx+kb*0.2), dy-kh*0.1 );
    for ( j=0; j<10; j++) glutBitmapCharacter(glut_font, string[j]);
  }
}


/* Button liefert 1 wenn maus im Button, sonst 0  */
/* dx,dy    Ursprung des Buttons (linkes unteres Eck)  */
/* bh, bb   Hoehe, Breite des Buttons  */
/* mx, my   Mauskoordinaten  */
int button(double dx, double dy, char *msg, double mx, double my, void *glut_font)
{
    register int   i, n;
    double bh, bb;

    i=0;
    bb=bh=0.05;

    while (*msg) {i++; msg++; }
    bb+=0.025*i;
    msg-=i;



    if ((mx > dx+bb*0.) & (mx < dx+bb*1.) & (my > dy+bh*0.) & (my < dy+bh*1.))
      {
      glColor3d ( 1., 0.0, 0.0 );
      glBegin ( GL_POLYGON );
        glVertex2d ( dx+bb*0., dy+bh*0. );
        glVertex2d ( dx+bb*1., dy+bh*0. );
        glVertex2d ( dx+bb*1., dy+bh*1. );
        glVertex2d ( dx+bb*0., dy+bh*1. );
      glEnd();
      glColor3d ( 0., 0., 0. );
      glRasterPos2d((dx+bh*0.3), (dy+bh*0.2));
      for (n=0; n<i; n++)
        glutBitmapCharacter(glut_font, msg[n]);
      glBegin ( GL_LINE_LOOP );
        glVertex2d ( dx+bb*0., dy+bh*0. );
        glVertex2d ( dx+bb*1., dy+bh*0. );
        glVertex2d ( dx+bb*1., dy+bh*1. );
        glVertex2d ( dx+bb*0., dy+bh*1. );
      glEnd();
      return (1);
      }
    else
      {
      glColor3d ( 0.7, 0.7, 0.7 );
      glBegin ( GL_POLYGON );
        glVertex2d ( dx+bb*0., dy+bh*0. );
        glVertex2d ( dx+bb*1., dy+bh*0. );
        glVertex2d ( dx+bb*1., dy+bh*1. );
        glVertex2d ( dx+bb*0., dy+bh*1. );
      glEnd();
      glColor3d ( 0., 0., 0. );
      glRasterPos2d((dx+bh*0.3), (dy+bh*0.2));
      for (n=0; n<i; n++)
        glutBitmapCharacter(glut_font, msg[n]);
      glBegin ( GL_LINE_LOOP );
        glVertex2d ( dx+bb*0., dy+bh*0. );
        glVertex2d ( dx+bb*1., dy+bh*0. );
        glVertex2d ( dx+bb*1., dy+bh*1. );
        glVertex2d ( dx+bb*0., dy+bh*1. );
      glEnd();
      return(0);
      }
}


/* Plaziert farbige Punkte  */
void polymark ( int n, double *col_r, double *col_g, double *col_b, double *x,
 double *y, double *z )
{
  GLint mode;
  int i;

  glGetIntegerv(GL_RENDER_MODE, &mode);

  if (mode == GL_RENDER)
    {
    for (i=1; i<=n; i++ )
      {
      glBegin ( GL_POINTS );
       glColor3d ( col_r[i], col_g[i], col_b[i] );
       glVertex3d ( x[i], y[i], z[i] );
      glEnd();
      }
    }
  if (mode == GL_SELECT)
    {
    for (i=1; i<=n; i++ )
      {
      glLoadName(i);
      glBegin ( GL_POINTS );
       glColor3d ( col_r[i], col_g[i], col_b[i] );
       glVertex3d ( x[i], y[i], z[i] );
      glEnd();
      }
    }
}
