/* pam_login_alert module */

/*
 * pam_login_alert.c - driver for pam_login_alert; contains PAM interface.
 *
 * The pam_login_alert module supports the auth, acct, and session stacks.
 * Note that this module will always return PAM_IGNORE since it doesn't
 * actually perform any PAM duties
 */

#include <stdio.h>
#include <unistd.h>
#include <syslog.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include "pam_login_alert.h"

#define PAM_SM_AUTH
#define PAM_SM_ACCOUNT
#define PAM_SM_SESSION

#include <security/pam_modules.h>

static int log_user(pam_handle_t *pamh, int flags, int argc, const char **argv);
static void log_login(alert_parms_t *parms, char *username);
static void mail_login(alert_parms_t *parms, char *username);
static int find_user(char *fname, char *username);
static int getfqdn(char *fqdn, int n);

/****
 * PAM management functions
 */

/* authentication management */

PAM_EXTERN
int pam_sm_authenticate(pam_handle_t *pamh, int flags,
						int argc, const char **argv)
{
	log_user(pamh, flags, argc, argv);
	return PAM_IGNORE;
}

PAM_EXTERN
int pam_sm_setcred(pam_handle_t *pamh, int flags,
					int argc, const char **argv)
{
	return PAM_IGNORE;
}

/* account management */

PAM_EXTERN
int pam_sm_acct_mgmt(pam_handle_t *pamh, int flags,
				int argc, const char **argv)
{
	log_user(pamh, flags, argc, argv);
	return PAM_IGNORE;
}

/* session management */

PAM_EXTERN
int pam_sm_open_session(pam_handle_t *pamh, int flags,
					int argc, const char **argv)
{
	log_user(pamh, flags, argc, argv);
	return PAM_IGNORE;
}

PAM_EXTERN
int pam_sm_close_session(pam_handle_t *pamh, int flags,
					int argc, const char **argv)
{
	return PAM_IGNORE;
}

/****
 * local functions
 */

/* 
 * log_user() - log an alert if the PAM user is in the user list
 */

static
int log_user(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	alert_parms_t parms;
	char *user;

	/* preset parms */
	parms.mail = ON;
	strncpy(parms.email, "root", MAX_EMAIL);
	parms.syslog = OFF;
	parms.syslog_facility = LOG_AUTHPRIV;
	parms.syslog_priority = LOG_INFO;
	strncpy(parms.users_fname, "/etc/login_alert.users", MAX_FILENAME);

	read_conf("/etc/login_alert.conf", &parms);

	/* get the user */
	pam_get_item(pamh, PAM_USER, (void *) &user);

	/* if the user is listed then send an alert */
	if (find_user(parms.users_fname, user))
	{
		if (parms.syslog == ON)
			log_login(&parms, user);
		if (parms.mail == ON)
			mail_login(&parms, user);
	}

	return 1;
}

/*
 * log_login() - log the alert via syslog
 */

static
void log_login(alert_parms_t *parms, char *username)
{
    openlog("PAM_login_alert", LOG_CONS, parms->syslog_facility);
    syslog(parms->syslog_priority, "LOGIN BY %s", username);
    closelog();
}

/*
 * mail_login() - log the alert via email
 */

static
void mail_login(alert_parms_t *parms, char *username)
{
	char fqdn[MAX_FQDN];
	char body[MAX_EMAIL_BODY];
	char mail[MAX_PROGRAMCALL];
	time_t t;
	struct tm *lt;
	FILE *mailp;

	/* get the fully qualitified domain name */
	getfqdn(fqdn, MAX_FQDN);
		
	/* get the time */
	t = time(NULL);
	lt = (struct tm *) localtime(&t);
	if (lt == NULL)
	{
		printerr("could not determine system time (user %s)", username);
		return;
	}

	/* create the email message */
	snprintf(mail, MAX_PROGRAMCALL, "/bin/mail -n -s 'LOGIN ALERT FROM %s' %s",
			fqdn, parms->email);
	snprintf(body, MAX_EMAIL_BODY, "LOGIN BY %s %.2d/%.2d/%.4d %.2d:%.2d:%.2d\n", 
			username, 
			lt->tm_mon, lt->tm_mday, lt->tm_year + 1900, 
			lt->tm_hour, lt->tm_min, lt->tm_sec);

	/* mail it */
	mailp = popen(mail, "w");
	if (mailp == NULL)
		printerr("could not invoke /bin/mail (user %s)", username);
	else
	{
		fprintf(mailp, body);
		pclose(mailp);
	}
}

/*
 * find_user() - return true if the specified user is found in the user list
 */

static 
int find_user(char *fname, char *user)
{
	FILE *fp;
	int found;
	char curuser[MAX_USER];

	fp = fopen(fname, "r");
	if (fp == NULL)
	{
		printerr("cannot open user file %s", fname);
		return 0;
	}

	/* scan each line */
	found = 0;
	while (getline(curuser, MAX_USER, fp) && !found)
	{
		if (!strcmp(user, curuser))
			found = 1;
	}

	fclose(fp);
	return found;
}

/*
 * getfqdn() - get the fully qualified domain name
 */

static
int getfqdn(char *fqdn, int n)
{
	char domainname[MAX_FQDN];

	if (gethostname(fqdn, n))
		fqdn[0] = '\0';
	if (getdomainname(domainname, MAX_FQDN))
		domainname[0] = '\0';
	if (strlen(domainname) > 0)
	{
		strncat(fqdn, ".", n - strlen(fqdn));
		strncat(fqdn, domainname, n - strlen(fqdn));
	}

	return 1;
}

/* if the module is compiled as static */
#ifdef PAM_STATIC

struct pam_module _pam_login_alert_modstruct = 
{
	"pam_login_alert",
	pam_sm_authenticate,
	pam_sm_setcred,
	pam_sm_acct_mgmt,
	pam_sm_open_session,
	pam_sm_close_session,
	NULL
};

#endif

