/*
**************************************************************************
                                 description
                             --------------------
    copyright            : (C) 2001 by Luis Carvalho
    email                : lpassos@mail.telepac.pt
**************************************************************************

**************************************************************************
*                                                                        *
*  This program is free software; you can redistribute it and/or modify  *
*  it under the terms of the GNU General Public License as published by  *
*  the Free Software Foundation; either version 2 of the License, or     *
*  (at your option) any later version.                                   *
*                                                                        *
**************************************************************************/


#include "pmlistpattern.h"

#include "pmoutputdevice.h"
#include "pmxmlhelper.h"
#include "pmlistpatternedit.h"
#include "pmcompositeobject.h"
#include "pmmemento.h"

#include <kdebug.h>
#include "pmglobals.h"

#include <klocale.h>

const PMVector brickSizeDefault = PMVector( 8, 6, 4.5 );
const double mortarDefault = 0.5;
const double depthDefault = 0;

PMListPattern::PMListPattern( )
      : Base( )
{
   m_listType = ListPatternChecker;
   m_brickSize = brickSizeDefault;
   m_mortar = mortarDefault;
}

PMListPattern::~PMListPattern( )
{
}

void PMListPattern::serialize( PMOutputDevice& dev ) const
{
   QString str1;

   switch( m_listType )
   {
      case ListPatternBrick:
         dev.writeLine( "brick" );
         break;
      case ListPatternChecker:
         dev.writeLine( "checker " );
         break;
      case ListPatternHexagon:
         dev.writeLine( "hexagon " );
         break;
   }
   
   Base::serialize( dev );
   
   if( m_listType == ListPatternBrick )
   {
      dev.writeLine( "brick_size " +
                     m_brickSize.serialize( ) );
      str1.setNum( m_mortar );
      dev.writeLine( "mortar " + str1 );
   }
}

void PMListPattern::serialize( QDomElement& e, QDomDocument& doc ) const
{
   switch( m_listType ) 
   {
      case ListPatternBrick:
         e.setAttribute( "listtype", "brick" );
         break;
      case ListPatternChecker:
         e.setAttribute( "listtype", "checker" );
         break;
      case ListPatternHexagon:
         e.setAttribute( "listtype", "hexagon" );
         break;
   };
   e.setAttribute( "bricksize", m_brickSize.serializeXML( ) );
   e.setAttribute( "mortar", m_mortar );
   Base::serialize( e, doc );
}

void PMListPattern::readAttributes( const PMXMLHelper& h )
{
   QString str = h.stringAttribute( "listtype", "checker" );
   if( str == "checker" )
      m_listType = ListPatternChecker;
   else if( str == "brick" )
      m_listType = ListPatternBrick;
   else
      m_listType = ListPatternHexagon;
   m_brickSize = h.vectorAttribute( "bricksize", brickSizeDefault );
   m_mortar = h.doubleAttribute( "mortar", mortarDefault );
   Base::readAttributes( h );
}

bool PMListPattern::isA( PMObjectType t ) const
{
   if( t == PMTListPattern )
      return true;
   return Base::isA( t );
}

bool PMListPattern::canInsert( PMObjectType t, const PMObject*,
                               const PMObjectList* objectsBetween ) const
{
   if( ( t == PMTComment ) || ( t == PMTRaw )  )
      return true;
   else if( t == listObjectType( ) )
   {
      int contains = 0;
      PMObject* o;

      // count child objects
      for( o = firstChild( ); o; o = o->nextSibling( ) )
         if( o->type( ) == listObjectType( ) )
            contains++;
      if( objectsBetween )
      {
         PMObjectListIterator it( *objectsBetween );
         for( ; it.current( ); ++it )
            if( it.current( )->type( ) == listObjectType( ) )
               contains++;
      }
      
      if( ( m_listType == ListPatternChecker ) || 
          ( m_listType == ListPatternBrick ) )
         return ( contains < 2 );
      if( m_listType == ListPatternHexagon )
         return ( contains < 3 );
   }
   return false;
}

void PMListPattern::setListType( PMListType l )
{
   if( l != m_listType )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTListPattern, PMListTypeID, m_listType );
      m_listType = l;
   } 
}

void PMListPattern::setBrickSize( const PMVector& n )
{
   if( n != m_brickSize )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTListPattern, PMBrickSizeID, m_brickSize );
      m_brickSize = n;
   } 
}

void PMListPattern::setMortar( double n )
{
   if( n != m_mortar )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTListPattern, PMMortarID, m_mortar );
      m_mortar = n;
   } 
}

PMDialogEditBase* PMListPattern::editWidget( QWidget* parent ) const
{
   return new PMListPatternEdit( parent );
}

void PMListPattern::restoreMemento( PMMemento* s )
{
   PMMementoDataIterator it( s );
   PMMementoData* data;

   for( ; it.current( ); ++it )
   {
      data = it.current( );
      if( data->objectType( ) == PMTListPattern )
      {
         switch( data->valueID( ) )
         {
            case PMListTypeID:
               setListType( ( PMListType ) data->intData( ) );
               break;
            case PMBrickSizeID:
               setBrickSize( data->vectorData( ) );
               break;
            case PMMortarID:
               setMortar( data->doubleData( ) );
               break;
            default:
               kdError( PMArea ) << "Wrong ID in PMListPattern::restoreMemento\n";
               break;
         }
      }
   }
   Base::restoreMemento( s );
}


PMTextureList::PMTextureList( )
      : Base( )
{
}

PMTextureList::~PMTextureList( )
{
}

bool PMTextureList::isA( PMObjectType t ) const
{
   if( t == PMTTextureList )
      return true;
   return Base::isA( t );
}

QString PMTextureList::description( ) const
{
   return i18n( "texture list" );
}

PMPigmentList::PMPigmentList( )
      : Base( )
{
}

PMPigmentList::~PMPigmentList( )
{
}

bool PMPigmentList::isA( PMObjectType t ) const
{
   if( t == PMTPigmentList )
      return true;
   return Base::isA( t );
}

QString PMPigmentList::description( ) const
{
   return i18n( "pigment list" );
}

PMColorList::PMColorList( )
      : Base( )
{
}

PMColorList::~PMColorList( )
{
}

bool PMColorList::isA( PMObjectType t ) const
{
   if( t == PMTColorList )
      return true;
   return Base::isA( t );
}

QString PMColorList::description( ) const
{
   return i18n( "color list" );
}

PMDensityList::PMDensityList( )
      : Base( )
{
}

PMDensityList::~PMDensityList( )
{
}

bool PMDensityList::isA( PMObjectType t ) const
{
   if( t == PMTDensityList )
      return true;
   return Base::isA( t );
}

QString PMDensityList::description( ) const
{
   return i18n( "density list" );
}

PMNormalList::PMNormalList( )
      : Base( )
{
   m_depth = depthDefault;
}

PMNormalList::~PMNormalList( )
{
}

bool PMNormalList::isA( PMObjectType t ) const
{
   if( t == PMTNormalList )
      return true;
   return Base::isA( t );
}

QString PMNormalList::description( ) const
{
   return i18n( "normal list" );
}

void PMNormalList::setDepth( double d )
{
   if( d != m_depth )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTNormalList, PMDepthID, m_depth );
      m_depth = d;
   } 
}

void PMNormalList::restoreMemento( PMMemento* s )
{
   PMMementoDataIterator it( s );
   PMMementoData* data;

   for( ; it.current( ); ++it )
   {
      data = it.current( );
      if( data->objectType( ) == PMTNormalList )
      {
         switch( data->valueID( ) )
         {
            case PMDepthID:
               setDepth( data->doubleData( ) );
               break;
            default:
               kdError( PMArea ) << "Wrong ID in PMNormalList::restoreMemento\n";
               break;
         }
      }
   }
   Base::restoreMemento( s );
}

bool PMNormalList::canInsert( PMObjectType t, const PMObject* after,
                               const PMObjectList* objectsBetween ) const
{
   // If a depth is specified we can't insert any child
   if( m_depth )
      return false;
   return Base::canInsert( t, after, objectsBetween );
}

void PMNormalList::serialize( PMOutputDevice& dev ) const
{
   QString str1;

   switch( listType( ) )
   {
      case ListPatternBrick:
         dev.writeLine( "brick " );
         break;
      case ListPatternChecker:
         dev.writeLine( "checker " );
         break;
      case ListPatternHexagon:
         dev.writeLine( "hexagon " );
         break;
   }
   if( m_depth )
   {
      str1.setNum( m_depth );
      dev.writeLine( str1 );
   }
   if( listType( ) == ListPatternBrick )
   {
      dev.writeLine( "brick_size " +
                     brickSize( ).serialize( ) );
      str1.setNum( mortar( ) );
      dev.writeLine( "mortar " + str1 );
   }
}

void PMNormalList::serialize( QDomElement& e, QDomDocument& doc ) const
{
   e.setAttribute( "depth", m_depth );
   Base::serialize( e, doc );
}

void PMNormalList::readAttributes( const PMXMLHelper& h )
{
   m_depth = h.doubleAttribute( "depth", depthDefault );
   Base::readAttributes( h );
}

PMDialogEditBase* PMNormalList::editWidget( QWidget* parent ) const
{
   return new PMListPatternEdit( parent );
}

