/***************************************************************************
                          fileread.cpp  -  description
                             -------------------
    begin                : Wed May 23 2001
    copyright            : (C) 2001 by Javier Campos
    email                : javi_cms@terra.es
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "fileread.h"

#include <assert.h>

#include <klocale.h>
#include <kio/netaccess.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <ktempfile.h>

#include <qtextstream.h>
#include <qfile.h>
#include <qfileinfo.h>

#include <iostream.h>

FileRead::FileRead( QObject *parent, const char *name )
    :QObject(parent, name),
     _changed(false),
     _tmpfile(0)
{
}

FileRead::~FileRead()
{
}

/** Open keduca file. This function can open a remote or local url. */
bool FileRead::openFile(const KURL &url) {
    QString tmpFile;
    bool returnval=false;
    if( KIO::NetAccess::download( url, tmpFile ) )
    {
        returnval=loadFile( tmpFile );
        if (returnval) {
            _currentURL = url;
            kdDebug()<<"... load sucessfull: "<<_currentURL.url()<<endl;
        }

        KIO::NetAccess::removeTempFile( tmpFile );
    }else
        kdDebug()<<"FileRead::openFile(): download NOT successfull: "<<url.url()<<endl;

    return returnval;
}

bool FileRead::loadFile( const QString &filename )
{
    QDomDocument doc("document.xml");

    QFile file(filename);

    if(!file.open(IO_ReadOnly))
    {
        return false;
    }
    doc.setContent( &file );

    if( doc.doctype().name() != "educa" )
    { return false; }

    QDomElement docElem = doc.documentElement();
    QDomNode n = docElem.firstChild();

    QDomNodeList dnList = n.childNodes();
    for( unsigned int i = 0; i < dnList.count(); ++i)
    {
        // ------------------- INFORMATION BODY -----------------------
        QDomElement element = dnList.item(i).toElement();
        if( element.tagName() == "default" || element.tagName() == "author" )
        {
            if( element.tagName() == "default" ) {	_header.insert( "image", element.attribute( "image", "default.png" ) ); }
            if( element.tagName() == "author" )	{
                QDomNodeList AuthordnList = element.childNodes();
                for( unsigned int i = 0; i < AuthordnList.count(); ++i) {
                    QDomElement authorelement = AuthordnList.item(i).toElement();
                    _header.insert( authorelement.tagName(), authorelement.text() );
                }
            }
        } else {
            _header.insert( element.tagName(), element.text() );
        }
    }

    n = n.nextSibling();

    dnList = n.childNodes();
    for( unsigned int i = 0; i < dnList.count(); ++i)
    {
        insertQuestion();
        // --------------------- QUESTION ATTRIBUTE------------------------
        QDomElement elementNODE = dnList.item(i).toElement();
        setQuestion( QF_TYPE, 		elementNODE.attribute( "type", "1" ).toInt() );
        setQuestion( QF_PICTURE, 	elementNODE.attribute( "image", "" ) );
        setQuestion( QF_TIME, 		elementNODE.attribute( "time", "0" ).toInt() );
        setQuestion( QF_POINTS, 	elementNODE.attribute( "points", "0" ).toInt() );

        QDomNodeList quList = elementNODE.childNodes();
        for( unsigned int x = 0; x < quList.count(); ++x)
        {
            // --------------------- QUESTION AND RESPONSES------------------
            QDomElement element = quList.item(x).toElement();
            if( element.tagName() == "text" ) 		setQuestion( QF_TEXT, element.text() );
            if( element.tagName() == "true" )		setAnswer( element.text(), true, element.attribute( "points", "0" ).toInt() );
            if( element.tagName() == "false" )		setAnswer( element.text(), false,element.attribute( "points", "0" ).toInt() );
            if( element.tagName() == "tip" ) 			setQuestion( QF_TIP, element.text() );
            if( element.tagName() == "explain" ) 	setQuestion( QF_EXPLAIN, element.text() );
        }
    }

    file.close();
    _changed=false;
    return true;
}

/** Set value to questions fields */
void FileRead::setQuestion( QuestionField field, QString text)
{
// QF_text, QF_picture, QF_type, QF_time, QF_tip, QF_explain
    switch( field )
    {
    case QF_TEXT:
        (*_recordQuestions).text = text;
        break;
    case QF_PICTURE:
        (*_recordQuestions).picture = text;
        break;
    case QF_TIP:
        (*_recordQuestions).tip = text;
        break;
    case QF_EXPLAIN:
        (*_recordQuestions).explain = text;
        break;
    default:
        kdDebug()<<"FileRead::setQuestion(QuestionField field, QString text) called for not handled field value "<<field <<endl;
        break;
    }
    _changed=true;
}

/** Set value to questions fields */
void FileRead::setQuestion( QuestionField field, int value )
{
    switch( field )
    {
    case QF_TYPE:
        (*_recordQuestions).type = value;
        break;
    case QF_TIME:
        (*_recordQuestions).time = value;
        break;
    case QF_POINTS:
        (*_recordQuestions).points = value;
        break;
    default:
        kdDebug()<<"FileRead::setQuestion(QuestionField field, int value) called for not handled field value "<<field <<endl;
        break;
    }
    _changed=true;
}

/** Insert an answer field */
void FileRead::setAnswer( QString text, bool value, int points)
{
    Answers tmpAnswers;

    tmpAnswers.text = text;
    tmpAnswers.value = value;
    tmpAnswers.points = points;

    (*_recordQuestions).listAnswers.append( tmpAnswers );
    _changed=true;
}

/** Insert a blank question */
void FileRead::insertQuestion()
{
    Questions tempQuestions;
    tempQuestions.text = "";
    _listQuestions.append( tempQuestions );
    recordLast();
    _changed=true;
}

/** Go to the First record */
void FileRead::recordFirst()
{
    if( _fileEOF = true ) _fileEOF = false;
    if( _fileBOF = false ) _fileBOF = true;
    _recordQuestions = _listQuestions.begin();
}

/** Go to the Last Record */
void FileRead::recordLast()
{
    if( _fileBOF = true ) 	_fileBOF = false;
    if( _fileEOF = false ) _fileEOF = true;
    _recordQuestions = _listQuestions.end();
    --_recordQuestions;
}

/** Record Next */
void FileRead::recordNext()
{
    ++_recordQuestions;
    if( _recordQuestions == _listQuestions.end() )
    {
        _fileEOF = true;
        --_recordQuestions;
    }
    else
    {
        if( _fileBOF = true ) _fileBOF = false;
    }
}

/** Record previous */
void FileRead::recordPrevious()
{
    if( _recordQuestions == _listQuestions.begin() )
    {
        _fileBOF = true;
    }
    else
    {
        if( _fileEOF = true ) _fileEOF = false;
        --_recordQuestions;
    }
}

/** Go to the First record */
void FileRead::recordAnswerFirst()
{
    if( _fileAnswerEOF = true ) _fileAnswerEOF = false;
    if( _fileAnswerBOF = false ) _fileAnswerBOF = true;
    (*_recordQuestions).recordAnswers = (*_recordQuestions).listAnswers.begin();
}

/** Go to the Last Record */
void FileRead::recordAnswerLast()
{
    if( _fileAnswerBOF = true ) 	_fileAnswerBOF = false;
    if( _fileAnswerEOF = false ) _fileAnswerEOF = true;
    (*_recordQuestions).recordAnswers = (*_recordQuestions).listAnswers.end();
    --(*_recordQuestions).recordAnswers;
}

/** Record Next */
void FileRead::recordAnswerNext()
{
    ++(*_recordQuestions).recordAnswers;
    if( (*_recordQuestions).recordAnswers == (*_recordQuestions).listAnswers.end() )
    {
        _fileAnswerEOF = true;
        --(*_recordQuestions).recordAnswers;
    }
    else
    {
        if( _fileAnswerBOF = true ) _fileAnswerBOF = false;
    }
}

/** Record previous */
void FileRead::recordAnswerPrevious()
{
    if( (*_recordQuestions).recordAnswers == (*_recordQuestions).listAnswers.begin() )
    {
        _fileBOF = true;
    }
    else
    {
        if( _fileAnswerEOF = true ) _fileAnswerEOF = false;
        --(*_recordQuestions).recordAnswers;
    }
}

/** Get the value of the field question */
QString FileRead::getQuestion( QuestionField field )
{
// QF_text, QF_picture, QF_type, QF_time, QF_tip, QF_explain
    switch( field )
    {
    case QF_TEXT:
        return (*_recordQuestions).text;
        break;
    case QF_PICTURE:
        return getPictureLocal( (*_recordQuestions).picture );
        break;
    case QF_POINTS:
        return QString().setNum( (*_recordQuestions).points );
        break;
    case QF_TIME:
        return QString().setNum( (*_recordQuestions).time );
        break;
    case QF_TIP:
        return (*_recordQuestions).tip;
        break;
    case QF_EXPLAIN:
        return (*_recordQuestions).explain;
        break;
    default:
        kdDebug()<<"FileRead::getQuestion() called for not handled field value "<<field <<endl;
        return "";
        break;
    }
}

/** Get the value of the field questions in integer */
int FileRead::getQuestionInt( QuestionField field )
{
    switch( field )
    {
    case QF_TYPE:
        return (*_recordQuestions).type;
        break;
    case QF_TIME:
        return (*_recordQuestions).time;
        break;
    case QF_POINTS:
        return (*_recordQuestions).points;
        break;
    default:
        kdDebug()<<"FileRead::getQuestionInt() called for not handled field value "<<field <<endl;
        return 0;
    }
}

/** Get Answer field */
QString FileRead::getAnswer( AnswerField field )
{
    // AField { AF_text, AF_value, AF_picture, AF_point };
    switch( field )
    {
    case AF_TEXT:
        return (*(*_recordQuestions).recordAnswers).text;
        break;
//		case AF_VALUE:
//			(*(*_recordQuestions).RecordAnswers).Value ? return i18n("True") : return i18n("False");
//			break;
    case AF_POINT:
        return QString().setNum( (*(*_recordQuestions).recordAnswers).points );
        break;
    default:
        kdDebug()<<"FileRead::getAnswer() called for not handled field value "<<field <<endl;
        return "";
    }
}

/** Get Answer field */
bool FileRead::getAnswerValue()
{
    return (*(*_recordQuestions).recordAnswers).value;
}

/** Get answer field */
int FileRead::getAnswerPoints()
{
    return (*(*_recordQuestions).recordAnswers).points;
}

/** Returns if this record is a end of file */
bool FileRead::recordEOF()
{
    return _fileEOF;
}

/** Returns if this record is a begin of file */
bool FileRead::recordBOF()
{
    return _fileBOF;
}

/** Returns if this record is a end of file */
bool FileRead::recordAnswerEOF()
{
    return _fileAnswerEOF;
}

/** Returns if this record is a begin of file */
bool FileRead::recordAnswerBOF()
{
    return _fileAnswerBOF;
}

/** Get an existing file */
QString FileRead::getPictureLocal( const QString &file )
{
    if( file.isEmpty() ) return "";

    QString picture = file;

    if( !QFileInfo(picture).exists() )
    {
        picture = _currentURL.directory(false,true) + QFileInfo(picture).fileName();
        if( !QFileInfo(picture).exists() ) { 	picture = ""; }
    }
    return picture;

}

/** Clear answers */
void FileRead::clearAnswers()
{
    ((*_recordQuestions).listAnswers).clear();
    _changed=true;
}

/** Delete current record */
void FileRead::recordDelete()
{
    _listQuestions.remove( _recordQuestions );
    _changed=true;
}

/** Swap two variables */
void FileRead::recordSwap( bool moveup )
{
    Questions listTMP;
    Questions listNEW;

    listTMP = (*_recordQuestions);

    if( moveup )
    {
        recordPrevious();
        listNEW = (*_recordQuestions);
        (*_recordQuestions) = listTMP;
        recordNext();
        (*_recordQuestions) = listNEW;
    }
    else
    {
        recordNext();
        listNEW = (*_recordQuestions);
        (*_recordQuestions) = listTMP;
        recordPrevious();
        (*_recordQuestions) = listNEW;
    }
    _changed=true;
}

/** Save file */
bool FileRead::saveFile( const KURL &url, bool copyimages )
{
    if (!url.isMalformed())
    {
        _currentURL = url;
    }
    kdDebug()<<"FileRead::saveFile() to "<<_currentURL.url()<<endl;
    // Local file
    if ( _currentURL.isLocalFile() )
    {
        if ( _tmpfile != 0 ) // get rid of a possible temp file first
        {              // (happens if previous _currentURL was remote)
             _tmpfile->unlink();
             delete _tmpfile;
             _tmpfile = 0;
        }
        if( saveFile(_currentURL.path(), copyimages) ) {
            emit completed();
            emit setWindowCaption( _currentURL.prettyURL() );
            return true;
        }
    }
    else
    { // Remote file
        // We haven't saved yet, or we did but locally - provide a temp file
        if ( _tmpfile == 0 )
        {
             _tmpfile = new KTempFile;
        }
        // otherwise, we already had a temp file
        if( saveFile(_tmpfile->name(), copyimages) ) {
            // upload the file
             KIO::Job * job = KIO::file_copy( _tmpfile->name(), _currentURL, -1, true /*overwrite*/ );
             connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotUploadFinished (KIO::Job *) ) );
            return true;
        }
    }

    // Save local file and upload local file
    return false;
}

bool FileRead::saveFile( const QString &filename, bool copyimages )
{
    KLocale *locale = KGlobal::locale();
    QDomDocument doc("document.xml");
    QTextStream stream;
    QString line;
    QFile file(filename);
    QStringList copyJOB;

    stream.setDevice(&file);

    if(!file.open(IO_WriteOnly))
    {
        // No puede abrir la base
        return false;
    }

#if QT_VERSION < 300
    QString head( "<?xml version='1.0' encoding='"+ locale->charset() +"' standalone='yes' ?><!DOCTYPE educa>" );
#else
    QString head( "<?xml version='1.0' encoding='"+ QString::fromLatin1(locale->encoding()) +"' standalone='yes' ?><!DOCTYPE educa>" );
#endif
    doc.setContent( head );

    QDomElement Root = doc.createElement("Document");
    doc.appendChild( Root );

    QDomElement NodeInfo = doc.createElement("Info");
    Root.appendChild( NodeInfo );

    insertXML( doc, NodeInfo, "title", _header["title"] );
    insertXML( doc, NodeInfo, "category", _header["category"] );
    insertXML( doc, NodeInfo, "type", _header["type"] );
    insertXML( doc, NodeInfo, "level", _header["level"] );
    insertXML( doc, NodeInfo, "language", _header["language"] );

    if( !(_header["image"]).isEmpty() )
    {
        QDomElement Nodedefault = doc.createElement("default");
        if( copyimages )
        {
            copyJOB.append( _header["image"] );
            Nodedefault.setAttribute( "image", QFileInfo(_header["image"]).fileName() );
        } else {
            Nodedefault.setAttribute( "image", _header["image"]);
        }
        NodeInfo.appendChild( Nodedefault );
    }

    if( !_header["name"].isEmpty() || !_header["email"].isEmpty() || !_header["www"].isEmpty() )
    {
        QDomElement Nodeauthor = doc.createElement("author");
        NodeInfo.appendChild( Nodeauthor );
        if( !_header["name"].isEmpty() ) 	insertXML( doc, Nodeauthor, "name", _header["name"] );
        if( !_header["email"].isEmpty() )	insertXML( doc, Nodeauthor, "email", _header["email"] );
        if( !_header["www"].isEmpty() )		insertXML( doc, Nodeauthor, "www", _header["www"] );
    }

    QDomElement NodeData = doc.createElement("Data");
    Root.appendChild( NodeData );

    recordFirst();
    while ( !recordEOF() )
    {
        QDomElement question = doc.createElement("question");
        if( !getQuestion( QF_PICTURE ).isEmpty() )
        {
            if( copyimages )
            {
                copyJOB.append( getQuestion( QF_PICTURE ) );
                question.setAttribute("image", QFileInfo( getQuestion( QF_PICTURE ) ).fileName() );
            } else {
                question.setAttribute("image",  getQuestion( QF_PICTURE ) );
            }
        }
        question.setAttribute( "type", getQuestionInt( QF_TYPE ) );
        if( getQuestionInt( QF_POINTS ) > 0 ) question.setAttribute( "points", getQuestion( QF_POINTS ) );
        if( getQuestionInt( QF_TIME ) > 0 ) question.setAttribute( "time", getQuestion( QF_TIME ) );
        insertXML( doc, question, "text", getQuestion( QF_TEXT ) );

        recordAnswerFirst();
        while( !recordAnswerEOF() )
        {
            if( getAnswerValue() )
            {
                QDomElement domELEMENT = doc.createElement( "true" );
                if( getAnswerPoints() > 0 ) domELEMENT.setAttribute("points", getAnswerPoints() );
                QDomText DATAelement = doc.createTextNode( getAnswer( AF_TEXT ) );
                question.appendChild( domELEMENT );
                domELEMENT.appendChild( DATAelement );
            }	else {
//				insertXML( doc, question, "false", getAnswer( AF_text ) );
                QDomElement domELEMENT = doc.createElement( "false" );
                if( getAnswerPoints() > 0 ) domELEMENT.setAttribute("points", getAnswerPoints() );
                QDomText DATAelement = doc.createTextNode( getAnswer( AF_TEXT ) );
                question.appendChild( domELEMENT );
                domELEMENT.appendChild( DATAelement );
            }
            recordAnswerNext();
        };

        if( !getQuestion( QF_TIP ).isEmpty() ) insertXML( doc, question, "tip", getQuestion( QF_TIP ) );
        if( !getQuestion( QF_EXPLAIN ).isEmpty() ) insertXML( doc, question, "explain", getQuestion( QF_EXPLAIN ) );

        NodeData.appendChild( question );
        recordNext();
    }

    doc.save( stream, 4);
    file.close();

    if( copyimages == true && copyJOB.count() > 0 )
    {
        KURL::List	KurlLIST( copyJOB );
        KIO::CopyJob *copyjob;

        copyjob = KIO::copy( KurlLIST, _currentURL.directory(false,true), true);
    }
    _changed=false;
    return true;

}

/** Insert xml format data */
void FileRead::insertXML( QDomDocument &doc, QDomElement &parent, const QString &tagName, const QString &data)
{
    QDomElement domELEMENT = doc.createElement( tagName );
    QDomText DATAelement = doc.createTextNode( data );

    parent.appendChild( domELEMENT );
    domELEMENT.appendChild( DATAelement );
}

/** Insert xml data format */
void FileRead::insertXML( QDomDocument &doc, QDomElement &parent, const QString &data)
{
    QDomText DATAelement = doc.createTextNode( data );
    parent.appendChild( DATAelement );
}

/** Get real picture */
QString FileRead::getPicture()
{
    if( getQuestion( QF_PICTURE).isEmpty() && _header["image"].isEmpty() )
    {
        return locate("data", "keduca/pics/default.png" );
    } else if ( getQuestion( QF_PICTURE ).isEmpty() )
    {
        return getPictureLocal( _header["image"] );
    } else {
        return getPictureLocal( getQuestion( QF_PICTURE ) ) ;
    }
}

/** Get Header */
QString FileRead::getHeader(const QString &head)
{
    return _header[head];
}

/** Set header data */
void FileRead::setHeader( const QString field, const QString value)
{
    if( (_header[field]).isEmpty() )
        _header.insert( field, value );
    else
        _header.replace( field, value );
}

/** is Multi Answer */
bool FileRead::isMultiAnswer()
{
    int numOKanswer = 0;

    recordAnswerFirst();

    while( !recordAnswerEOF() )
    {
        if( (*(*_recordQuestions).recordAnswers).value == true ) numOKanswer++;
        recordAnswerNext();
    }

    if( numOKanswer > 1 )	{
        return true;
    }	else {
        return false;
    }
}

void FileRead::slotUploadFinished( KIO::Job * job )
{
  if (job->error()) {
    emit canceled( job->errorString() );
    kdDebug()<< "FileRead::slotUploadFinished(): " <<job->errorString()<<endl;
  }
  else
  {
    if ( _tmpfile!=0 ) // We're finished with this document -> remove temp file
    {
       _tmpfile->unlink();
       delete _tmpfile;
       _tmpfile=0;
    }
    emit setWindowCaption( _currentURL.prettyURL() );
    emit completed();
  }
}

#include "fileread.moc"
