/*
 *  Copyright (C) 2001-2002, Richard J. Moore <rich@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

#include <qcombobox.h>
#include <qlayout.h>
#include <qpushbutton.h>

#include <kdebug.h>
#include <kdialog.h>
#include <kjs/interpreter.h>
#include <kjs/ustring.h>
#include <kjs/types.h>

#include "jsbinding.h"
#include "messagelogwidget.h"

#include "jsconsolewidget.h"
#include "jsconsolewidget.moc"

//
// JScript Widget
//

namespace KJSEmbed {

JSConsoleWidget::JSConsoleWidget( QWidget *parent, const char *name )
    : QFrame( parent, name ? name : "JSConsoleWidget" )
{
    setFocusPolicy( QWidget::StrongFocus );
    js = new KJS::Interpreter();
    createView();
}

JSConsoleWidget::JSConsoleWidget( KJS::Interpreter *js, QWidget *parent, const char *name )
    : QFrame( parent, name ? name : "JSConsoleWidget" )
{
    setFocusPolicy( QWidget::StrongFocus );
    createView();
    this->js = js;
}

JSConsoleWidget::~JSConsoleWidget()
{
    delete js;
}

void JSConsoleWidget::createView()
{
    //
    // Contents
    //
    log = new MessageLogWidget( this, "LogView" );

    cmd = new QComboBox( true, this, "CmdEdit" );
    cmd->setAutoCompletion( true );
    cmd->setInsertionPolicy( QComboBox::AtTop );
    cmd->setFocus();

    run = new QPushButton( "&Run", this, "RunButton" );
    run->setFixedSize( run->sizeHint() );

    connect( cmd, SIGNAL( activated(int) ), run, SLOT( animateClick() ) );
    connect( run, SIGNAL( clicked() ), SLOT( execute() ) );

    //
    // Layout
    //
    QVBoxLayout *vertical = new QVBoxLayout( this, KDialog::marginHint(), KDialog::spacingHint() );
    vertical->addWidget( log );

    QHBoxLayout *horizontal = new QHBoxLayout( vertical );
    horizontal->addWidget( cmd );
    horizontal->addWidget( run );

    log->message( QString("-- Enter a JS expression and press enter --") );
}

void JSConsoleWidget::execute()
{
    QString code = cmd->currentText();
    execute( code );
}

bool JSConsoleWidget::execute( const QString &cmd )
{
    log->raw( QString( "<b>kjs&gt; %1</b><br>" ).arg( cmd ) );
    KJS::Completion res = js->evaluate( KJS::UString(cmd), js->globalObject() );

    QString s = res.value().toString( js->globalExec() ).qstring();
    KJS::ComplType ct = res.complType();

    // Report Errors
    if ( ct == KJS::Throw ) {
	log->error( s );
	return false;
    }
    // Handle Return Values
    else if ( ct == KJS::ReturnValue ) {
	log->message( s );
	return true;
    }
    // Handle Normal
    else if ( ct == KJS::Normal ) {
	log->message( s );
	return true;
    }
    // Handle Break
    else if ( ct == KJS::Break ) {
	log->error( s );
	return false;
    }
    // Handle Continue
    else if ( ct == KJS::Continue ) {
	log->error( s );
	return false;
    }

    return false;
}

//
// JScript <-> Widget Bindings
//

void JSConsoleWidget::addBindings( KJS::ExecState *state, KJS::Object &object )
{
    object.put( state ,"print", KJS::Object( new MethodImp( MethodImp::MethodPrint, this ) ) );
}


KJS::Value JSConsoleWidget::MethodImp::call( KJS::ExecState *state, KJS::Object &/*self*/,
					     const KJS::List &args )
{
    if ( state->interpreter() != w->jscript() ) {
	kdWarning() << "JSConsole::Method call from unknown interpreter!" << endl;
	return KJS::Undefined();
    }

    kdDebug() << "JSConsole::Method print called " << (int) id << endl;

    w->messages()->message( args[0].toString( state ).qstring() );

    return KJS::Undefined();
}

}; // namespace KJSEmbed

// Local Variables:
// c-basic-offset: 4
// End:
