/* CallableStatement.java -- A statement for calling stored procedures.
   Copyright (C) 1999, 2000 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.
 
GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */


package java.sql;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Map;

/**
  * This interface provides a mechanism for calling stored procedures.
  *
  * @author Aaron M. Renn (arenn@urbanophile.com)
  */
public interface CallableStatement extends PreparedStatement
{

/*************************************************************************/

/**
  * This method tests whether the value of the last parameter that was fetched
  * was actually a SQL NULL value.
  *
  * @return <code>true</code> if the last parameter fetched was a NULL,
  * <code>false</code> otherwise.
  * 
  * @exception SQLException If an error occurs.
  */
public abstract boolean
wasNull() throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>String</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>String</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract String
getString(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>Object</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as an <code>Object</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract Object
getObject(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>Object</code>.
  *
  * @param index The index of the parameter to return.
  * @param map The mapping to use for conversion from SQL to Java types.
  *
  * @return The parameter value as an <code>Object</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract Object
getObject(int index, Map map) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>boolean</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>boolean</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract boolean
getBoolean(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>byte</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>byte</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract byte
getByte(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>short</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>short</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract short
getShort(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>int</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>int</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract int
getInt(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>long</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>long</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract long
getLong(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>float</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>float</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract float
getFloat(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>double</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>double</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract double
getDouble(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>BigDecimal</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>BigDecimal</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract BigDecimal
getBigDecimal(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>BigDecimal</code>.
  *
  * @param index The index of the parameter to return.
  * @param scale The number of digits to the right of the decimal to return.
  *
  * @return The parameter value as a <code>BigDecimal</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract BigDecimal
getBigDecimal(int index, int scale) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * byte array.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a byte array
  *
  * @exception SQLException If an error occurs.
  */
public abstract byte[]
getBytes(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>java.sql.Date</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>java.sql.Date</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract java.sql.Date
getDate(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>java.sql.Date</code>.
  *
  * @param index The index of the parameter to return.
  * @param calendar The <code>Calendar</code> to use for timezone and locale.
  *
  * @return The parameter value as a <code>java.sql.Date</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract java.sql.Date
getDate(int index, Calendar calendar) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>java.sql.Time</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>java.sql.Time</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract java.sql.Time
getTime(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>java.sql.Time</code>.
  *
  * @param index The index of the parameter to return.
  * @param calendar The <code>Calendar</code> to use for timezone and locale.
  *
  * @return The parameter value as a <code>java.sql.Time</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract java.sql.Time
getTime(int index, Calendar calendar) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>java.sql.Timestamp</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>java.sql.Timestamp</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract java.sql.Timestamp
getTimestamp(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>java.sql.Timestamp</code>.
  *
  * @param index The index of the parameter to return.
  * @param calendar The <code>Calendar</code> to use for timezone and locale.
  *
  * @return The parameter value as a <code>java.sql.Timestamp</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract java.sql.Timestamp
getTimestamp(int index, Calendar calendar) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>Ref</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>Ref</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract Ref
getRef(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>Blob</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>Blob</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract Blob
getBlob(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>Clob</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>Clob</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract Clob
getClob(int index) throws SQLException;

/*************************************************************************/

/**
  * This method returns the value of the specified parameter as a Java
  * <code>Array</code>.
  *
  * @param index The index of the parameter to return.
  *
  * @return The parameter value as a <code>Array</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract Array
getArray(int index) throws SQLException;

/*************************************************************************/

/**
  * This method registers the specified parameter as an output parameter
  * of the specified SQL type.
  *
  * @param index The index of the parameter to register as output.
  * @param type The SQL type value from <code>Types</code>.
  *
  * @exception SQLException If an error occurs.
  */
public abstract void
registerOutParameter(int index, int type) throws SQLException;

/*************************************************************************/

/**
  * This method registers the specified parameter as an output parameter
  * of the specified SQL type.
  *
  * @param index The index of the parameter to register as output.
  * @param type The SQL type value from <code>Types</code>.
  * @param name The user defined data type name.
  *
  * @exception SQLException If an error occurs.
  */
public abstract void
registerOutParameter(int index, int type, String name) throws SQLException;

/*************************************************************************/

/**
  * This method registers the specified parameter as an output parameter
  * of the specified SQL type and scale.
  *
  * @param index The index of the parameter to register as output.
  * @param type The SQL type value from <code>Types</code>.
  * @param scale The scale of the value that will be returned.
  *
  * @exception SQLException If an error occurs.
  */
public abstract void
registerOutParameter(int index, int type, int scale) throws SQLException;

} // interface CallableStatement


