/***************************************************************************
	kjprefs.cpp  -  Preferences-Dialog for KJfol-Skinloader
	--------------------------------------------------------
	Maintainer: Stefan Gehn <sgehn@gmx.net>
 
 ***************************************************************************/

// local includes
#include "kjprefs.h"
#include "kjloader.h"
#include "kjwidget.h"
#include "parser.h"

//#include "kjprefswidget.h"

// system includes
#include <qcombobox.h>
#include <qfileinfo.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qstringlist.h>
#include <qregexp.h>
#include <qpixmap.h>

#include <kconfig.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kio/job.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmimemagic.h>
#include <knotifyclient.h>
#include <kprocess.h>
#include <kstandarddirs.h>


KJPrefs::KJPrefs(QObject* parent)
	: CModule(i18n("K-Jöfol Skins"), i18n("Skin Selection For the K-Jöfol Plugin"), "style", parent)
{
	QVBoxLayout *vbox = new QVBoxLayout( this );
	vbox->setSpacing( 0 );
	vbox->setMargin( 0 );

	prefsWidget = new KJPrefsWidget(this);
	vbox->addWidget(prefsWidget);

	connect ( prefsWidget->mSkins, SIGNAL(activated(const QString&)), SLOT(showPreview(const QString&)) );
	connect ( prefsWidget->installButton, SIGNAL(clicked()), this, SLOT(installNewSkin()) );
	connect ( prefsWidget->mRemoveButton, SIGNAL(clicked()), this, SLOT(removeSelectedSkin()) );

	reopen(); // fill the skinlist and draw a preview
}


void KJPrefs::installNewSkin( void )
{
	bool skinInstalled = false; // flag showing wether a skindir got installed
	KURL src, dst; // sourcedir and destinationdir for skin-installation

	KURL srcFile ( prefsWidget->mSkinRequester->url() );

	kdDebug(66666) << "file to work on: " << srcFile.path().latin1() << endl;

	if ( srcFile.isMalformed() || srcFile.isEmpty() ) // stop working on broken URLs
	{
		kdDebug(66666) << "srcFile is malformed or empty !!!" << endl;
		return;
	}

	if ( !srcFile.isLocalFile() )  // TODO: Download file into tmp dir + unpack afterwards
	{
		KMessageBox::sorry ( this, i18n("Non-Local files aren't supported yet") );
		return;
	}

	// Determine file-format trough mimetype (no stupid .ext test)
	KMimeMagicResult * result = KMimeMagic::self()->findFileType( srcFile.path() );

	if ( !result->isValid() )
	{
		kdDebug(66666) << "Could not determine filetype of srcFile !!!" << endl;
		return;
	}

	if ( result->mimeType() != "application/x-zip" )
	{
		KMessageBox::error ( this, i18n("The selected file does not appear to be a valid zip-archive") );
		return;
	}

	// create a dir with name of the skinarchive
	// path to unpack to: pathToTmp/filename.ext/
	QString tmpUnpackPath = locateLocal("tmp", srcFile.fileName()+"/" );
	kdDebug(66666) << "tmpUnpackPath: " << tmpUnpackPath.latin1() << endl;

	// Our extract-process, TODO: wanna have kio_(un)zip instead :)
	KShellProcess proc;

	// "unzip -d whereToUnpack whatToUnpack"
	proc << "unzip -d " << proc.quote(tmpUnpackPath) << " " << proc.quote(srcFile.path());
	kdDebug(66666) << "unzip -d " << tmpUnpackPath.latin1() << " " << srcFile.path().latin1() << endl;

	proc.start( KProcess::Block, KProcess::NoCommunication );

	// "unzip" spits out errorcodes > 0 only, 0 on success
	if ( proc.exitStatus() != 0 )
	{
		KMessageBox::error ( this, i18n("Extracting skin-archive failed") );
		// FIXME: Do I have to wait for the job to finish?
		// I'd say no because I don't care about the temp-dir
		// anyway after leaving this method :)
		KIO::del( tmpUnpackPath );
		return;
	}

	QDir tmpCnt = QDir ( tmpUnpackPath );
	tmpCnt.setFilter ( QDir::Dirs );

	QStringList dirList = tmpCnt.entryList();
	// Iterate trough all subdirs of tmpUnpackPath (including "."!)
	for ( int i = 0; i < dirList.count(); i++ )
	{
		// FIXME: is the following portable?
		if ( dirList[i] == ".." )
			continue;

		QDir tmpSubCnt = QDir( tmpUnpackPath + dirList[i], "*.rc;*.RC;*.Rc;*.rC", QDir::Name|QDir::IgnoreCase, QDir::Files );
		kdDebug(66666) << "Searching for *.rc in " << QString(tmpUnpackPath+dirList[i]).latin1() << endl;

		// oh, no .rc file in current dir, let's go to next dir in list
		if ( tmpSubCnt.count() == 0 )
			continue;

		src = KURL::encode_string(tmpUnpackPath+dirList[i]);
		dst = KURL::encode_string(locateLocal("data","noatun/kjskins/")); // destination to copy skindir into

		if ( dirList[i] == "." ) // zip did not contain a subdir, we have to create one
		{
			// skindir is named like the archive without extension (FIXME: extension is not stripped from name)
			
			int dotPos = srcFile.fileName().findRev('.');
			if ( dotPos > 0 ) // found a dot -> (hopefully) strip the extension
			{
				dst.addPath( srcFile.fileName().left(dotPos) );
			}
			else // we don't seem to have any extension, just append the archivename
			{
				dst.addPath( srcFile.fileName() );
			}

			kdDebug(66666) << "want to create: " << dst.path().latin1() << endl;

			if ( dst.isMalformed() )
			{
				KMessageBox::error ( this, i18n("Installing new skin failed: Destination path is invalid.\nPlease report a bug to the K-Jöfol maintainer") );
				KIO::del( tmpUnpackPath );
				return;
			}
			KIO::mkdir( dst );
		}

		if ( src.isMalformed() || dst.isMalformed() )
			KMessageBox::error ( this, i18n("Installing new skin failed: Either source or destination path is invalid.\nPlease report a bug to the K-Jöfol maintainer") );
		else
		{
			kdDebug(66666) << "src: " << src.path().latin1() << endl;
			kdDebug(66666) << "dst: " << dst.path().latin1() << endl;
			KIO::Job *job = KIO::copy(src,dst);
			connect ( job, SIGNAL(result(KIO::Job*)), this, SLOT(slotResult(KIO::Job*)) );
			skinInstalled = true;
		}
	} // END iterate trough dirList

	if ( !skinInstalled )
	{
		KMessageBox::sorry ( this, i18n("No new skin has been installed.\nMake sure the archive contains a valid K-Jöfol skin") );
	}
	else
	{
		KMessageBox::information ( this, i18n("The new skin has been successfully installed") );
	}

	KIO::del( tmpUnpackPath );
}


void KJPrefs::removeSelectedSkin( void )
{
	QString question = i18n("Are you sure you want to remove %1?\n"
		"This will delete the files installed by this skin ").
		arg ( prefsWidget->mSkins->currentText() );

	QString loadedSkin = KGlobal::config()->readEntry("SkinResource", "kjofol");
//	kdDebug(66666) << "loaded Skin Name: " << QFileInfo(loadedSkin).baseName().latin1() << endl;

	int r = KMessageBox::questionYesNo ( this, question, i18n("Confirmation") );
	if ( r != KMessageBox::Yes )
		return;

	bool deletingCurrentSkin = ( prefsWidget->mSkins->currentText() == QFileInfo(loadedSkin).baseName() );

	// Now find the dir to delete !!!

	QString dirToDelete = QString ("");
	QStringList skinLocations = KGlobal::dirs()->findDirs("data", "noatun/kjskins");

	// iterate through all paths where Noatun is searching for kjofol-skins
	for (uint i = 0; i < skinLocations.count(); ++i )
	{
		QStringList skinDirs = QDir(skinLocations[i]).entryList();

		// iterate trough all dirs containing a skin
		for (uint k = 0; k < skinDirs.count(); ++k )
		{
			QDir skinDirCnt = QDir ( skinLocations[i]+skinDirs[k], "*.rc", QDir::Name|QDir::IgnoreCase, QDir::Files );
			 // make a list of all .rc-files in a skindir
			QStringList rcFiles = skinDirCnt.entryList();

			// iterate trough all those rc.-files in a skindir
			for (uint j = 0; j < rcFiles.count(); j++ )
			{
				if ( rcFiles[j].left(rcFiles[j].length()-3) == prefsWidget->mSkins->currentText() ) // found skinname.rc :)
				{
					dirToDelete = QString ( skinLocations[i]+skinDirs[k] );
					kdDebug(66666) << "FOUND SKIN @ " << dirToDelete.latin1() << endl;
				}
			}
		}
	}

	if ( dirToDelete.length() != 0 )
	{
		kdDebug(66666) << "Deleting Skindir: " << dirToDelete.latin1() << endl;
		KIO::Job *job = KIO::del( dirToDelete, false, true );
		connect ( job, SIGNAL(result(KIO::Job*)), this, SLOT(slotResult(KIO::Job*)) );
 }

	int item = -1;
	// Fallback to kjofol-skin (the default one) if we've deleted the current skin
	if ( deletingCurrentSkin )
	{
		for ( uint i = 0; i < prefsWidget->mSkins->count(); i++ )
		{ // FIXME: no check wether "kjofol" is ever found, well, it HAS to be in the list
			if ( prefsWidget->mSkins->text(i) == "kjofol" )
				item = i;
		}
	}
	else
		item = prefsWidget->mSkins->currentItem();

	if ( item != -1 )
		prefsWidget->mSkins->setCurrentItem( item );

	// update configuration
	if ( deletingCurrentSkin )
		save();
}

void KJPrefs::slotResult(KIO::Job *job )
{
	if ( job->error() )
	{
		job->showErrorDialog(this);
	}
	else
	{
		// Reload Skinlist
		reopen();
	}
}


void KJPrefs::reopen() // reload config and set stuff in dialog
{
	KGlobal::config()->setGroup("KJofol-Skins");
/*
	if ( (KGlobal::config()->readNumEntry("TimeCountMode", KJTime::Up)) == KJTime::Up )
	{
		prefsWidget->timeCountUp->setChecked(true);
		prefsWidget->timeCountDown->setChecked(false);
	}
	else
	{
		prefsWidget->timeCountDown->setChecked(true);
		prefsWidget->timeCountUp->setChecked(false);
	}

	switch ( KGlobal::config()->readNumEntry("AnalyzerType", KJVisScope::FFT ) )
	{
		case KJVisScope::Null:
			prefsWidget->visNone->setChecked(true);
			prefsWidget->visOsci->setChecked(false);
			prefsWidget->visFft->setChecked(false);
			break;
		case KJVisScope::FFT:
			prefsWidget->visNone->setChecked(false);
			prefsWidget->visOsci->setChecked(false);
			prefsWidget->visFft->setChecked(true);
			break;
		case KJVisScope::Mono:
			prefsWidget->visNone->setChecked(false);
			prefsWidget->visOsci->setChecked(true);
			prefsWidget->visFft->setChecked(false);
			break;
	}
*/
	QStringList skins;
	QStringList skinLocations = KGlobal::dirs()->findDirs("data", "noatun/kjskins");
	// iterate through all paths where Noatun is searching for kjofol-skins
	for (uint i = 0; i < skinLocations.count(); ++i )
	{
		QStringList skinDirs = QDir(skinLocations[i]).entryList();
		// iterate trough all dirs (normally, users can fsck every dir-struct *g*) containing a skin
		for (uint k = 2; k < skinDirs.count(); ++k )
		{
			QDir skinDirCnt = QDir ( skinLocations[i]+skinDirs[k], "*.rc", QDir::Name|QDir::IgnoreCase, QDir::Files );
			// make a list of all .rc-files in a skindir
			QStringList rcFiles = skinDirCnt.entryList();
			// iterate trough all those rc.-files in a skindir
			for (uint j = 0; j < rcFiles.count(); j++ )
			{
//				kdDebug(66666) << "found: " <<  rcFiles[j].latin1() << endl;
				skins += ( rcFiles[j] );
			}
		}
	}

	QString loaded = KGlobal::config()->readEntry("SkinResource", "kjofol");

	prefsWidget->mSkins->clear();

	loaded = loaded.mid(loaded.findRev("/")+1);  // remove path
	loaded = loaded.left( loaded.length() - 3 ); // remove ".rc"
	int index;
	for (QStringList::Iterator i=skins.begin(); i!=skins.end(); ++i)
	{
		*i = (*i).left( (*i).length() - 3 );
		prefsWidget->mSkins->insertItem(*i);
		if ( (*i) == loaded )
			index = prefsWidget->mSkins->count()-1; // save index no. to set active item later on
	}
	prefsWidget->mSkins->setCurrentItem(index);

	showPreview( prefsWidget->mSkins->currentText() );
}

// takes name of rc-file without .rc at the end and returns full path to rc-file
static QString expand(QString s)
{
//	kdDebug(66666) << "expand( "<< s.latin1() << " )" << endl;

	QStringList skinLocations = KGlobal::dirs()->findDirs("data", "noatun/kjskins");

	// iterate through all paths where Noatun is searching for kjofol-skins
	for (uint i = 0; i < skinLocations.count(); ++i )
	{
		QStringList skinDirs = QDir(skinLocations[i]).entryList();

		// iterate trough all dirs containing a skin
		for (uint k = 0; k < skinDirs.count(); ++k )
		{
			QDir skinDirCnt = QDir ( skinLocations[i]+skinDirs[k], "*.rc", QDir::Name|QDir::IgnoreCase, QDir::Files );
			// make a list of all .rc-files in a skindir
			QStringList rcFiles = skinDirCnt.entryList();

			// iterate trough all those rc.-files in a skindir
			for (uint j = 0; j < rcFiles.count(); j++ )
			{
				if ( rcFiles[j].left(rcFiles[j].length()-3) == s ) // found $s.rc :)
				{
//					kdDebug(66666) << "expand() found: " << QString(skinLocations[i]+skinDirs[k]+"/"+rcFiles[j]).latin1() << endl;
					return (skinLocations[i]+skinDirs[k]+"/"+rcFiles[j]);
				}
			}
		}
	}
	return QString();
}

void KJPrefs::save()
{
	kdDebug(66666) << "KJPrefs::save();" << endl;
	QString skin=::expand ( prefsWidget->mSkins->currentText() );

/*
	if ( prefsWidget->timeCountUp->isChecked() )
		config->writeEntry("TimeCountMode", KJTime::Up);
	else
		config->writeEntry("TimeCountMode", KJTime::Down);
*/

	// first load skin and then save config to prevent
	// reloading a broken skin after a crash
	KJLoader *l=KJLoader::kjofol;
	if (l)
		l->loadSkin(skin);

	KConfig *config=KGlobal::config();
	config->setGroup("KJofol-Skins");
	config->writeEntry("SkinResource", skin);
	config->sync();

//	emit configChanged();
}

void KJPrefs::showPreview(const QString &_skin)
{
	QString skin= ::expand(_skin);

	Parser p;
	p.open(skin);
	QImage image = p.image(p["BackgroundImage"][1]);
	if (!image.isNull())
	{
		mPixmap.convertFromImage(image);
		mPixmap.setMask(KJWidget::getMask(image));
	}
	else
		mPixmap=QPixmap();

	prefsWidget->mPreview->setPixmap(mPixmap);
}


QString filenameNoCase(const QString &filename, int badNodes)
{
	QStringList names=QStringList::split('/', filename);
	QString full;
	int number=(int)names.count();
	for (QStringList::Iterator i=names.begin(); i!=names.end(); ++i)
	{
		full+="/";
		if (number<=badNodes)
		{
			QDir d(full);
			QStringList files=d.entryList();
			files=files.grep(QRegExp("^"+ (*i) + "$", false));
			if (!files.count())
				return "";
			*i=files.grep(*i, false)[0];
		}

		full+=*i;

		number--;
	}

	if (filename.right(1)=="/")
		full+="/";
	return full;
}

#include "kjprefs.moc"
