/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <pthread.h>
#include <glib.h>

#include <pan/base/debug.h>
#include <pan/base/pan-object.h>

static glong obj_count = 0;

/* #define PAN_OBJECT_BOOKKEEPING */
#ifdef PAN_OBJECT_BOOKKEEPING
static GPtrArray * live = NULL;
#endif

static pthread_mutex_t ref_lock = PTHREAD_MUTEX_INITIALIZER;

void
pan_object_destroy (PanObject* o)
{
	debug1 (DEBUG_PAN_OBJECT, "pan_object_destroy: %p", o);
	g_return_if_fail (o!=NULL);

	if (o->ref_count)
		g_warning ("Destroying a PanObject with a refcount of %d", o->ref_count);

	(*o->destructor)(o);
	g_free (o);
}


void
pan_object_ref (PanObject* o)
{
	g_return_if_fail (o!=NULL);

	pthread_mutex_lock (&ref_lock);
	++o->ref_count;
	debug2 (DEBUG_PAN_OBJECT, "pan_object_ref: %p up to %d", o, o->ref_count);
	pthread_mutex_unlock (&ref_lock);
}
void
pan_object_unref (PanObject* o)
{
	g_return_if_fail (o!=NULL);

	pthread_mutex_lock (&ref_lock);
	--o->ref_count;
	debug2 (DEBUG_PAN_OBJECT, "pan_object_unref: %p down to %d", o, o->ref_count);
	pthread_mutex_unlock (&ref_lock);

	g_return_if_fail (o->ref_count>=0);
	if (o->ref_count<1)
		pan_object_destroy (o);
}

/**
***
**/

void
pan_object_constructor (PanObject* o, PanObjectDestructor destructor)
{
	g_return_if_fail (o!=NULL);

	++obj_count;
#ifdef PAN_OBJECT_BOOKKEEPING
	if (live == NULL)
		live = g_ptr_array_new ();
	g_ptr_array_add (live, o);
#endif
	debug2 (DEBUG_PAN_OBJECT, "pan_object_constructor: %p (%ld PanObjects alive)", o, obj_count);

	o->ref_count = (gint16)1;
	o->destructor = destructor;
}
void
pan_object_destructor (PanObject *o)
{
#ifdef PAN_OBJECT_BOOKKEEPING
	if (live != NULL)
		g_ptr_array_remove (live, o);
#endif
	--obj_count;
	debug2 (DEBUG_PAN_OBJECT, "pan_object_destructor: %p (%ld PanObjects alive)", o, obj_count);

	g_return_if_fail (o!=NULL);
}

void
pan_object_bookkeeping (void)
{
#ifdef PAN_OBJECT_BOOKKEEPING
	if (live != NULL)
	{
		guint i;
		for (i=0; i!=live->len; ++i)
			g_message ("PanObject still alive: %p", g_ptr_array_index(live,i));
		g_ptr_array_free (live, TRUE);
	}
#endif
}

gboolean
pan_object_is_valid (const PanObject * o)
{
	g_return_val_if_fail (o!=NULL, FALSE);
	g_return_val_if_fail (o->ref_count>0, FALSE);

	return TRUE;
}
