/*
 * GQview
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "collect-io.h"

#include "collect.h"
#include "menu.h"
#include "rcfile.h"
#include "thumb.h"
#include "ui_fileops.h"


static void collection_load_thumb_step(CollectionData *cd);


static gint scan_geometry(gchar *buffer, gint *x, gint *y, gint *w, gint *h)
{
	gint nx, ny, nw, nh;

	if(sscanf(buffer, "%d %d %d %d", &nx, &ny, &nw, &nh) != 4) return FALSE;

	*x = nx;
	*y = ny;
	*w = nw;
	*h = nh;

	return TRUE;
}

gint collection_load(CollectionData *cd, const gchar *path, gint append)
{
	gchar s_buf[2048];
	FILE *f;

	collection_load_stop(cd);

	if (!append)
		{
		collection_list_free(cd->list);
		cd->list = NULL;
		}

	if (!path && !cd->path) return FALSE;

	if (!path) path = cd->path;

	/* load it */
	f = fopen(path, "r");
	if (!f)
		{
		printf("Failed to open collection file: \"%s\"\n", path);
		return FALSE;
		}

	while (fgets(s_buf, sizeof(s_buf), f))
		{
		gchar *buf;
		if (s_buf[0]=='#')
			{
			if (strncmp(s_buf, "#geometry:", 10 ) == 0 &&
			    scan_geometry(s_buf + 10, &cd->window_x, &cd->window_y, &cd->window_w, &cd->window_h) )
				{
				cd->window_read = TRUE;
				}
			continue;
			}
		if (s_buf[0]=='\n') continue;

		buf = quoted_value(s_buf);
		if (buf)
			{
			collection_add(cd, buf, FALSE);
			g_free(buf);
			}
		}

	fclose(f);

	cd->list = collection_list_sort(cd->list, cd->sort_method);
	if (!append) cd->changed = FALSE;

	recent_menu_add_path(cd->path);

	return TRUE;
}

static void collection_load_thumb_do(CollectionData *cd)
{
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	if (!cd->thumb_loader || !g_list_find(cd->list, cd->thumb_info)) return;

	if (thumb_loader_to_pixmap(cd->thumb_loader, &pixmap, &mask) < 0)
		{
		thumb_from_xpm_d(img_unknown, thumb_max_width, thumb_max_height, &pixmap, &mask);
		}

	collection_info_set_thumb(cd->thumb_info, pixmap, mask);

	if (pixmap) gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);

	if (cd->info_updated_func) cd->info_updated_func(cd, cd->thumb_info, cd->info_updated_data);
}

static void collection_load_thumb_error_cb(ThumbLoader *tl, gpointer data)
{
	CollectionData *cd = data;

	collection_load_thumb_do(cd);
	collection_load_thumb_step(cd);
}

static void collection_load_thumb_done_cb(ThumbLoader *tl, gpointer data)
{
	CollectionData *cd = data;

	collection_load_thumb_do(cd);
	collection_load_thumb_step(cd);
}

static void collection_load_thumb_step(CollectionData *cd)
{
	GList *work;
	CollectInfo *ci;

	if (!cd->list)
		{
		collection_load_stop(cd);
		return;
		}

	work = cd->list;
	ci = work->data;
	work = work->next;
	/* find first unloaded pixmap */
	while (work && ci->pixmap)
		{
		ci = work->data;
		work = work->next;
		}

	if (!ci || ci->pixmap)
		{
		/* done */
		collection_load_stop(cd);

		/* send a NULL CollectInfo to notify end */
		if (cd->info_updated_func) cd->info_updated_func(cd, NULL, cd->info_updated_data);

		return;
		}

	/* setup loader and call it */
	cd->thumb_info = ci;
	thumb_loader_free(cd->thumb_loader);
	cd->thumb_loader = thumb_loader_new(ci->path, thumb_max_width, thumb_max_height);

	/* setup error */
	thumb_loader_set_error_func(cd->thumb_loader, collection_load_thumb_error_cb, cd);

	/* start it */
	if (!thumb_loader_start(cd->thumb_loader, collection_load_thumb_done_cb, cd))
		{
		/* error, handle it, do next */
		if (debug) printf("error loading thumb for %s\n", ci->path);
		collection_load_thumb_do(cd);
		collection_load_thumb_step(cd);
		}
}

void collection_load_thumb_idle(CollectionData *cd)
{
	if (!cd->thumb_loader) collection_load_thumb_step(cd);
}

gint collection_load_begin(CollectionData *cd, const gchar *path, gint append)
{
	if (!collection_load(cd, path, append)) return FALSE;

	collection_load_thumb_idle(cd);

	return TRUE;
}

void collection_load_stop(CollectionData *cd)
{
	if (!cd->thumb_loader) return;

	thumb_loader_free(cd->thumb_loader);
	cd->thumb_loader = NULL;
}

gint collection_save(CollectionData *cd, const gchar *path)
{
	FILE *f;
	GList *work;

	if (!path && !cd->path) return FALSE;

	if (!path)
		{
		path = cd->path;
		}

	f = fopen (path, "w");
	if (!f)
		{
		/* file open failed */
		printf("failed to open collection (write) \"%s\"\n", path);
		return FALSE;
		}

	fprintf(f, "#GQview collection\n");
	fprintf(f, "#created with GQview version %s\n", VERSION);

	collection_update_geometry(cd);
	if (cd->window_read)
		{
		fprintf(f, "#geometry: %d %d %d %d\n", cd->window_x, cd->window_y, cd->window_w, cd->window_h);
		}

	work = cd->list;
	while(work)
		{
		CollectInfo *ci = work->data;
		if (!fprintf(f, "\"%s\"\n", ci->path))
			{
			fclose(f);
			printf("Error writing to %s\n", path);
			return FALSE;
			}
		work = work->next;
		}

	fprintf(f, "#end\n");

	fclose(f);

	if (!cd->path || strcmp(path, cd->path) != 0)
		{
		gchar *buf = cd->path;
		cd->path = g_strdup(path);
		path = cd->path;
		g_free(buf);

		g_free(cd->name);
		cd->name = g_strdup(filename_from_path(cd->path));

		collection_path_changed(cd);
		}

	cd->changed = FALSE;

	recent_menu_add_path(cd->path);

	return TRUE;
}

gint collection_load_only_geometry(CollectionData *cd, const gchar *path)
{
	gchar s_buf[2048];
	FILE *f;

	if (!path && !cd->path) return FALSE;

	if (!path) path = cd->path;

	/* load it */
	f = fopen(path, "r");
	if (!f) return FALSE;

	while (fgets(s_buf, sizeof(s_buf), f))
		{
		if (s_buf[0]=='#' &&
		    strncmp(s_buf, "#geometry:", 10 ) == 0 &&
		    scan_geometry(s_buf + 10, &cd->window_x, &cd->window_y, &cd->window_w, &cd->window_h) )
			{
			cd->window_read = TRUE;
			fclose(f);
			return TRUE;
			}
		}
	fclose(f);
	return FALSE;
}

