#include "codepage.h"

/* MS-DOS doesn't use the same ASCII code as Unix does. The appearance
 * of the characters is defined using code pages. These code pages
 * aren't the same for all countries. For instance, some code pages
 * don't contain upper case accented characters. This affects two
 * things, relating to filenames:

 * 1. upper case characters. In short names, only upper case
 * characters are allowed.  This also holds for accented characters.
 * For instance, in a code page which doesn't contain accented
 * uppercase characters, the accented lowercase characters get
 * transformed into their unaccented counterparts. This is very bad
 * design. Indeed, stuff like national language support should never
 * affect filesystem integrity. And it does: A filename which is legal
 * in one country could be illegal in another one. Bad News for
 * frequent travellers.

 * 2. long file names: Micro$oft has finally come to their senses and
 * uses a more standard mapping for the long file names.  They use
 * Unicode, which is basically a 32 bit version of ASCII. Its first
 * 256 characters are identical to Unix ASCII. Thus, the code page
 * also affects the correspondence between the codes used in long
 * names and those used in short names.

 * Such a bad design is rather unbelievable. That's why I quoted the
 * translation tables. BEGIN FAIR USE EXCERPT:
 */

unsigned char toucase[][128]={
	/* 0 */
	/* 437 German Umlauts upcased, French accents 
	 * upcased and lose accent */
	{ 0x80, 0x9a, 0x45, 0x41, 0x8e, 0x41, 0x8f, 0x80, 
	  0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x8e, 0x8f, 
	  0x90, 0x92, 0x92, 0x4f, 0x99, 0x4f, 0x55, 0x55, 
	  0x59, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
	  0x41, 0x49, 0x4f, 0x55, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
	  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 1 */
	{ 0x43, 0x55, 0x45, 0x41, 0x41, 0x41, 0x86, 0x43, 
	  0x45, 0x45, 0x45, 0x49, 0x49, 0x8d, 0x41, 0x8f, 
	  0x45, 0x45, 0x45, 0x4f, 0x45, 0x49, 0x55, 0x55, 
	  0x98, 0x4f, 0x55, 0x9b, 0x9c, 0x55, 0x55, 0x9f, 
	  0xa0, 0xa1, 0x4f, 0x55, 0xa4, 0xa5, 0xa6, 0xa7, 
	  0x49, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
	  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 2 */
	{ 0x80, 0x9a, 0x90, 0x41, 0x8e, 0x41, 0x8f, 0x80, 
	  0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x8e, 0x8f, 
	  0x90, 0x92, 0x92, 0x4f, 0x99, 0x4f, 0x55, 0x55, 
	  0x59, 0x99, 0x9a, 0x9d, 0x9c, 0x9d, 0x9e, 0x9f, 
	  0x41, 0x49, 0x4f, 0x55, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
	  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 3 */
	{ 0x80, 0x9a, 0x90, 0x41, 0x8e, 0x41, 0x8f, 0x80, 
	  0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x8e, 0x8f, 
	  0x90, 0x92, 0x92, 0x4f, 0x99, 0x4f, 0x55, 0x55, 
	  0x59, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
	  0x41, 0x49, 0x4f, 0x55, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
	  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 4 
	 * 437: all accented characters lose their accent */
	{ 0x80, 0x55, 0x45, 0x41, 0x41, 0x41, 0x8f, 0x80, 
	  0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x41, 0x8f, 
	  0x45, 0x92, 0x92, 0x4f, 0x4f, 0x4f, 0x55, 0x55, 
	  0x98, 0x4f, 0x55, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
	  0x41, 0x49, 0x4f, 0x55, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
	  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 5 */
	{ 0x80, 0x9a, 0x90, 0x8f, 0x8e, 0x91, 0x86, 0x80, 
	  0x89, 0x89, 0x92, 0x8b, 0x8c, 0x98, 0x8e, 0x8f, 
	  0x90, 0x91, 0x92, 0x8c, 0x99, 0xa9, 0x96, 0x9d, 
	  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
	  0x86, 0x8b, 0x9f, 0x96, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
	  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 6 All accented characters lose their accent
	   when upcased. C loses cedilla.  upcased. 
	   loses slash. , ,  intact */
	{ 0x43, 0x55, 0x45, 0x41, 0x41, 0x41, 0x41, 0x43, 
	  0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x41, 0x41, 
	  0x45, 0x92, 0x92, 0x4f, 0x4f, 0x4f, 0x55, 0x55, 
	  0x59, 0x4f, 0x55, 0x4f, 0x9c, 0x4f, 0x9e, 0x9f, 
	  0x41, 0x49, 0x4f, 0x55, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0x41, 0x41, 0x41, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0x41, 0x41, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd1, 0xd1, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 
	  0x49, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0x49, 0xdf, 
	  0x4f, 0xe1, 0x4f, 0x4f, 0x4f, 0x4f, 0xe6, 0xe8, 
	  0xe8, 0x55, 0x55, 0x55, 0x59, 0x59, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 7: As 6, but German Umlauts keep their Umlaut */
	{ 0x43, 0x9a, 0x45, 0x41, 0x8e, 0x41, 0x41, 0x43, 
	  0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x8e, 0x41, 
	  0x45, 0x92, 0x92, 0x4f, 0x99, 0x4f, 0x55, 0x55, 
	  0x59, 0x99, 0x9a, 0x4f, 0x9c, 0x4f, 0x9e, 0x9f, 
	  0x41, 0x49, 0x4f, 0x55, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0x41, 0x41, 0x41, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0x41, 0x41, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd1, 0xd1, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 
	  0x49, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0x49, 0xdf, 
	  0x4f, 0xe1, 0x4f, 0x4f, 0x4f, 0x4f, 0xe6, 0xe8, 
	  0xe8, 0x55, 0x55, 0x55, 0x59, 0x59, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 8: All characters except  keep their accent
	 */
	{ 0x80, 0x9a, 0x90, 0xb6, 0x8e, 0xb7, 0x8f, 0x80, 
	  0xd2, 0xd3, 0xd4, 0xd8, 0xd7, 0xde, 0x8e, 0x8f, 
	  0x90, 0x92, 0x92, 0xe2, 0x99, 0xe3, 0xea, 0xeb, 
	  0x59, 0x99, 0x9a, 0x9d, 0x9c, 0x9d, 0x9e, 0x9f, 
	  0xb5, 0xd6, 0xe0, 0xe9, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc7, 0xc7, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd1, 0xd1, 0xd2, 0xd3, 0xd4, 0x49, 0xd6, 0xd7, 
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe5, 0xe5, 0xe6, 0xe8, 
	  0xe8, 0xe9, 0xea, 0xeb, 0xed, 0xed, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 9: As 6, but  and  preserved */
	{ 0x80, 0x55, 0x45, 0x41, 0x41, 0x41, 0x8f, 0x80, 
	  0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x41, 0x8f, 
	  0x45, 0x92, 0x92, 0x4f, 0x4f, 0x4f, 0x55, 0x55, 
	  0x98, 0x4f, 0x55, 0x4f, 0x9c, 0x4f, 0x9e, 0x9f, 
	  0x41, 0x49, 0x4f, 0x55, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0x41, 0x41, 0x41, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0x41, 0x41, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd1, 0xd1, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 
	  0x49, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0x49, 0xdf, 
	  0x4f, 0xe1, 0x4f, 0x4f, 0x4f, 0x4f, 0xe6, 0xe8, 
	  0xe8, 0x55, 0x55, 0x55, 0x59, 0x59, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 10: every accented character keeps its accent */
	{ 0x80, 0x9a, 0x90, 0xb6, 0x8e, 0xb7, 0x8f, 0x80, 
	  0xd2, 0xd3, 0xd4, 0xd8, 0xd7, 0xde, 0x8e, 0x8f, 
	  0x90, 0x92, 0x92, 0xe2, 0x99, 0xe3, 0xea, 0xeb, 
	  0x98, 0x99, 0x9a, 0x9d, 0x9c, 0x9d, 0x9e, 0x9f, 
	  0xb5, 0xd6, 0xe0, 0xe9, 0xa5, 0xa5, 0xa6, 0xa7, 
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc7, 0xc7, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd1, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe5, 0xe5, 0xe6, 0xe8, 
	  0xe8, 0xe9, 0xea, 0xeb, 0xed, 0xed, 0xee, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

	/* 11 */
	{ 0x80, 0x9a, 0x90, 0xb6, 0x8e, 0xde, 0x8f, 0x80, 
	  0x9d, 0xd3, 0x8a, 0x8a, 0xd7, 0x8d, 0x8e, 0x8f, 
	  0x90, 0x91, 0x91, 0xe2, 0x99, 0x95, 0x95, 0x97, 
	  0x97, 0x99, 0x9a, 0x9b, 0x9b, 0x9d, 0x9e, 0xac, 
	  0xb5, 0xd6, 0xe0, 0xe9, 0xa4, 0xa4, 0xa6, 0xa6, 
	  0xa8, 0xa8, 0xaa, 0x8d, 0xac, 0xb8, 0xae, 0xaf, 
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbd, 0xbf, 
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc6, 
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	  0xd1, 0xd1, 0xd2, 0xd3, 0xd2, 0xd5, 0xd6, 0xd7, 
	  0xb7, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe3, 0xd5, 0xe6, 0xe6, 
	  0xe8, 0xe9, 0xe8, 0xeb, 0xed, 0xed, 0xdd, 0xef, 
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
	  0xf8, 0xf9, 0xfa, 0xeb, 0xfc, 0xfc, 0xfe, 0xff },


	/* 14 All accented characters lose their accent, C loses cedilla,
	 *  loses slash.   upcased. , ,  intact */
	{ 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,

	  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,

	  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,

	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,

	  0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0xc6, 0x43,
	  0x45, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x49,

	  0xd0, 0xd1, 0x4f, 0x4f, 0x4f, 0x4f, 0x4f, 0xd7,
	  0x4f, 0x55, 0x55, 0x55, 0x55, 0x59, 0xde, 0xdf,

	  0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0xc6, 0x43,
	  0x45, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x49,

	  0xd0, 0xd1, 0x4f, 0x4f, 0x4f, 0x4f, 0x4f, 0xf7,
	  0x4f, 0x55, 0x55, 0x55, 0x55, 0x59, 0xde, 0x59  },



	/* 15 as 14, but German Umlauts (, , ) keep their Umlaut when
	   upcased */
	{ 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,

	  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,

	  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,

	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,

	  0x41, 0x41, 0x41, 0x41, 0xc4, 0x41, 0xc6, 0x43,
	  0x45, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x49,

	  0xd0, 0xd1, 0x4f, 0x4f, 0x4f, 0x4f, 0xd6, 0xd7,
	  0x4f, 0x55, 0x55, 0x55, 0xdc, 0x59, 0xde, 0xdf,

	  0x41, 0x41, 0x41, 0x41, 0xc4, 0x41, 0xc6, 0x43,
	  0x45, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x49,

	  0xd0, 0xd1, 0x4f, 0x4f, 0x4f, 0x4f, 0xd6, 0xf7,
	  0x4f, 0x55, 0x55, 0x55, 0xdc, 0x59, 0xde, 0x59  },


	/* 16 every accented character except  keeps its accent */
	{ 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,

	  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,

	  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,

	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,

	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,

	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,

	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,

	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xf7,
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0x59 },



	/* 17: As 6, but ,  and  preserved */
	{ 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,

	  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,

	  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,

	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,

	  0x41, 0x41, 0x41, 0x41, 0x41, 0xc5, 0xc6, 0xc7,
	  0x45, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x49,

	  0xd0, 0xd1, 0x4f, 0x4f, 0x4f, 0x4f, 0x4f, 0xd7,
	  0x4f, 0x55, 0x55, 0x55, 0x55, 0x59, 0xde, 0xdf,

	  0x41, 0x41, 0x41, 0x41, 0x41, 0xc5, 0xc6, 0xc7,
	  0x45, 0x45, 0x45, 0x45, 0x49, 0x49, 0x49, 0x49,

	  0xd0, 0xd1, 0x4f, 0x4f, 0x4f, 0x4f, 0x4f, 0xf7,
	  0x4f, 0x55, 0x55, 0x55, 0x55, 0x59, 0xde, 0xff  },


	/* 18 every accented character keeps its accent */
	{ 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,

	  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,

	  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,

	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,

	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,

	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,

	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,

	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xf7,
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xff },


	/* 19 */
	{ 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	  0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	  0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	  0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	  0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	  0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	  0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	  0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	  0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },

};

country_t countries[]={
	{   1, 437, 437,   0 },
	{   1, 850, 437,   6 },
	{   1, 819, 437,  14 },
	{  44, 437, 437,   0 },
	{  44, 850, 437,   6 },
	{  44, 819, 437,  14 },
	{  33, 437, 437,   0 },
	{  33, 850, 437,   6 },
	{  33, 819, 437,  14 },
	{  49, 437, 437,   0 },
	{  49, 850, 437,   7 },
	{  49, 819, 437,  15 },
	{  34, 850, 437,   8 },
	{  34, 819, 437,  16 },
	{  34, 437, 437,   0 },
	{  39, 437, 437,   0 },
	{  39, 850, 437,   6 },
	{  39, 819, 437,  14 },
	{  46, 437, 437,   3 },
	{  46, 850, 437,   8 },
	{  46, 819, 437,  16 },
	{  45, 850, 865,   8 },
	{  45, 819, 865,  16 },
	{  45, 865, 865,   2 },
	{  41, 850, 437,   8 },
	{  41, 819, 437,  16 },
	{  41, 437, 437,   3 },
	{  47, 850, 865,   8 },
	{  47, 819, 865,  16 },
	{  47, 865, 865,   2 },
	{  31, 437, 437,   4 },
	{  31, 850, 437,   9 },
	{  31, 819, 437,  17 },
	{  32, 850, 437,   8 },
	{  32, 819, 437,  16 },
	{  32, 437, 437,   3 },
	{ 358, 850, 437,   8 },
	{ 358, 819, 437,  16 },
	{ 358, 437, 437,   3 },
	{   2, 863, 863,   1 },
	{   2, 850, 863,   6 },
	{   2, 819, 863,  14 },
	{ 351, 850, 860,   6 },
	{ 351, 819, 860,  14 },
	{ 351, 860, 860,   5 },
	{   3, 850, 437,   8 },
	{   3, 819, 437,  16 },
	{   3, 437, 437,   0 },
	{  61, 437, 437,   0 },
	{  61, 850, 437,   6 },
	{  61, 819, 437,  16 },
	{  81, 437, 437,   0 },
	{  81, 819, 437,  14 },
	{  82, 437, 437,   0 },
	{  82, 819, 437,  14 },
	{  86, 437, 437,   0 },
	{  86, 819, 437,  14 },
	{  88, 437, 437,   0 },
	{  88, 819, 437,  14 },
	{  55, 850, 850,  10 },
	{  55, 819, 850,  18 },
	{  55, 437, 850,   0 },
	{ 354, 850, 850,  10 },
	{ 354, 819, 850,  18 },
	{  90, 850, 850,  10 },
	{  90, 819, 850,  18 },
	{  38, 852, 852,  11 },
	{  38, 850, 852,  10 },
	{  38, 819, 852,  18 },
	{  42, 852, 852,  11 },
	{  42, 850, 852,  10 },
	{  42, 819, 852,  18 },
	{  48, 852, 852,  11 },
	{  48, 850, 852,  10 },
	{  48, 819, 852,  18 },
	{  36, 852, 852,  11 },
	{  36, 850, 852,  10 },
	{  36, 819, 852,  18 },
	{ 886, 950, 950,  19 }, 	/* for Taiwan support (Country code) */
	{   0,   0,   0,   0 }
};

/* END FAIR USE EXCERPT */
