/* libwpd
 * Copyright (C) 2003 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2003-2004 Marc Maurer (uwog@uwog.net)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * For further information visit http://libwpd.sourceforge.net
 */

/* "This product is not manufactured, approved, or supported by
 * Corel Corporation or Corel Corporation Limited."
 */

#include "WPDocument.h"
#include "WPXHeader.h"
#include "WPXParser.h"
#include "WP1Parser.h"
#include "WP3Parser.h"
#include "WP42Parser.h"
#include "WP1Heuristics.h"
#include "WP42Heuristics.h"
#include "WP5Parser.h"
#include "WP6Parser.h"
#include "WPXStream.h"
#include "libwpd_internal.h"

/**
\mainpage libwpd documentation
This document contains both the libwpd API specification and the normal libwpd
documentation.
\section api_docs libwpd API documentation
The external libwpd API is provided by the WPDocument class. This class, combined
with the WPXHLListenerImpl class, are the only two classes that will be of interest
for the application programmer using libwpd.
\section lib_docs libwpd documentation
If you are interrested in the structure of libwpd itself, this whole document
would be a good starting point for exploring the interals of libwpd. Mind that
this document is a work-in-progress, and will most likely not cover libwpd for
the full 100%.
*/

/**
Analyzes the content of an input stream to see if it can be parsed
\param input The input stream
\param partialContent A boolean which states if the content from the input stream
represents the full contents of a WordPerfect file, or just the first X bytes
\return A confidence value which represents the likelyhood that the content from
the input stream can be parsed
*/
WPDConfidence WPDocument::isFileFormatSupported(WPXInputStream *input, bool partialContent)
{
	WPDConfidence confidence = WPD_CONFIDENCE_NONE;

	WPXHeader *header = 0;

	WPD_DEBUG_MSG(("WPDocument::isFileFormatSupported()\n"));

	// by-pass the OLE stream (if it exists) and returns the (sub) stream with the
	// WordPerfect document.
	WPXInputStream *document = 0;
	bool isDocumentOLE = false;

	// BIG BIG NOTE: very unsafe on partial content!!!
	if (input->isOLEStream())
	{
		document = input->getDocumentOLEStream();
		if (document)
			isDocumentOLE = true;
		else
		{
			if (partialContent)
				return WPD_CONFIDENCE_LIKELY; // HACK: too high, but for now, we'll live with it
			else
				return WPD_CONFIDENCE_NONE;
		}
	}
	else
		document = input;

	try
	{
		// NOTE: even when passed partial content, we for now just assume that
		// we can extract the header from it. We could also read the major version
		// and the preceding -1 WPC stuff manually.
		header = WPXHeader::constructHeader(document);
		if (header)
		{
			switch (header->getFileType())
			{
				case 0x0a: // WordPerfect File
					switch (header->getMajorVersion())
					{
						case 0x00: // WP5
						case 0x02: // WP6+
							confidence = WPD_CONFIDENCE_EXCELLENT;
							break;
						default:
							// unhandled file format
							confidence = WPD_CONFIDENCE_NONE;
							break;
					}
					break;
				case 0x2c: // WP Mac File
					switch (header->getMajorVersion())
					{
						case 0x02: // WP Mac 2.x
						case 0x03: // WP Mac 3.0-3.5
						case 0x04: // WP Mac 3.5e
							confidence = WPD_CONFIDENCE_EXCELLENT;
							break;
						default:
							// unhandled file format
							confidence = WPD_CONFIDENCE_NONE;
							break;
					}
					break;
				default:
					// unhandled file type
					confidence = WPD_CONFIDENCE_NONE;
					break;
			}
			if (header->getDocumentEncryption())
				confidence = WPD_CONFIDENCE_NONE; // do not handle password protected documents						
			DELETEP(header);
		}
		else
			confidence = WP1Heuristics::isWP1FileFormat(input, partialContent);
			if (confidence != WPD_CONFIDENCE_EXCELLENT)
				confidence = LIBWPD_MAX(confidence, WP42Heuristics::isWP42FileFormat(input, partialContent));
			

		// dispose of the reference to the ole input stream, if we allocated one
		if (document && isDocumentOLE)
			DELETEP(document);

		return confidence;
	}
	catch (FileException)
	{
		WPD_DEBUG_MSG(("File Exception trapped\n"));

		// dispose of the reference to the ole input stream, if we allocated one
		if (document && isDocumentOLE)
			DELETEP(document);

		return WPD_CONFIDENCE_NONE;
	}
	catch (...)
	{
		WPD_DEBUG_MSG(("Unknown Exception trapped\n"));

		// dispose of the reference to the ole input stream, if we allocated one
		if (document && isDocumentOLE)
			DELETEP(document);

		return WPD_CONFIDENCE_NONE;
	}
}

/**
Parses the input stream content. It will make callbacks to the functions provided by a
WPXHLListenerImpl class implementation when needed. This is often commonly called the
'main parsing routine'.
\param input The input stream
\param listenerImpl A WPXListener implementation
*/
WPDResult WPDocument::parse(WPXInputStream *input, WPXHLListenerImpl *listenerImpl)
{
	WPXParser *parser = 0;

	// by-pass the OLE stream (if it exists) and returns the (sub) stream with the
	// WordPerfect document.

	WPXInputStream *document = 0;
	bool isDocumentOLE = false;

	WPD_DEBUG_MSG(("WPDocument::parse()\n"));
	if (input->isOLEStream())
	{
		document = input->getDocumentOLEStream();
		if (document)
			isDocumentOLE = true;
		else
			return WPD_OLE_ERROR;
	}
	else
		document = input;

	WPDResult error = WPD_OK;

	try
	{
		WPXHeader *header = WPXHeader::constructHeader(document);

		if (header && !header->getDocumentEncryption())
		{
			switch (header->getFileType())
			{
				case 0x0a: // WordPerfect File
					switch (header->getMajorVersion())
					{
						case 0x00: // WP5
							WPD_DEBUG_MSG(("WordPerfect: Using the WP5 parser.\n"));
							parser = new WP5Parser(document, header);
							parser->parse(listenerImpl);
							break;
						case 0x02: // WP6
							WPD_DEBUG_MSG(("WordPerfect: Using the WP6 parser.\n"));
							parser = new WP6Parser(document, header);
							parser->parse(listenerImpl);
							break;
						default:
							// unhandled file format
							WPD_DEBUG_MSG(("WordPerfect: Unsupported file format.\n"));
							break;
					}
					break;
				case 0x2c: // WP Mac File
					switch (header->getMajorVersion())
					{
						case 0x02: // WP Mac 2.x
						case 0x03: // WP Mac 3.0-3.5
						case 0x04: // WP Mac 3.5e
							WPD_DEBUG_MSG(("WordPerfect: Using the WP3 parser.\n"));
							parser = new WP3Parser(document, header);
							parser->parse(listenerImpl);
							break;
						default:
							// unhandled file format
							WPD_DEBUG_MSG(("WordPerfect: Unsupported file format.\n"));
							break;
					}
					break;
				default:
					// unhandled file format
					WPD_DEBUG_MSG(("WordPerfect: Unsupported file type.\n"));
					break;
			}

			DELETEP(parser); // deletes the header as well
		}
		else if (header && header->getDocumentEncryption())
		{
			DELETEP(header);
			throw UnsupportedEncryptionException();
		}
		else
		{
			// WP file formats prior to version 5.x do not contain a generic
			// header which can be used to determine which parser to instanciate.
			// Use heuristics to determine with some certainty if we are dealing with
			// a file in the WP4.2 format or WP Mac 1.x format.
			if (WP1Heuristics::isWP1FileFormat(document, false) != WPD_CONFIDENCE_NONE)
			{
				WPD_DEBUG_MSG(("WordPerfect: Mostly likely the file format is WP Mac 1.x.\n\n"));
				WPD_DEBUG_MSG(("WordPerfect: Using the WP Mac 1.x parser.\n\n"));
				parser = new WP1Parser(document);
				parser->parse(listenerImpl);
				DELETEP(parser);
			}
			else if (WP42Heuristics::isWP42FileFormat(document, false) != WPD_CONFIDENCE_NONE)
			{
				WPD_DEBUG_MSG(("WordPerfect: Mostly likely the file format is WP4.2.\n\n"));
				WPD_DEBUG_MSG(("WordPerfect: Using the WP4.2 parser.\n\n"));
				parser = new WP42Parser(document);
				parser->parse(listenerImpl);
				DELETEP(parser);
			}
			else

				error = WPD_FILE_ACCESS_ERROR;
		}
	}

	catch (FileException)
	{
		WPD_DEBUG_MSG(("File Exception trapped\n"));
		error = WPD_FILE_ACCESS_ERROR;
	}
	catch (ParseException)
	{
		WPD_DEBUG_MSG(("Parse Exception trapped\n"));
		error = WPD_PARSE_ERROR;
	}
	catch (UnsupportedEncryptionException)
	{
		WPD_DEBUG_MSG(("Encrypted document exception trapped\n"));
		error = WPD_UNSUPPORTED_ENCRYPTION_ERROR;
	}
	catch (...)
	{
		WPD_DEBUG_MSG(("Unknown Exception trapped\n"));
		error = WPD_UNKNOWN_ERROR; 
	}

	DELETEP(parser);
	if (document && isDocumentOLE)
		DELETEP(document);

	return error;
}
