/* 
 * xfce-colorbutton - color-selector button for gtk 2.2, GtkColorButton
 *                    wrapper for gtk 2.4
 *
 * Copyright (c) 2004 Brian Tarricone <bjt23@cornell.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Portions based on GtkColorButton, Copyright (C) 1998, 1999 Red Hat, Inc.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gtk/gtkversion.h>
#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>

#include "xfce-colorbutton.h"

#if !GTK_CHECK_VERSION(2, 4, 0)

#include <gobject/gmarshal.h>

#define CHECK_SIZE  4
#define CHECK_DARK  21845  /* 65535 / 3     */
#define CHECK_LIGHT 43690 

enum {
	CB_SIGNAL_COLOR_SET,
	CB_N_SIGNALS
};
static gboolean _cb_signal_inited = FALSE;
static guint _cb_signals[CB_N_SIGNALS] = { 0 };

static GdkPixbuf *
render(GtkWidget *color_button)
{
	gint dark_r, dark_g, dark_b;
	gint light_r, light_g, light_b;
	gint i, j, rowstride;
	gint width, height;
	gint c1[3], c2[3];
	guchar *pixels;
	guint8 insensitive_r = 0;
	guint8 insensitive_g = 0;
	guint8 insensitive_b = 0;
	GtkWidget *drawing_area;
	GdkPixbuf *pix;
	GdkColor *color;
	guint16 alpha;
	
	drawing_area = g_object_get_data(G_OBJECT(color_button), "xfce-cb-da");
	pix = g_object_get_data(G_OBJECT(color_button), "xfce-cb-pixbuf");
	color = g_object_get_data(G_OBJECT(color_button), "xfce-cb-color");
	alpha = (guint16)GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(color_button), "xfce-cb-alpha"));
	
	width = drawing_area->allocation.width;
	height = drawing_area->allocation.height;
	if(!pix || gdk_pixbuf_get_width(pix) != width
			|| gdk_pixbuf_get_height(pix) != height)
	{
		if(pix)
			g_object_unref(pix);
		pix = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, width, height);
		g_object_set_data(G_OBJECT(color_button), "xfce-cb-pixbuf", pix);
	}
	
	/* Compute dark and light check colors */
	
	insensitive_r = GTK_WIDGET(color_button)->style->bg[GTK_STATE_INSENSITIVE].red >> 8;
	insensitive_g = GTK_WIDGET(color_button)->style->bg[GTK_STATE_INSENSITIVE].green >> 8;
	insensitive_b = GTK_WIDGET(color_button)->style->bg[GTK_STATE_INSENSITIVE].blue >> 8;
	
	if(GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(color_button), "xfce-cb-use_alpha"))) {
		dark_r = ((CHECK_DARK << 16) + (color->red - CHECK_DARK) * alpha) >> 24;
		dark_g = ((CHECK_DARK << 16) + (color->green - CHECK_DARK) * alpha) >> 24;
		dark_b = ((CHECK_DARK << 16) + (color->blue - CHECK_DARK) * alpha) >> 24;
		
		light_r = ((CHECK_LIGHT << 16) + (color->red - CHECK_LIGHT) * alpha) >> 24;
		light_g = ((CHECK_LIGHT << 16) + (color->green - CHECK_LIGHT) * alpha) >> 24;
		light_b = ((CHECK_LIGHT << 16) + (color->blue - CHECK_LIGHT) * alpha) >> 24;
	} else {
		dark_r = light_r = color->red >> 8;
		dark_g = light_g = color->green >> 8;
		dark_b = light_b = color->blue >> 8;
	}

	/* Fill image buffer */
	pixels = gdk_pixbuf_get_pixels(pix);
	rowstride = gdk_pixbuf_get_rowstride(pix);
	for(j = 0; j < height; j++) {
		if((j / CHECK_SIZE) & 1) {
			c1[0] = dark_r;
			c1[1] = dark_g;
			c1[2] = dark_b;
			
			c2[0] = light_r;
			c2[1] = light_g;
			c2[2] = light_b;
		} else {
			c1[0] = light_r;
			c1[1] = light_g;
			c1[2] = light_b;
			
			c2[0] = dark_r;
			c2[1] = dark_g;
			c2[2] = dark_b;
		}
      
		for (i = 0; i < width; i++) {
			if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(color_button)) && (i+j)%2) {
				*(pixels + j * rowstride + i * 3) = insensitive_r;
				*(pixels + j * rowstride + i * 3 + 1) = insensitive_g;
				*(pixels + j * rowstride + i * 3 + 2) = insensitive_b;
			} else if((i / CHECK_SIZE) & 1) {
				*(pixels + j * rowstride + i * 3)     = c1[0];
				*(pixels + j * rowstride + i * 3 + 1) = c1[1];
				*(pixels + j * rowstride + i * 3 + 2) = c1[2];
			} else {
				*(pixels + j * rowstride + i * 3)     = c2[0];
				*(pixels + j * rowstride + i * 3 + 1) = c2[1];
				*(pixels + j * rowstride + i * 3 + 2) = c2[2];
			}
		}
	}
	
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-pixbuf", pix);
	
	gtk_widget_queue_draw(drawing_area);
	
	return pix;
}

static void
_cp_ok_clicked_cb(GtkWidget *w, gpointer user_data)
{
	GtkWidget *color_button = user_data, *cp, *sel;
	GdkColor *color;
	guint16 alpha;
	
	cp = g_object_get_data(G_OBJECT(color_button), "xfce-cb-cp");
	gtk_widget_hide(cp);
	sel = GTK_COLOR_SELECTION_DIALOG(cp)->colorsel;
	
	color = g_object_get_data(G_OBJECT(color_button), "xfce-cb-color");
	gtk_color_selection_get_current_color(GTK_COLOR_SELECTION(sel), color);
	
	alpha = gtk_color_selection_get_current_alpha(GTK_COLOR_SELECTION(sel));
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-alpha",
			GUINT_TO_POINTER((guint32)alpha));
	
	render(color_button);
	
	g_signal_emit(G_OBJECT(color_button), _cb_signals[CB_SIGNAL_COLOR_SET], 0);
}

static gboolean
_cp_destroy_cb(GtkWidget *w, gpointer user_data)
{
	GtkWidget *color_button = user_data;
	
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-cp", NULL);
	
	return FALSE;
}

static void
_cp_cancel_clicked_cb(GtkWidget *w, gpointer user_data)
{
	GtkWidget *cp = user_data;
	gtk_widget_hide(cp);
}

static void
_clicked_cb(GtkWidget *w, gpointer user_data)
{
	GtkWidget *cp, *sel, *parent;
	gchar *title = NULL;
	GdkColor *color;
	guint16 alpha;
	
	cp = g_object_get_data(G_OBJECT(w), "xfce-cb-cp");
	if(!cp) {
		title = g_object_get_data(G_OBJECT(w), "xfce-cb-title");
		cp = gtk_color_selection_dialog_new(title);
		g_signal_connect(G_OBJECT(cp), "destroy",
				G_CALLBACK(_cp_destroy_cb), w);
		
		parent = gtk_widget_get_toplevel(w);
		gtk_window_set_transient_for(GTK_WINDOW(cp), GTK_WINDOW(parent));
		
		g_object_set_data_full(G_OBJECT(w), "xfce-cb-cp", cp,
				(GDestroyNotify)gtk_widget_destroy);
		
		g_signal_connect(GTK_COLOR_SELECTION_DIALOG(cp)->ok_button, "clicked",
				G_CALLBACK(_cp_ok_clicked_cb), w);
		g_signal_connect(GTK_COLOR_SELECTION_DIALOG(cp)->cancel_button,
				"clicked", G_CALLBACK(_cp_cancel_clicked_cb), cp);
		gtk_widget_destroy(GTK_COLOR_SELECTION_DIALOG(cp)->help_button);
	}
	
	if(!title) {
		title = g_object_get_data(G_OBJECT(w), "xfce-cb-title");
		if(title)
			gtk_window_set_title(GTK_WINDOW(cp), title);
		else
			gtk_window_set_title(GTK_WINDOW(cp), _("Pick a Color"));  /* default title */
	}
	
	alpha = (guint16)GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(w), "xfce-cb-alpha"));
	color = g_object_get_data(G_OBJECT(w), "xfce-cb-color");
	sel = GTK_COLOR_SELECTION_DIALOG(cp)->colorsel;
	
	gtk_color_selection_set_has_opacity_control(GTK_COLOR_SELECTION(sel),
			GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(w), "xfce-cb-use_alpha")));
	
	gtk_color_selection_set_previous_color(GTK_COLOR_SELECTION(sel), color);
	gtk_color_selection_set_previous_alpha(GTK_COLOR_SELECTION(sel), alpha);
	
	gtk_color_selection_set_current_color(GTK_COLOR_SELECTION(sel), color);
	gtk_color_selection_set_current_alpha(GTK_COLOR_SELECTION(sel), alpha);
	
	gtk_widget_show_all(cp);
}

static gboolean
_da_expose_cb(GtkWidget *w, GdkEventExpose *evt, gpointer user_data)
{
	GtkWidget *color_button = user_data, *drawing_area;
	gint width, height;
	GdkPixbuf *pix;
	GdkGC *gc;
	
	drawing_area = g_object_get_data(G_OBJECT(color_button), "xfce-cb-da");
	
	width = drawing_area->allocation.width;
	height = drawing_area->allocation.height;
	
	pix = g_object_get_data(G_OBJECT(color_button), "xfce-cb-pixbuf");
	if(!pix || width != gdk_pixbuf_get_width(pix)
			|| height != gdk_pixbuf_get_height(pix))
	{
		pix = render(color_button);
	}
	
	gc = g_object_get_data(G_OBJECT(color_button), "xfce-cb-gc");
	
	gdk_draw_pixbuf(w->window, gc, pix, evt->area.x, evt->area.y, evt->area.x,
			evt->area.y, evt->area.width, evt->area.height, GDK_RGB_DITHER_MAX,
			evt->area.x, evt->area.y);
	
	return FALSE;
}

static void
_cb_realize_cb(GtkWidget *w, gpointer user_data)
{
	GdkGC *gc;
	
	gc = g_object_get_data(G_OBJECT(w), "xfce-cb-gc");
	if(!gc) {
		gc = gdk_gc_new(w->window);
		g_object_set_data_full(G_OBJECT(w), "xfce-cb-gc", gc,
				(GDestroyNotify)g_object_unref);
	}
	
	render(w);
}

static GtkWidget *
xfce_color_button_init()
{
	GtkWidget *color_button, *align, *frame, *drawing_area;
	PangoLayout *layout;
	PangoRectangle rect;
	GdkPixbuf *pix;
	GdkColor *color;
	
	if(!_cb_signal_inited) {
		GtkButtonClass *button_class = g_type_class_ref(GTK_TYPE_BUTTON);
		
		/* this is somewhat evil, but... eh, what the hell */
		_cb_signals[CB_SIGNAL_COLOR_SET] = g_signal_new("color-set",
				G_OBJECT_CLASS_TYPE(G_OBJECT_CLASS(button_class)),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET(GtkButtonClass, _gtk_reserved4),
				NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
		_cb_signal_inited = TRUE;
	}
	
	color_button = gtk_button_new();
	g_signal_connect(G_OBJECT(color_button), "realize",
			G_CALLBACK(_cb_realize_cb), NULL);
	g_signal_connect(G_OBJECT(color_button), "clicked",
			G_CALLBACK(_clicked_cb), NULL);
	
	gtk_widget_push_composite_child();
	
	align = gtk_alignment_new(0.5, 0.5, 0.5, 1.0);
	gtk_container_set_border_width(GTK_CONTAINER(align), 1);
	gtk_container_add(GTK_CONTAINER(color_button), align);
	gtk_widget_show(align);
	
	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_container_add(GTK_CONTAINER (align), frame);
	gtk_widget_show(frame);
	
	drawing_area = gtk_drawing_area_new();
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-da", drawing_area);
	
	layout = gtk_widget_create_pango_layout(GTK_WIDGET(color_button), "Black");
	pango_layout_get_pixel_extents(layout, NULL, &rect);
	gtk_widget_set_size_request(drawing_area, rect.width - 2, rect.height - 2);
	g_signal_connect(G_OBJECT(drawing_area), "expose-event",
			G_CALLBACK(_da_expose_cb), color_button);
	gtk_container_add(GTK_CONTAINER(frame), drawing_area);
	gtk_widget_show(drawing_area);
	
	pix = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, rect.width, rect.height);
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-pixbuf", pix);
	
	color = g_new0(GdkColor, 1);
	color->red = color->blue = color->green = 0;
	g_object_set_data_full(G_OBJECT(color_button), "xfce-cb-color", color,
			(GDestroyNotify)g_free);
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-alpha",
			GUINT_TO_POINTER(65535));
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-use_alpha",
			GUINT_TO_POINTER(FALSE));
		
	gtk_widget_pop_composite_child();
	
	return color_button;
}
#endif
	

GtkWidget *
xfce_color_button_new()
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_color_button_new();
#else
	return xfce_color_button_init();
#endif
}

void
xfce_color_button_set_color(XfceColorButton *color_button,
		const GdkColor *color)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_color_button_set_color(GTK_COLOR_BUTTON(color_button), color);
#else
	GdkPixbuf *pix;
	guint32 rgba;
	GdkColor *our_color;

	g_return_if_fail(XFCE_IS_COLOR_BUTTON(color_button) && color != NULL);
	
	rgba = (((color->red & 0xff00) << 8) | ((color->green & 0xff00))
			| ((color->blue & 0xff00) >> 8)) << 8;
	pix = g_object_get_data(G_OBJECT(color_button), "xfce-cb-pixbuf");
	gdk_pixbuf_fill(pix, rgba);

	our_color = g_new0(GdkColor, 1);
        memcpy (our_color, color, sizeof (GdkColor));
	g_object_set_data_full(G_OBJECT(color_button), "xfce-cb-color", our_color,
			(GDestroyNotify)g_free);
#endif
}

GtkWidget *
xfce_color_button_new_with_color(const GdkColor *color)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_color_button_new_with_color(color);
#else
	GtkWidget *color_button;
        
        color_button = xfce_color_button_init();
        xfce_color_button_set_color(XFCE_COLOR_BUTTON(color_button), color);

	return color_button;
#endif
}
void
xfce_color_button_get_color(XfceColorButton *color_button, GdkColor *color)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_color_button_get_color(GTK_COLOR_BUTTON(color_button), color);
#else
	GdkColor *our_color;
	
	g_return_if_fail(XFCE_IS_COLOR_BUTTON(color_button) && color != NULL);
	
	our_color = g_object_get_data(G_OBJECT(color_button), "xfce-cb-color");
	color->red = our_color->red;
	color->green = our_color->green;
	color->blue = our_color->blue;
#endif
}

void
xfce_color_button_set_alpha(XfceColorButton *color_button, guint16 alpha)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_color_button_set_alpha(GTK_COLOR_BUTTON(color_button), alpha);
#else
	g_return_if_fail(XFCE_IS_COLOR_BUTTON(color_button));
	
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-alpha",
			GUINT_TO_POINTER((guint)alpha));
#endif
}

guint16
xfce_color_button_get_alpha(XfceColorButton *color_button)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_color_button_get_alpha(GTK_COLOR_BUTTON(color_button));
#else
	g_return_val_if_fail(XFCE_IS_COLOR_BUTTON(color_button), (guint16) 65535);
	
	return (guint16)GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(color_button),
			"xfce-cb-alpha"));
#endif
}

void
xfce_color_button_set_use_alpha(XfceColorButton *color_button,
		gboolean use_alpha)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_color_button_set_use_alpha(GTK_COLOR_BUTTON(color_button), use_alpha);
#else
	g_return_if_fail(XFCE_IS_COLOR_BUTTON(color_button));
	
	g_object_set_data(G_OBJECT(color_button), "xfce-cb-use_alpha",
			GUINT_TO_POINTER(use_alpha));
#endif
}

gboolean
xfce_color_button_get_use_alpha(XfceColorButton *color_button)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_color_button_get_use_alpha(GTK_COLOR_BUTTON(color_button));
#else
	g_return_val_if_fail(XFCE_IS_COLOR_BUTTON(color_button), FALSE);
	
	return (gboolean)GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(color_button),
			"xfce-cb-use_alpha"));
#endif
}

void
xfce_color_button_set_title(XfceColorButton *color_button, const gchar *title)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_color_button_set_title(GTK_COLOR_BUTTON(color_button), title);
#else
	GtkWidget *cp;
	gchar *oldtitle;
	
	g_return_if_fail(XFCE_IS_COLOR_BUTTON(color_button));
	
	oldtitle = g_object_get_data(G_OBJECT(color_button), "xfce-cb-title");
	if(oldtitle)
		g_free(oldtitle);
	g_object_set_data_full(G_OBJECT(color_button), "xfce-cb-title",
			g_strdup(title), (GDestroyNotify)g_free);
	
	cp = g_object_get_data(G_OBJECT(color_button), "xfce-cb-cp");
	if(cp)
		gtk_window_set_title(GTK_WINDOW(cp), title);
#endif
}

G_CONST_RETURN gchar *
xfce_color_button_get_title(XfceColorButton *color_button)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_color_button_get_title(GTK_COLOR_BUTTON(color_button));
#else
	g_return_val_if_fail(XFCE_IS_COLOR_BUTTON(color_button), NULL);
	
	return g_object_get_data(G_OBJECT(color_button), "xfce-cb-title");
#endif
}
