/*
 * Copyright (C) 2002 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "entry.h"

#include "misc.h"


G_MODULE_EXPORT
void 
free_entry (		GtkTreeModel * treemodel,
			GtkTreeIter *iter,
			tree_entry_t *en)
{
    if (!en && !iter){
        g_warning("!en && !iter");
	return;
    } else if (!en) {
       	gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
    }
    if (en) {
	g_free(en->filter);
	g_free(en->st);
	g_free(en->path);
	g_free(en->tag);
    }
    g_free(en);
}


G_MODULE_EXPORT
gint entry_compare (int caso,tree_entry_t *en_a,tree_entry_t *en_b){
    gboolean d1d, d2d; /* d1, d2 is directory flag */
    gboolean asc;
    int ascending = get_ascending();
    switch (ascending %3){
	case 0:
	case 1:	  asc=TRUE; break; 
	default:  asc=FALSE;
    }
   	if (!en_a && !en_b) return 0;
	if (!en_a ) return (asc)?-1:1;
	if (!en_b ) return (asc)?1:-1;
	if (IS_DUMMY_TYPE(en_a->type)) return (asc)?-1:1;
	if (IS_DUMMY_TYPE(en_b->type)) return (asc)?1:-1;

	if (!en_a->path && !en_b->path) return 0;
	if (!en_a->path ) return (asc)?-1:1;
	if (!en_b->path ) return (asc)?1:-1;
	
	d1d = IS_DIR(en_a->type) | IS_NETDIR(en_a->subtype);
	d2d = IS_DIR(en_b->type) | IS_NETDIR(en_b->subtype);

	if (d1d && !d2d)  return (asc)?-1:1;
	if (!d1d && d2d)  return (asc)?1:-1;
	
	
	/* subsorting... */
	if (caso ==GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID  && IS_FILE(en_a->type) && IS_FILE(en_b->type)){
	    char *a,*b;
	    a=strrchr(en_a->path,'.');
	    b=strrchr(en_b->path,'.');
	    if (a || b) {
		if (!a) return -1;
		if (!b) return 1;
		if (strcmp(a,b)) return strcmp(a,b); 
	    }
	}
		
	

	switch (caso){   
	    case GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID:
	    case NAME_COLUMN: 
	    {
		gchar *a=en_a->path,*b=en_b->path;
		if (strchr(a,G_DIR_SEPARATOR)) a=strrchr(a,G_DIR_SEPARATOR)+1;
		if (strchr(b,G_DIR_SEPARATOR)) b=strrchr(b,G_DIR_SEPARATOR)+1;
		return strcmp(a,b);
	    }
	    default:
	  	if (!en_a->st && !en_b->st) return 0;
		if (!en_a->st ) return (asc)?-1:1;
		if (!en_b->st ) return (asc)?1:-1;
		switch (caso) {
		    case SIZE_COLUMN: return en_a->st->st_size - en_b->st->st_size;
		    case DATE_COLUMN: return en_a->st->st_mtime - en_b->st->st_mtime;
		    case OWNER_COLUMN: return en_a->st->st_uid - en_b->st->st_uid;
		    case GROUP_COLUMN: return en_a->st->st_gid - en_b->st->st_gid;
		    case MODE_COLUMN: return en_a->st->st_mode - en_b->st->st_mode;
		}
	}
	return 0;   
}


G_MODULE_EXPORT
int get_entry_from_reference (GtkTreeView * treeview, GtkTreeRowReference * reference, GtkTreeIter * iter, tree_entry_t ** en_p)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreePath *treepath = gtk_tree_row_reference_get_path(reference);
    if(!treepath || !treemodel)
	return FALSE;
    if(!tree_details->window)
	return FALSE;
    if(!gtk_tree_model_get_iter(treemodel, iter, treepath))
	return FALSE;
    gtk_tree_path_free(treepath);
    if(!tree_details->window)
	return FALSE;
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, en_p, -1);
    return TRUE;
}

static GtkTreePath *first_path;
static void first_selection(GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data){
	if (first_path) return;
	first_path=gtk_tree_path_copy(path);
}

G_MODULE_EXPORT
const gchar *get_selected_chdir (void)
{
    gint tree_id = get_active_tree_id();
    GtkTreeView *treeview = tree_details->treestuff[tree_id].treeview;
    GtkTreeModel *treemodel = tree_details->treestuff[tree_id].treemodel;
    GtkTreeSelection *selection = tree_details->treestuff[tree_id].selection;
    static gchar *Cdir=NULL;
    tree_entry_t *en;
    
    if (Cdir) {
	g_free(Cdir);
	Cdir=NULL;
    }

    first_path=NULL;
    gtk_tree_selection_selected_foreach(selection, first_selection, (gpointer) treeview);
#if 0
    if (!first_path){
	//FIXME: variable number of treeviews
      GtkTreeView * treeview2;
      treeview2=(GtkTreeView *)WIDGET("treeview2");
      selection = gtk_tree_view_get_selection(treeview2);
      gtk_tree_selection_selected_foreach(selection, 
		    first_selection, (gpointer) treeview2);
    }
#endif
    if (first_path)
    {
        GtkTreeIter iter;
	if (gtk_tree_model_get_iter (treemodel,&iter,first_path)){
	  gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1); 
	  if (en && en->path) { 
	   gchar *g;
	   if (IS_DIR(en->type)) g=g_strdup(en->path);
	   else g=g_path_get_dirname(en->path);
	   if (strcmp(g,".")==0) {
		   g_free(g);
		   g=NULL;
	   }
	   else Cdir=g;
	  } 	   
	}
	gtk_tree_path_free(first_path);
    }
    if (!Cdir) Cdir = g_strdup(GETWD);

#ifdef DEBUG
    printf("DBG: doing chdir to %s\n",Cdir);
#endif
    
    return Cdir;
}

/* this function always returns a local path (or NULL)*/
G_MODULE_EXPORT
tree_entry_t *get_selected_entry (GtkTreeIter * iter)
{
    gint tree_id = get_active_tree_id();
    GtkTreeModel *treemodel = tree_details->treestuff[tree_id].treemodel;
    GtkTreeSelection *selection = tree_details->treestuff[tree_id].selection;
    static tree_entry_t *en;

    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
    if(gtk_tree_selection_get_selected(selection, &treemodel, iter))
    {
	gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
	if(!en)
	    assert_not_reached();
	if(!IS_PATH(en->type))
	{
	    if(IS_DUMMY_TYPE(en->type))
	    {
		GtkTreeIter *child;
		child = gtk_tree_iter_copy(iter);
		gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
		if(gtk_tree_model_iter_parent(treemodel, iter, child))
		    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
		if(!IS_PATH(en->type))
		    en = NULL;
		gtk_tree_iter_free(child);
	    }
	}
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
	return en;

    }

    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
    if(!en)
	return NULL;

    if(!get_selectpath_iter(iter, &en))
	return NULL;
    return en;
}

G_MODULE_EXPORT
tree_entry_t *mk_entry (int type)
{
    tree_entry_t *en;
    en = (tree_entry_t *) malloc(sizeof(tree_entry_t));
    if(!en)
	assert_not_reached();
    en->type = 0;
    en->subtype = 0;
    en->filter = g_strdup("*");
    en->count = -1;
    en->load_time = 0;
    en->path = NULL;
    en->tag = NULL;
    en->st = NULL;
    INHERIT_TYPE(en->type, type);
    return en;
}

G_MODULE_EXPORT
tree_entry_t *mk_entry_path (char *path, int type)
{
    tree_entry_t *en;
    en = mk_entry(type);
    en->path = g_strdup(path);
    if(access(path, W_OK) < 0)
    {
	SET_NOWRITE(en->type);
    }
    if(access(path, R_OK) < 0)
    {
	SET_NOACCESS(en->type);
    }
    if(access(path, X_OK) >= 0)
    {
	/* this fails for priviledged users so retest */
	struct stat s;
	if (stat (path, &s) != -1 && 
	    ((s.st_mode & S_IXUSR) || 
	     (s.st_mode & S_IXGRP) || 
	     (s.st_mode & S_IXOTH)) ){
		SET_EXE(en->type);
	}
    }
    return en;
}

G_MODULE_EXPORT
tree_entry_t *stat_entry (char *path, int type)
{
    tree_entry_t *en;
    struct stat *st, s;
    st = (struct stat *)malloc(sizeof(struct stat));

    en = mk_entry_path(path, type);
    /*SET_LOCAL_TYPE(en->type);  default */

    if(!st)
	assert_not_reached();
    else if(stat(path, st) < 0)
    {
	if(lstat(path, st) < 0)
	{
	    destroy_entry(en);	    
	    g_free(st);
	    st=NULL;
	    return NULL;
	}
	SET_BROKEN_LNK(en->type);
	en->st = st;
	return en;
    }
    en->st = st;
    if (lstat(path, &s) < 0){
	    destroy_entry(en);	    
	    g_free(st);
	    st=NULL;
	    return NULL;
    }
    
    if(S_ISLNK(s.st_mode))
    {
	SET_XF_LNK(en->type);
    }
    else if(S_ISDIR(s.st_mode))
    {
	SET_XF_DIR(en->type);
    }
    else if(S_ISSOCK(s.st_mode))
    {
	SET_XF_SOCK(en->type);
    }
    else if(S_ISBLK(s.st_mode))
    {
	SET_XF_BLK(en->type);
    }
    else if(S_ISCHR(s.st_mode))
    {
	SET_XF_CHR(en->type);
    }
    else if(S_ISFIFO(s.st_mode))
    {
	SET_XF_FIFO(en->type);
    }
    else
    {
	SET_XF_REG(en->type);
    }

    if(S_ISDIR(st->st_mode))
    {
	SET_DIR(en->type);
	en->count = count_files(en->path);
    }

    return en;
}

G_MODULE_EXPORT
tree_entry_t *mk_net_entry (char *path,unsigned type){
   tree_entry_t *en;
    char *p,*t;
   
    /* posible stuff (printers not yet defined as netstuff):
	     * XF_NETWS    smb://XXX@YYY:
	     * XF_NETSHARE smb://XXX@YYY:ZZZ
	     * NETFILE     smb://XXX@YYY:ZZZ/UUU
	     * NETDIR      smb://XXX@YYY:ZZZ/VVV/
     * */

   p=g_strdup(path+strlen("smb://"));

   en=mk_entry(type);
   if (p[strlen(p)-1]==':') SET_XF_NETWS(en->subtype);
   else if (p[strlen(p)-1]=='/') SET_NETDIR(en->subtype);
   else if (strchr(p,'/')) SET_NETFILE(en->subtype);
   else SET_XF_NETSHARE(en->subtype);
			  
	  	  
   en->st=(struct stat *)malloc(sizeof(struct stat));
   en->st->st_size=0;
   en->st->st_mtime=0;
   en->st->st_ctime=0;
   en->st->st_gid=-1;
   en->st->st_uid=-1;
   en->st->st_mode=0;
   t=strtok(p,"@");
   if (!t) assert_not_reached();
   en->tag=g_strdup(t);

   if (IS_XF_NETWS(en->subtype)){
       t=strtok(NULL,":"); 
   } 
   else {
        t=t+strlen(t)+1;
        *(strchr(t,':'))='/';
   }
   en->path=(char *)malloc(strlen(t)+3);
   sprintf(en->path,"//%s",t);
   g_free(p);
   p=NULL;
   return en;
}


G_MODULE_EXPORT
void destroy_entry (tree_entry_t * en)
{
    if(!en)
	return;
    if(en->filter){
	g_free(en->filter);
	en->filter=NULL;
    }
    if(en->st){
	g_free(en->st);
	en->st=NULL;
    }
    if(en->path){
	g_free(en->path);
	en->path=NULL;
    }
    if(en->tag){
	g_free(en->tag);
	en->tag=NULL;
    }
    g_free(en);
    en=NULL;
}


