//
// Class: documentPageCache
//
// Cache that holds a number of pages in a document.
// Part of KDVI- A previewer for TeX DVI files.
//
// (C) 2004 Stefan Kebekus. Distributed under the GPL.


#ifndef _documentpagecache_h_
#define _documentpagecache_h_

#include <qobject.h>
#include <qpixmap.h>
#include <qrect.h>
#include <qstring.h>
#include <qvaluevector.h>

#include "documentPagePixmap.h"
#include "pageNumber.h"
#include "pageSize.h"

class documentPage;
class documentRenderer;


class documentPageCache: public QObject
{
 Q_OBJECT

 public:
  documentPageCache();
  ~documentPageCache();

  /** The maximal number of elements in the cache. */
  static const Q_UINT16 maxSize = 4;

  /** This method is used to name the documentRenderer, that the
      documentPageCache uses to render the page. The renderer can be
      used any time (e.g., triggered by an internal timer event), and
      must not be deleted before either the documentRenderer is
      deleted, or another renderer has been set. */
  void     setRenderer(documentRenderer *_renderer);

  void     setResolution(double res) {resolutionInDPI = res;};
  double   getResolution() const {return resolutionInDPI;};


  /** Returns the size of page 'page'. If the document does not
      specify a size (which happens, e.g., for some DVI-files), then
      the userPreferredSize is returned. */
  simplePageSize sizeOfPage(PageNumber page = 1) const;

  /** Returns the size of page 'page', in pixels, using the resolution
      set with setResolution(). If the document does not specify a
      size (which happens, e.g., for some DVI-files), the
      userPreferredSize is used. */
  QSize sizeOfPageInPixel(PageNumber page) const;

  /** Returns a pointer to a documentPage structure, or 0, if the
      documentPage could not be generated for some reason (e.g.,
      because no renderer has been set). */
  documentPagePixmap *getPage(PageNumber pageNr);

  /** Returns a "width" pixels width thumbnail of the given page. This
      method returns an empty QPixmap if one of the arguments is
      invalid, or if the page cannot be rendered for any reason. */
  QPixmap createThumbnail(PageNumber pageNr, int width);

 signals:
  /** This signal is emitted when setUserPreferredSize() or
      setUseUserPreferredSize() is called, and the page size
      changes accordingly. */
  void paperSizeChanged(void);

 public slots:
  /** Clears the contents of the cache. */
  void clear();

  /** Sets the userPreferredSize, which is explained below */
  void setUserPreferredSize(simplePageSize t);
  void setUseDocumentSpecifiedSize(bool);

 private:
  /** This field contains resolution of the display device. In
      principle. In fact, kviewshell implements zooming by calling the
      setResolution()-method with values that are not exactly the
      resolution of the display, but multiplied with the zoom
      factor. Bottom line: the documentRenderer should act as if this
      field indeed contains resolution of the display device. Whene a
      documentRenderer is constructed, this field is set to the actual
      resolution to give a reasonable default value. */
  double              resolutionInDPI;

  simplePageSize      userPreferredSize;
  bool                useDocumentSpecifiedSize;


  /** This list holds the cache. The least recently accessed page is
      always the last. */
  QPtrList<documentPagePixmap> LRUCache;

  /** If a page is removed from the LRUCache, it is not deleted, but
      clear()ed, and stored in this recycleBin. The page can then
      later be re-used. This makes for a substantial preformance
      gain. */
  QPtrList<documentPagePixmap> recycleBin;

  /** This is a dummy documentPage structure which is used internally
      by the 'createThumbnail' method. */
  documentPagePixmap thumbnailPage;

  documentRenderer* renderer;
};


#endif
