/*
 * Copyright (C) 2001-2004 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gtkvideo.c,v 1.26 2004/12/16 00:36:55 dsalt Exp $
 *
 * gtk xine video widget
 *
 * some code originating from totem's gtkxine widget
 */

#include "config.h"
#include "i18n.h"

#include "defs.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <pwd.h>
#include <sys/types.h>
#include <pthread.h>
#include <sched.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <X11/Xatom.h>
#ifdef HAVE_XTESTEXTENSION
#include <X11/extensions/XTest.h>
#endif

#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkmain.h>
#include <gdk/gdkx.h>
#include <xine.h>

#include "gtkvideo.h"

/* missing stuff from libxine includes (<= 1-rc7) */
#ifndef XINE_GUI_SEND_WILL_DESTROY_DRAWABLE
#define XINE_GUI_SEND_WILL_DESTROY_DRAWABLE 9
#endif

/* missing stuff from X includes */
#ifndef XShmGetEventBase
extern int XShmGetEventBase(Display *);
#endif

static void gtk_video_class_init    (GtkVideoClass   *klass);
static void gtk_video_instance_init (GtkVideo        *gvideo);

static void gtk_video_finalize      (GObject        *object);
static void gtk_video_realize       (GtkWidget      *widget);
static void gtk_video_unrealize     (GtkWidget      *widget);

static gint gtk_video_expose        (GtkWidget      *widget,
				     GdkEventExpose *event);

static void gtk_video_size_request (GtkWidget      *widget,
		       		    GtkRequisition *requisition);
static void gtk_video_size_allocate (GtkWidget      *widget,
 				     GtkAllocation  *allocation);

static GtkWidgetClass *parent_class = NULL;
static pthread_mutex_t resize_lock = PTHREAD_MUTEX_INITIALIZER;

/*
 * private data 
 */

struct gtk_video_private_s {

  xine_t                  *xine;
  xine_stream_t           *stream;
  xine_post_out_t         *out;
  char                    *video_driver_id;
  xine_video_port_t       *video_port;

  Display                 *display;
  Window                   video_window;
  int                      screen;
  int                      completion_event;
  pthread_t                thread;

  double                   display_ratio;
 
  int                      xpos, ypos;
  float                    resize_factor;
  gboolean                 resize, auto_resize, allow_resize, force_resize;
  int                      video_width, video_height;
  int                      oldwidth, oldheight;

  int			   button_press_mask; /* pass to front end? */
  int			   button_release_mask; /* pass to front end? */
  int			   button_event_shifted; /* was shifted? */

  /* fullscreen stuff */

  int                      fullscreen_mode;
  int                      fullscreen_width, fullscreen_height;
  Window                   fullscreen_window, toplevel;
  Cursor                   no_cursor;
  Cursor                   on_cursor;
  gboolean                 cursor_visible;

  Bool                   have_xtest;
#ifdef HAVE_XTESTEXTENSION
  KeyCode                kc_shift_l;      /* Fake key to send */
#endif

  /* visualization */
  gboolean		   vis_active;
  xine_post_t		  *vis_plugin;
  char			  *vis_plugin_id;
  xine_audio_port_t       *vis_audio;
};

static int gtv_table_signals[LAST_SIGNAL] = { 0 };

static int shift = 0; /* Shift key state */

GtkType gtk_video_get_type (void) {
  static GtkType gtk_video_type = 0;

  if (!gtk_video_type) {
    static const GTypeInfo gtk_video_info = {
      sizeof (GtkVideoClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gtk_video_class_init,
      (GClassFinalizeFunc) NULL,
      NULL /* class_data */,
      sizeof (GtkVideo),
      0 /* n_preallocs */,
      (GInstanceInitFunc) gtk_video_instance_init,
    };
    
    gtk_video_type = g_type_register_static (GTK_TYPE_WIDGET,
					     "GtkVideo", &gtk_video_info, (GTypeFlags)0);
  }

  return gtk_video_type;
}

static void gtk_video_class_init (GtkVideoClass *class) {

  GObjectClass    *object_class;
  GtkWidgetClass  *widget_class;

  object_class = (GObjectClass *) class;
  widget_class = (GtkWidgetClass *) class;

  parent_class = gtk_type_class (gtk_widget_get_type ());

  /* GtkWidget */
  widget_class->realize       = gtk_video_realize;
  widget_class->unrealize     = gtk_video_unrealize;
  widget_class->size_allocate = gtk_video_size_allocate;
  widget_class->size_request  = gtk_video_size_request;
  widget_class->expose_event  = gtk_video_expose;
  
  /* GObject */
  object_class->set_property  = NULL;
  object_class->get_property  = NULL;
  object_class->finalize      = gtk_video_finalize;

  gtv_table_signals[GTK_VIDEO_KEYPRESS] =
    g_signal_new ("keypress",
	G_TYPE_FROM_CLASS (object_class),
	G_SIGNAL_RUN_LAST,
	G_STRUCT_OFFSET (GtkVideoClass, keypress),
	NULL, NULL,
	g_cclosure_marshal_VOID__POINTER,
	G_TYPE_NONE, 1, G_TYPE_POINTER);

  gtv_table_signals[GTK_VIDEO_KEYRELEASE] =
    g_signal_new ("keyrelease",
	G_TYPE_FROM_CLASS (object_class),
	G_SIGNAL_RUN_LAST,
	G_STRUCT_OFFSET (GtkVideoClass, keyrelease),
	NULL, NULL,
	g_cclosure_marshal_VOID__POINTER,
	G_TYPE_NONE, 1, G_TYPE_POINTER);
}

static void gtk_video_instance_init (GtkVideo *this) {
  this->widget.requisition.width  = 480;
  this->widget.requisition.height = 205;
}

static void gtk_video_finalize (GObject *object) {

  GtkVideo *gtv = (GtkVideo *) object;

  G_OBJECT_CLASS (parent_class)->finalize (object);

  gtv = NULL;
}


static void dest_size_cb (void *gv_gen,
			  int video_width, int video_height,
			  double video_pixel_aspect,
			  int *dest_width, int *dest_height,
			  double *dest_pixel_aspect)  {

  GtkVideo            *gv = (GtkVideo *) gv_gen;
  gtk_video_private_t *priv = gv->priv;

  /* correct size with video_pixel_aspect */
  if (video_pixel_aspect >= priv->display_ratio)
    video_width  = video_width * video_pixel_aspect / priv->display_ratio + .5;
  else
    video_height = video_height * priv->display_ratio / video_pixel_aspect + .5;

  if (priv->fullscreen_mode) {
    *dest_width  = priv->fullscreen_width;
    *dest_height = priv->fullscreen_height;
  } else {
    *dest_width  = gv->widget.allocation.width;
    *dest_height = gv->widget.allocation.height;
  }

  *dest_pixel_aspect = priv->display_ratio;
}

static gboolean gtk_video_idle_resize (GtkVideo *gtv) {

  GtkWindow           *toplevel;
  GtkRequisition       toplevel_size;
  gtk_video_private_t *priv = gtv->priv;
  int                  video_width  = priv->video_width * priv->resize_factor;
  int                  video_height = priv->video_height * priv->resize_factor;

  logprintf ("gtkvideo: idle resize to %d x %d\n", video_width, video_height);

  priv->resize = priv->allow_resize & priv->auto_resize;

  gdk_threads_enter ();
  XLockDisplay (priv->display);

  toplevel = GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (gtv)));

  gtk_window_set_resizable (toplevel, FALSE);
  gtv->widget.allocation.width = video_width;
  gtv->widget.allocation.height = video_height;
  gtk_widget_set_size_request (&gtv->widget,
			       video_width, video_height);
  gtk_widget_size_request (GTK_WIDGET(toplevel), &toplevel_size);
  if (gtv->widget.allocation.width < toplevel_size.width)
    gtk_widget_set_size_request
      (&gtv->widget,
       gtv->widget.allocation.width = toplevel_size.width,
       gtv->widget.allocation.height);

  gtk_widget_queue_resize (gtk_widget_get_parent (GTK_WIDGET (gtv)));
  while (gtk_events_pending ())
    gtk_main_iteration ();

  gtk_window_set_resizable (toplevel, TRUE);

  XUnlockDisplay (priv->display);
  gdk_threads_leave ();

  priv->resize = priv->auto_resize;
  priv->force_resize = FALSE;
  priv->allow_resize = TRUE;

  logprintf ("gtkvideo: idle signal done\n");

  return FALSE;
}

static void frame_output_cb (void *gv_gen,
			     int video_width, int video_height,
			     double video_pixel_aspect,
			     int *dest_x, int *dest_y,
			     int *dest_width, int *dest_height,
			     double *dest_pixel_aspect,
			     int *win_x, int *win_y) {

  GtkVideo            *gtv = (GtkVideo *) gv_gen;
  gtk_video_private_t *priv;

  if (gtv == NULL)
    return;

  priv = gtv->priv;

  /* correct size with video_pixel_aspect */
  if (video_pixel_aspect >= priv->display_ratio)
    video_width  = video_width * video_pixel_aspect / priv->display_ratio + .5;
  else
    video_height = video_height * priv->display_ratio / video_pixel_aspect + .5;

  priv->video_width = video_width;
  priv->video_height = video_height;

  *dest_x = 0;
  *dest_y = 0;

  /* process windowed mode resize even if we're in full-screen mode */
  if (priv->resize || (priv->allow_resize && priv->force_resize))
  {
    video_width  *= priv->resize_factor;
    video_height *= priv->resize_factor;

    /* size changed? */
    if (video_width != priv->oldwidth || video_height != priv->oldheight ||
	priv->force_resize)
    {
      /* need to disable until resizing is done */
      priv->resize = priv->allow_resize = FALSE;

      pthread_mutex_lock (&resize_lock);
      priv->oldwidth = video_width;
      priv->oldheight = video_height;
      pthread_mutex_unlock (&resize_lock);

      g_idle_add ((GSourceFunc) gtk_video_idle_resize, gtv);
    }
  }

  if (priv->fullscreen_mode) {
    *win_x = 0;
    *win_y = 0;
    *dest_width  = priv->fullscreen_width;
    *dest_height = priv->fullscreen_height;
  } else {
    *win_x = priv->xpos;
    *win_y = priv->ypos;
    *dest_width  = gtv->widget.allocation.width;
    *dest_height = gtv->widget.allocation.height;
  }

  *dest_pixel_aspect = priv->display_ratio;
}

static xine_video_port_t *load_video_out_driver(GtkVideo *this) {

  double                   res_h, res_v;
  x11_visual_t             vis;
  const char              *video_driver_id;
  xine_video_port_t       *video_port;
  gtk_video_private_t     *priv = this->priv;

  vis.display           = priv->display;
  vis.screen            = priv->screen;
  vis.d                 = priv->video_window;
  res_h                 = (DisplayWidth  (priv->display, priv->screen)*1000
			   / DisplayWidthMM (priv->display, priv->screen));
  res_v                 = (DisplayHeight (priv->display, priv->screen)*1000
			   / DisplayHeightMM (priv->display, priv->screen));
  priv->display_ratio   = res_v / res_h;

  if (fabs(priv->display_ratio - 1.0) < 0.01) {
    priv->display_ratio   = 1.0;
  }

  vis.dest_size_cb      = dest_size_cb;
  vis.frame_output_cb   = frame_output_cb;
  vis.user_data         = this;

  if (priv->video_driver_id)
    video_driver_id = priv->video_driver_id;
  else {

    char **driver_ids = (char **) xine_list_video_output_plugins (priv->xine) ;
    char **choices;
    int i;

    choices = malloc (sizeof (char *) * 100);
    choices[0] = "auto"; i=0;
    while (driver_ids[i]) {
      choices[i+1] = strdup(driver_ids[i]);
      i++;
    }
    choices[i+1]=0;
    

    /* try to init video with stored information */
    i = xine_config_register_enum (priv->xine,
				   "video.driver", 0,
				   choices,
				   N_("video driver to use"),
				   NULL, 10, NULL, NULL);
    video_driver_id = choices[i];
  }
  if (strcmp (video_driver_id, "auto")) {

    video_port=xine_open_video_driver (priv->xine,
				      video_driver_id,
				      XINE_VISUAL_TYPE_X11,
				      (void *) &vis);
    if (video_port)
      return video_port;
    else
      printf (_("gtkxine: video driver %s failed.\n"),
	      video_driver_id); /* => auto-detect */
  }

  return xine_open_video_driver (priv->xine, NULL,
				 XINE_VISUAL_TYPE_X11,
				 (void *) &vis);
}

struct signal_s {
  GObject *this;
  guint signal;
  GQuark quark;
  GdkEvent event;
};

static gboolean gtv_idle_send_key_event (struct signal_s *event)
{
  gdk_threads_enter ();
  g_signal_emit (event->this, event->signal, event->quark, &event->event);
  gdk_threads_leave ();
  free (event); /* allocated in gtv_send_key_event */
  return FALSE;
}

static void gtv_send_key_event (GtkVideo *this, gboolean press, guint keysym,
				guint32 time, guint state, gchar *buffer)
{
  struct signal_s *event = malloc (sizeof (struct signal_s));

  event->this = G_OBJECT(this);
  event->signal = gtv_table_signals[press ? GTK_VIDEO_KEYPRESS
					  : GTK_VIDEO_KEYRELEASE];
  event->quark = 0;

  event->event.key.keyval	= keysym;
  event->event.key.type		= press ? GDK_KEY_PRESS : GDK_KEY_RELEASE;
  event->event.key.window	= this->widget.window;
  event->event.key.send_event	= 1;
  event->event.key.time		= time;
  event->event.key.state	= (GdkModifierType) state;
  event->event.key.string	= buffer;
  event->event.key.length	= strlen (buffer);
  event->event.key.hardware_keycode = 0;
  event->event.key.group	= 0;

  /* We're in the wrong thread for this, so use an idle event.
   * The called-back function will free(event).
   */
  gdk_threads_enter();
  g_idle_add ((GSourceFunc) gtv_idle_send_key_event, event);
  gdk_threads_leave();
}

static gboolean have_focus = FALSE;
static GdkWindow *focus_window = NULL;

static void gtv_event_handler (GdkEvent *event, gpointer data)
{
  if (event->type == GDK_FOCUS_CHANGE)
  {
    if (event->focus_change.in)
    {
      focus_window = event->focus_change.window;
      have_focus = TRUE;
    }
    else if (focus_window == event->focus_change.window)
      have_focus = FALSE;
  }
  gtk_main_do_event (event);
}

static void* xine_thread (void *this_gen) {

  GtkVideo            *this   = (GtkVideo *) this_gen;
  gtk_video_private_t *priv = this->priv;
  /*   GtkWidget  *widget = &this->widget; */

  while (1) {

    XEvent event;

    XNextEvent (priv->display, &event);

    /* printf ("gtkxine: got an event (%d)\n", event.type);  */

    switch (event.type) {
    case Expose:
      if (event.xexpose.count != 0)
	break;
      xine_port_send_gui_data (priv->video_port,
			       XINE_GUI_SEND_EXPOSE_EVENT, &event);
      break;

    case FocusOut: /* when in fullscreen mode, keep the focus */
      sched_yield ();
      if (priv->fullscreen_mode && !have_focus) {
	XLockDisplay (priv->display);
	XSetInputFocus (priv->display,
			priv->fullscreen_window, RevertToNone, CurrentTime);
	XUnlockDisplay (priv->display);
      }
      break;

    case MotionNotify: 
      {
	XMotionEvent     *mevent = (XMotionEvent *) &event;
	x11_rectangle_t   rect;
	xine_event_t      event;
	xine_input_data_t input;

	logprintf("gtkvideo: mouse event: mx=%d my=%d\n",mevent->x, mevent->y); 

	rect.x = mevent->x;
	rect.y = mevent->y;
	rect.w = 0;
	rect.h = 0;
	
	xine_port_send_gui_data (priv->video_port,
			         XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO, 
			         (void*)&rect);
	
	event.type        = XINE_EVENT_INPUT_MOUSE_MOVE;
	event.data        = &input;
	event.data_length = sizeof(input);
	input.button      = 0; /*  no buttons, just motion */
	input.x           = rect.x;
	input.y           = rect.y;
	xine_event_send (priv->stream, &event);

	if (priv->fullscreen_mode && !priv->cursor_visible) {

	  /*
	   * make mouse pointer visible
	   */
	  
	  XLockDisplay (priv->display);
	  XDefineCursor (priv->display, priv->fullscreen_window,
			 priv->on_cursor);
	  XFlush (priv->display);
	  XUnlockDisplay (priv->display);

	  priv->cursor_visible = TRUE;
	}
      }
      break;

    case ButtonPress:
      {
	XButtonEvent *bevent = &event.xbutton;

	logprintf("gtkvideo: mouse button event: mx=%d my=%d\n",
		  bevent->x, bevent->y); 

	/* send a GTK keypress event if allowed and neither Shift is pressed;
	 * otherwise, tell libxine (unles we're sending GTK keyrelease events
	 * for this button)
	 */
	if (!shift && (priv->button_press_mask & (1 << bevent->button)))
	  gtv_send_key_event (this, TRUE,
			      GDK_Pointer_Button1+event.xbutton.button-Button1,
			      event.xbutton.time, event.xbutton.state, "");
	else if (!(priv->button_release_mask & (1 << bevent->button)))
	{
	  x11_rectangle_t   rect;
	  xine_event_t      event;
	  xine_input_data_t input;

          /* record if either Shift is pressed for this button
           * (so that we know *not* to send the release event to the front end)
           */
	  if (shift)
	    priv->button_event_shifted |= 1 << bevent->button;

	  rect.x = bevent->x;
	  rect.y = bevent->y;
	  rect.w = 0;
	  rect.h = 0;

	  xine_port_send_gui_data (priv->video_port,
				   XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO, 
				   (void*)&rect);

	  event.type        = XINE_EVENT_INPUT_MOUSE_BUTTON;
	  event.data        = &input;
	  event.data_length = sizeof(input);
	  input.button      = bevent->button;
	  input.x           = rect.x;
	  input.y           = rect.y;
	  xine_event_send (priv->stream, &event);
	}
      }
      break;

    case ButtonRelease:
      /* send a GTK key release event if allowed *and* Shift wasn't pressed when
       * the corresponding press event was received
       */
      logprintf("gtkvideo: mouse button release: mx=%d my=%d\n",
		bevent->x, bevent->y);

      if (priv->button_event_shifted & (1 << event.xbutton.button))
	priv->button_event_shifted &= ~(1 << event.xbutton.button);
      else if (priv->button_release_mask & (1 << event.xbutton.button))
	gtv_send_key_event (this, FALSE,
			    GDK_Pointer_Button1+event.xbutton.button-Button1,
			    event.xbutton.time, event.xbutton.state, "");
      break;

    case KeyPress:
      {
	XKeyEvent     *kevent = (XKeyEvent *) &event;
	static char    buffer [20];
	KeySym         keysym;
	XComposeStatus compose;

	XLookupString (kevent, buffer, sizeof (buffer), &keysym, &compose);
	logprintf ("gtkvideo: key press %d\n", keysym);
	switch (keysym)
	{
	case XK_Shift_L: shift |= 1; break;
	case XK_Shift_R: shift |= 2; break;
	}
	gtv_send_key_event (this, TRUE, keysym, kevent->time, kevent->state,
			    buffer);
      }
      break;

    case KeyRelease:
      {
	XKeyEvent     *kevent = (XKeyEvent *) &event;
	static char    buffer [20];
	KeySym         keysym;
	XComposeStatus compose;

	XLookupString (kevent, buffer, sizeof (buffer), &keysym, &compose);
	logprintf ("gtkvideo: key release %d\n", keysym);
	switch (keysym)
	{
	case XK_Shift_L: shift &= ~1; break;
	case XK_Shift_R: shift &= ~2; break;
	}
	gtv_send_key_event (this, FALSE, keysym, kevent->time, kevent->state,
			    buffer);
      }
      break;
    }
  }

  pthread_exit(NULL);
  return NULL;
}

static gboolean configure_cb  (GtkWidget *widget,
			       GdkEventConfigure *event,
			       gpointer user_data) {

  GtkVideo            *this;
  gtk_video_private_t *priv;

  this = GTK_VIDEO (user_data);
  priv = this->priv;

  priv->xpos = event->x + this->widget.allocation.x;
  priv->ypos = event->y + this->widget.allocation.y;

  priv->resize = priv->allow_resize & priv->auto_resize;
  return FALSE;
}

static gint gtk_timeout_cb (gpointer data) {

  GtkVideo            *gtv  = (GtkVideo *) data;
  gtk_video_private_t *priv = gtv->priv;

  if (priv->fullscreen_mode) {
    XLockDisplay (priv->display);
    if (priv->cursor_visible) {
      XDefineCursor (priv->display, priv->fullscreen_window,
	  priv->no_cursor);
      XFlush (priv->display);
      priv->cursor_visible = FALSE;
    } 

#ifdef HAVE_XTESTEXTENSION
    if (priv->have_xtest == True) {
      XTestFakeKeyEvent(priv->display, priv->kc_shift_l, True, CurrentTime);
      XTestFakeKeyEvent(priv->display, priv->kc_shift_l, False, CurrentTime); 
      XSync(priv->display, False);
    }
    else
#endif
    {
      XResetScreenSaver(priv->display);
    }
    XUnlockDisplay (priv->display);
  }

  return TRUE;
}

#define MWM_HINTS_DECORATIONS   (1L << 1)
#define PROP_MWM_HINTS_ELEMENTS 5
typedef struct {
  uint32_t                    flags;
  uint32_t                    functions;
  uint32_t                    decorations;
  int32_t                     input_mode;
  uint32_t                    status;
} MWMHints;

static void create_fullscreen_window (gtk_video_private_t *priv)
{
  static char *window_title = N_("gxine fullscreen window");
  XSizeHints   hint;
  Atom         prop;
  MWMHints     mwmhints;

  int screen = DefaultScreen (priv->display);

  priv->fullscreen_window =
    XCreateSimpleWindow (priv->display, RootWindow (priv->display, screen),
			 0, 0, priv->fullscreen_width, priv->fullscreen_height,
			 1, BlackPixel(priv->display, screen),
			 BlackPixel(priv->display, screen));

  hint.flags       = PPosition | PSize | PMinSize | PWinGravity;
  hint.x           = 0;
  hint.y           = 0;
  hint.width       = priv->fullscreen_width;
  hint.height      = priv->fullscreen_height;
  hint.min_width   = priv->fullscreen_width;
  hint.min_height  = priv->fullscreen_height;
  hint.win_gravity = StaticGravity;

  /* set Properties for window manager (always before mapping) */
  XSetStandardProperties (priv->display, priv->fullscreen_window, window_title,
			  gettext (window_title), None, NULL, 0, &hint);
  XSetWMNormalHints (priv->display, priv->fullscreen_window, &hint);

  /* XSetWMHints (priv->display, priv->fullscreen_window, &hint);*/

  /* layer above most other things, like gnome panel */
  {
    static Atom XA_WIN_LAYER = None;
    long        propvalue[1];

    if (XA_WIN_LAYER == None)
      XA_WIN_LAYER = XInternAtom(priv->display, "_WIN_LAYER", False);

    propvalue[0] = 10; /* WIN_LAYER_ABOVE_DOCK */
    XChangeProperty(priv->display, priv->fullscreen_window,
		    XA_WIN_LAYER, XA_CARDINAL, 32, PropModeReplace,
		    (unsigned char *)propvalue, 1);
  }

  /* fullscreen the modern (e.g. metacity) way */
  {
    static Atom XA_WIN_STATE = None;
    long        propvalue[2];

    if (XA_WIN_STATE == None)
      XA_WIN_STATE = XInternAtom (priv->display, "_NET_WM_STATE", False);

    propvalue[0] = XInternAtom (priv->display, "_NET_WM_STATE_FULLSCREEN",
				False);
    propvalue[1] = 0;

    XChangeProperty (priv->display, priv->fullscreen_window,
		     XA_WIN_STATE, XA_ATOM, 32, PropModeReplace,
		     (unsigned char *)propvalue, 1);
    XFlush(priv->display);
  }

  /* wm, no borders please */
  prop = XInternAtom(priv->display, "_MOTIF_WM_HINTS", False);
  mwmhints.flags = MWM_HINTS_DECORATIONS;
  mwmhints.decorations = 0;
  XChangeProperty (priv->display, priv->fullscreen_window, prop, prop, 32,
		   PropModeReplace, (unsigned char *)&mwmhints,
		   PROP_MWM_HINTS_ELEMENTS);

  XSetTransientForHint(priv->display, priv->fullscreen_window, None);
}

static void gtk_video_realize (GtkWidget *widget)
{
  /* GdkWindowAttr attributes; */
  /* gint          attributes_mask; */
  GtkVideo            *this;
  Pixmap               bm_no;
  unsigned long        black_pixel;
  gtk_video_private_t *priv;

  static unsigned char bm_no_data[] = { 0,0,0,0, 0,0,0,0 };

  g_return_if_fail(widget != NULL);
  g_return_if_fail(GTK_IS_VIDEO(widget));

  this = GTK_VIDEO(widget);
  priv = this->priv;

  gdk_event_handler_set (gtv_event_handler, NULL, NULL);

  /* set realized flag */
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  if (!XInitThreads ()) {
    printf (_("gtkvideo: XInitThreads failed - looks like you don't have a thread-safe xlib.\n"));
    return;
  }

  priv->display = XOpenDisplay (NULL);

  if (!priv->display) {
    printf (_("gtkvideo: XOpenDisplay failed!\n"));
    return;
  }

  XLockDisplay (priv->display);

  priv->screen = DefaultScreen (priv->display);

  /*
   * create our own video window
   */

  black_pixel = BlackPixel (gdk_display, priv->screen);

  priv->video_window = XCreateSimpleWindow (gdk_display,
					    GDK_WINDOW_XWINDOW (gtk_widget_get_parent_window(widget)),
					    0, 0,
					    widget->allocation.width,
					    widget->allocation.height, 1,
					    black_pixel, black_pixel);

  widget->window = gdk_window_foreign_new (priv->video_window);

  /*
   * prepare for fullscreen playback
   */

  priv->fullscreen_width  = DisplayWidth (gdk_display, priv->screen);
  priv->fullscreen_height = DisplayHeight (gdk_display, priv->screen);
  priv->fullscreen_mode   = 0;

  priv->toplevel = GDK_WINDOW_XWINDOW (gdk_window_get_toplevel (gtk_widget_get_parent_window(widget)));

  /*
   * create our fullscreen window
   */

  create_fullscreen_window (priv);

  /*
   * track configure events of toplevel window
   */
  g_signal_connect (gtk_widget_get_toplevel (widget),
		    "configure-event",
		    G_CALLBACK (configure_cb), this);

#ifdef HAVE_XTESTEXTENSION
  {
    int dummy1 = 0, dummy2 = 0, dummy3 = 0, dummy4 = 0;

    priv->have_xtest = XTestQueryExtension(priv->display, &dummy1, &dummy2, &dummy3, &dummy4);
    priv->kc_shift_l = XKeysymToKeycode(priv->display, XK_Shift_L);
  }
#else
  priv->have_xtest = False;
#endif

  XSelectInput (priv->display, priv->video_window,
		StructureNotifyMask | ExposureMask |
		ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  XSelectInput (priv->display, priv->fullscreen_window,
		ExposureMask | KeyPressMask | KeyReleaseMask |
		ButtonPressMask | ButtonReleaseMask | StructureNotifyMask |
		FocusChangeMask | PointerMotionMask);

  XUnlockDisplay (priv->display);

  /*
   * load audio, video drivers
   */

  priv->video_port = load_video_out_driver (this) ;

  if (!priv->video_port) {
    printf (_("gtkvideo: couldn't open video driver\n"));
    return ;
  }

  xine_post_wire_video_port (priv->out, priv->video_port);

  /*
   * create mouse cursors
   */

  bm_no = XCreateBitmapFromData(priv->display,
                                priv->video_window,
                                bm_no_data, 8, 8);
  priv->no_cursor = XCreatePixmapCursor(priv->display, bm_no, bm_no,
					(XColor *) &black_pixel,
					(XColor *) &black_pixel,
					0, 0);
  priv->on_cursor = XCreateFontCursor (priv->display, XC_left_ptr);
  priv->cursor_visible = TRUE;

  /*
   * allow resizing
   */

  priv->allow_resize = TRUE;
  priv->resize = priv->auto_resize;

  /*
   * now, create a xine thread
   */

  pthread_create (&priv->thread, NULL, xine_thread, this);

  return;
}


xine_video_port_t *gtk_video_get_port  (GtkVideo *gtv) {
  gtk_video_private_t *priv;

  priv = gtv->priv;

  return priv->video_port;
}

static void gtk_video_unrealize (GtkWidget *widget) {
  GtkVideo            *this;
  gtk_video_private_t *priv;

  g_return_if_fail(widget != NULL);
  g_return_if_fail(GTK_IS_VIDEO(widget));

  this = GTK_VIDEO(widget);
  priv = this->priv;

  xine_port_send_gui_data (priv->video_port,
			   XINE_GUI_SEND_WILL_DESTROY_DRAWABLE,
			   (void*)priv->fullscreen_window);

  /* stop event thread */

  pthread_cancel (priv->thread);

  /* prevent resizing */

  priv->allow_resize = priv->resize = FALSE;

  /* Hide all windows */

  if (GTK_WIDGET_MAPPED (widget))
    gtk_widget_unmap (widget);

  GTK_WIDGET_UNSET_FLAGS (widget, GTK_MAPPED);

  XDestroyWindow (priv->display, priv->fullscreen_window);

  /* This destroys widget->window and unsets the realized flag */
  if (GTK_WIDGET_CLASS(parent_class)->unrealize)
    (* GTK_WIDGET_CLASS(parent_class)->unrealize) (widget);
}


GtkWidget *gtk_video_new (xine_t *xine, xine_stream_t *stream,
    		          xine_post_out_t *out,
                          const gchar *video_driver_id,
                          int default_width, int default_height,
                          int button_press_mask, int button_release_mask)
{
  GtkWidget           *this=GTK_WIDGET (gtk_type_new (gtk_video_get_type ()));
  GtkVideo            *gtv = GTK_VIDEO(this);
  gtk_video_private_t *priv;

  gtv->priv = priv = malloc (sizeof (gtk_video_private_t));
  memset (priv, 0, sizeof (gtk_video_private_t));

  if (video_driver_id)
    priv->video_driver_id = strdup(video_driver_id);
  else
    priv->video_driver_id = NULL;

  priv->xine = xine;
  priv->stream = stream;
  priv->out = out;
  priv->resize_factor = 0.5;
  priv->resize = FALSE;
  priv->allow_resize = FALSE;
  priv->auto_resize = TRUE;
  priv->video_width = default_width;
  priv->video_height = default_height;
  priv->oldwidth = default_width;
  priv->oldheight = default_height;
  priv->button_press_mask = button_press_mask;
  priv->button_release_mask = button_release_mask;
  priv->vis_active = FALSE;
  priv->vis_plugin_id = NULL;
  priv->vis_plugin = NULL;
  return this;
}

static gint gtk_video_expose (GtkWidget *widget,
		 	      GdkEventExpose *event) {

  /*
  GtkVideo *this = GTK_VIDEO (widget);
  */

  return TRUE;
}

static void gtk_video_size_request (GtkWidget      *widget,
		       		    GtkRequisition *requisition) {
  GtkVideo *this;

  g_return_if_fail (widget != NULL);
  g_return_if_fail(GTK_IS_VIDEO(widget));

  this = GTK_VIDEO(widget);

  pthread_mutex_lock (&resize_lock);
  requisition->width = this->priv->oldwidth;
  requisition->height = this->priv->oldheight;
  pthread_mutex_unlock (&resize_lock);
}

static void gtk_video_size_allocate (GtkWidget *widget, 
    				     GtkAllocation *allocation) {

  GtkVideo *this;

  g_return_if_fail (widget != NULL);
  g_return_if_fail(GTK_IS_VIDEO(widget));

  this = GTK_VIDEO(widget);

  if (GTK_WIDGET_REALIZED (widget))
  {
    XLockDisplay (this->priv->display);
    widget->allocation = *allocation;
    gdk_window_move_resize (widget->window,
			    allocation->x, allocation->y,
			    allocation->width, allocation->height);
    XUnlockDisplay (this->priv->display);
  }
  else
    widget->allocation = *allocation;
}

void gtk_video_resize (GtkVideo *gtv,
		       gint x, gint y,
		       gint width, gint height)
{
  GtkAllocation allocation = { x, y, width, height };
  gtk_video_size_allocate (GTK_WIDGET (gtv), &allocation);
}

void gtk_video_rescale (GtkVideo *gtv, float scale)
{
  GtkRequisition size;
  GtkWidget *toplevel = gtk_widget_get_toplevel (&gtv->widget);
  gint w = gtv->priv->video_width * scale + .5;
  gint h = gtv->priv->video_height * scale + .5;

  XLockDisplay (gtv->priv->display);
  gtv->priv->resize = gtv->priv->force_resize = FALSE;
  gtv->priv->resize_factor = scale;
  /* set the widget's size now so that the window's size is recalculated... */
  gtk_widget_set_size_request (&gtv->widget, w, h);
  /* ... and get the recalculated size (we want the width) */
  gtk_widget_size_request (toplevel, &size);
  gtv->widget.allocation.width = w > size.width ? w : size.width;
  gtv->widget.allocation.height = h;
  gtv->priv->resize = gtv->priv->force_resize = TRUE;
  XUnlockDisplay (gtv->priv->display);
}

void gtk_video_set_fullscreen (GtkVideo *gtv, gint fullscreen) {

  static guint          timeout_id;
  gtk_video_private_t  *priv;

  g_return_if_fail (gtv != NULL);
  g_return_if_fail (GTK_IS_VIDEO (gtv));
  priv = gtv->priv;
  g_return_if_fail (priv->stream != NULL);

  if (fullscreen == priv->fullscreen_mode)
    return;

  XLockDisplay (priv->display);

  if (fullscreen)
  {
    XEvent xev;

    XRaiseWindow(priv->display, priv->fullscreen_window);

    XMapWindow (priv->display, priv->fullscreen_window);

    XFlush(priv->display);

    /* Wait for map. */

    do  {
      XWindowEvent(priv->display, priv->fullscreen_window,
		   StructureNotifyMask, &xev);
    } while (xev.type != MapNotify || xev.xmap.event != priv->fullscreen_window);

    XSetInputFocus (priv->display, priv->fullscreen_window, 
		    RevertToNone, CurrentTime);

    XMoveWindow (priv->display, priv->fullscreen_window, 0, 0); 

    XUnlockDisplay (priv->display);
    xine_port_send_gui_data (priv->video_port,
			     XINE_GUI_SEND_WILL_DESTROY_DRAWABLE,
			     (void*)priv->video_window);
    xine_port_send_gui_data (priv->video_port,
			   XINE_GUI_SEND_DRAWABLE_CHANGED,
			   (void*)priv->fullscreen_window);
    XLockDisplay (priv->display);

    /*
     * switch off mouse cursor
     */

    XDefineCursor(priv->display, priv->fullscreen_window,
		  priv->no_cursor);
    XFlush(priv->display);

    priv->cursor_visible = FALSE;

    /*
     * add timeout to switch off mouse cursor, disable screen saver
     */

    timeout_id = gtk_timeout_add (4000, gtk_timeout_cb, gtv);

  } else {

    gtk_timeout_remove (timeout_id);

    XUnlockDisplay (priv->display);
    xine_port_send_gui_data (priv->video_port,
			     XINE_GUI_SEND_WILL_DESTROY_DRAWABLE,
			     (void*)priv->fullscreen_window);
    xine_port_send_gui_data (priv->video_port,
			   XINE_GUI_SEND_DRAWABLE_CHANGED,
			   (void*)priv->video_window);
    XLockDisplay (priv->display);

    XUnmapWindow (priv->display, priv->fullscreen_window);

    XFlush(priv->display);
  }

  priv->fullscreen_mode = fullscreen;

  XUnlockDisplay (priv->display);
}

gint gtk_video_is_fullscreen (GtkVideo *gtv) {

  g_return_val_if_fail (gtv != NULL, 0);
  g_return_val_if_fail (GTK_IS_VIDEO (gtv), 0);

  return gtv->priv->fullscreen_mode;
}

void gtk_video_set_auto_resize (GtkVideo *gtv, gboolean resize) {

  g_return_if_fail (gtv != NULL);
  g_return_if_fail (GTK_IS_VIDEO (gtv));

  gtv->priv->auto_resize = resize;
  gtv->priv->resize &= resize;
}

gboolean gtk_video_get_auto_resize (GtkVideo *gtv) {

  g_return_val_if_fail (gtv != NULL, 0);
  g_return_val_if_fail (GTK_IS_VIDEO (gtv), 0);

  return gtv->priv->auto_resize;
}

static void gtv_hide_vis (gtk_video_private_t *priv,
			  xine_audio_port_t *audio_port)
{
  if (priv->vis_plugin)
  {
    xine_post_wire_audio_port (xine_get_audio_source (priv->stream),
			       audio_port);
    xine_post_dispose (priv->xine, priv->vis_plugin);
    priv->vis_plugin = NULL;
  }
}

static gboolean gtv_show_vis (gtk_video_private_t *priv,
			      xine_audio_port_t **audio_port)
{
  if (!priv->vis_plugin && priv->vis_plugin_id &&
      strcmp (priv->vis_plugin_id, "none"))
  {
    priv->vis_plugin = xine_post_init (priv->xine, priv->vis_plugin_id, 0,
				       audio_port, &priv->video_port);
    if (!priv->vis_plugin)
      return FALSE;
    xine_post_wire (xine_get_audio_source (priv->stream),
		    xine_post_input (priv->vis_plugin, "audio in"));
    return TRUE;
  }
  return FALSE;
}

gboolean gtk_video_set_vis (GtkVideo *gtv, const char *id,
			    xine_audio_port_t **audio_port)
{
  gtk_video_private_t *priv;

  g_return_val_if_fail (gtv != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_VIDEO (gtv), FALSE);
  priv = gtv->priv;
  g_return_val_if_fail (priv->stream != NULL, FALSE);

  free (priv->vis_plugin_id);
  priv->vis_plugin_id = strdup (id ? id : "none");

  if (!priv->vis_active)
    return priv->vis_plugin != NULL;

  gtv_hide_vis (priv, *audio_port);
  return gtv_show_vis (priv, audio_port);
}

void gtk_video_hide_vis (GtkVideo *gtv, xine_audio_port_t *audio_port)
{
  gtk_video_private_t  *priv;

  g_return_if_fail (gtv != NULL);
  g_return_if_fail (GTK_IS_VIDEO (gtv));
  priv = gtv->priv;
  g_return_if_fail (priv->stream != NULL);

  priv->vis_active = FALSE;
  gtv_hide_vis (priv, audio_port);
}

gboolean gtk_video_show_vis (GtkVideo *gtv, xine_audio_port_t **audio_port)
{
  gtk_video_private_t  *priv;

  g_return_val_if_fail (gtv != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_VIDEO (gtv), FALSE);
  priv = gtv->priv;
  g_return_val_if_fail (priv->stream != NULL, FALSE);

  if (priv->vis_active)
    return priv->vis_plugin != NULL;

  priv->vis_active = TRUE;
  return gtv_show_vis (priv, audio_port);
}
