/* 
 * Copyright (C) 2003-2005 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menu.c,v 1.41.2.1 2005/04/29 20:01:44 dsalt Exp $
 *
 * main/context menu creation / handling
 */

#include "globals.h"

#include <string.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>

#include "menu.h"
#include "playlist.h"
#include "engine.h"
#include "utils.h"
#include "gtkvideo.h"
#include "vis.h"

/* global pointers to menus: */

GtkWidget *popup_menu = NULL, *menubar = NULL;

GtkMenuShell *media_menu[2] = { NULL, NULL },
	     *vis_menu[2] = { NULL, NULL };
GtkCheckMenuItem *fullscreen_menu_item[2] = { NULL, NULL },
		 *deinterlace_menu_item[2] = { NULL, NULL },
		 *pp_video_menu_item[2] = { NULL, NULL },
		 *pp_audio_menu_item[2] = { NULL, NULL },
		 *auto_resize_menu_item[2] = { NULL, NULL },
		 *auto_rescale_menu_item[2] = { NULL, NULL },
		 *toolbar_item[3][2] = { { NULL, NULL }, { NULL, NULL },
					 { NULL, NULL } };
GSList *aspect_menu_items[2] = { NULL, NULL };

/*
 * context menu
 */

#if 0
static gint context_menu_cb (GtkWidget *widget, GdkEvent *event) {

  if (event->type == GDK_BUTTON_PRESS) {
    GdkEventButton *bevent = (GdkEventButton *) event; 

    if (bevent->button == 3) {
      gtk_menu_popup (GTK_MENU(widget), NULL, NULL, NULL, NULL,
		      bevent->button, bevent->time);
      /* Tell calling code that we have handled this event; the buck
       * stops here. */
      return TRUE;
    }
  }
  
  /* Tell calling code that we have not handled this event; pass it on. */
  return FALSE;
}
#endif

static void autoplay_cb (GtkWidget* widget, gpointer data) {

  char **mrls, *mrl;
  int    n, i, pos;

  mrls = xine_get_autoplay_mrls (xine, data, &n);

  if (!mrls) {
    playlist_logo();
    display_error (_("Autoplay input plugin failed"),
		   _("Autoplay for input plugin '%s' failed.\n"
		     "Check engine output for further details."),
		   data);
    return;
  }

  mrl = mrls[0]; i = 0; pos = -1;
  while (mrl) {

    if (pos<0)
      pos = playlist_add_mrl (mrl, -1);
    else
      playlist_add_mrl (mrl, -1);

    mrl = mrls[++i];
  }

  playlist_play (pos);
}

static void add_autoplay_entries (GtkMenuShell *submenu, int index) {

  GtkWidget    *item = NULL;
  int           i;
  const char *const *autoplay_ids = xine_get_autoplay_input_plugin_ids (xine);

  static const struct {
    char id[8];
    const char *stock;
  } stock[] = {
    { "VCD", GTK_STOCK_CDROM },
    { "VCDO", GTK_STOCK_CDROM },
    { "DVD", GTK_STOCK_CDROM },
    { "CD", GTK_STOCK_CDROM },
  };

  if (!autoplay_ids) {
    logprintf ("menu: warning: didn't get any autoplay entries\n");
    return;
  }

  i=0;
  while (autoplay_ids[i]) {
    int j = sizeof (stock) / sizeof (stock[0]);
    while (j--)
      if (!strcmp (autoplay_ids[i], stock[j].id))
      {
	item = gtk_image_menu_item_new_with_label (autoplay_ids[i]);
	gtk_image_menu_item_set_image
	  (GTK_IMAGE_MENU_ITEM(item),
	   gtk_image_new_from_stock (stock[j].stock, GTK_ICON_SIZE_MENU));
	break;
      }
    if (j < 0)
      item = gtk_menu_item_new_with_label (autoplay_ids[i]);

    g_signal_connect (GTK_OBJECT(item), "activate", 
			G_CALLBACK (autoplay_cb), 
			(gpointer) strdup (autoplay_ids[i]));

    gtk_menu_shell_insert (submenu, item, ++index);
    gtk_widget_show(item);

    i++;
  }
}

static void vis_none_cb (gpointer callback_data, guint callback_action,
			 GtkWidget *widget)
{
  vis_set (GTK_VIDEO(gtv), "none", &audio_port);
}

static void vis_cb (GtkWidget* widget, gpointer data)
{
  vis_set (GTK_VIDEO(gtv), data, &audio_port);
}

static void add_vis_entries (GtkMenuShell *submenu, GtkRadioMenuItem *none_item)
{
  const char *const *pol =
    xine_list_post_plugins_typed (xine, XINE_POST_TYPE_AUDIO_VISUALIZATION);
  if (pol)
  {
    xine_cfg_entry_t entry;
    gboolean ticked = FALSE;
    int i;
    GSList *group =
      gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (none_item));
    const char *vis = xine_config_lookup_entry
			(xine, "gui.post_plugins.audio_visualisation", &entry)
		      ? entry.enum_values[entry.num_value] : NULL;

    for (i = 0; pol[i]; ++i)
    {
      GtkWidget *item = gtk_radio_menu_item_new_with_label (group, pol[i]);
//      group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM(item));
      g_object_set_data (G_OBJECT (item), "vis", (gpointer) strdup (pol[i]));
      g_signal_connect (GTK_OBJECT(item), "activate",
			G_CALLBACK (vis_cb), (gpointer) strdup (pol[i]));
      gtk_menu_shell_insert (submenu, item, i + 1);
      gtk_widget_show(item);
      if (vis && !strcasecmp (vis, pol[i]))
	GTK_CHECK_MENU_ITEM(item)->active = ticked = TRUE;
    }
    GTK_CHECK_MENU_ITEM(none_item)->active = !ticked;
  }
}

enum {
  /* File */
  JS_CB_OPEN,
  JS_CB_OPEN_MRL,
  JS_CB_PLAYLIST,
  JS_CB_PREFERENCES,
  JS_CB_KEYBINDINGS,
  JS_CB_SNAPSHOT,
  JS_CB_EXIT,
  /* View */
  JS_CB_FULLSCREEN,
  JS_CB_TOOLBAR,
  JS_CB_ZOOM_IN,
  JS_CB_ZOOM_OUT,
  JS_CB_ZOOM_100,
  JS_CB_AV_SETTINGS,
  JS_CB_POSTPROC_DEINTERLACE,
  JS_CB_POSTPROC_VIDEO,
  JS_CB_POSTPROC_AUDIO,
  /* Media */
  JS_CB_MEDIAMARKS_ADD,
  JS_CB_MEDIAMARKS_MANAGE,
  JS_CB_MEDIAMARKS_IMPORT,
  /* Help */
  JS_CB_ABOUT,
  JS_CB_LOG,
  JS_CB_STREAM_INFO,
  JS_CB_WIZARDS,
};

static void js_cb (gpointer data, guint action, GtkWidget *widget)
{
  static const char *const cmds[] = {
    /* File */
    [JS_CB_OPEN]			= "open_show ();",
    [JS_CB_OPEN_MRL]			= "open_mrl_show ();",
    [JS_CB_PLAYLIST]			= "playlist_show ();",
    [JS_CB_PREFERENCES]			= "preferences_show ();",
    [JS_CB_KEYBINDINGS]			= "keybindings_show ();",
    [JS_CB_SNAPSHOT]			= "snapshot ();",
    [JS_CB_EXIT]			= "exit ();",
    /* View */
    [JS_CB_FULLSCREEN]			= "set_fullscreen ();",
    [JS_CB_TOOLBAR]			= "toolbar_show ();",
    [JS_CB_ZOOM_IN]			= "set_zoom (get_zoom()+5);",
    [JS_CB_ZOOM_OUT]			= "set_zoom (get_zoom()-5);",
    [JS_CB_ZOOM_100]			= "set_zoom (100);",
    [JS_CB_AV_SETTINGS]			= "settings_show ();",
    [JS_CB_POSTPROC_DEINTERLACE]	= "deinterlace_show ();",
    [JS_CB_POSTPROC_VIDEO]		= "postproc_video_show ();",
    [JS_CB_POSTPROC_AUDIO]		= "postproc_audio_show ();",
    /* Media */
    [JS_CB_MEDIAMARKS_ADD]		= "mm_add_show ();",
    [JS_CB_MEDIAMARKS_MANAGE]		= "mm_manage_show ();",
    [JS_CB_MEDIAMARKS_IMPORT]		= "import_mediamarks ();",
    /* Help */
    [JS_CB_ABOUT]			= "about_show ();",
    [JS_CB_LOG]				= "log_show ();",
    [JS_CB_STREAM_INFO]			= "stream_info_show ();",
    [JS_CB_WIZARDS]			= "run_wizards ();",
  };
  engine_exec (cmds[action], NULL, NULL);
}

enum {
  JS_TOGGLE_CB_DEINTERLACE,
  JS_TOGGLE_CB_AUTO_RESIZE,
  JS_TOGGLE_CB_AUTO_RESCALE,
  JS_TOGGLE_CB_POSTPROC_VIDEO,
  JS_TOGGLE_CB_POSTPROC_AUDIO,
};

static void js_toggle_cb (gpointer data, guint action, GtkWidget *widget)
{
  static const char *const cmds[] = {
    [JS_TOGGLE_CB_DEINTERLACE]		= "set_deinterlace",
    [JS_TOGGLE_CB_AUTO_RESIZE]		= "set_auto_resize",
    [JS_TOGGLE_CB_AUTO_RESCALE]		= "set_auto_rescale",
    [JS_TOGGLE_CB_POSTPROC_VIDEO]	= "set_postproc_video",
    [JS_TOGGLE_CB_POSTPROC_AUDIO]	= "set_postproc_audio",
  };
  v_engine_exec ("%s (%d);", NULL, NULL, cmds[action],
		 gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)));
}

static void tbar_position_cb(gpointer data, guint action, GtkWidget *widget)
{
  v_engine_exec ("set_toolbar_position (%d);", NULL, NULL, action);
}

static void factor_cb(gpointer data, guint action, GtkWidget *widget)
{
  v_engine_exec ("set_video_size (%d);", NULL, NULL, action);
}

static void aspect_cb(gpointer data, guint action, GtkWidget *widget)
{
  v_engine_exec ("set_aspect (%d);", NULL, NULL, action);
}

static void sub_cb (gpointer data, guint action, GtkWidget *widget)
{
  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)))
    v_engine_exec ("set_sub (%d);", NULL, NULL, action);
}

static void popdown_idle_cb (GtkWidget *widget)
{
  gdk_threads_enter ();
  gtk_menu_popdown (GTK_MENU(widget));
  gdk_threads_leave ();
}

static gboolean popup_key_cb (GtkWidget *widget, GdkEventKey *event,
			      gpointer data)
{
  if (event->keyval != GDK_F10 || (event->state & GXINE_MODIFIER_MASK))
    return FALSE;
  g_idle_add ((GSourceFunc) popdown_idle_cb, widget);
  return TRUE;
}

#define STOCK(x) "<StockItem>", GTK_STOCK_##x
#define STOCKQ(x) "<StockItem>", "gtk-" #x
#define ASPECT(x) aspect_cb, XINE_VO_ASPECT_##x
#define JS(x) js_cb, JS_CB_##x
#define JSTOGGLE(x) js_toggle_cb, JS_TOGGLE_CB_##x
static GtkItemFactoryEntry menu_items[] = {
  { N_("/_File"),			NULL,		NULL, 0,		"<Branch>" },
  { N_("/File/_Open..."),		"<control>O",	JS(OPEN),		STOCK(OPEN) },
  { N_("/File/Open _MRL..."),		"<control>M",	JS(OPEN_MRL),		STOCK(OPEN) },
  { N_("/File/Play_list..."),		NULL,     	JS(PLAYLIST),		NULL },
  { "/File/sep0",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/File/_Preferences..."),	NULL,		JS(PREFERENCES),	STOCK(PREFERENCES) },
  { N_("/File/_Keybindings..."),	NULL,		JS(KEYBINDINGS),	STOCK(PREFERENCES) },
  { "/File/sep1",			NULL,		NULL, 0,		"<Separator>" },
  { "/File/sep2",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/File/_Snapshot..."),		"<control>S",	JS(SNAPSHOT),		NULL },
  { "/File/sep3",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/File/_Quit"),			"<control>Q",	JS(EXIT),		STOCK(QUIT) },
  { N_("/_View"),			NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/_Fullscreen mode"),	"<control>F",	JS(FULLSCREEN),		"<CheckItem>" },
  { N_("/View/Fullscreen _toolbar"),	NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Fullscreen toolbar/_Visible"),	NULL,	JS(TOOLBAR),		"<CheckItem>" },
  { N_("/View/Fullscreen toolbar/At _top"),	NULL,	tbar_position_cb, 1,	"<RadioItem>" },
  { N_("/View/Fullscreen toolbar/At _bottom"),	NULL,	tbar_position_cb, 0,	"/View/Fullscreen toolbar/At top" },
  { "/View/sep3",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/_Window"),		NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Window/_50%"),		NULL,		factor_cb, 50,		NULL },
  { N_("/View/Window/_100%"),		NULL,		factor_cb, 100,		STOCK(ZOOM_100) },
  { N_("/View/Window/_200%"),		NULL,		factor_cb, 200,		NULL },
  { N_("/View/Window/Auto _resize"),	NULL,		JSTOGGLE(AUTO_RESIZE),	"<CheckItem>" },
  { N_("/View/Window/_Magnify low-res video"),	NULL,	JSTOGGLE(AUTO_RESCALE),	"<CheckItem>" },
  { N_("/View/Zoom _in"),		NULL,		JS(ZOOM_IN),		STOCK(ZOOM_IN) },
  { N_("/View/Zoom _out"),		NULL,		JS(ZOOM_OUT),		STOCK(ZOOM_OUT) },
  { N_("/View/_Zoom 100%"),		NULL,		JS(ZOOM_100),		STOCK(ZOOM_100) },
  { "/View/sep5",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/_Deinterlace"),		"<control>I",	JSTOGGLE(DEINTERLACE),	"<CheckItem>" },
  { N_("/View/_Aspect ratio"),		NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Aspect ratio/_Auto"),	NULL,		ASPECT(AUTO),		"<RadioItem>" },
  { N_("/View/Aspect ratio/_Square"),	NULL,		ASPECT(SQUARE),		"/View/Aspect ratio/Auto"},
  { N_("/View/Aspect ratio/_4:3"),	NULL,		ASPECT(4_3),		"/View/Aspect ratio/Auto"},
  { N_("/View/Aspect ratio/_16:9"),	NULL,		ASPECT(ANAMORPHIC),	"/View/Aspect ratio/Auto"},
  { N_("/View/Aspect ratio/_2:1"),	NULL,		ASPECT(DVB),		"/View/Aspect ratio/Auto"},
  { N_("/View/A\\/V _settings..."),	NULL,		JS(AV_SETTINGS),	NULL },
  { N_("/View/_Post-processing"),	NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Post-processing/_Deinterlace..."), NULL,	JS(POSTPROC_DEINTERLACE), STOCK(PREFERENCES) },
  { N_("/View/Post-processing/_Video..."), NULL,	JS(POSTPROC_VIDEO),	STOCK(PREFERENCES) },
  { N_("/View/Post-processing/_Audio..."), NULL,	JS(POSTPROC_AUDIO),	STOCK(PREFERENCES) },
  { N_("/View/Post-processing/Enable _video"), NULL,	JSTOGGLE(POSTPROC_VIDEO), "<CheckItem>" },
  { N_("/View/Post-processing/Enable _audio"), NULL,	JSTOGGLE(POSTPROC_AUDIO), "<CheckItem>" },
  { "/View/sep6",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/_Visualisations"),	NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Visualisations/_None"),	NULL,		vis_none_cb, 0,		"<RadioItem>"},
  { "/View/sep7",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/_Subtitles"),		NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Subtitles/_Auto"),	NULL,		sub_cb, -1,		"<RadioItem>"},
  { N_("/View/Subtitles/_None"),	NULL,		sub_cb, -2,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _0"),	NULL,		sub_cb, 0,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _1"),	NULL,		sub_cb, 1,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _2"),	NULL,		sub_cb, 2,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _3"),	NULL,		sub_cb, 3,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _4"),	NULL,		sub_cb, 4,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _5"),	NULL,		sub_cb, 5,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _6"),	NULL,		sub_cb, 6,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _7"),	NULL,		sub_cb, 7,		"/View/Subtitles/Auto"},
  { N_("/_Media"),			NULL,		NULL, 0,		"<Branch>" },
  { N_("/Media/_Add media mark..."),	"<control>D",	JS(MEDIAMARKS_ADD),	STOCK(NEW) },
  { N_("/Media/_Manage media marks..."), "<control>B",	JS(MEDIAMARKS_MANAGE),	NULL },
  { N_("/Media/_Import media marks..."), NULL,		JS(MEDIAMARKS_IMPORT),	STOCK(OPEN) },
  { "/Media/sep8",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/_Help"),			NULL,		NULL, 0,		"<LastBranch>" },
  { N_("/Help/_About..."),		NULL,		JS(ABOUT),		STOCKQ(about) },
  { N_("/Help/Engine _Log..."),		"<control>L",	JS(LOG),		NULL },
  { N_("/Help/Stream _Info..."),	NULL,		JS(STREAM_INFO),	NULL },
  { "/Help/sep6",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/Help/_Re-run setup wizards..."), NULL,		JS(WIZARDS),		NULL },
};

GtkWidget *create_menu_tree (GtkItemFactory **factory, GtkWidget *window,
			     GtkAccelGroup **accel,
			     GtkItemFactoryEntry *menu_items,
			     gint nmenu_items)
{
  if (window)
  {
    GtkAccelGroup *accel_group =
      (accel && *accel) ? *accel : gtk_accel_group_new ();
    if (accel)
      *accel = accel_group;
    *factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>",
				     accel_group);
#ifdef ENABLE_NLS
    gtk_item_factory_set_translate_func (GTK_ITEM_FACTORY (*factory),
					 (GtkTranslateFunc) gettext,
					 NULL, NULL);
#endif
    gtk_item_factory_create_items (*factory, nmenu_items, menu_items, NULL);
    gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);
  }
  else
  {
    *factory = gtk_item_factory_new (GTK_TYPE_MENU, "<main>", NULL);
#ifdef ENABLE_NLS
    gtk_item_factory_set_translate_func (GTK_ITEM_FACTORY (*factory),
					 (GtkTranslateFunc) gettext,
					 NULL, NULL);
#endif
    gtk_item_factory_create_items (*factory, nmenu_items, menu_items, NULL);
  }

  return gtk_item_factory_get_widget (*factory, "<main>");
}


void create_menus (GtkWidget *window)
{
  GtkItemFactory *item_factory[2];
  GtkRadioMenuItem *none_item;
  int i;

  if (window)
    menubar = create_menu_tree (&item_factory[0], window, NULL,
				menu_items, num_menu_items (menu_items));
  popup_menu = create_menu_tree (&item_factory[1], NULL, NULL,
				 menu_items, num_menu_items (menu_items));

  for (i = !window; i < 2; ++i)
  {
    GtkMenuShell *menu;
    xine_cfg_entry_t entry;

    media_menu[i] = GTK_MENU_SHELL (gtk_item_factory_get_widget (item_factory[i], "/Media"));

    fullscreen_menu_item[i]  = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Fullscreen mode"));

    deinterlace_menu_item[i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Deinterlace"));
    if (xine_config_lookup_entry (xine, "gui.post_plugins.deinterlace_enable",
				  &entry))
      deinterlace_menu_item[i]->active = entry.num_value;

    pp_video_menu_item[i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Post-processing/Enable video"));
    if (xine_config_lookup_entry (xine, "gui.post_plugins.video_enable",
				  &entry))
      pp_video_menu_item[i]->active = entry.num_value;

    pp_audio_menu_item[i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Post-processing/Enable audio"));
    if (xine_config_lookup_entry (xine, "gui.post_plugins.audio_enable",
				  &entry))
      pp_audio_menu_item[i]->active = entry.num_value;

    auto_resize_menu_item[i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Window/Auto resize"));
    auto_resize_menu_item[i]->active = TRUE;

    auto_rescale_menu_item[i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Window/Magnify low-res video"));
    if (xine_config_lookup_entry (xine, "gui.magnify_lowres_video", &entry))
      auto_rescale_menu_item[i]->active = entry.num_value;

    toolbar_item[0][i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Fullscreen toolbar/Visible"));
    toolbar_item[1][i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Fullscreen toolbar/At bottom"));
    toolbar_item[2][i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Fullscreen toolbar/At top"));
    if (xine_config_lookup_entry (xine, "gui.fullscreen_toolbar", &entry))
    {
      toolbar_item[0][i]->active = (entry.num_value >> 1) & 1;
      toolbar_item[1][i]->active = (entry.num_value & 1);
      toolbar_item[2][i]->active = !(entry.num_value & 1);
    }

    none_item = GTK_RADIO_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Aspect ratio/Auto"));
    aspect_menu_items[i] = gtk_radio_menu_item_get_group (none_item);

    menu = GTK_MENU_SHELL (gtk_item_factory_get_widget (item_factory[i],
							"/File"));
    add_autoplay_entries (menu, g_list_index (menu->children,
				  gtk_item_factory_get_widget (item_factory[i],
							       "/File/sep1")));

    none_item = GTK_RADIO_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Visualisations/None"));
    vis_menu[i] = GTK_MENU_SHELL (gtk_item_factory_get_widget (item_factory[i], "/View/Visualisations"));
    add_vis_entries (vis_menu[i], none_item);
  }

  g_signal_connect (GTK_OBJECT(popup_menu), "key-press-event",
		    G_CALLBACK(popup_key_cb), NULL);
}
