#!/bin/bash
# ======================================================================
# $Source: /home/cvs/cvsroot/src/cron/cron_diagnose.sh,v $
# $Date: 2004/12/13 17:09:13 $
# $Revision: 1.2 $
# $Name:  $
# $State: Exp $
# $Author: corinna $
# ======================================================================

# ======================================================================
# This script checks that the various files, directories, and mount
# points needed by the cron daemon exist and have the proper settings,
# permissions, etc.  This script is based primarily on the requirements
# listed in the text file /usr/share/doc/Cygwin/cron.README.
# ======================================================================

# ======================================================================
# Routine: check_program
# Check to see that a specified program ($1) is installed and accessible
# by this script.  If it is not, then alert the user about which package
# ($2) should be installed to provide that program.
# ======================================================================
function check_program() {
	if ! type $1 > /dev/null 2>&1; then
		echo "The '$1' program is missing or not in your PATH."
		echo "This program is included in the '$2' package."
		echo "Please install this program and run this script again."
		return 1
	fi
} # === End of check_program() === #


# ======================================================================
# Routine: sanity_check
# Check for the set of programs that are used by this script.
# ======================================================================
function sanity_check() {
	# Check for programs that this script uses.
	check_program awk gawk || return
	check_program ls fileutils || return
	check_program grep grep || return
	check_program sed sed || return
	check_program id sh-utils || return
	check_program cut textutils || return
	check_program mount cygwin || return
	check_program cygcheck cygwin || return
} # === End of sanity_check() === #

# ======================================================================
# Routine: get_NT
# ======================================================================
function get_NT() {
    NT2003=""
    NT=$(uname -s | sed -ne 's/^CYGWIN_NT-\([^ ]*\)/\1/p')
    if [ "$NT" \> 5.1 ]; then NT2003=yes; fi
    if [ -z "$NT" ]; then
	echo "This script is only meant to run on NT."
	exit 0
    fi
} # === End of get_NT() === #

# ======================================================================
# Routine: warning_for_etc_file
# Display a warning message for the user about overwriting the specified
# file in /etc.
# ======================================================================
function warning_for_etc_file() {
	echo
	echo "WARNING: The command above overwrites any existing /etc/$1."
	echo "You may want to preserve /etc/$1 before generating a new,"
	echo "one, and then compare your saved /etc/$1 file with the"
	echo "newly-generated one in case you need to restore other"
	echo "entries."
} # === warning_for_etc_file() === #


# ======================================================================
# Routine: check_system_in_group
# Check to see whether the first gid 18 maps to the SYSTEM sid
# in /etc/group.
# ======================================================================
function check_system_in_group() {
    if [ "$(sed -ne '/^[^:]*:[^:]*:18:/{s/^[^:]*:S-1-5-18:.*$/yes/p;q}' /etc/group)" != yes ]; then 
		echo "It appears that you do not have an entry mapping"
		echo "gid 18 to the SYSTEM sid in /etc/group."
		echo
		echo "Use the 'mkgroup' utility to generate it"
		echo "   mkgroup -l > /etc/group"
		echo "and then run this script again."
		warning_for_etc_file group
		return 1;
    fi
}  # === check_system_in_group() === #


# ======================================================================
# Routine: get_system_and_admins_gids
# Get the ADMINs ids from /etc/group and /etc/passwd
# ======================================================================
function get_system_and_admins_ids() {
    ADMINSGID=$(sed -ne '/^[^:]*:S-1-5-32-544:.*:/{s/[^:]*:[^:]*:\([0-9]*\):.*$/\1/p;q}' /etc/group)
    SYSTEMGID=$(sed -ne '/^[^:]*:S-1-5-18:.*:/{s/[^:]*:[^:]*:\([0-9]*\):.*$/\1/p;q}' /etc/group)
    if [ -z "$ADMINSGID" -o -z "$SYSTEMGID" ]; then
		echo "It appears that you do not have entries for the"
		echo "ADMINISTRATORS and/or SYSTEM sids in /etc/group."
		echo
		echo "Use the 'mkgroup' utility to generate them"
		echo "   mkgroup -l > /etc/group"
		echo "and then run this script again."
		warning_for_etc_file group
		return 1;
    fi

    ADMINSUID=$(sed -ne '/^[^:]*:[^:]*:[0-9]*:[0-9]*:[^:]*,S-1-5-32-544:.*:/{s/[^:]*:[^:]*:\([0-9]*\):.*$/\1/p;q}' /etc/passwd)
    SYSTEMUID=$(sed -ne '/^[^:]*:[^:]*:[0-9]*:[0-9]*:[^:]*,S-1-5-18:.*:/{s/[^:]*:[^:]*:\([0-9]*\):.*$/\1/p;q}' /etc/passwd)
    if [ -z "$ADMINSUID" -o -z "$SYSTEMUID" ]; then
		echo "It appears that you do not have an entry for the"
		echo "ADMINISTRATORS and/or SYSTEM sids in /etc/passwd."
		echo
		echo "Use the 'mkpasswd' utility to generate it"
		echo "   mkpasswd -l > /etc/passwd"
		echo "and then run this script again."
		warning_for_etc_file passwd
		return 1;
    fi
}  # === get_system_and_admins_ids() === #    


# ======================================================================
# Routine: check_passwd_and_group
# Check to see whether the user's password ID and group exist in the
# system /etc/passwd and /etc/group files, respectively.
# ======================================================================
function check_passwd_and_group() {
	if [ "$(id -gn)" = mkpasswd ]; then
		echo "It appears that you do not have an entry for your user ID"
		echo "in /etc/passwd.  If this check is incorrect, then re-run"
		echo "this script with the '-f' command-line option."
		echo
		echo "Otherwise, use the 'mkpasswd' utility to generate an"
		echo "entry for your User ID in the password file:"
		echo "   mkpasswd -l -u User_ID >> /etc/passwd"
		echo "or"
		echo "   mkpasswd -d -u User_ID >> /etc/passwd"
		echo "and then run this script again."
		return 1

	elif [ -n "$USERDOMAIN" ] && [ -n "$USERNAME" ]; then
		if ! grep -E -q -i "^$(id -un):.*U-$USERDOMAIN\\\\$USERNAME" /etc/passwd; then
			echo "It appears that you do not have an entry for:"
			echo "   $USERDOMAIN\\$USERNAME"
			echo "in /etc/passwd."
			echo
			echo "Use the 'mkpasswd' utility to generate an entry for"
			echo "your User ID in the password file:"
			echo "   mkpasswd -d -u User_ID >> /etc/passwd"
			echo "and then run this script again."
			return 1
		fi
	fi

	if [ "$(id -gn)" = mkgroup ]; then
		echo "It appears that you do not have an entry for your group ID"
		echo "in /etc/group.  If this check is incorrect, then re-run"
		echo "this script with the '-f' command-line option."
		echo
		echo "Otherwise, use the 'mkgroup' utility to generate an"
		echo "entry for your group ID in the password file:"
		echo "   mkgroup -l -g Group_id  >> /etc/group"
		echo "or"
		echo "   mkgroup -d -g Group_id >> /etc/group"
		echo "and then run this script again."
		return 1
	fi

	local fname=""
	for fname in /etc/passwd /etc/group; do
		if ! ls -ld $fname | grep -E -q '^-r..r..r..+? '; then
			echo "The permissions on the file $fname are not correct."
			echo "Please run 'chmod +r $fname', and then run this script again."
			return 1
		fi
	done
} # === End of check_passwd_and_group() === #


# ======================================================================
# Routine: check_dir
# Check to see that the specified directory ($1) exists.
# ======================================================================
function check_dir() {
	if [ ! -d $1 ]; then
		echo "Your computer does not appear to have a $1 directory."
		echo "Please investigate this problem, and then run this script again."
		return 1
	fi

	if ! ls -ld $1 | grep -E -q '^dr[-w]x.*'; then
		echo "The permissions on the directory $1 are not correct."
		echo "Please run 'chmod u+rx $1', and then run this script again."
		return 1
	fi
} # === End of check_dir() === #


# ======================================================================
# Routine: check_dir_perms
# Check to see that the specified directory ($1) exists and has the
# required permissions, as described in /usr/share/doc/Cygwin/cron.README.
# ======================================================================
function check_dir_perms() {
	check_dir $1 || return

	if ! ls -ld $1 | grep -F -q 'drwxrwxrwt'; then
		echo "The permissions on the directory $1 are not correct."
		echo "Please run 'chmod 1777 $1', and then run this script again."
		return 1
	fi
} # === End of check_dir_perms() === #


# ======================================================================
# Routine: check_access
# Check to see that the owner and Administrators have
# proper access to the file or directory.
# On installations older than Windows 2003, allow access by System
# ======================================================================
function check_access() {
	local file="$1"
        local perm="$2"
	local notify=0;
	local ls="$(ls -dLln "$file" 2> /dev/null)"

	# If the owner of the file does not have access,
	# then notify the user.
	if [ -z "$(echo "$ls" | sed -n /^."$perm"/p)" ]; then
		notify=1;
	# If 'Administrators' has owner or group access to the file,
	# but does not have the desired access, then notify the user.
	elif [ "$(echo "$ls" | awk '{ print $3 }')" -eq $ADMINSUID \
	       -o \( -z "$NT2003" -a "$(echo "$ls" | awk '{ print $3 }')" -eq $SYSTEMUID \) ]; then
	        true;
	elif [ "$(echo "$ls" | awk '{ print $4 }')" -eq $ADMINSGID \
	       -o \( -z "$NT2003" -a "$(echo "$ls" | awk '{ print $4 }')" -eq $SYSTEMGID \) ]; then
		if [ -z "$(echo "$ls" | sed -n /^...."$perm"/p)" ]; then notify=1; fi
	elif [ -z "$(echo "$ls" | sed -n /^......."$perm"/p)" ]; then notify=1; fi

	if [ $notify -eq 1 ]; then
		echo "The owner and the Administrators need";
		echo "to have $perm permission to $file.";
		echo "Here are the current permissions:";
		echo;
		ls -dlL "$file";
		echo;
		echo "Please change the user and/or group ownership and";
		echo "permissions of $file, and then run this script again.";
		return 1;
	fi
} # === End of check_access() === #


# ======================================================================
# Routine: check_sys_mount
# Check to see that the SYSTEM account has access to the specified
# directory.
# ======================================================================
function check_sys_mount() {
	local mnt_point=$1
	local dos_dir=$2

	local SYSTEM_MOUNTS='/proc/registry/HKey_Local_Machine/Software/Cygnus Solutions/Cygwin/mounts v2'
	if ! ls "$SYSTEM_MOUNTS" | grep -Eq "^${mnt_point}\$"; then
		echo;
		echo "The SYSTEM user cannot access the mount point ${mnt_point}."
		echo "Please run the following command to add a system mount point:"
		echo '   mount -f -s -b "[DOS path to Cygwin]'$dos_dir\" \"$mnt_point\"
		echo "where [DOS path to Cygwin] is something like c:/cygwin."
		echo
		echo "For more information, run 'mount -m' and 'mount -h'"
		echo
		echo "After adding this mount point, please re-run this script."
		return 1
	fi
} # === End of check_sys_mount() === #


# ======================================================================
# Routine: check_cron_table
# Check for the existence of a crontab for the user, and check its
# permissions and ownership.
# ======================================================================
function check_cron_table() {
	local user_id=$(id -un)
	local cron_table=/var/cron/tabs/$user_id

	if [ ! -f $cron_table ]; then
		echo "Your computer does not appear to have a cron table for $user_id."
		echo "It is missing the file:"
		echo
		ls $cron_table
		echo
		echo "Please generate a cron table for $user_id using 'crontab -e',"
		echo "and run this script again."
		return 1
	fi

	local ls="$(ls -ln "$cron_table")"
	if ! echo "$ls" | grep -F -q 'rw-r-----'; then
		echo "The file permissions of your cron table need to"
		echo "provide read/write access for $user_id."
		echo "The permissions of your cron table file are set to:"
		echo
		ls -l $cron_table
		echo
		echo "You can set the file permissions with:"
		echo "	chmod 640 $cron_table"
		echo "Please change your cron table's permissions, and run"
		echo "this script again."
		return 1
	fi

	if [ "$(echo "$ls"  | awk '{ print $4 }')" -ne "$ADMINSGID" \
	    -a \( -n "$NT2003" -o "$(echo "$ls" | awk '{ print $4 }')" -ne "$SYSTEMGID" \) ]; then
		echo "The group membership of your cron table file should be ADMINISTRATORS,"
		echo "as documented in the file /usr/share/doc/Cygwin/cron.README."
		echo "Here is your cron table file:"
		echo
		ls -l $cron_table
		echo
		echo "You can change the group membership setting with:"
		echo "	 chgrp $ADMINSGID $cron_table"
		echo "Please change your cron table's group membership, and"
		echo "run this script again."
		return 1
	fi
} # === End of check_cron_table() === #

function check_myself() {
	echo
	if [ ! -x /usr/sbin/cron ]; then
	    echo "If you are running cron as yourself,";
	    echo "you need x access to /usr/sbin/cron.";
	fi
	if [ ! -w /var/run ]; then
	    echo "If you are running cron as yourself,";
	    echo "you need w access to /var/run.";
        fi
	if [ ! -x /usr/sbin/sendmail ]; then
	    echo "If you are running cron as yourself,";
	    echo "/usr/sbin/sendmail should point to an executable mailer";
	fi
        echo
} # === End of check_myself() === #

function main() {
	echo -e "cron_diagnose.sh 1.10\n"

	get_NT

	sanity_check || return

	check_system_in_group || return
	get_system_and_admins_ids || return

	if [ "$1" != '-f' ]; then  # Allow the user to skip the password check.
		check_passwd_and_group || return
	fi

	# Check the integrity of the files in the 'cron' package:
	if cygcheck -c cron | grep -F -q 'Incomplete'; then
		echo "'cygcheck -c cron' reports that your cron installation"
		echo "is incomplete.  Please consider running 'setup.exe' and"
		echo "selecting 'Reinstall' from the install options, and then"
		echo "run this script again."
		return 1
	fi

	check_dir /etc/cron.d || return
	check_dir /var || return

	check_dir_perms /var/cron || return
	check_dir_perms /var/cron/tabs || return

	# Check write access to /var/run, to create cron_pid
	check_access /var/run .w. || return

	# Check x access to /usr/sbin/cron
 	check_access /usr/sbin/cron ..x || return

	# Check x access to /usr/sbin/sendmail
 	check_access /usr/sbin/sendmail ..x ||
	   echo "     ssmtp and exim are suitable mailers."

	check_sys_mount /usr/bin /bin || return
	check_sys_mount /usr/lib /lib || return
	check_sys_mount / / || return
	echo;

	check_cron_table || return

	echo "This script did not find any errors in your cron setup."
	check_myself

	echo "If you are still unable to get cron to work, then try"
	echo "shutting down the cron service, uninstalling it,"
	echo "reinstalling it, and restarting it."
	echo
	echo "The following commands will do that:"
	echo "  $ cygrunsrv --stop cron"
	echo "  $ cygrunsrv --remove cron"
	echo "  $ cygrunsrv --install cron -p /usr/sbin/cron -a -D"
	echo "      (You can also add a -u switch)"
	echo "  $ cygrunsrv --start cron"
	echo 

	if [ -f /var/run/cron.pid ]; then
		echo "If the cron service does not start, try deleting the file"
		echo "/var/run/cron.pid and then repeating the commands above."
		echo
	fi

	echo "Also, examine the log file for cron, /var/log/cron.log, for"
	echo "information that it might give you about the problem cron is"
	echo "having."
	echo

	echo "If none of this fixes the problem, then report your problem"
	echo "to cygwin@cygwin.com.  Please include a copy of your crontab,"
	echo "('crontab -l') and the output of 'cygcheck -srv > cygcheck.txt'."
	echo
	echo "Please include the generated file 'cygcheck.txt' *as an attachment*,"
	echo "and NOT in the body of the mail message."
} # === End of main() === #

# Entry point:

# Set PATH to use the Cygwin programs
PATH=/usr/bin:/bin:$PATH

main $@
exit $?

# === End of $RCSfile: cron_diagnose.sh,v $ === #
