/*!
 * \file
 * \brief Random number generator test program
 * \author Adam Piatyszek
 *
 * -------------------------------------------------------------------------
 *
 * Copyright (C) 1995-2012  (see AUTHORS file for a list of contributors)
 *
 * This file is part of IT++ - a C++ library of mathematical, signal
 * processing, speech processing, and communications classes and functions.
 *
 * IT++ is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * IT++ is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with IT++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * -------------------------------------------------------------------------
 */

#include <itpp/itbase.h>
#include <iomanip>
#include "gtest/gtest.h"

using namespace itpp;
using namespace std;

TEST(Rand, All)
{
  const double eps = 2*numeric_limits<double>::epsilon();
  int i,j;

  {
    RNG_reset(4357U);
    Bernoulli_RNG b_rng;
    ASSERT_EQ(bin(0), b_rng());
    bvec v_expect = "0 1 1 0 1 0 1 0 0 0";
    bvec v = b_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_EQ(v_expect(i), v(i)) << i;
    }
    bmat m_expect = "1 1 1 0 0; 1 0 1 0 1; 1 0 0 1 1";
    bmat m = b_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_EQ(m_expect(i,j), m(i,j)) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Uniform_RNG u_rng;
    ASSERT_NEAR(0.6776523052396253, u_rng(), eps);
    vec v_expect = "0.6941770584246434 0.4674258031190679 0.4892392302568118 0.6187760132324469 0.3122587757056146 0.9778712826640192 "
    "0.2195994049039265 0.8667504766402725 0.55711296390849 0.951397435272191";
    vec v = u_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "0.3585771136429858 0.1794607099116465 0.07970432382915571 0.7640365697101184 0.5859759976605128; "
    "0.2040934355571897 0.8051015464399693 0.2878831162446251 0.8554638029408097 0.4979499314960036; "
    "0.1683366265617352 0.6664940489317837 0.8546382205321217 0.4453365941302123 0.1443036942242235";
    mat m = u_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    I_Uniform_RNG iu_rng(0, 9);
    ASSERT_EQ(6, iu_rng());
    ivec v_expect = "6 4 4 6 3 9 2 8 5 9";
    ivec v = iu_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_EQ(v_expect(i), v(i)) << i;
    }
    imat m_expect = "3 2 1 1 8; 6 0 2 8 7; 8 4 5 4 1";
    imat m = iu_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_EQ(m_expect(i,j), m(i,j)) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Normal_RNG n_rng;
    ASSERT_NEAR(1.893097921430992, n_rng(), eps);
    vec v_expect = "0.6402717419207782 -0.6697680324312685 0.7272643156549641 -0.9828709673793459 0.08733966325286896 "
    "-0.4069743415888314 1.50369957083693 1.708256250743253 -0.3616838528627909 0.5491090382229717";
    vec v = n_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "-0.3664826823907004 0.2362398029921575 0.08634252616292722 0.7341571423564899 -0.1914601183139686; "
    "-0.5005183411687337 0.4395163301393568 0.4302349815876011 -1.572718128851201 0.7524087638759094; "
    "-1.7451782785168886 -0.4690060611195704 0.1806300166341613 -0.4310872094225911 -0.3197472822012825";
    mat m = n_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Complex_Normal_RNG cn_rng;
    complex<double> scalar = cn_rng();
    ASSERT_NEAR(1.338622377694012, scalar.real(), eps);
    ASSERT_NEAR(0.4527404905143053, scalar.imag(), eps);
    cvec v_expect = "-0.4735975175541214+0.5142535293146189i -0.6949947260653173+0.06175846815265315i -0.2877743167063931+1.0632761634060945i "
    "1.2079195789048613-0.2557491050049569i 0.3882787245382864-0.2591423899058999i -0.3539199131486535-1.234027395118757i "
    "0.1670467666819286+0.3107849774837645i -0.3316373662352405+0.06105338575458272i 0.3042220729842621+0.1277247096478543i "
    "0.5191274938168114-1.1120796538057025i";
    cvec v = cn_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i).real(), v(i).real(), eps) << i;
      ASSERT_NEAR(v_expect(i).imag(), v(i).imag(), eps) << i;
    }
    cmat m_expect = "-0.3048246890654995-0.1353827479865859i 0.3709934479396093-0.4304478413979487i -0.1411731426937465-0.1305120155848639i "
    "-1.364128829828384+0.8726592365224727i 0.7522775207792468+0.4907727192185704i; 0.5320333391608433-0.2260954715104955i "
    "0.6321816921398413-0.2125745439342603i -0.4673563954429671+0.210388106679149i 0.7942460734239527-0.30166557762441i "
    "0.1343997864207408-0.001025126590916942i; 0.4398725368242644-0.3404948025920709i 0.6039173090419301+1.294761311710982i "
    "0.7862861082444027+0.3639643058675799i 1.306621454125742+0.2472936168443791i 0.2281929123280635-0.2856873610294465i";
    cmat m = cn_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j).real(), m(i,j).real(), eps) << i << "x" << j;
        ASSERT_NEAR(m_expect(i,j).imag(), m(i,j).imag(), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Exponential_RNG e_rng;
    ASSERT_NEAR(1.132124518607137, e_rng(), eps);
    vec v_expect = "1.184748966799 0.6300330542394361 0.6718539593722301 0.9643681848177544 0.3743426392815667 3.810879087015543 "
    "0.2479479076808403 2.015531792980273 0.8144405389904958 3.0240789772909515";
    vec v = e_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "0.4440663104809407 0.2282734813807377 0.1843275192661373 0.1977934840550751 1.6352766070304716; "
    "1.098094569503158 0.08306027343061394 0.3395132184655891 1.928529613170319 1.444078442362317; "
    "1.934225304119688 0.5893938251269003 0.8818313301719766 0.6890554262081474 0.155839748762987";
    mat m = e_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Gamma_RNG g_rng;
    ASSERT_NEAR(2.7345773127255977, g_rng(), eps);
    vec v_expect = "1.0552274663898724 0.1385497867625918 0.04651410856398541 0.2536327119715227 2.4374544334557107 0.2769546973504614 "
    "0.2744349992171708 6.406185233749446 0.3737814962396139 0.2994641096082811";
    vec v = g_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "0.2174735527040358 2.6329647388627526 1.5954090302534676 1.509676054135247 1.48296617377091; "
    "0.1621948306841281 0.368791785415268 0.9301993138404147 0.06886240299088112 0.5479410473121604; "
    "0.3100194244301691 0.1418546047377545 2.161406742662151 3.905612567710304 1.565773718382246";
    mat m = g_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Laplace_RNG lap_rng;
    ASSERT_NEAR(0.3104038525203892, lap_rng(), eps);
    vec v_expect = "0.3476149566930553 -0.04763599642043614 -0.0153841723257936 0.1917822113109224 -0.3328887924675951 "
    "2.2045693729764233 -0.5818096506718954 0.9350671267791583 0.08576735625913316 1.648217679951838";
    vec v = lap_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "-0.2350878039510808 -0.6335890249101929 -0.7697864192777591 -0.7245382366575881 0.6661861062127028; "
    "0.2863410447455327 -1.2984490544054614 -0.390360806687952 0.8735472954575282 0.5309885874254274; "
    "0.8775747571513712 -0.08186720446773095 0.1334198416930847 -0.002905194608440886 -0.8787131343057741";
    mat m = lap_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    AR1_Normal_RNG ar1_rng;
    ASSERT_NEAR(-0.6758828355068255, ar1_rng(), eps);
    vec v_expect = "1.382996864155973 -1.134989861747717 -0.2355969704269518 -0.6352883048864264 0.5874470599446935 0.6974029497987103 "
    "0.09759580407854659 0.8544948376298329 0.9480082544181549 0.8988055902561795";
    vec v = ar1_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "0.283339252286234 0.1727133610040758 -0.582086877933999 0.775548110744278 -1.63373141672122; "
    "-0.2041721536057245 0.3527524323845667 -0.463068243353409 -1.908566747153011 0.1732392237249772; "
    "1.9591934002549476 -1.2504661196749935 -0.4472104022592917 -0.5582363180520757 -0.00719101756607896";
    mat m = ar1_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Weibull_RNG w_rng;
    ASSERT_NEAR(1.132124518607137, w_rng(), eps);
    vec v_expect = "1.184748966799 0.6300330542394361 0.6718539593722301 0.9643681848177544 0.3743426392815667 3.810879087015543 "
    "0.2479479076808403 2.015531792980273 0.8144405389904958 3.0240789772909515";
    vec v = w_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "0.4440663104809407 0.2282734813807377 0.1843275192661373 0.1977934840550751 1.6352766070304716; "
    "1.098094569503158 0.08306027343061394 0.3395132184655891 1.928529613170319 1.444078442362317; "
    "1.934225304119688 0.5893938251269003 0.8818313301719766 0.6890554262081474 0.155839748762987";
    mat m = w_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Rayleigh_RNG ray_rng;
    ASSERT_NEAR(1.998441303523476, ray_rng(), eps);
    vec v_expect = "0.9886873125978385 0.9867439157625122 1.557799895380286 1.746125604767615 0.6601744408491138 1.815534586190379 "
    "0.498982814310214 0.4768875309662541 0.4666147686164047 1.7356350487617695";
    vec v = ray_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "0.4716918051361034 0.8175312057850639 0.7866696374519958 0.8036435560349791 0.9432302249245959; "
    "2.0204568644066563 0.2718942530839984 0.7248243315362419 1.2253292291982576 2.290144802171662; "
    "1.201523154893684 1.880645504774169 1.27025928864855 0.1900755295900456 0.5170865952405508";
    mat m = ray_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }

  {
    RNG_reset(4357U);
    Rice_RNG ric_rng;
    ASSERT_NEAR(2.9631003166431262, ric_rng(), eps);
    vec v_expect = "0.7987280746470597 0.0890034860870291 1.616413261164721 2.7323007025420125 1.5918694572094734 1.815249060409068 "
    "1.312045523204609 0.5379680240980399 1.441596096507667 2.3410987393109353";
    vec v = ric_rng(10);
    for (i = 0; i < v.length(); ++i) {
      ASSERT_NEAR(v_expect(i), v(i), eps) << i;
    }
    mat m_expect = "0.6002655580552441 1.781340731074266 1.692039164166575 1.641697565452313 1.91774948912197; "
    "2.6058361496697944 0.8213577410479846 0.4510951183425345 2.173794994082281 1.544805564133882; "
    "2.165669649217335 2.8692352774305294 2.1774574176104475 1.1900708837802285 1.383042143204967";
    mat m = ric_rng(3, 5);
    for (i = 0; i < m.rows(); ++i) {
      for (j = 0; j < m.cols(); ++j) {
        ASSERT_NEAR(m_expect(i,j), m(i,j), eps) << i << "x" << j;
      }
    }
  }
}
