/*  ksim - a system monitor for kde
 *
 *  Copyright (C) 2001  Robbie Ward <linuxphreak@gmx.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "fsysscroller.h"
#include "fsysscroller.moc"
#include <pluginmodule.h>
#include <common.h>
#include <themetypes.h>
#include <progress.h>

#include <kmessagebox.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kprocess.h>
#include <kdebug.h>

#include <qwidget.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qcursor.h>
#include <qlayout.h>
#include <qpopupmenu.h>

#define CHECK_LIST if (id > m_progressList.count()) return;
#define CHECK_LIST_TYPE(x) if (id > m_progressList.count()) return x;

FSysScroller::FSysScroller(bool stackItems, uint speed,
   QWidget *parent, const char *name)
   : QScrollView(parent, name)
{
  setMinimumWidth(1);
  setVScrollBarMode(QScrollView::AlwaysOff);
  setHScrollBarMode(QScrollView::AlwaysOff);
  setFrameShape(QScrollView::NoFrame);
  setFrameShadow(QScrollView::Plain);

  m_stackItems = stackItems;
  m_dummyProgress = 0;
  m_minHeight = 0;
  m_mountProcess = 0;

  m_main = new QWidget(this);
  addChild(m_main);
  m_layout = new QBoxLayout(m_main, QBoxLayout::LeftToRight);

  m_scrollTimer = new QTimer(this);
  connect(m_scrollTimer, SIGNAL(timeout()), this, SLOT(startScrolling()));

  if (stackItems) {
    m_layout->setDirection(QBoxLayout::TopToBottom);
  }
  else {
    m_layout->setDirection(QBoxLayout::LeftToRight);
    setScrollSpeed(speed);
  }
}

FSysScroller::~FSysScroller()
{
  delete m_mountProcess;
}

void FSysScroller::clear()
{
  ProgressList::ConstIterator it;
  for (it = m_progressList.begin(); it != m_progressList.end(); ++it) {
    delete (*it).first;
  }

  m_minHeight = 0;
  m_progressList.clear();
}

const QString &FSysScroller::text(uint id) const
{
  CHECK_LIST_TYPE(QString::null)
  return m_progressList[id].first->text();
}

int FSysScroller::value(uint id) const
{
  CHECK_LIST_TYPE(0)
  return m_progressList[id].first->value();
}

QSize FSysScroller::sizeHint() const
{
  return QSize(m_main->width(), height());
}

QSize FSysScroller::minimumSizeHint() const
{
  return sizeHint();
}

void FSysScroller::append(uint id, int maxValue, const QString &mountPoint)
{
  KSim::Progress *progress = new KSim::Progress(maxValue, m_main);
  progress->installEventFilter(this);
  progress->show();
  m_layout->addWidget(progress);
  m_progressList.append(qMakePair(progress, mountPoint));

  if (m_stackItems)
    m_minHeight += progress->sizeHint().height();
  else
    m_minHeight = progress->sizeHint().height();

  setMinimumHeight(m_minHeight);
  updateGeometry();

  if (id == 0)
   m_dummyMax = maxValue;
}

void FSysScroller::setText(uint id, const QString &text)
{
  CHECK_LIST
  m_progressList[id].first->setText(text);
}

void FSysScroller::setValue(uint id, int value)
{
  CHECK_LIST
  m_progressList[id].first->setValue(value);
}

void FSysScroller::setScrollSpeed(uint speed)
{
  if (m_scrollTimer->isActive())
    m_scrollTimer->stop();

  m_scrollTimer->start(speed);
}

void FSysScroller::insertDummy()
{
  if (!m_dummyProgress) {
    m_dummyProgress = new KSim::Progress(m_dummyMax, m_main);
    m_layout->addWidget(m_dummyProgress);
    m_dummyProgress->installEventFilter(this);
  }

  m_dummyProgress->setValue(value(0));
  m_dummyProgress->setText(text(0));
  m_dummyProgress->show();
}

void FSysScroller::resizeScroller()
{
  // should come up with a better fix here
  // but since this function hardly gets called
  // then its not that important
  int theWidth = 0;
  ProgressList::ConstIterator it;
  for (it = m_progressList.begin(); it != m_progressList.end(); ++it) {
    if ((*it).first->sizeHint().width() > theWidth)
      theWidth = (*it).first->sizeHint().width();
  }

  setMinimumWidth(theWidth);

  int sizeWidth = width();
  if (m_layout->direction() == QBoxLayout::LeftToRight)
    sizeWidth *= (m_progressList.count() + 1);

  m_main->resize(sizeWidth, m_main->height());

  if (m_progressList.isEmpty()) {
    setMinimumHeight(1);
  }

  updateGeometry();
}

void FSysScroller::setStackScroller(bool value)
{
  if (value) {
    setDirection(QBoxLayout::TopToBottom);
    stopScrolling();
    removeDummy();
  }
  else {
    setDirection(QBoxLayout::LeftToRight);
  }
}

void FSysScroller::wheelEvent(QWheelEvent *ev) // jasonkb's pride and joy :)
{
  if (!m_dummyProgress)
    return;

  int startPos = m_dummyProgress->width() - width();
  int delta = ev->delta();

  delta /= 4;
  if (delta < 0) { // Scroll with the flow, the way normally scrolls
    if (startPos == 0)
      startPos = delta;

    if (contentsX() < -delta) // we will bump into the beginning!
      setContentsPos(contentsWidth() - visibleWidth() - (contentsX() + delta), 0);
    //else
    // for some reason, the setContentsPos likes to be accompanied
    // by a scrollby.. don't really know why. thus, the commented out 'else' :)
    scrollBy(delta, 0);
  }
  else {
    delta /= 2;
    if (startPos == 0)
      startPos = delta;

    if (contentsX() == (contentsWidth() - visibleWidth()))
      setContentsPos(startPos, 0);
    else
      scrollBy(delta, 0);
  }
}

void FSysScroller::resizeEvent(QResizeEvent *ev)
{
  resizeScroller();
  QScrollView::resizeEvent(ev);
}

bool FSysScroller::eventFilter(QObject *o, QEvent *e)
{
  if (!o->isA("KSim::Progress"))
    return QScrollView::eventFilter(o, e);

  KSim::Progress *progressBar = 0;
  int i = 0;
  ProgressList::ConstIterator it;
  for (it = m_progressList.begin(); it != m_progressList.end(); ++it) {
    if (o == (*it).first) {
      progressBar = (*it).first;
      break;
    }
    ++i;
  }

  if ((o == progressBar || m_dummyProgress)
        && e->type() == QEvent::MouseButtonPress)
  {
    switch(static_cast<QMouseEvent *>(e)->button()) {
      case QMouseEvent::RightButton:
        showMenu(i);
        break;
      default:
        break;
      case QMouseEvent::LeftButton:
        if (parentWidget()->inherits("KSim::PluginView"))
          static_cast<KSim::PluginView *>(parentWidget())->doCommand();
        break;
    }

    return true;
  }

  return QScrollView::eventFilter(o, e);
}

void FSysScroller::receivedStderr(KProcess *, char *buffer, int length)
{
  m_stderrString.setLatin1(buffer, length);
}

void FSysScroller::processExited(KProcess *)
{
  delete m_mountProcess;
  m_mountProcess = 0;
  kdDebug() << "Deleting KProcess pointer" << endl;

  if (m_stderrString.isEmpty())
    return;

  QStringList errorList = QStringList::split("\n", m_stderrString);
  QString message = i18n("<qt>The following errors occured:<ul>");

  QStringList::Iterator it;
  for (it = errorList.begin(); it != errorList.end(); ++it) {
    message += QString::fromLatin1("<li>%1</li>")
       .arg((*it).replace(QRegExp("[u]?mount: "), QString::null));
  }

  message += QString::fromLatin1("</ul></qt>");
  KMessageBox::sorry(0, message);
}

void FSysScroller::startScrolling()
{
  if (!m_dummyProgress)
    return;

  int startPos = m_dummyProgress->width() - width();
  if (startPos == 0)
    startPos = 1;

  if (contentsX() == (contentsWidth() - visibleWidth()))
    setContentsPos(startPos, 0);
  else
    scrollBy(1, 0);
}

void FSysScroller::stopScrolling()
{
  if (m_scrollTimer)
    m_scrollTimer->stop();
}

void FSysScroller::createProcess(const QString &command, const QString &point)
{
  m_mountProcess = new KProcess();
  connect(m_mountProcess,
     SIGNAL(receivedStderr(KProcess *, char *, int)),
     SLOT(receivedStderr(KProcess *, char *, int)));
  connect(m_mountProcess,
     SIGNAL(processExited(KProcess *)),
     SLOT(processExited(KProcess *)));

  *m_mountProcess << command << point;
  (void)m_mountProcess->start(KProcess::NotifyOnExit, KProcess::Stderr);
}

void FSysScroller::showMenu(uint id)
{
  CHECK_LIST
  QPopupMenu menu;
  menu.insertItem(SmallIcon("hdd_mount"), i18n("&Mount Device"), 1);
  menu.insertItem(SmallIcon("hdd_unmount"), i18n("&Unmount Device"), 2);

  switch (menu.exec(QCursor::pos())) {
    case 1:
      createProcess("mount", m_progressList[id].second);
      break;
    case 2:
      createProcess("umount", m_progressList[id].second);
      break;
  }
}

void FSysScroller::setDirection(int direction)
{
  m_layout->setDirection((QBoxLayout::Direction)direction);

  if (m_progressList.isEmpty())
    return;

  m_minHeight = 0;

  if (direction == QBoxLayout::TopToBottom) {
    ProgressList::ConstIterator it;
    for (it = m_progressList.begin(); it != m_progressList.end(); ++it)
      m_minHeight += (*it).first->height();
  }
  else
    if (m_progressList.first().first)
      m_minHeight = m_progressList.first().first->height();

  setMinimumHeight(m_minHeight);
}

void FSysScroller::removeDummy()
{
  if (!m_dummyProgress)
    return;

  delete m_dummyProgress;
  m_dummyProgress = 0;
}
