/***************************************************************************
                            KDirWatch.h -  description
                             -------------------
    begin                : Thu Mar 21 01:01:29 2002
    copyright            : (C) 2000-2002 Lost Highway Ltd. All rights reserved.
    email                : Lost_Highway@tipitina.demon.co.uk
    generated by         : duke@tipitina on Thu Mar 21 01:01:29 2002, using kalyptus 0.9.
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 ***************************************************************************/

#ifndef KDIRWATCH_H
#define KDIRWATCH_H

#include <Foundation/NSObject.h>
#include <Foundation/NSString.h>
#include <qtobjc/QObject.h>

/**

 Watch directories and files for changes.
 The watched directories or files doesn't have to exist yet.

 When a watched directory is changed, i.e. when files therein are
 created or deleted, KDirWatch will emit the signal @ref -dirty.

 When a watched, but previously not existing directory gets created,
 KDirWatch will emit the signal @ref -created.

 When a watched directory gets deleted, KDirWatch will emit the
 signal @ref -deleted. The directory is still watched for new
 creation.

 When a watched file is changed, i.e. attributes changed or written
 to, KDirWatch will emit the signal @ref -fileDirty.

 Scanning of particular directories or files can be stopped temporarily
 and restarted. The whole class can be stopped and restarted.
 Directories and files can be added/removed from list in any state.

 The implementation uses the FAM service when available;
 if FAM is not available, the DNOTIFY functionality is used on LINUX.
 As a last resort, a regular polling for change of modification times
 is done; the polling interval is a global config option:
 DirWatch/PollInterval and DirWatch/NFSPollInterval for NFS mounted
 directories.

 @author Sven Radej <sven@lisa.exp.univie.ac.at>
  
 @short Class for watching directory and file changes.

*/
@interface KDirWatch : QObject 

/** 
 Constructor.

 Scanning begins immediatly when a dir/file watch
 is added.
    
*/
- initWithObject: (QObject *)parent name: (NSString *)name;
- init;

/** 
 Destructor.

 Stops scanning and cleans up.
    
*/
- (void) dealloc;

/** 
 Adds a directory to be watched.

 The directory does not have to exist. When @param watchFiles is
 false (the default), the signals -dirty, -created, -deleted
 can be emitted, all for the watched directory.
 When @param watchFiles is true, all files in the watched directory
 are looked for changes, too. Thus, the signals -fileDirty,
 -fileCreated, -fileDeleted can be emitted.

 When @param recursive is true, also all sub directories are watched.
    
*/
- addDir: (NSString *)path watchFiles: (BOOL)watchFiles recursive: (BOOL)recursive;
- addDir: (NSString *)path ;

/** 
 Adds a file to be watched.

    
*/
- addFile: (NSString *)file;

/** 
 Returns the time the directory/file was last changed.
    
*/
- (NSDate *) ctime: (NSString *)path;

/** 
 Removes a directory from the list of scanned directories.

 If specified path is not in the list this does nothing.
    
*/
- removeDir: (NSString *)path;

/** 
 Removes a file from the list of watched files.

 If specified path is not in the list this does nothing.
    
*/
- removeFile: (NSString *)file;

/** 
 Stops scanning the specified path.

 The @p path is not deleted from the interal just, it is just skipped.
 Call this function when you perform an huge operation
 on this directory (copy/move big files or many files). When finished,
 call @ref restartDirScan (path).
 Returns @p false if specified path is not in list, @p true otherwise.
    
*/
- (BOOL) stopDirScan: (NSString *)path;

/** 
 Restarts scanning for specified path.

 Resets ctime. It doesn't notify
 the change (by emitted a signal), since the ctime value is reset.

 Call it when you are finished with big operations on that path,
 @em and when @em you have refreshed that path.  Returns @p false
 if specified path is not in list, @p true otherwise.  
    
*/
- (BOOL) restartDirScan: (NSString *)path;

/** 
 Starts scanning of all dirs in list.

 If notify is @p true, all changed directories (since @ref
 -stopScan call) will be notified for refresh. If notify is
 @p false, all ctimes will be reset (except those who are stopped,
 but only if @p skippedToo is @p false) and changed dirs won't be
 notified. You can start scanning even if the list is
 empty. First call should be called with @p false or else all 
 directories 
 in list will be notified.  If
 @p skippedToo is true, the skipped directoris (scanning of which was
 stopped with @ref -stopDirScan ) will be reset and notified
 for change.  Otherwise, stopped directories will continue to be
 unnotified.
    
*/
- startScan: (BOOL)notify skippedToo: (BOOL)skippedToo;
- startScan;

/** 
 Stops scanning of all directories in internal list.

 The timer is stopped, but the list is not cleared.
    
*/
- stopScan;

/** 
 Is scanning stopped?
 After creation of a KDirWatch instance, this is false.
    
*/
- (BOOL) isStopped;

/** 
 Check if a directory is being watched by this KDirWatch instance
    
*/
- (BOOL) contains: (NSString *)path;

/** 
 @ref -created
    
*/
- setDirCreated: (NSString *)arg1;

/** 
 @ref -dirty
    
*/
- setDirDirty: (NSString *)arg1;

/** 
 @ref -deleted
    
*/
- setDirDeleted: (NSString *)arg1;

/** 
 @ref -fileCreated
    
*/
- setFileCreated: (NSString *)arg1;

/** 
 @ref -fileDirty
    
*/
- setFileDirty: (NSString *)arg1;

/** 
 @ref -fileDeleted
    
*/
- setFileDeleted: (NSString *)arg1;

/** 
 Dump statistic information about all KDirWatch instances.
 This checks for consistency, too.
    
*/
+ statistics;

/** 
 The KDirWatch instance usually globally used in an application.
 However, you can create an arbitrary number of KDirWatch instances
 aside from this one.
    
*/
+ (KDirWatch*) self;
@end

/** Override these methods in subclasses of KDirWatch to customise event handling behaviour */
@protocol KDirWatchEventHandling
- childEvent:(QChildEvent*) arg1;
- customEvent:(QCustomEvent*) arg1;
- timerEvent:(QTimerEvent*) arg1;
@end


#endif
