/* This file is part of the KDE project

   Copyright (C) 2001 Lukas Tinkl <lukas@kde.org>
                      Andreas Schlapbach <schlpbch@iam.unibe.ch>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <qlabel.h>
#include <qvbox.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qcombobox.h>
#include <qwhatsthis.h>

#include <klocale.h>
#include <kurl.h>
#include <kfontdialog.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <klineedit.h>

#include "imgallerydialog.h"
#include "imgallerydialog.moc"

RecursionSpinBox::RecursionSpinBox(QWidget *parent, const char *name)
  : QSpinBox(0, 99, 1, parent, name)
{
}

RecursionSpinBox::RecursionSpinBox(int lower, int upper, int step, QWidget *parent, const char *name)
  : QSpinBox(lower, upper, step, parent, name)
{
}

RecursionSpinBox::~RecursionSpinBox()
{
}

QString RecursionSpinBox::mapValueToText(int v)
{
  return v == 0 ? QString(i18n("endless")) : QString::number(v);
}


KIGPDialog::KIGPDialog(QWidget *parent, const QString& path, const char *name )
  : KDialogBase( IconList, i18n("Configure"), Ok|Cancel,
		 Ok, parent, name, true, true ),
    m_dialogOk( false )
{
  setCaption(i18n("Create Image Gallery"));
  setupLookPage(path);
  setupDirectoryPage(path);
}

void KIGPDialog::setupLookPage(const QString& path) {
  QFrame *page = addPage( i18n("Look"), i18n("Page Look"),
			  BarIcon("colorize", KIcon::SizeMedium ) );

  QVBoxLayout *vlay = new QVBoxLayout( page, spacingHint() );
  vlay->addSpacing( fontMetrics().lineSpacing());

  vlay->addWidget(new QLabel( i18n("Page Title:"), page));

  m_title = new QLineEdit(i18n("Image Gallery for %1").arg(path), page);
  vlay->addWidget( m_title );

  QHBoxLayout *hlay = new QHBoxLayout( spacingHint() );
  vlay->addLayout( hlay );

  hlay->addWidget(new QLabel( i18n("Images per Row :"), page) );
  hlay->addStretch( 100 );

  m_imagesPerRow = new QSpinBox(1,8,1,page );
  m_imagesPerRow->setValue(4);
  hlay->addWidget( m_imagesPerRow );

  QHBoxLayout *hlay0 = new QHBoxLayout( spacingHint() );
  vlay->addLayout( hlay0 );

  m_imageName = new QCheckBox( i18n("Add Image Name"), page);
  m_imageName->setChecked( true );
  hlay0->addWidget( m_imageName );

  m_imageSize = new QCheckBox( i18n("Add Image File Size"), page);
  m_imageSize->setChecked( false );
  hlay0->addWidget( m_imageSize );

  m_imageProperty = new QCheckBox( i18n("Add Image Dimension"), page);
  m_imageProperty->setChecked( false );
  vlay->addWidget( m_imageProperty );

  QHBoxLayout *hlay11  = new QHBoxLayout( spacingHint() );
  vlay->addLayout( hlay11 );

  m_fontName = new QComboBox( false,page );
  QStringList standardFonts;
  KFontChooser::getFontList(standardFonts, 0);
  m_fontName->insertStringList( standardFonts );

  hlay11->addWidget(new QLabel( i18n("Font Name"), page ));
  hlay11->addStretch( 100 );
  hlay11->addWidget(m_fontName);

  QHBoxLayout *hlay12  = new QHBoxLayout( spacingHint() );
  vlay->addLayout( hlay12 );

  m_fontSize = new QSpinBox( 6,30,1,page );
  m_fontSize->setValue(14);

  hlay12->addWidget(new QLabel( i18n("Font Size"), page ));
  hlay12->addStretch( 100 );
  hlay12->addWidget(m_fontSize);

  QHBoxLayout *hlay1  = new QHBoxLayout( spacingHint() );
  vlay->addLayout( hlay1 );

  m_foregroundColor = new KColorButton(page);
  m_foregroundColor->setColor(QColor("#d0ffd0"));

  hlay1->addWidget(new QLabel( i18n("Foreground Color:"), page));
  hlay1->addStretch( 100 );
  hlay1->addWidget(m_foregroundColor);

  QHBoxLayout *hlay2 = new QHBoxLayout( spacingHint() );
  vlay->addLayout( hlay2 );

  m_backgroundColor = new KColorButton(page);
  m_backgroundColor->setColor(QColor(33,33,33));

  hlay2->addWidget(new QLabel( i18n("Background Color:"), page));
  hlay2->addStretch( 100 );
  hlay2->addWidget(m_backgroundColor);
}

void KIGPDialog::setupDirectoryPage(const QString& path) {
  QFrame *page = addPage( i18n("Directories"), i18n("Directories"),
			  BarIcon("folder", KIcon::SizeMedium ) );

  QVBoxLayout *dvlay = new QVBoxLayout( page, spacingHint() );
  dvlay->addSpacing( fontMetrics().lineSpacing());

  dvlay->addWidget(new QLabel(i18n("Save to:"), page));

  m_imageNameReq = new KURLRequester(path + "images.html", page);
  dvlay->addWidget(m_imageNameReq);
  connect( m_imageNameReq->lineEdit(),SIGNAL(textChanged ( const QString & )),this, SLOT( imageUrlChanged(const QString & )));

  QHBoxLayout *hlay1  = new QHBoxLayout( spacingHint() );
  dvlay->addLayout(hlay1);

  m_recurseSubDirectories = new QCheckBox(i18n("&Recurse Subdirectories"), page);
  m_recurseSubDirectories->setChecked(false);
  connect(m_recurseSubDirectories, SIGNAL( clicked() ), this, SLOT( slotRecurseSubDirectories() ) );

  m_recursionLevel = new RecursionSpinBox(0, 99, 1, page);
  m_recursionLevel->setEnabled(false);

  hlay1->addWidget(m_recurseSubDirectories);
  hlay1->addStretch( 100 );
  hlay1->addWidget(new QLabel( i18n("Recursion level:"), page ));
  hlay1->addWidget(m_recursionLevel);

  m_copyOriginalFiles = new QCheckBox(i18n("Copy &Original Files"), page);
  m_copyOriginalFiles->setChecked(false);
  dvlay->addWidget(m_copyOriginalFiles);

  m_generateXHTML = new QCheckBox(i18n("Generate &XHTML"), page);
  m_generateXHTML->setChecked(false);
  dvlay->addWidget(m_generateXHTML);

  m_useCommentFile = new QCheckBox(i18n("Use comment file"), page);
  m_useCommentFile->setChecked(false);
  connect(m_useCommentFile, SIGNAL( clicked() ), this, SLOT( slotUseCommentFile() ) );
  dvlay->addWidget(m_useCommentFile);

  dvlay->addWidget(new QLabel(i18n("Comments File:"), page));

  m_commentFileReq = new KURLRequester(path + "comments", page);
  m_commentFileReq->setEnabled(false);
  dvlay->addWidget(m_commentFileReq);
  enableButtonOK( !m_imageNameReq->lineEdit()->text().isEmpty());
}

KIGPDialog::~KIGPDialog()
{
}

void KIGPDialog::imageUrlChanged(const QString &url )
{
    enableButtonOK( !url.isEmpty());
}

bool  KIGPDialog::printImageName()  const
{
  return m_imageName->isChecked();
}

bool  KIGPDialog::printImageSize() const
{
  return m_imageSize->isChecked();
}

bool  KIGPDialog::printImageProperty() const
{
  return m_imageProperty->isChecked();
}

bool KIGPDialog::recurseSubDirectories() const
{
  return m_recurseSubDirectories->isChecked();
}

int KIGPDialog::recursionLevel() const
{
  return m_recursionLevel->value();
}

bool KIGPDialog::copyOriginalFiles() const
{
  return m_copyOriginalFiles->isChecked();
}

bool KIGPDialog::useCommentFile() const
{
  return m_useCommentFile->isChecked();
}

bool KIGPDialog::generateXHTML() const
{
  return m_generateXHTML->isChecked();
}

int KIGPDialog::getImagesPerRow() const
{
  return m_imagesPerRow->value();
}

const QString KIGPDialog::getTitle() const
{
  return m_title->text();
}

const QString KIGPDialog::getImageName() const
{
  return m_imageNameReq->url();
}

const QString KIGPDialog::getCommentFile() const
{
  return m_commentFileReq->url();
}

const QString KIGPDialog::getFontName() const
{
  return m_fontName->currentText();
}

const QString KIGPDialog::getFontSize() const
{
  return m_fontSize->text();
}

const QColor KIGPDialog::getBackgroundColor() const
{
  return m_backgroundColor->color();
}

const QColor KIGPDialog::getForegroundColor() const
{
  return m_foregroundColor->color();
}

bool KIGPDialog::isDialogOk() const
{
  return m_dialogOk;
}

void KIGPDialog::slotUseCommentFile()
{
  m_commentFileReq->setEnabled(m_useCommentFile->isChecked());
}

void KIGPDialog::slotRecurseSubDirectories()
{
  m_recursionLevel->setEnabled(m_recurseSubDirectories->isChecked());
}

void KIGPDialog::slotOk()
{
  m_dialogOk = true;
  this->accept();
}

void KIGPDialog::slotCancel()
{
  m_dialogOk = false;
  this->done(Cancel);
}
