
/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <ctype.h>
#include <string.h>
#include <pan/base/run.h>

/**
***  Run operations
**/

gchar*
run_strdup (const Run * run)
{
	g_return_val_if_fail (run!=NULL, NULL);

	return g_strndup (run->str, run->len);
}

Run*
run_strstrip (Run * run)
{
	g_return_val_if_fail (run!=NULL, NULL);

	/* remove leading spaces */
	while (run->len>0 && isspace((guchar)run->str[0])) {
		--run->len;
		++run->str;
	}

	/* remove trailing spaces */
	while (run->len>0 && isspace((guchar)run->str[run->len-1]))
		--run->len;

	return run;
}

gint
run_runcmp (const Run * r1, const Run * r2)
{
	gint val;
	gint shortlen;

	g_return_val_if_fail (r1!=NULL && r2!=NULL, 0);
	g_return_val_if_fail (r1!=NULL, -1);
	g_return_val_if_fail (r2!=NULL, 1);

	shortlen = MIN(r1->len, r2->len);
	val = strncmp (r1->str, r2->str, shortlen);
	if (val)
		return val;

	if (r1->len == r2->len)
		return 0;

	return r1->len==shortlen ? 1 : -1;
}

gint
run_strcmp (const Run * r1, const gchar * str)
{
	Run r2;
	(void) run_init (&r2, str, strlen(str));
	return run_runcmp (r1, &r2);
}


const gchar*
run_strchr (const Run     * run,
            gchar           ch)
{
	const gchar * pch;

	g_return_val_if_fail (run!=NULL, FALSE);

	for (pch=run->str; pch!=run->str+run->len; ++pch)
		if (*pch == ch)
			return pch;

	return NULL;
}



/**
***  For hashtables of runs
**/

gint
run_equal (gconstpointer v, gconstpointer v2)
{
       	return run_runcmp ((const Run*)v, (const Run*)v2) == 0;
}
 
/* 31 bit hash function */
guint
run_hash (gconstpointer key)
{
	const Run * run = (const Run*) key;
	const gchar * pch;
	guint h = 0;

	for (pch=run->str; pch!=run->str+run->len; ++pch)
		h = (h << 5) - h + *pch;

	return h;
}
