/*
 * phase3d.c
 *
 * Copyright (C) 1994, 1998, 2000 Klaus Kudielka, Viktor Kudielka
 *
 * This program reads the P3D 512-byte blocks from stdin
 * and displays the formatted output on stdout.
 *
 * Last update: 2000-12-28 oe1vkw
 * Corresponding to TLMspec.doc Release 1.4 2000-11-01
 * by DB2OS, DJ4ZC, G3RUH, W4SM
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <stdio.h>
#include <ctype.h>
#include <math.h>


/* Display block */

#define C(n)  ((unsigned char)block[256+n])
#define C2(n) ((unsigned short)block[257+n]*256+block[256+n])
#define T(n)  (0.659*C(n)-69.7)            /* #140 - #16f */
#define T2(n) (-0.413*C(n)+103.8)          /* #12b - #12c */
#define Sr(n) (pow(C(n),-3.7545)*2.84e8)   /* #100 */
#define I1(n) ((C(n)*0.1014-0.7098)*1000)  /* #171 - #174, #176, #177 corr. */
#define I2(n) ((C(n)*0.2410-31.28)*1000)   /* #106 */
#define I3(n) ((C(n)*0.2035-2.85)*1000)    /* #107 */
#define I4(n) ((C(n)*0.197-0.739)*1000)    /* #108 */
#define I5(n) ((C(n)*0.0412-0.76)*1000)    /* #109 */
#define I6(n) ((C(n)*0.1024-0.653)*1000)   /* #10a */
#define I7(n) ((C(n)*0.0429-0.333)*1000)   /* #17a */
#define I8(n) ((C(n)*0.0125-0.0875)*1000)  /* #175, #178 */
#define I9(n) (C(n)*0.0503-0.3154)         /* #103 */
#define I10(n) (C(n)*0.103-0.95)           /* #119 */
#define V1(n) (C(n)*0.1548-1.484)          /* #10b - #10d */
#define V2(n) (C(n)*0.1522-1.06)           /* #10e */ 
#define V3(n) (C(n)*0.1318-0.923)          /* #10f, #111 */
#define V4(n) (C(n)*0.0657-0.712)          /* #110, #112 corr. */
#define V5(n) (C(n)*1.221-263.0537)        /* #104 */
#define Prs1(n) (C(n)*0.815-1.381)         /* #101 */
#define Prs2(n) (C(n)*0.835-1.253)         /* #102 */
#define Prs3(n) (C(n)*2.131-183.8)         /* #114 */
#define Prs4(n) (C(n)*0.1235-1.235)        /* #115 */
#define SA1(n) (-31.501 +0.3682*C(n) -0.001539*pow(C(n),2) \
		+0.00000361*pow(C(n),3))   /* #12e, #12f */
#define SA2(n) (57.296*asin(C(n)/255.0))   /* #130 - #133 */
#define SA3(n) (-55.179 +0.64187*C(n) -0.002447*pow(C(n),2) \
                +0.00000581*pow(C(n),3))   /* #134,  #135 */
#define Ws(n) ((1./(C2(n)+2)-1./24576)*121.263e6) 
                                           /* #1C0 - #1C5 */


#define N_RX 9
#define N_TX 8
#define N_RXTX 7
#define N_TXMod 14
#define N_Exp 10

struct _rxtx {
  int index;
  unsigned char mask;
  int rxtx_index;
};

struct _rx {
  unsigned char mask, value;
  char *name;
  int agc_index;
  double agc2, agc1, agc0;
  struct _rxtx rxtx[N_RXTX-1];
};

struct _txmod {
  int index;
  unsigned char mask;
  int txmod_index;
};

struct _tx {
  unsigned short mask, value;
  char *name;
  int pow_index;
  double pow2, pow1, pow0;
  struct _txmod txmod[N_TXMod];
};

struct experiment {
  unsigned char index, mask;
  char *string;
};

static char *rxtx_names[N_RXTX] =
{
  "V", "U", "S1", "X", "S2/Ku", "Leila1", "Leila2"
};

static char *txmod_names[N_TXMod+1] =
{
  "21/24 MHz", "V", "U", "L1", "L2", "S1", "S2/C", "Rudak1", "Rudak2", 
  "Leila1", "Leila2", "GB", "MB", "EB", ""
};

static struct _rx rx[N_RX] =
{
  {
    0x01, 0x01, "21 MHz", 0x1c, 0, 0, 0,
    {
      { 0xf4, 0x01, 0  },
      { 0xf4, 0x02, 2  },
      { 0xf4, 0x04, 3  },
      { 0xf4, 0x08, 4  },
      { 0xf4, 0x10, 5  },
      { 0xf4, 0x20, 6  }
    }
  },
  {
    0x02, 0x02, "24 MHz", 0x1c, 0, 0, 0,
    {
      { 0xf4, 0x01, 0  },
      { 0xf4, 0x02, 2  },
      { 0xf4, 0x04, 3  },
      { 0xf4, 0x08, 4  },
      { 0xf4, 0x10, 5  },
      { 0xf4, 0x20, 6  }
    }
  },
  {
    0x04, 0x04, "V", 0x1e, 0, 0.254, -14.8,
    {
      { 0xf2, 0x01, 1  },
      { 0xf2, 0x02, 2  },
      { 0xf2, 0x04, 3  },
      { 0xf2, 0x08, 4  },
      { 0xf2, 0x10, 5  },
      { 0xf2, 0x20, 6  }
    }
  },
  {
    0x04, 0x00, "U", 0x1f, 0, 0.457, -31.9,
    {
      { 0xf4, 0x40, 0  },
      { 0xf4, 0x80, 2  },
      { 0xf5, 0x01, 3  },
      { 0xf5, 0x02, 4  },
      { 0xf5, 0x04, 5  },
      { 0xf5, 0x08, 6  }
    }
  },
  {
    0x00, 0x00, "L1", 0x20, 0, 0.129, -7.9,
    {
      { 0xfe, 0x40, 0  },
      { 0xfe, 0x80, 1  },
      { 0xff, 0x01, 2  },
      { 0xff, 0x02, 3  },
      { 0xff, 0x04, 4  },
      { 0xff, 0x08, 5  }
    }
  },
  {
    0x20, 0x20, "L2", 0x16, 0, 0.154, -10.6,
    {
      { 0xfc, 0x01, 0  },
      { 0xfc, 0x02, 1  },
      { 0xfc, 0x04, 2  },
      { 0xfc, 0x08, 3  },
      { 0xfc, 0x10, 4  },
      { 0xfc, 0x20, 6  }
    }
  },
  {
    0x40, 0x40, "S1", 0x1d, 0.004, 1.2, -72,
    {
      { 0xf6, 0x40, 0  },
      { 0xf6, 0x80, 1  },
      { 0xf7, 0x01, 3  },
      { 0xf7, 0x02, 4  },
      { 0xf7, 0x04, 5  },
      { 0xf7, 0x08, 6  }
    }
  },
  {
    0x08, 0x08, "S2", 0x1b, 0.011, 3.7, -284,
    {
      { 0xfe, 0x01, 0  },
      { 0xfe, 0x02, 1  },
      { 0xfe, 0x04, 2  },
      { 0xfe, 0x08, 3  },
      { 0xfe, 0x10, 4  },
      { 0xfe, 0x20, 5  }
    }
  },
  {
    0x80, 0x80, "C", 0x1b, 0.011, 3.7, -284,
    {
      { 0xfe, 0x01, 0  },
      { 0xfe, 0x02, 1  },
      { 0xfe, 0x04, 2  },
      { 0xfe, 0x08, 3  },
      { 0xfe, 0x10, 4  },
      { 0xfe, 0x20, 5  }
    }
  }
};

static struct _tx tx[N_TX] =
{
  {
    0x0200, 0x0200, "V", 0x23, 0, 0, 0,
    {
      { 0xf4, 0x01, 0 },
      { 0xf4, 0x40, 2 },
      { 0xfe, 0x40, 3 },
      { 0xfc, 0x01, 4 },
      { 0xf6, 0x40, 5 },
      { 0xfe, 0x01, 6 },
      { 0xf2, 0x40, 7 },
      { 0xf8, 0x01, 8 },
      { 0xf6, 0x01, 9 },
      { 0xf8, 0x40, 10 },
      { 0xfa, 0x40, 11 },
      { 0xfc, 0x40, 12 },
      { 0xfa, 0x01, 13 }
    }
  },
  {
    0x0404, 0x0404, "U-low", 0x25, 0, 0, 0,
    {
      { 0xf2, 0x01, 1 },
      { 0xfe, 0x80, 3 },
      { 0xfc, 0x02, 4 },
      { 0xf6, 0x80, 5 },
      { 0xfe, 0x02, 6 },
      { 0xf2, 0x40, 7 },
      { 0xf8, 0x02, 8 },
      { 0xf6, 0x02, 9 },
      { 0xf8, 0x80, 10 },
      { 0xfa, 0x80, 11 },
      { 0xfc, 0x80, 12 },
      { 0xfa, 0x02, 13 },
      { 0xff, 0x00, 14 }
    }
  },
  {
    0x0404, 0x0400, "U", 0x25, 0, 0, 0,
    {
      { 0xf2, 0x01, 1 },
      { 0xfe, 0x80, 3 },
      { 0xfc, 0x02, 4 },
      { 0xf6, 0x80, 5 },
      { 0xfe, 0x02, 6 },
      { 0xf2, 0x40, 7 },
      { 0xf8, 0x02, 8 },
      { 0xf6, 0x02, 9 },
      { 0xf8, 0x80, 10 },
      { 0xfa, 0x80, 11 },
      { 0xfc, 0x80, 12 },
      { 0xfa, 0x02, 13 },
      { 0xff, 0x00, 14 }
    }
  },
  {
    0x0800, 0x0800, "S1", 0x22, 0, 0, 0,
    {
      { 0xf4, 0x02, 0 },
      { 0xf2, 0x02, 1 },
      { 0xf4, 0x80, 2 },
      { 0xff, 0x01, 3 },
      { 0xfc, 0x04, 4 },
      { 0xfe, 0x04, 6 },
      { 0xf2, 0x80, 7 },
      { 0xf8, 0x04, 8 },
      { 0xf9, 0x01, 10 },
      { 0xfb, 0x01, 11 },
      { 0xfd, 0x01, 12 },
      { 0xfa, 0x04, 13 },
      { 0xff, 0x00, 14 }
    }
  },
  {
    0x0001, 0x0001, "S2", 0x38, 0, 0, 0,
    {
      { 0xf4, 0x08, 0 },
      { 0xf2, 0x08, 1 },
      { 0xf5, 0x02, 2 },
      { 0xff, 0x04, 3 },
      { 0xfc, 0x10, 4 },
      { 0xf7, 0x02, 5 },
      { 0xfe, 0x10, 6 },
      { 0xf3, 0x02, 7 },
      { 0xf8, 0x08, 8 },
      { 0xf6, 0x08, 9 },
      { 0xf9, 0x02, 10 },
      { 0xfd, 0x04, 12 },
      { 0xff, 0x00, 14 }
    }
  },
  {
    0x0020, 0x0020, "X", 0x17, 0, 0, 0,
    {
      { 0xf4, 0x04, 0 },
      { 0xf2, 0x04, 1 },
      { 0xf5, 0x01, 2 },
      { 0xff, 0x02, 3 },
      { 0xfc, 0x08, 4 },
      { 0xf7, 0x01, 5 },
      { 0xfe, 0x08, 6 },
      { 0xf3, 0x01, 7 },
      { 0xf6, 0x04, 9 },
      { 0xfb, 0x02, 11 },
      { 0xfd, 0x02, 12 },
      { 0xfa, 0x08, 13 },
      { 0xff, 0x00, 14 }
    }
  },
  {
    0x2000, 0x2000, "X-TWT", 0x17, 0, 0, 0,
    {
      { 0xf4, 0x04, 0 },
      { 0xf2, 0x04, 1 },
      { 0xf5, 0x01, 2 },
      { 0xff, 0x02, 3 },
      { 0xfc, 0x08, 4 },
      { 0xf7, 0x01, 5 },
      { 0xfe, 0x08, 6 },
      { 0xf3, 0x01, 7 },
      { 0xf6, 0x04, 9 },
      { 0xfb, 0x02, 11 },
      { 0xfd, 0x02, 12 },
      { 0xfa, 0x08, 13 },
      { 0xff, 0x00, 14 }
    }
  },
  {
    0x4000, 0x0400, "Ku", 0x1a, 0, 0, 0,
    {
      { 0xf4, 0x08, 0 },
      { 0xf2, 0x08, 1 },
      { 0xf5, 0x02, 2 },
      { 0xff, 0x04, 3 },
      { 0xfc, 0x10, 4 },
      { 0xf7, 0x02, 5 },
      { 0xfe, 0x10, 6 },
      { 0xf3, 0x02, 7 },
      { 0xf8, 0x08, 8 },
      { 0xf6, 0x08, 9 },
      { 0xf9, 0x02, 10 },
      { 0xfd, 0x04, 12 },
      { 0xff, 0x00, 14 }
    }
  }
};

struct experiment elist[N_Exp] = {
  { 0xc9, 0x01, " IHU-2"  },
  { 0x88, 0x01, " ARU"    },
  { 0x88, 0x02, " RUDAK"  },
  { 0x88, 0x04, " GPS"    },
  { 0x88, 0x08, " Mon-Rx" },
  { 0x88, 0x10, " A-Cam"  },
  { 0x88, 0x20, " B-Cam"  },
  { 0x88, 0x40, " Cedex-ctrl"},
  { 0x88, 0x80, " Cedex-pwr"},
  { 0xa2, 0x10, " IR-beacon"}
};

/* Display list of active receivers (with AGC and connected transmitters) */

void display_rx(char *block) {
  int i, j;

  printf("Rx: ");
  
  for (i = 0; i < N_RX; i++) {
    if((C(0xca) & rx[i].mask) == rx[i].value) {
      if((rx[i].agc2+rx[i].agc1+rx[i].agc0) != 0)
	printf("%s(%.1fdB",
	       rx[i].name,
	       pow(C(rx[i].agc_index),2) * rx[i].agc2 +
	       C(rx[i].agc_index) * rx[i].agc1 + rx[i].agc0);
      else printf("%s(0x%2xdB",rx[i].name,C(rx[i].agc_index));
      for (j = 0; j < N_RXTX-1; j++) {
	if ((C(rx[i].rxtx[j].index) & rx[i].rxtx[j].mask)) {
	  printf(" %s", rxtx_names[rx[i].rxtx[j].rxtx_index]);
	}
      }
      printf(") ");
    }
  }

  printf("\n");
}

/* Display list of active transmitters               */
/* (with output power and connected receivers and beacons) */

void display_tx(char *block) {
  int i, j;

  printf("Tx: "); 

  for (i = 0; i < N_TX; i++) {
    if((C2(0xc8) & tx[i].mask) == tx[i].value) {
      if((tx[i].pow2+tx[i].pow1+tx[i].pow0) != 0)
	printf("%s(%.1fW",tx[i].name,
	       pow(C(tx[i].pow_index),2) * tx[i].pow2 +
	       C(tx[i].pow_index) * tx[i].pow1 + tx[i].pow0);
      else printf("%s(0x%2xW", tx[i].name,C(tx[i].pow_index));
      for (j = 0; j < N_TXMod; j++) {
	if ((C(tx[i].txmod[j].index) & tx[i].txmod[j].mask)) {
	  printf(" %s", txmod_names[tx[i].txmod[j].txmod_index]);
	}
      }
      printf(") ");
    }
  }
  printf("\n");
}

void display_block(char *block)
{
  int row, i;

  putchar('\n');
  switch (block[0])
  {
    case 'A':
    case 'E':
      printf("######## AO-40 IHU-1 Telemetry Display -"
	     " OE1KIB, OE1VKW ########\n");
      for (row = 0; row < 4; row++)
      {
	int col;
	char c;
	for (col = 0; col < 64; col++) {
	  c = block[row*64+col];
	  if (isprint(c)) putchar(c); else putchar('.');
	}
	putchar('\n');
      }
      printf("________________________ Orbit data "
	     "____________________________\n");
      printf("AMSAT day %05u  Time %02u:%02u:%05.2f UTC",
	     C(0xad)*256+C(0xac), C(0xab), C(0xaa), C(0xa8)/100.0+C(0xa9));
      printf("  Orbit No %05u  Phase %03u\n",
	     C(0xa7)*256+C(0xa6), C(0xa5));

      printf("________________________ Status "
	     "________________________________\n");

      display_rx(block);

      display_tx(block);

      printf("Beacons:");
      if(!(C(0x94) & 0x01))
      {
	printf(" GB(");
	if(!(C(0x94) & 0x02)) printf("PSK");
	if(C(0x94) & 0x01) printf("FSK");
	printf(";");
	if(C(0xfa) & 0x40) printf(" V");
	if(C(0xfa) & 0x80) printf(" U");
	if(C(0xfb) & 0x01) printf(" S1");
	if(C(0xfb) & 0x02) printf(" X");
	if(C(0xfb) & 0x04) printf(" Leila1");
	if(C(0xfb) & 0x08) printf(" Leila2");
	printf(")");
      }

      printf(" MB(");
      switch(C(0x94) & 0x30)
      {
	case 0x00: printf("No PSK"); break;
	case 0x10: printf("Ranging"); break;
	case 0x20: printf("EB source");
      }
      printf(";");
      if(C(0xfc) & 0x40) printf(" V");
      if(C(0xfc) & 0x80) printf(" U");
      if(C(0xfd) & 0x01) printf(" S1");
      if(C(0xfd) & 0x02) printf(" X");
      if(C(0xfd) & 0x04) printf(" S2/Ku");
      if(C(0xfd) & 0x08) printf(" Leila2");
      printf(")");

      if(C(0x94) & 0x08) 
      {
        printf(" EB(PSK;");
        if(C(0xfa) & 0x01) printf(" V");
        if(C(0xfa) & 0x02) printf(" U");
        if(C(0xfa) & 0x04) printf(" S1");
        if(C(0xfa) & 0x08) printf(" X");
        if(C(0xfa) & 0x10) printf(" Leila1");
        if(C(0xfa) & 0x20) printf(" Leila2");
      printf(")");
      }
      printf("\n");

      printf("Ant/Pre-amps:");
      if(C(0x89) & 0x01) printf(" V-Rx=om V-Tx=hi");
	else printf(" V-Rx=hi V-Tx=om");
      if(C(0x89) & 0x02) printf(" U-Rx=om U-Tx=hi");
	else printf(" U-Rx=hi U-Tx=om");
      if(C(0x89) & 0x04) printf(" L-om+preamp");
      if(C(0x89) & 0x08) printf(" L-hi+preamp");
      printf("\n");

      printf("Experiments:");
      for (i = 0; i < N_Exp; i++)
      if (C(elist[i].index) & elist[i].mask) printf(elist[i].string);
      printf("\n");

      if(C(0x8B) & 0x01) printf("Main battery on;");
      if(C(0x8B) & 0x02) printf(" Aux battery on;");
      if(!(C(0x8B) & 0x03)) printf(" Main+aux battery on;");
      if(C(0x8B) & 0x04) printf(" Charger on;");
      if(C(0x8B) & 0x08) printf(" Aux heater on;");
      printf(" Memory soft error count: %u;\n", (C(0xd9) & 0xe0) >> 5);

      printf("________________________ Temperatures [C] "
	     "_____________________\n");
      printf(" SolPan1  %5.1f  SolPan2  %5.1f  SolPan3  %5.1f"
	     "  SolPan4  %5.1f\n",
	     T(0x4c), T(0x4d), T(0x4e), T(0x4f));
      printf(" SolPan5  %5.1f  SolPan6  %5.1f\n",
	     T(0x50), T(0x51));
      printf(" BCR1     %5.1f  BCR2     %5.1f  BCR3     %5.1f  AuxBatB1 %5.1f\n",
	     T(0x42), T(0x44), T(0x43), T(0x47));
      printf(" MainBatB2%5.1f  MainBatB4%5.1f  MainBatB6%5.1f  AuxBatB5 %5.1f\n",
	     T(0x49), T(0x4a), T(0x4c), T(0x46));
      printf(" V-Tx     %5.1f  U-Excit  %5.1f  U-Amp    %5.1f\n",
	     T(0x5e), T(0x58), T(0x5f));
      printf(" S1-Tx    %5.1f  S2-Tx    %5.1f", T(0x5b), T(0x5c));
      if(C(0xc8) & 0x10) printf("  X-Tx     %5.1f", T2(0x2b));
      if(C(0xc9) & 0x20) printf("  X-Tx TWT %5.1f", T2(0x2c));
      printf("\n");
      printf(" U&V-Rx   %5.1f  L1-Rx    %5.1f  L2-Rx    %5.1f"
	     "  S1-HF Rx %5.1f\n",
	     T(0x59), T(0x5a), T(0x52), T(0x57));
      printf(" S2-C Rx  %5.1f  IHU      %5.1f  SEU      %5.1f  EPU      %5.1f\n",
	     T(0x56), T(0x61), T(0x40), T(0x41));
      printf(" HP1+X-Y  %5.1f  HP2+X+Y  %5.1f  HP2+X-Y  %5.1f  HP2-X    %5.1f\n",
	     T(0x69), T(0x68), T(0x53), T(0x54));
      printf(" HP3+X    %5.1f  HP3-X    %5.1f  HP4+X+Y  %5.1f\n",
	     T(0x6a), T(0x67), T(0x66));
      printf(" NH3 B2   %5.1f  MMH B3   %5.1f  N2O4-X-Y %5.1f  N2O4+X+Y %5.1f\n",
	     T(0x48), T(0x45), T(0x6b), T(0x6c));
      printf(" Top      %5.1f  Bottom   %5.1f  Back     %5.1f  Side2Pan %5.1f\n",
	     T(0x62), T(0x63), T(0x64), T(0x6d));
      printf(" Side4Pan %5.1f  S ant    %5.1f  He Tank  %5.1f\n",
	     T(0x65), T(0x6e), T(0x6f));
      printf("________________________ Voltages [V] "
	     "__________________________\n");
      printf(" Main-Bat   %5.1f  Aux-Bat    %5.1f\n",
	     V1(0x0b), V1(0x0c));
      printf(" BCR1-In    %5.1f  BCR2-In    %5.1f  BCR3-In    %5.1f\n",
	     V2(0x0e), V3(0x11), V3(0x0f));
      printf(" 28V Bus    %5.1f  10V-C1BCR2 %5.1f  10V-C2BCR3 %5.1f\n", 
             V1(0x0d),V4(0x12), V4(0x10));
      printf("________________________ Currents [mA] "
	     "_________________________\n");
      printf(" SP1-BCR1   %5.0f  SP4-BCR2   %5.0f  SP2-BCR3   %5.0f\n",
	     I1(0x71), I1(0x77), I1(0x74));
      printf(" SP6-BCR1   %5.0f  SP5-BCR2   %5.0f  SP3-BCR3   %5.0f\n",
	     I1(0x72), I1(0x76), I1(0x73));
      printf(" 10V-C1BCR2 %5.0f  10V-C2BCR3 %5.0f  28V-SSEU    0x%02x\n",
	     I8(0x78), I8(0x75), C(0x70));
      printf(" 28V-Bat    %5.0f  28VU1EPU   %5.0f  28VU2Bus   %5.0f\n",
	     I2(0x06), I3(0x07), I4(0x08));
      printf(" 28VU3-S    %5.0f  28V-BCR    %5.0f  X-Helix    %5.0f\n",
	     I5(0x09), I6(0x0a), I10(0x19));
      printf(" 28VS-PA/M  %5.0f  10VS-PA/M   0x%02x  K-Tx        0x%02x\n",
	     I7(0x7a), C(0x7b), C(0x79));
      printf("________________________ Sun/Earth Sensors "
	     "_____________________\n");
      printf("Sensor mode:");
      if(C(0x9f))
      {
	printf(" spin\n");
	printf(" SS1 PLL: ");
	if((C(0x90) == 0xff)|(C(0x90) == 0x00))
	{
	  printf("locked");
	  printf("  SS2 count: %u",C(0x91));
        }
	else printf("not locked"); 
        printf("  SpinRate[rpm] %5.1f\n",Sr(0x00));
      }
      else printf(" 3-axis\n");
      printf(" SS Z-25  %4.0f  SS Y-25  %4.0f  ",SA1(C(0x2e)),SA1(C(0x2f)));
      printf("SS Up-X  %4.0f  SS Up-Y  %4.0f\n",SA2(C(0x30)),SA2(C(0x31)));
      printf(" SS Z-45  %4.0f  SS Y-45  %4.0f  ",SA3(C(0x34)),SA3(C(0x35)));
      printf("SS Dwn-X %4.0f  SS Dwn-Y %4.0f\n",SA2(C(0x32)),SA2(C(0x33)));

      if(C(0xA0) & 0x01)
      {
        printf("________________________ Magnetorquer ___"
	       "_______________________\n");
	printf(" Magnet system is on;");
	if(C(0xA2) & 0x08)printf("  Magnet power ON");
	printf("\n");
	printf(" Desired magnet vector angle (256): %u\n",C(0xA1));
	printf(" Current polarity: Arm 1:");
	if(C(0xA2) & 0x01)printf(" +");
	else printf(" -");
	printf("  Arm 2:");
	if(C(0xA2) & 0x02)printf(" +");
	else printf(" -");
	printf("  Arm 3:");
	if(C(0xA2) & 0x04)printf(" +\n");
	else printf(" -\n");
      }

      if(C(0x87) & 0x07)
      {
        printf("________________________ Momentum Wheels "
	       "_______________________\n");
	printf(" Wheel no. power on (speed):");
	if(C(0x87) & 0x01)printf("1(%5f rpm),",Ws(0xc0));
	if(C(0x87) & 0x02)printf("2(%5f rpm),",Ws(0xc2));
	if(C(0x87) & 0x04)printf("3(%5f rpm)",Ws(0xc4)); 
	printf("\n");
      }

      printf("________________________ Arcjet Motor "
	     "__________________________\n");
      if(C(0xd9) & 0x04)printf("EPU power ON;  ");
      else printf("EPU power OFF;  ");
      if(!(C(0x81) & 0x01))printf("Configuration OFF;\n");
      else
      {
	printf("Gas generator ON; Configuration ON;");
	if(C(0x81) & 0x02) printf(" EPU run;\n");
	else printf("\n");
	if(C(0x81) & 0x04) printf(" Flow control 1. valve C\n");
	if(C(0x81) & 0x08) printf(" Flow control 2. valve B\n");
	if(C(0x81) & 0x20) printf(" Input valve A\n");
	if(C(0x81) & 0x40) printf(" Output valve B\n");
	if(C(0x81) & 0x40) printf(" Output valve C\n");
      }
      if(!(C(0xc6) & 0x08))printf("Instruments OFF;\n");
      else
      {       
        printf(" Tank Pressure %.0fBar",Prs1(0x01));
        printf("  Motor Pressure %.0fBar\n",Prs2(0x02));
        printf(" Motor Current %5.1fA",I9(0x03));
        printf("  Motor Voltage %5.1fV",V5(0x04));
        printf("  Flow 0x%02x",C(0x05));
        printf("\n");
      }

      printf("________________________ 400 N Motor "
	     "___________________________\n");
      if(C(0xd9) & 0x01)
      {
        printf("LIU power ON; ");
        printf("Motor valve:");
        if(C(0x13) <= 0x59)printf(" closed\n");
        else printf(" open\n");
      }
      else printf("LIU power OFF;\n");
      if(!(C(0xc6) & 0x08))printf("Instruments OFF;\n");
      else
      {
        printf("High pressure: %.0fBar ",Prs3(0x14));
        printf("Low pressure: %.0fBar\n",Prs4(0x15));
      }

      printf("###### AO-40 IHU-1 Telemetry Display -"
	     "-- End of A/E Block ######\n");

   break;
    default:
      for (row = 0; row < 8; row++)
      {
	int col;
	char c;
	for (col = 0; col < 64; col++) {
	  c = block[row*64+col];
	  if (isprint(c)) putchar(c); else putchar('.');
	}
	putchar('\n');
      }
  }
  fflush(stdout);
}


/* Main program */

int main(int argc, char **argv)
{
  char block[512];

  if (argc != 1)
  {
    fprintf(stderr, "Usage: [z]cat file | %s\n", argv[0]);
    return 1;
  }

  while (!feof(stdin))
    if (fread(block, 512, 1, stdin) == 1)
      display_block(block);

   return 0;
}

