/* kasbar.h
**
** Copyright (C) 2001-2004 Richard Moore <rich@kde.org>
** Contributor: Mosfet
**     All rights reserved.
**
** KasBar is dual-licensed: you can choose the GPL or the BSD license.
** Short forms of both licenses are included below.
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
** MA 02111-1307, USA.
*/

/*
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

/*
** Bug reports and questions can be sent to kde-devel@kde.org
*/
// -*- c++ -*-


#ifndef __KASBAR_H
#define __KASBAR_H

#include <qwidget.h>
#include <qpoint.h>
#include <qptrlist.h>

#include <kpixmap.h>

class KasItem;
class KRootPixmap;

typedef QPtrList<KasItem> KasItemList;

/**
 * The main view for KasBar.
 */
class KasBar : public QWidget
{
   Q_OBJECT

   friend class KasItem;
public:
   KasBar( Orientation o, QWidget* parent = 0, const char* name = 0, WFlags f = 0 );
   virtual ~KasBar();

   //
   // Item management
   //
   void append( KasItem *i );
   void insert( int index, KasItem *i );
   void remove( KasItem *i );
   void clear();
   KasItem *itemAt( uint i ) { return items.at( i ); }
   int indexOf( KasItem *i ) { return items.find( i ); }

   //
   // Layout options.
   //

   /**
    * The possible item sizes.
    */
   enum ItemSize { Large, Medium, Small };

   int itemSize() const { return itemSize_; }
   int itemExtent() const { return itemExtent_; }

   /**
    * The number of items in the bar.
    */
   unsigned int itemCount() const { return items.count(); }

   int maxBoxes() const { return maxBoxes_; }
   uint boxesPerLine() const { return boxesPerLine_; }

   void setOrientation( Orientation o ) { orient = o; updateLayout(); }
   Orientation orientation() { return orient; }

   QSize sizeHint( Orientation,  QSize max );

   //
   // Look and feel options
   //

   /**
    *  Is transparency enabled?
    */
   bool isTransparent() const { return transparent_; }

   /**
    * Is tinting enabled?
    */
   bool hasTint() const { return enableTint_; }

   /**
    * Sets the amount and color of the tint.
    */
   void setTint( double amount, QColor color );

   /**
    * Sets the amount of tinting.
    */
   void setTintAmount( double amount ) { setTint( amount, tintColour_ ); }

   /**
    * Get the amount of tinting.
    */
   double tintAmount() const { return tintAmount_; }

   /**
    * Get the color of the tint.
    */
   QColor tintColor() const { return tintColour_; }

   QColor labelPenColor() const { return labelPenColor_; }
   QColor labelBgColor() const { return labelBgColor_; }
   QColor inactivePenColor() const { return inactivePenColor_; }
   QColor inactiveBgColor() const { return inactiveBgColor_; }
   QColor activePenColor() const { return activePenColor_; }
   QColor activeBgColor() const { return activeBgColor_; }
   QColor progressColor() const { return progressColor_; }

   //
   // Utilities
   //

   void updateItem( KasItem *i );

   /**
    * Redraws the specified item.
    */
   void repaintItem(KasItem *i, bool erase = true );

   /**
    * Returns the item at p or 0.
    */
   KasItem* itemAt(const QPoint &p);

   /**
    * Get the position of the specified item.
    */
   QPoint itemPos( KasItem *i );

   /**
    * The item under the mouse pointer (or 0).
    */
   KasItem *itemUnderMouse() const { return itemUnderMouse_; }

public slots:
   //
   // Layout slots
   //
   void setMaxBoxes( int count );
   void setBoxesPerLine( int count );

   void setItemSize( int size );
   virtual void updateLayout();

   void updateMouseOver();
   void updateMouseOver( QPoint pos );

   /**
    * Enable or disable tinting.
    */
   void setTint( bool enable );

   /**
    * Enable or disable transparency.
    */
   void setTransparent( bool enable );

   /**
    * Set the color of the tint.
    */
   void setTintColor( const QColor &c );

   /**
    * Set the strength of the tint (as a percentage).
    */
   void setTintAmount( int percent );

   void setBackground( const QPixmap &pix );

   void setLabelPenColor( const QColor &color );
   void setLabelBgColor( const QColor &color );
   void setInactivePenColor( const QColor &color );
   void setInactiveBgColor( const QColor &color );
   void setActivePenColor( const QColor &color );
   void setActiveBgColor( const QColor &color );
   void setProgressColor( const QColor &color );

signals:
   /**
    * Emitted when kasbar wants to resize. This happens when a new window is added.
    */
   void layoutChanged();

   /**
    * Emitted when the item size is changed.
    */
   void itemSizeChanged( int );

   void configChanged();

protected:
    KRootPixmap *rootPixmap() const { return rootPix; }

   /**
    * Displays the popup menus, hides/shows windows.
    */
   void mousePressEvent(QMouseEvent *ev);

   /**
    * Overridden to implement the mouse-over highlight effect.
    */
   void mouseMoveEvent(QMouseEvent *ev);

   /**
    * Overridden to implement the drag-over task switching.
    */
   void dragMoveEvent(QDragMoveEvent *ev);

   /**
    * Paints the background of the item to the painter.
    */
   void paintBackground( QPainter *p, const QRect &r );

   /**
    * Calls the paint methods for the items in the rectangle specified by the event.
    */
   void paintEvent(QPaintEvent *ev);

   /**
    * Forces the widget to re-layout it's contents.
    */
   void resizeEvent(QResizeEvent *ev);

private:
   /**
    * Internal factory method used by items to get the active bg fill.
    */
   KPixmap *activeBg();

   /**
    * Internal factory method used by items to get the inactive bg fill.
    */
   KPixmap *inactiveBg();

private:
   // Core data
   KasItemList items;
   Orientation orient;
   KasItem *itemUnderMouse_;
   uint boxesPerLine_;
   int maxBoxes_;
   int itemSize_;
   int itemExtent_;

   // Implements pseudo-transparency
   bool transparent_;
   KPixmap bg;
   KRootPixmap *rootPix;
   bool enableTint_;
   double tintAmount_;
   QColor tintColour_;

   // Look and feel resources
   KPixmap *actBg;
   KPixmap *inactBg;
   QColor labelPenColor_;
   QColor labelBgColor_;
   QColor activePenColor_;
   QColor activeBgColor_;
   QColor inactivePenColor_;
   QColor inactiveBgColor_;
   QColor progressColor_;
};



#endif
