/*
        This program is free software; you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation; You may only use version 2 of the License,
        you have no option to use any other version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program; if not, write to the Free Software
        Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

        gnome theme-switcher capplet - (c) Jonathan Blandford <jrb@gnome.org>
        xfce4 mcs plugin   - (c) 2002 Olivier Fourdan

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gtk/gtk.h>

#include <libxfce4mcs/mcs-common.h>
#include <libxfce4mcs/mcs-manager.h>
#include <libxfce4util/i18n.h>
#include <libxfce4util/util.h>
#include <xfce-mcs-manager/manager-plugin.h>
#include <libxfcegui4/libxfcegui4.h>

#include "plugins/gtk_common/gtk_common.h"
#include "inline-icon.h"

#define RCDIR   "settings"
#define RCFILE  "gtk.xml"
#define CHANNEL "settings"
#define PLUGIN_NAME "ui"
#define DEFAULT_THEME "Default"
#define INITIAL_THEME "Xfce"
#define DEFAULT_TOOLBAR_STYLE "icons"
#define DEFAULT_FONT "Sans 10"

#define DEFAULT_ICON_SIZE 48
#define MAX_ELEMENTS_BEFORE_SCROLLING 6

static void create_channel(McsPlugin * mcs_plugin);
static gboolean write_options(McsPlugin * mcs_plugin);
static void run_dialog(McsPlugin * mcs_plugin);

static gboolean setting_model = FALSE;
static gboolean initial_scroll = TRUE;
static gboolean is_running = FALSE;
static gchar *current_theme = NULL;
static gchar *current_font = NULL;
static gchar *current_toolbar_style = NULL;

typedef struct _Itf Itf;
struct _Itf
{
    McsPlugin *mcs_plugin;

    GtkWidget *theme_dialog;
    GtkWidget *dialog_vbox1;
    GtkWidget *dialog_header;
    GtkWidget *hbox1;
    GtkWidget *frame1;
    GtkWidget *vbox2;
    GtkWidget *hbox2;
    GtkWidget *theme_swindow;
    GtkWidget *theme_treeview;
    AtkObject *atko;
    GtkWidget *label1;
    GtkWidget *vbox3;
    GtkWidget *frame2;
    GtkWidget *button3;
    GtkWidget *frame3;
    GtkWidget *optionmenu1;
    GtkWidget *menu1;
    GtkWidget *menuitem1;
    GtkWidget *menuitem2;
    GtkWidget *menuitem3;
    GtkWidget *menuitem4;
    GtkWidget *dialog_action_area1;
    GtkWidget *helpbutton1;
    GtkWidget *closebutton1;

    GtkWidget *font_selection;
};

static void theme_selection_changed(GtkTreeSelection * selection, gpointer data)
{
    GtkTreeModel *model;
    gchar *new_key;
    GtkTreeIter iter;
    McsPlugin *mcs_plugin;

    if(setting_model)
        return;

    mcs_plugin = (McsPlugin *) data;

    if(gtk_tree_selection_get_selected(selection, &model, &iter))
    {
        gtk_tree_model_get(model, &iter, THEME_NAME_COLUMN, &new_key, -1);
    }
    else
    {
        new_key = NULL;
    }

    if(new_key != NULL)
    {
        if(current_theme && strcmp(current_theme, new_key))
        {
            g_free(current_theme);
            current_theme = new_key;
            mcs_manager_set_string(mcs_plugin->manager, "Net/ThemeName", CHANNEL, current_theme);
            mcs_manager_notify(mcs_plugin->manager, CHANNEL);
            write_options(mcs_plugin);
        }
    }
}

static void read_themes(Itf * dialog)
{
    static GList *gtk_theme_list = NULL;
    GList *list;
    GtkTreeModel *model;
    GtkTreeView *tree_view;
    gint i = 0;
    gboolean current_theme_found = FALSE;
    GtkTreeRowReference *row_ref = NULL;

    gtk_theme_list = theme_common_get_list(gtk_theme_list);
    tree_view = GTK_TREE_VIEW(dialog->theme_treeview);
    model = gtk_tree_view_get_model(tree_view);

    setting_model = TRUE;
    gtk_list_store_clear(GTK_LIST_STORE(model));

    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dialog->theme_swindow), GTK_POLICY_NEVER, GTK_POLICY_NEVER);
    gtk_widget_set_size_request(dialog->theme_swindow, -1, -1);

    for(list = gtk_theme_list; list; list = list->next)
    {
        ThemeInfo *info = list->data;
        GtkTreeIter iter;

        if(!info->has_gtk)
            continue;

        gtk_list_store_prepend(GTK_LIST_STORE(model), &iter);
        gtk_list_store_set(GTK_LIST_STORE(model), &iter, THEME_NAME_COLUMN, info->name, -1);

        if(strcmp(current_theme, info->name) == 0)
        {
            GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
            row_ref = gtk_tree_row_reference_new(model, path);
            gtk_tree_path_free(path);
            current_theme_found = TRUE;
        }

        if(i == MAX_ELEMENTS_BEFORE_SCROLLING)
        {
            GtkRequisition rectangle;
            gtk_widget_size_request(GTK_WIDGET(tree_view), &rectangle);
            gtk_widget_set_size_request(dialog->theme_swindow, -1, rectangle.height);
            gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dialog->theme_swindow), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
        }
        i++;
    }

    if(!current_theme_found)
    {
        GtkTreeIter iter;
        GtkTreePath *path;

        gtk_list_store_prepend(GTK_LIST_STORE(model), &iter);
        gtk_list_store_set(GTK_LIST_STORE(model), &iter, THEME_NAME_COLUMN, DEFAULT_THEME, -1);

        path = gtk_tree_model_get_path(model, &iter);
        row_ref = gtk_tree_row_reference_new(model, path);
        gtk_tree_path_free(path);
    }

    if(row_ref)
    {
        GtkTreePath *path;

        path = gtk_tree_row_reference_get_path(row_ref);
        gtk_tree_view_set_cursor(tree_view, path, NULL, FALSE);

        if(initial_scroll)
        {
            gtk_tree_view_scroll_to_cell(tree_view, path, NULL, TRUE, 0.5, 0.0);
            initial_scroll = FALSE;
        }

        gtk_tree_path_free(path);
        gtk_tree_row_reference_free(row_ref);
    }
    setting_model = FALSE;
}

static void font_selection_ok(GtkWidget * w, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    gchar *new_font = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(itf->font_selection));
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    if(new_font != NULL)
    {
        if(current_font && strcmp(current_font, new_font))
        {
            g_free(current_font);
            current_font = new_font;
            gtk_button_set_label(GTK_BUTTON(itf->button3), current_font);
            mcs_manager_set_string(mcs_plugin->manager, "Gtk/FontName", CHANNEL, current_font);
            mcs_manager_notify(mcs_plugin->manager, CHANNEL);
            write_options(mcs_plugin);
        }
    }

    gtk_widget_destroy(GTK_WIDGET(itf->font_selection));
    itf->font_selection = NULL;
}

static void show_font_selection(GtkWidget * widget, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;

    if(!(itf->font_selection))
    {
        itf->font_selection = gtk_font_selection_dialog_new(_("Font Selection Dialog"));

        gtk_window_set_position(GTK_WINDOW(itf->font_selection), GTK_WIN_POS_MOUSE);
        gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(itf->font_selection), current_font);
        g_signal_connect(itf->font_selection, "destroy", G_CALLBACK(gtk_widget_destroyed), &itf->font_selection);

        g_signal_connect(GTK_FONT_SELECTION_DIALOG(itf->font_selection)->ok_button, "clicked", G_CALLBACK(font_selection_ok), user_data);
        g_signal_connect_swapped(GTK_FONT_SELECTION_DIALOG(itf->font_selection)->cancel_button, "clicked", G_CALLBACK(gtk_widget_destroy), itf->font_selection);

        gtk_widget_show(itf->font_selection);
    }
    else
    {
        gtk_widget_destroy(itf->font_selection);
        itf->font_selection = NULL;
    }
}

static gint sort_func(GtkTreeModel * model, GtkTreeIter * a, GtkTreeIter * b, gpointer user_data)
{
    gchar *a_str = NULL;
    gchar *b_str = NULL;

    gtk_tree_model_get(model, a, 0, &a_str, -1);
    gtk_tree_model_get(model, b, 0, &b_str, -1);

    if(a_str == NULL)
        a_str = g_strdup("");
    if(b_str == NULL)
        b_str = g_strdup("");

    if(!strcmp(a_str, DEFAULT_THEME))
        return -1;
    if(!strcmp(b_str, DEFAULT_THEME))
        return 1;

    return g_utf8_collate(a_str, b_str);
}

static void cb_dialog_response(GtkWidget * dialog, gint response_id)
{
    if(response_id == GTK_RESPONSE_HELP)
    {
        g_message("HELP: TBD");
    }
    else
    {
        is_running = FALSE;
        gtk_widget_destroy(dialog);
    }
}

static void on_icons_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    if(current_toolbar_style && strcmp(current_toolbar_style, "icons"))
    {
        g_free(current_toolbar_style);
        current_toolbar_style = g_strdup("icons");
        mcs_manager_set_string(mcs_plugin->manager, "Gtk/ToolbarStyle", CHANNEL, current_toolbar_style);
        mcs_manager_notify(mcs_plugin->manager, CHANNEL);
        write_options(mcs_plugin);
    }
}


static void on_text_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    if(current_toolbar_style && strcmp(current_toolbar_style, "text"))
    {
        g_free(current_toolbar_style);
        current_toolbar_style = g_strdup("text");
        mcs_manager_set_string(mcs_plugin->manager, "Gtk/ToolbarStyle", CHANNEL, current_toolbar_style);
        mcs_manager_notify(mcs_plugin->manager, CHANNEL);
        write_options(mcs_plugin);
    }
}


static void on_both_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    if(current_toolbar_style && strcmp(current_toolbar_style, "both"))
    {
        g_free(current_toolbar_style);
        current_toolbar_style = g_strdup("both");
        mcs_manager_set_string(mcs_plugin->manager, "Gtk/ToolbarStyle", CHANNEL, current_toolbar_style);
        mcs_manager_notify(mcs_plugin->manager, CHANNEL);
        write_options(mcs_plugin);
    }
}


static void on_both_horiz_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    if(current_toolbar_style && strcmp(current_toolbar_style, "both-horiz"))
    {
        g_free(current_toolbar_style);
        current_toolbar_style = g_strdup("both-horiz");
        mcs_manager_set_string(mcs_plugin->manager, "Gtk/ToolbarStyle", CHANNEL, current_toolbar_style);
        mcs_manager_notify(mcs_plugin->manager, CHANNEL);
        write_options(mcs_plugin);
    }
}

Itf *create_theme_dialog(McsPlugin * mcs_plugin)
{
    Itf *dialog;
    GdkPixbuf *icon;

    dialog = g_new(Itf, 1);

    dialog->mcs_plugin = mcs_plugin;

    dialog->theme_dialog = gtk_dialog_new();

    dialog->font_selection = NULL;

    icon = inline_icon_at_size(default_icon_data, 32, 32);
    gtk_window_set_icon(GTK_WINDOW(dialog->theme_dialog), icon);
    g_object_unref(icon);

    gtk_window_set_title(GTK_WINDOW(dialog->theme_dialog), _("Theme Preferences"));
    gtk_window_set_default_size(GTK_WINDOW(dialog->theme_dialog), 320, 200);
    gtk_dialog_set_has_separator(GTK_DIALOG(dialog->theme_dialog), FALSE);

    dialog->dialog_vbox1 = GTK_DIALOG(dialog->theme_dialog)->vbox;
    gtk_widget_show(dialog->dialog_vbox1);

    dialog->dialog_header = create_header(icon, _("User Interface Preferences"));
    gtk_widget_show(dialog->dialog_header);
    gtk_box_pack_start(GTK_BOX(dialog->dialog_vbox1), dialog->dialog_header, FALSE, TRUE, 0);

    dialog->hbox1 = gtk_hbox_new(FALSE, 5);
    gtk_widget_show(dialog->hbox1);
    gtk_box_pack_start(GTK_BOX(dialog->dialog_vbox1), dialog->hbox1, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(dialog->hbox1), 5);

    dialog->frame1 = xfce_framebox_new (_("Select theme"), FALSE);
    gtk_widget_show(dialog->frame1);
    gtk_box_pack_start(GTK_BOX(dialog->hbox1), dialog->frame1, TRUE, TRUE, 0);

    dialog->vbox2 = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(dialog->vbox2);
    xfce_framebox_add (XFCE_FRAMEBOX (dialog->frame1), dialog->vbox2);

    dialog->hbox2 = gtk_hbox_new(FALSE, 8);
    gtk_widget_show(dialog->hbox2);
    gtk_box_pack_start(GTK_BOX(dialog->vbox2), dialog->hbox2, TRUE, TRUE, 0);

    dialog->theme_swindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(dialog->theme_swindow);
    gtk_box_pack_start(GTK_BOX(dialog->hbox2), dialog->theme_swindow, TRUE, TRUE, 0);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(dialog->theme_swindow), GTK_SHADOW_IN);

    dialog->theme_treeview = gtk_tree_view_new();
    gtk_widget_show(dialog->theme_treeview);
    gtk_container_add(GTK_CONTAINER(dialog->theme_swindow), dialog->theme_treeview);
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(dialog->theme_treeview), FALSE);

    dialog->vbox3 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(dialog->vbox3);
    gtk_box_pack_start(GTK_BOX(dialog->hbox1), dialog->vbox3, TRUE, TRUE, 0);

    dialog->frame2 = xfce_framebox_new (_("Font"), TRUE);
    gtk_widget_show(dialog->frame2);
    gtk_box_pack_start(GTK_BOX(dialog->vbox3), dialog->frame2, TRUE, FALSE, 0);

    dialog->button3 = gtk_button_new();
    gtk_button_set_label(GTK_BUTTON(dialog->button3), current_font);
    gtk_widget_show(dialog->button3);
    xfce_framebox_add (XFCE_FRAMEBOX (dialog->frame2), dialog->button3);

    dialog->frame3 = xfce_framebox_new (_("Toolbar Style"), TRUE);
    gtk_widget_show(dialog->frame3);
    gtk_box_pack_start(GTK_BOX(dialog->vbox3), dialog->frame3, TRUE, FALSE, 0);

    dialog->optionmenu1 = gtk_option_menu_new();
    gtk_widget_show(dialog->optionmenu1);
    xfce_framebox_add (XFCE_FRAMEBOX (dialog->frame3), dialog->optionmenu1);

    dialog->menu1 = gtk_menu_new();

    dialog->menuitem1 = gtk_menu_item_new_with_mnemonic(_("Icons"));
    gtk_widget_show(dialog->menuitem1);
    gtk_container_add(GTK_CONTAINER(dialog->menu1), dialog->menuitem1);

    dialog->menuitem2 = gtk_menu_item_new_with_mnemonic(_("Text"));
    gtk_widget_show(dialog->menuitem2);
    gtk_container_add(GTK_CONTAINER(dialog->menu1), dialog->menuitem2);

    dialog->menuitem3 = gtk_menu_item_new_with_mnemonic(_("Both"));
    gtk_widget_show(dialog->menuitem3);
    gtk_container_add(GTK_CONTAINER(dialog->menu1), dialog->menuitem3);

    dialog->menuitem4 = gtk_menu_item_new_with_mnemonic(_("Both horizontal"));
    gtk_widget_show(dialog->menuitem4);
    gtk_container_add(GTK_CONTAINER(dialog->menu1), dialog->menuitem4);

    gtk_option_menu_set_menu(GTK_OPTION_MENU(dialog->optionmenu1), dialog->menu1);

    if(!strcmp(current_toolbar_style, "text"))
    {
        gtk_option_menu_set_history(GTK_OPTION_MENU(dialog->optionmenu1), 1);
    }
    else if(!strcmp(current_toolbar_style, "both"))
    {
        gtk_option_menu_set_history(GTK_OPTION_MENU(dialog->optionmenu1), 2);
    }
    else if(!strcmp(current_toolbar_style, "both-horiz"))
    {
        gtk_option_menu_set_history(GTK_OPTION_MENU(dialog->optionmenu1), 3);
    }
    else
    {
        gtk_option_menu_set_history(GTK_OPTION_MENU(dialog->optionmenu1), 0);
    }

    dialog->dialog_action_area1 = GTK_DIALOG(dialog->theme_dialog)->action_area;
    gtk_widget_show(dialog->dialog_action_area1);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog->dialog_action_area1), GTK_BUTTONBOX_END);

    dialog->helpbutton1 = gtk_button_new_from_stock("gtk-help");
    /* gtk_widget_show (helpbutton1); */
    gtk_dialog_add_action_widget(GTK_DIALOG(dialog->theme_dialog), dialog->helpbutton1, GTK_RESPONSE_HELP);
    GTK_WIDGET_SET_FLAGS(dialog->helpbutton1, GTK_CAN_DEFAULT);

    dialog->closebutton1 = gtk_button_new_from_stock("gtk-close");
    gtk_widget_show(dialog->closebutton1);
    gtk_dialog_add_action_widget(GTK_DIALOG(dialog->theme_dialog), dialog->closebutton1, GTK_RESPONSE_CLOSE);
    GTK_WIDGET_SET_FLAGS(dialog->closebutton1, GTK_CAN_DEFAULT);

    dialog->atko = gtk_widget_get_accessible(dialog->theme_treeview);
    atk_object_set_description(dialog->atko, _("List of available GTK+ themes"));

    gtk_widget_grab_focus(dialog->closebutton1);
    gtk_widget_grab_default(dialog->closebutton1);

    return dialog;
}

static void setup_dialog(Itf * itf)
{
    GtkTreeModel *model;
    GtkTreeSelection *selection;

    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(itf->theme_treeview), -1, NULL, gtk_cell_renderer_text_new(), "text", THEME_NAME_COLUMN, NULL);
    model = (GtkTreeModel *) gtk_list_store_new(N_COLUMNS, G_TYPE_STRING);
    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(model), 0, sort_func, NULL, NULL);
    gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model), 0, GTK_SORT_ASCENDING);
    gtk_tree_view_set_model(GTK_TREE_VIEW(itf->theme_treeview), model);
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(itf->theme_treeview));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
    g_signal_connect(G_OBJECT(selection), "changed", (GCallback) theme_selection_changed, itf->mcs_plugin);

    read_themes(itf);

    g_signal_connect(G_OBJECT(itf->theme_dialog), "response", G_CALLBACK(cb_dialog_response), itf->mcs_plugin);
    g_signal_connect(G_OBJECT(itf->button3), "clicked", G_CALLBACK(show_font_selection), itf);
    g_signal_connect(G_OBJECT(itf->menuitem1), "activate", G_CALLBACK(on_icons_activate), itf);
    g_signal_connect(G_OBJECT(itf->menuitem2), "activate", G_CALLBACK(on_text_activate), itf);
    g_signal_connect(G_OBJECT(itf->menuitem3), "activate", G_CALLBACK(on_both_activate), itf);
    g_signal_connect(G_OBJECT(itf->menuitem4), "activate", G_CALLBACK(on_both_horiz_activate), itf);

    gtk_window_set_position(GTK_WINDOW(itf->theme_dialog), GTK_WIN_POS_CENTER);
    gtk_widget_show(itf->theme_dialog);
}

McsPluginInitResult mcs_plugin_init(McsPlugin * mcs_plugin)
{
#if 0
#ifdef ENABLE_NLS
    /* This is required for UTF-8 at least - Please don't remove it  */
    bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
    textdomain (GETTEXT_PACKAGE);
#endif
#else
    xfce_textdomain(GETTEXT_PACKAGE, LOCALEDIR, "UTF-8");
#endif

    create_channel(mcs_plugin);
    mcs_plugin->plugin_name = g_strdup(PLUGIN_NAME);
    mcs_plugin->caption = g_strdup(_("User interface"));
    mcs_plugin->run_dialog = run_dialog;
    mcs_plugin->icon = inline_icon_at_size(default_icon_data, DEFAULT_ICON_SIZE, DEFAULT_ICON_SIZE);
    mcs_manager_notify(mcs_plugin->manager, CHANNEL);
    return (MCS_PLUGIN_INIT_OK);
}

static void create_channel(McsPlugin * mcs_plugin)
{
    McsSetting *setting;
    gchar *rcfile;
    
    rcfile = xfce_get_userfile(RCDIR, RCFILE, NULL);
    mcs_manager_add_channel_from_file(mcs_plugin->manager, CHANNEL, rcfile);
    g_free(rcfile);

    setting = mcs_manager_setting_lookup(mcs_plugin->manager, "Net/ThemeName",
                    CHANNEL);
    if(setting)
    {
        if(current_theme)
        {
            g_free(current_theme);
        }
        current_theme = g_strdup(setting->data.v_string);
    }
    else
    {
        if(current_theme)
        {
            g_free(current_theme);
        }

        current_theme = g_strdup(INITIAL_THEME);
        mcs_manager_set_string(mcs_plugin->manager, "Net/ThemeName", CHANNEL, current_theme);
    }

    setting = mcs_manager_setting_lookup(mcs_plugin->manager, "Gtk/FontName", CHANNEL);
    if(setting)
    {
        if(current_font)
        {
            g_free(current_font);
        }
        current_font = g_strdup(setting->data.v_string);
    }
    else
    {
        if(current_font)
        {
            g_free(current_font);
        }

        current_font = g_strdup(DEFAULT_FONT);
        mcs_manager_set_string(mcs_plugin->manager, "Gtk/FontName", CHANNEL, current_font);
    }

    setting = mcs_manager_setting_lookup(mcs_plugin->manager, "Gtk/ToolbarStyle", CHANNEL);
    if(setting)
    {
        if(current_toolbar_style)
        {
            g_free(current_toolbar_style);
        }
        current_toolbar_style = g_strdup(setting->data.v_string);
    }
    else
    {
        if(current_toolbar_style)
        {
            g_free(current_toolbar_style);
        }

        current_toolbar_style = g_strdup(DEFAULT_TOOLBAR_STYLE);
        mcs_manager_set_string(mcs_plugin->manager, "Gtk/ToolbarStyle", CHANNEL, current_toolbar_style);
    }
}

static gboolean
write_options(McsPlugin *mcs_plugin)
{
    gboolean result;
    gchar *rcfile;

    rcfile = xfce_get_userfile(RCDIR, RCFILE, NULL);
    result = mcs_manager_save_channel_to_file(mcs_plugin->manager, CHANNEL,
                    rcfile);
    g_free(rcfile);

    return(result);
}

static void
run_dialog(McsPlugin *mcs_plugin)
{
    Itf *dialog;

    if (!is_running) {
        is_running = TRUE;

        dialog = create_theme_dialog(mcs_plugin);
        setup_dialog(dialog);
    }
}

/* macro defined in manager-plugin.h */
MCS_PLUGIN_CHECK_INIT
