/*
 *  Copyright (C) 2008 Nicolas Vion <nico@picapo.net>
 *
 *   This file is part of swac-explore.
 *
 *   Swac-get is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   Swac-get is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with swac-explore; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <iostream>
#include <string>

#include "sqlite3++.hh"

std::string SqlDatabase::str(std::string str) {
	char formated[512];
	sqlite3_snprintf (512, formated, "'%q'", str.c_str());
	return formated;
}

void SqlDatabase::errorMsg() {
	std::cerr << "SQLite error: " << sqlite3_errmsg(handle) << std::endl;
}

bool SqlDatabase::open(std::string const path) {
	int rc = sqlite3_open(path.c_str(), &handle);
	if (rc){
		errorMsg();
		sqlite3_close(handle);
	}
	return (rc == SQLITE_OK);
}

bool SqlDatabase::close() {
	sqlite3_close(handle);
	return true;
}

bool SqlDatabase::exec(std::string const sql, SqlQueryResult &result) {
	SqlQuery query(this);
	if (query.prepare(sql) == SQLITE_OK) {
		query.exec(result);
		return true;
	}
	return 
		false;
}

bool SqlDatabase::exec(std::string const sql,  int (*callback)(void*,int,char**,char**), void *params) {
	char *zErrMsg = 0;
	int rc;

	rc = sqlite3_exec(handle, sql.c_str(), callback, params, &zErrMsg);

	if (rc != SQLITE_OK) {
		fprintf(stderr, "SQL error: %s\n", zErrMsg);
		sqlite3_free(zErrMsg);
		return false;
	}
	return true;
}

bool SqlDatabase::exec(std::string const sql) {
	return exec(sql, NULL, NULL);
}

int SqlDatabase::lastInsertRowId() {
	return sqlite3_last_insert_rowid(handle);
}

int SqlDatabase::changes() {
	return sqlite3_changes(handle);
}

SqlQuery::SqlQuery(SqlDatabase *parent) {
	database = parent;
}

SqlQuery::~SqlQuery() {
	sqlite3_finalize(handle);
}

int SqlQuery::prepare(std::string const sql) {
	char const *unused;
	int res = sqlite3_prepare(database->handle, sql.c_str(), -1, &handle, &unused);
	if (res != SQLITE_OK)
		database->errorMsg();
	return res;
}

int SqlQuery::step() {
	return sqlite3_step(handle);
}

int SqlQuery::reset() {
	return sqlite3_reset(handle);
}

int SqlQuery::bind(std::string const value, int const n) {
	return sqlite3_bind_text(handle, n, value.c_str(), -1, SQLITE_TRANSIENT);
}

int SqlQuery::clearBindings() {
	return sqlite3_clear_bindings(handle);
}

int SqlQuery::bindParameterCount() {
	return sqlite3_bind_parameter_count(handle);
}

int SqlQuery::columnCount() {
	return sqlite3_column_count(handle);
}

std::string SqlQuery::columnName(int const iCol) {
	char const *name;
	name = sqlite3_column_name(handle, iCol);
	if (name != NULL)
		return name;
	else
		return "";
}

std::string SqlQuery::columnText(int const iCol) {
	unsigned char const *str;
	str = sqlite3_column_text(handle, iCol);
	if (str != NULL)
		return std::string((char const*) str);
	else
		return "";
}

int SqlQuery::columnInt(int const iCol) {
	return sqlite3_column_int(handle, iCol);
}

double SqlQuery::columnDouble(int const iCol) {
	return sqlite3_column_double(handle, iCol);
}

bool SqlQuery::exec(SqlQueryResult &result) {
	SqlQueryResultRaw raw;
	int res;
	while ((res = step()) == SQLITE_ROW) {
		raw.clear();
		for (int i = 0; i < columnCount(); i++)		
			raw.push_back(columnText(i));
		result.push_back(raw);
	}
	if (res != SQLITE_DONE)
		database->errorMsg();
	return (res == SQLITE_DONE);
}
