/*
 *   Copyright 2009 - 2010 by Giulio Camuffo <giuliocamuffo@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "playwolf.h"

///Qt includes
#include <QVariantMap>
#include <QtGui/QGraphicsGridLayout>
#include <QtGui/QGraphicsSceneHoverEvent>
#include <QtGui/QGraphicsSceneWheelEvent>

///KDE includes
#include <KConfigDialog>
#include <KIcon>

///Plasma includes
#include <Plasma/ExtenderItem>
#include <Plasma/Extender>
#include <Plasma/TextEdit>

///PlayWolf includes
#include "infomanager.h"
#include "wolfbuttons.h"
#include "playercontroller.h"
#include "infowidget.h"
#include "wolfmeter.h"
#include "wolflabel.h"
#include "dialogwidget.h"
#include "lyricswidget.h"

PlayWolf::PlayWolf(QObject *parent, const QVariantList &args)
    : Plasma::PopupApplet(parent, args)
{
    setBackgroundHints(StandardBackground);

    resize(200,200);
}

PlayWolf::~PlayWolf()
{
}

void PlayWolf::init()
{
    KConfigGroup cg = config();
    m_widthOnPanel = cg.readEntry("widthOnPanel", 250); //defines the width of the applet when in a horizontal panel
    m_heightOnPanel = cg.readEntry("heightOnPanel", 150); //defines the height of the applet when in a vertical panel
    bool amarokButtonShown = cg.readEntry("amarokButtonShown", true);
    m_showCover = cg.readEntry("showCover", true);
    m_wheelSeek = cg.readEntry("wheelSeek", true);
    bool resizeFont = cg.readEntry("resizeFont", true);
    int minFont = cg.readEntry("minFont", 6.0);
    bool useCustomFont = cg.readEntry("useCustomFont", false);
    m_customFont = cg.readEntry("customFont", QFont());
    m_showInformationsOnHPanel = cg.readEntry("showInformationsOnHPanel", true);
    QString amarokBinPath = cg.readEntry("customAmarokBinPath", "/usr/bin/amarok");

    m_layout = new QGraphicsGridLayout(this);
    m_layout->setContentsMargins(0,0,0,0);
    m_layout->setSpacing(0);
    setLayout(m_layout);

    m_infoManager = new InfoManager(this);
    m_buttons = new WolfButtons(this);
    m_infos = new InfoWidget(this);
    m_meter = new WolfMeter(this);
    m_cover = new WolfLabel(this);

    m_infoManager->registerInfoWidget(m_infos);
    m_infoManager->registerButtonsWidget(m_buttons);
    m_infoManager->registerCoverWidget(m_cover);
    m_infoManager->registerMeterWidget(m_meter);

    m_meter->setMinimum(0);
    m_meter->setMinimumHeight(5);
    m_meter->setMinimumWidth(5);
    m_buttons->setZValue(5);

    m_buttons->setIsAmarokButtonVisible(amarokButtonShown);
    m_infos->setResizeFont(resizeFont, minFont);

    if (useCustomFont) {
        m_infos->setFont(m_customFont);
    }

    Plasma::Extender *ex = extender();
    if (!ex->hasItem("info")) {
        Plasma::ExtenderItem *infosItem = new Plasma::ExtenderItem(ex);
        infosItem->setName("info");
        initExtenderItem(infosItem);
    }
    if (!ex->hasItem("misc")) {
        Plasma::ExtenderItem *miscItem = new Plasma::ExtenderItem(ex);
        miscItem->setName("misc");
        initExtenderItem(miscItem);
    }
    if (!ex->hasItem("lyrics")) {
        Plasma::ExtenderItem *lyricsItem = new Plasma::ExtenderItem(ex);
        lyricsItem->setName("lyrics");
        initExtenderItem(lyricsItem);
    }

    PlayerController::self()->setAmarokPath(amarokBinPath);
    PlayerController::self()->updateInfos();
}

void PlayWolf::constraintsEvent(Plasma::Constraints constraints)
{
    if (formFactor() == Plasma::Horizontal) {
        if (constraints >= (Plasma::SizeConstraint | Plasma::LocationConstraint)) {
            resetWidgets();

            m_meter->setMeterType(Plasma::Meter::BarMeterHorizontal);
            m_meter->setMaximumHeight(6);

            m_layout->addItem(m_meter,1,1,1,1);
            if (m_showInformationsOnHPanel) {
                m_infos->show();
                m_layout->addItem(m_infos,0,1,1,1);
            } else {
                m_buttons->show();
                m_layout->addItem(m_buttons,0,1,1,1);
            }

            if (m_showCover) {
                m_cover->show();
                m_layout->addItem(m_cover,0,0,2,1);
            }

        }
        if (m_showInformationsOnHPanel) {
            m_buttons->setMaximumHeight(20);
            m_buttons->resize(m_infos->contentsRect().size());
            m_buttons->setPos(m_infos->x(),m_infos->y() + (contentsRect().height() - m_buttons->contentsRect().height())/2.0);
        }
    }
    else if (formFactor() == Plasma::Vertical) {
        if (constraints > (Plasma::SizeConstraint | Plasma::LocationConstraint)) {
            resetWidgets();

            m_meter->setMeterType(Plasma::Meter::BarMeterVertical);

            m_layout->addItem(m_buttons,1,0,1,1);
            m_layout->addItem(m_meter,1,1,1,1);
            if (m_showCover) {
                m_cover->show();
                m_layout->addItem(m_cover,0,0,1,2);
            }

            m_buttons->show();
        }
        if (m_showCover) {
            m_cover->setMinimumHeight(size().width());
        } else {
            m_cover->setMinimumHeight(-1);
        }
    }
    else if (formFactor() == Plasma::Planar) {
        if (constraints > (Plasma::SizeConstraint | Plasma::LocationConstraint)) {
            resetWidgets();

            m_meter->setMeterType(Plasma::Meter::BarMeterHorizontal);

            m_layout->addItem(m_infos,0,1,1,1);
            m_layout->addItem(m_meter,1,0,1,2);
            m_layout->addItem(m_buttons,2,0,1,2);
            if (m_showCover) {
                m_cover->show();
                m_layout->addItem(m_cover,0,0,1,1);
            }
            m_layout->setRowSpacing(1,5);

            m_infos->show();
            m_buttons->show();
        }
        m_meter->setMaximumHeight(contentsRect().height()/6);
        m_buttons->setMaximumHeight(contentsRect().height()/6);
    }
}

void PlayWolf::resetWidgets()
{
    m_meter->setMaximumHeight(-1);
    m_meter->setMinimumHeight(-1);
    m_cover->setMinimumHeight(-1);
    m_buttons->setMaximumHeight(-1);
    m_buttons->setGeometry(0,0,0,0);
    while (m_layout->count() > 0) {
        m_layout->removeAt(0);
    }
    m_infos->hide();
    m_buttons->hide();
    m_cover->hide();
}

void PlayWolf::initExtenderItem(Plasma::ExtenderItem *item)
{
    if (item->name() == QString("misc")) {
        DialogWidget *miscDialog = new DialogWidget(item);
        miscDialog->setPreferredSize(250, 150);
        item->setWidget(miscDialog);
        item->setTitle("Miscellaneous");
        m_infoManager->registerRatingWidget(miscDialog);
        m_infoManager->registerVolumeWidget(miscDialog);
        m_infoManager->registerPlaylistWidget(miscDialog);
    } else if (item->name() == QString("info")) {
        InfoWidget *infosDialog = new InfoWidget(item);
        infosDialog->setPreferredSize(250, 100);
        item->setWidget(infosDialog);
        item->setTitle("Information");
        m_infoManager->registerInfoWidget(infosDialog);
    } else if (item->name() == QString("lyrics")) {
        LyricsWidget *lyricsDialog = new LyricsWidget(item);
        lyricsDialog->setPreferredSize(250, 300);
        item->setWidget(lyricsDialog);
        item->setTitle("Lyrics");
    } else {
        delete item;
        return;
    }
}

void PlayWolf::popupEvent(bool show)
{
//     if (show) {
//         miscDialog->updateContents();
//     }
}

void PlayWolf::createConfigurationInterface(KConfigDialog *parent)
{
    QWidget *generalWidget = new QWidget();
    QWidget *horizontalWidget = new QWidget();
    QWidget *verticalWidget = new QWidget();
    m_generalUi.setupUi(generalWidget);
    m_horizontalUi.setupUi(horizontalWidget);
    m_verticalUi.setupUi(verticalWidget);
    parent->addPage(generalWidget, i18n("General"), icon());
    KPageWidgetItem *horizontalPage = parent->addPage(horizontalWidget, i18n("Horizontal FormFactor"), icon());
    KPageWidgetItem *verticalPage = parent->addPage(verticalWidget, i18n("Vertical FormFactor"), icon());
    parent->setButtons( KDialog::Ok | KDialog::Cancel);
    connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
    parent->resize(600,380);

    m_generalUi.amarokButtonShown->setChecked(m_buttons->isAmarokButtonVisible());
    m_generalUi.showCover->setChecked(m_showCover);
    m_generalUi.wheelSeek->setChecked(m_wheelSeek);
    m_generalUi.resizeFont->setChecked(m_infos->resizeFont());
    m_generalUi.minFont->setEnabled(m_infos->resizeFont());
    m_generalUi.minFont->setValue(m_infos->minimumFont());
    m_generalUi.scrollLabel->setChecked(!m_infos->resizeFont());
    m_generalUi.useCustomFont->setChecked(m_infos->font() != QFont());
    m_generalUi.fontChooser->setFont(m_customFont);
    m_generalUi.customAmarokBinPath->setText(PlayerController::self()->amarokPath());

    if (formFactor() == Plasma::Horizontal) {
        verticalPage->setEnabled(false);
        m_horizontalUi.widthOnPanel->setValue(m_widthOnPanel);
        m_horizontalUi.showInformationsOnHPanel->setChecked(m_showInformationsOnHPanel);
    } else if (formFactor() == Plasma::Vertical) {
        horizontalPage->setEnabled(false);
        m_verticalUi.heightOnPanel->setValue(m_heightOnPanel);
    } else {
        horizontalPage->setEnabled(false);
        verticalPage->setEnabled(false);
    }
}

void PlayWolf::configAccepted()
{
    KConfigGroup cg = config();

    bool amarokButtonShown = m_generalUi.amarokButtonShown->isChecked();
    m_showCover = m_generalUi.showCover->isChecked();
    m_wheelSeek = m_generalUi.wheelSeek->isChecked();
    bool resizeFont = m_generalUi.resizeFont->isChecked();
    int minFont = m_generalUi.minFont->value();
    bool useCustomFont = m_generalUi.useCustomFont->isChecked();
    QFont font = m_generalUi.fontChooser->font();
    m_showInformationsOnHPanel = m_horizontalUi.showInformationsOnHPanel->isChecked();
    QString amarokBinPath = m_generalUi.customAmarokBinPath->text();

    m_buttons->setIsAmarokButtonVisible(amarokButtonShown);
    m_infos->setResizeFont(resizeFont,minFont);
    if (!useCustomFont) {
        font = QFont();
    } else {
        m_customFont = font;
    }
    m_infos->setFont(font);
    PlayerController::self()->setAmarokPath(amarokBinPath);

    cg.writeEntry("customAmarokBinPath", amarokBinPath);
    cg.writeEntry("amarokButtonShown", amarokButtonShown);
    cg.writeEntry("showCover", m_showCover);
    cg.writeEntry("wheelSeek", m_wheelSeek);
    cg.writeEntry("resizeFont", resizeFont);
    cg.writeEntry("minFont", minFont);
    cg.writeEntry("useCustomFont", useCustomFont);
    cg.writeEntry("customFont", m_customFont);
    cg.writeEntry("showInformationsOnHPanel", m_showInformationsOnHPanel);
    if (formFactor() == Plasma::Horizontal) {
        int newWidth;
        if (m_widthOnPanel != (newWidth = m_horizontalUi.widthOnPanel->value())) {
            cg.writeEntry("widthOnPanel" ,newWidth);
            m_widthOnPanel = newWidth;
            resize(newWidth,30);
        }
    } else if (formFactor() == Plasma::Vertical) {
        int newHeight;
        if (m_heightOnPanel != (newHeight = m_verticalUi.heightOnPanel->value())) {
            cg.writeEntry("heightOnPanel" ,newHeight);
            m_heightOnPanel = newHeight;
            resize(30,newHeight);
        }
    }

    updateConstraints();
}

QSizeF PlayWolf::sizeHint(Qt::SizeHint which, const QSizeF &constraint) const {
    QSizeF hint = Plasma::Applet::sizeHint(which,constraint);

    if (formFactor() == Plasma::Horizontal) {
        return QSizeF(m_widthOnPanel, hint.height());
    } else if (formFactor() == Plasma::Vertical) {
        return QSizeF(hint.width(), m_heightOnPanel);
    }

    return hint;
}

void PlayWolf::hoverEnterEvent(QGraphicsSceneHoverEvent *event)
{
    if (event->pos().x() <= (contentsRect().width() - contentsRect().x())) {
        if ((formFactor() == Plasma::Horizontal) && m_showInformationsOnHPanel) {
            m_buttons->show();
        }
    }
}

void PlayWolf::hoverLeaveEvent(QGraphicsSceneHoverEvent *)
{
    if ((formFactor() == Plasma::Horizontal) && m_showInformationsOnHPanel) {
        m_buttons->hide();
    }
}

void PlayWolf::wheelEvent(QGraphicsSceneWheelEvent *event)
{
    if (m_wheelSeek) {
        if (m_meter->contains(event->pos())) {
            if (event->delta() > 0) {
                PlayerController::self()->seekForward(5);
            } else {
                PlayerController::self()->seekBack(5);
            }
        }
        else if (m_cover->contains(event->pos())) {
            if (event->delta() > 0) {
                PlayerController::self()->volumeUp();
            } else {
                PlayerController::self()->volumeDown();
            }
        }
        else if (m_infos->contains(event->pos())) {
            if (event->delta() > 0) {
                PlayerController::self()->next();
            } else {
                PlayerController::self()->prev();
            }
        }
    }
}

#include "playwolf.moc"