package net.sf.picard.illumina.parser;

import org.testng.Assert;
import org.testng.annotations.Test;

import java.io.File;
import java.util.*;

/**
 * @author mccowan
 */
public class TileMetricsUtilTest {
    static final File TEST_DATA_DIRECTORY = new File("testdata/net/sf/picard/illumina/parser/TileMetricsUtilTest");
    
    static class Testcase {
        final File tileMetricsFile;
        final Set<HashableTile> tiles;

        Testcase(final File tileMetricsFile, final Set<HashableTile> tiles) {
            this.tileMetricsFile = tileMetricsFile;
            this.tiles = tiles;
        }
    }

    /** Wrapper for Tile so we can use it in sets. */
    static class HashableTile extends TileMetricsUtil.Tile {
        private HashableTile(final TileMetricsUtil.Tile tile) {
            super(tile.getLaneNumber(), tile.getTileNumber(), tile.getClusterDensity(), tile.getClusterCount());
        }

        @Override
        public int hashCode() {
            return getTileNumber();
        }

        @Override
        public boolean equals(final Object o) {
            if (!(o instanceof TileMetricsUtil.Tile))
                return false;
            final TileMetricsUtil.Tile that = (TileMetricsUtil.Tile) o;
            return (that.getClusterCount() == this.getClusterCount() && that.getClusterDensity() == this.getClusterDensity()
                    && that.getLaneNumber() == this.getLaneNumber() && that.getTileNumber() == this.getTileNumber());
        }

        public static Set<HashableTile> from(final TileMetricsUtil.Tile... tiles) {
            final HashSet<HashableTile> hashableTiles = new HashSet<HashableTile>();
            for (final TileMetricsUtil.Tile tile : tiles) {
                hashableTiles.add(new HashableTile(tile));
            }
            return hashableTiles;
        }
    }

    final Comparator<TileMetricsUtil.Tile> TILE_COMPARATOR = new Comparator<TileMetricsUtil.Tile>() {
        @Override
        public int compare(final TileMetricsUtil.Tile o1, final TileMetricsUtil.Tile o2) {
            return o1.getTileNumber() - o2.getTileNumber();
        }
    };

    @Test
    public void testParseTileData() throws Exception {
        for (final Testcase testcase : TESTCASES) {
            final List<TileMetricsUtil.Tile> tiles = new ArrayList<TileMetricsUtil.Tile>(TileMetricsUtil.parseTileMetrics(testcase.tileMetricsFile));
            System.out.println(testcase.tileMetricsFile);
            for (final TileMetricsUtil.Tile tile : tiles) {
                System.out.println(String.format("%s\t%s\t%s\t%s", tile.getLaneNumber(), tile.getTileNumber(), tile.getClusterDensity(), tile.getClusterCount()));
            }
            // TestNG does not handle set equality properly.
            Assert.assertTrue(testcase.tiles.equals(HashableTile.from(tiles.toArray(new TileMetricsUtil.Tile[tiles.size()]))));
        }
    }

    final Collection<Testcase> TESTCASES = Arrays.asList(
            new Testcase(
                    new File(TEST_DATA_DIRECTORY, "MiSeqTileMetricsOut.bin"),
                    HashableTile.from(
                            new TileMetricsUtil.Tile(1, 1108, 699852.94f, 482438.0f),
                            new TileMetricsUtil.Tile(1, 1107, 699791.7f, 482610.0f),
                            new TileMetricsUtil.Tile(1, 1109, 702601.44f, 484489.0f),
                            new TileMetricsUtil.Tile(1, 1112, 718619.8f, 495960.0f),
                            new TileMetricsUtil.Tile(1, 1113, 725377.25f, 500602.0f),
                            new TileMetricsUtil.Tile(1, 1101, 698977.6f, 482926.0f),
                            new TileMetricsUtil.Tile(1, 1110, 704098.25f, 485448.0f),
                            new TileMetricsUtil.Tile(1, 1102, 693865.75f, 478827.0f),
                            new TileMetricsUtil.Tile(1, 1111, 710164.2f, 488936.0f),
                            new TileMetricsUtil.Tile(1, 1103, 698093.56f, 479817.0f),
                            new TileMetricsUtil.Tile(1, 1104, 693708.2f, 479144.0f),
                            new TileMetricsUtil.Tile(1, 1114, 734585.6f, 503048.0f),
                            new TileMetricsUtil.Tile(1, 1105, 700553.2f, 479396.0f),
                            new TileMetricsUtil.Tile(1, 1106, 694881.1f, 480108.0f)
                    )
            ),
            new Testcase(new File(TEST_DATA_DIRECTORY, "HiSeq2000TileMetricsOut.bin"),
                    HashableTile.from(
                            new TileMetricsUtil.Tile(4, 1101, 544523.94f, 1568229.0f),
                            new TileMetricsUtil.Tile(4, 1209, 668636.44f, 1925673.0f),
                            new TileMetricsUtil.Tile(4, 1105, 602357.25f, 1734789.0f),
                            new TileMetricsUtil.Tile(4, 1104, 582857.25f, 1678629.0f),
                            new TileMetricsUtil.Tile(4, 1103, 571702.06f, 1646502.0f),
                            new TileMetricsUtil.Tile(4, 1309, 664930.2f, 1914999.0f),
                            new TileMetricsUtil.Tile(4, 1102, 559316.3f, 1610831.0f),
                            new TileMetricsUtil.Tile(4, 1308, 659291.6f, 1898760.0f),
                            new TileMetricsUtil.Tile(4, 1109, 650628.8f, 1873811.0f),
                            new TileMetricsUtil.Tile(4, 1108, 640943.0f, 1845916.0f),
                            new TileMetricsUtil.Tile(4, 1107, 629443.75f, 1812798.0f),
                            new TileMetricsUtil.Tile(4, 1106, 623904.1f, 1796844.0f),
                            new TileMetricsUtil.Tile(4, 1302, 563119.44f, 1621784.0f),
                            new TileMetricsUtil.Tile(4, 1303, 583720.1f, 1681114.0f),
                            new TileMetricsUtil.Tile(4, 1301, 551133.3f, 1587264.0f),
                            new TileMetricsUtil.Tile(4, 1306, 628892.0f, 1811209.0f),
                            new TileMetricsUtil.Tile(4, 1307, 637835.06f, 1836965.0f),
                            new TileMetricsUtil.Tile(4, 1304, 592503.1f, 1706409.0f),
                            new TileMetricsUtil.Tile(4, 1305, 608899.3f, 1753630.0f),
                            new TileMetricsUtil.Tile(4, 1207, 638901.4f, 1840036.0f),
                            new TileMetricsUtil.Tile(4, 1208, 662913.9f, 1909192.0f),
                            new TileMetricsUtil.Tile(4, 1205, 607855.56f, 1750624.0f),
                            new TileMetricsUtil.Tile(4, 1206, 628009.7f, 1808668.0f),
                            new TileMetricsUtil.Tile(4, 1203, 578895.1f, 1667218.0f),
                            new TileMetricsUtil.Tile(4, 1204, 592665.25f, 1706876.0f),
                            new TileMetricsUtil.Tile(4, 1201, 545689.94f, 1571587.0f),
                            new TileMetricsUtil.Tile(4, 1202, 562946.5f, 1621286.0f),
                            new TileMetricsUtil.Tile(8, 2201, 580058.7f, 1670569.0f),
                            new TileMetricsUtil.Tile(1, 1216, 882619.06f, 2541943.0f),
                            new TileMetricsUtil.Tile(4, 1110, 541470.1f, 1559434.0f),
                            new TileMetricsUtil.Tile(1, 1215, 860050.3f, 2476945.0f),
                            new TileMetricsUtil.Tile(1, 1214, 840209.3f, 2419803.0f),
                            new TileMetricsUtil.Tile(4, 1112, 723241.6f, 2082936.0f),
                            new TileMetricsUtil.Tile(8, 2309, 706817.3f, 2035634.0f),
                            new TileMetricsUtil.Tile(1, 1213, 820381.2f, 2362698.0f),
                            new TileMetricsUtil.Tile(4, 1111, 680317.0f, 1959313.0f),
                            new TileMetricsUtil.Tile(8, 2308, 636503.1f, 1833129.0f),
                            new TileMetricsUtil.Tile(8, 2205, 625169.06f, 1800487.0f),
                            new TileMetricsUtil.Tile(1, 1212, 798143.4f, 2298653.0f),
                            new TileMetricsUtil.Tile(8, 2307, 679264.56f, 1956282.0f),
                            new TileMetricsUtil.Tile(8, 2204, 613535.06f, 1766981.0f),
                            new TileMetricsUtil.Tile(1, 1211, 664055.5f, 1912480.0f),
                            new TileMetricsUtil.Tile(8, 2306, 664856.94f, 1914788.0f),
                            new TileMetricsUtil.Tile(8, 2203, 598923.56f, 1724900.0f),
                            new TileMetricsUtil.Tile(1, 1210, 760540.25f, 2190356.0f),
                            new TileMetricsUtil.Tile(8, 2305, 668689.9f, 1925827.0f),
                            new TileMetricsUtil.Tile(8, 2202, 587381.56f, 1691659.0f),
                            new TileMetricsUtil.Tile(8, 2304, 657572.9f, 1893810.0f),
                            new TileMetricsUtil.Tile(8, 2209, 640729.8f, 1845302.0f),
                            new TileMetricsUtil.Tile(8, 2303, 644017.3f, 1854770.0f),
                            new TileMetricsUtil.Tile(8, 2208, 508849.62f, 1465487.0f),
                            new TileMetricsUtil.Tile(8, 2302, 633281.25f, 1823850.0f),
                            new TileMetricsUtil.Tile(8, 2207, 622214.2f, 1791977.0f),
                            new TileMetricsUtil.Tile(8, 2301, 628018.4f, 1808693.0f),
                            new TileMetricsUtil.Tile(8, 2206, 624249.6f, 1797839.0f),
                            new TileMetricsUtil.Tile(4, 1114, 757904.1f, 2182764.0f),
                            new TileMetricsUtil.Tile(4, 1113, 735293.75f, 2117646.0f),
                            new TileMetricsUtil.Tile(4, 1116, 807057.25f, 2324325.0f),
                            new TileMetricsUtil.Tile(4, 1115, 774449.6f, 2230415.0f),
                            new TileMetricsUtil.Tile(4, 1315, 771113.9f, 2220808.0f),
                            new TileMetricsUtil.Tile(4, 1316, 807589.9f, 2325859.0f),
                            new TileMetricsUtil.Tile(4, 1311, 682684.0f, 1966130.0f),
                            new TileMetricsUtil.Tile(4, 1312, 714275.7f, 2057114.0f),
                            new TileMetricsUtil.Tile(4, 1313, 725538.9f, 2089552.0f),
                            new TileMetricsUtil.Tile(4, 1314, 748121.9f, 2154591.0f),
                            new TileMetricsUtil.Tile(4, 1310, 570019.06f, 1641655.0f),
                            new TileMetricsUtil.Tile(8, 2210, 681208.6f, 1961881.0f),
                            new TileMetricsUtil.Tile(1, 1203, 489748.25f, 1410475.0f),
                            new TileMetricsUtil.Tile(1, 1202, 610748.94f, 1758957.0f),
                            new TileMetricsUtil.Tile(8, 2212, 730553.44f, 2103994.0f),
                            new TileMetricsUtil.Tile(1, 1205, 679378.8f, 1956611.0f),
                            new TileMetricsUtil.Tile(8, 2211, 703858.6f, 2027113.0f),
                            new TileMetricsUtil.Tile(1, 1204, 669089.56f, 1926978.0f),
                            new TileMetricsUtil.Tile(8, 2214, 775678.1f, 2233953.0f),
                            new TileMetricsUtil.Tile(8, 2316, 883011.06f, 2543072.0f),
                            new TileMetricsUtil.Tile(8, 2213, 755248.94f, 2175117.0f),
                            new TileMetricsUtil.Tile(8, 2216, 835654.5f, 2406685.0f),
                            new TileMetricsUtil.Tile(1, 1201, 637199.6f, 1835135.0f),
                            new TileMetricsUtil.Tile(8, 2215, 803531.2f, 2314170.0f),
                            new TileMetricsUtil.Tile(8, 2313, 817312.1f, 2353859.0f),
                            new TileMetricsUtil.Tile(8, 2312, 785540.6f, 2262357.0f),
                            new TileMetricsUtil.Tile(8, 2315, 859228.8f, 2474579.0f),
                            new TileMetricsUtil.Tile(8, 2314, 834483.6f, 2403313.0f),
                            new TileMetricsUtil.Tile(8, 2311, 757663.2f, 2182070.0f),
                            new TileMetricsUtil.Tile(8, 2310, 730759.4f, 2104587.0f),
                            new TileMetricsUtil.Tile(1, 1206, 694945.44f, 2001443.0f),
                            new TileMetricsUtil.Tile(1, 1207, 714633.3f, 2058144.0f),
                            new TileMetricsUtil.Tile(1, 1208, 728221.5f, 2097278.0f),
                            new TileMetricsUtil.Tile(1, 1209, 738294.75f, 2126289.0f),
                            new TileMetricsUtil.Tile(7, 2110, 673538.9f, 1939792.0f),
                            new TileMetricsUtil.Tile(7, 2112, 704191.6f, 2028072.0f),
                            new TileMetricsUtil.Tile(7, 2111, 685426.4f, 1974028.0f),
                            new TileMetricsUtil.Tile(7, 2114, 721513.9f, 2077960.0f),
                            new TileMetricsUtil.Tile(7, 2113, 703654.1f, 2026524.0f),
                            new TileMetricsUtil.Tile(7, 2116, 832204.1f, 2396748.0f),
                            new TileMetricsUtil.Tile(7, 2115, 800356.56f, 2305027.0f),
                            new TileMetricsUtil.Tile(1, 1310, 748198.56f, 2154812.0f),
                            new TileMetricsUtil.Tile(1, 1311, 760848.56f, 2191244.0f),
                            new TileMetricsUtil.Tile(1, 1314, 822341.6f, 2368344.0f),
                            new TileMetricsUtil.Tile(1, 1315, 842715.94f, 2427022.0f),
                            new TileMetricsUtil.Tile(1, 1312, 782347.9f, 2253162.0f),
                            new TileMetricsUtil.Tile(1, 1313, 797066.6f, 2295552.0f),
                            new TileMetricsUtil.Tile(4, 1213, 754260.06f, 2172269.0f),
                            new TileMetricsUtil.Tile(4, 1212, 738124.6f, 2125799.0f),
                            new TileMetricsUtil.Tile(4, 1215, 791817.3f, 2280434.0f),
                            new TileMetricsUtil.Tile(4, 1214, 773102.4f, 2226535.0f),
                            new TileMetricsUtil.Tile(4, 2215, 813619.75f, 2343225.0f),
                            new TileMetricsUtil.Tile(4, 1216, 819997.56f, 2361593.0f),
                            new TileMetricsUtil.Tile(4, 2216, 849781.94f, 2447372.0f),
                            new TileMetricsUtil.Tile(4, 2213, 766563.9f, 2207704.0f),
                            new TileMetricsUtil.Tile(4, 2214, 788909.7f, 2272060.0f),
                            new TileMetricsUtil.Tile(4, 2211, 687051.4f, 1978708.0f),
                            new TileMetricsUtil.Tile(4, 2212, 744198.56f, 2143292.0f),
                            new TileMetricsUtil.Tile(1, 1316, 862726.4f, 2484652.0f),
                            new TileMetricsUtil.Tile(4, 2210, 521713.2f, 1502534.0f),
                            new TileMetricsUtil.Tile(4, 1211, 688227.75f, 1982096.0f),
                            new TileMetricsUtil.Tile(4, 1210, 524505.56f, 1510576.0f),
                            new TileMetricsUtil.Tile(1, 1301, 639037.8f, 1840429.0f),
                            new TileMetricsUtil.Tile(1, 1302, 627089.2f, 1806017.0f),
                            new TileMetricsUtil.Tile(1, 1303, 571894.75f, 1647057.0f),
                            new TileMetricsUtil.Tile(1, 1304, 678610.4f, 1954398.0f),
                            new TileMetricsUtil.Tile(4, 2206, 626362.8f, 1803925.0f),
                            new TileMetricsUtil.Tile(4, 2207, 638985.75f, 1840279.0f),
                            new TileMetricsUtil.Tile(4, 2208, 659187.5f, 1898460.0f),
                            new TileMetricsUtil.Tile(4, 2209, 662527.4f, 1908079.0f),
                            new TileMetricsUtil.Tile(8, 1201, 580114.56f, 1670730.0f),
                            new TileMetricsUtil.Tile(4, 2202, 565767.0f, 1629409.0f),
                            new TileMetricsUtil.Tile(8, 1202, 588968.4f, 1696229.0f),
                            new TileMetricsUtil.Tile(4, 2203, 579432.94f, 1668767.0f),
                            new TileMetricsUtil.Tile(8, 1203, 601895.1f, 1733458.0f),
                            new TileMetricsUtil.Tile(4, 2204, 593594.75f, 1709553.0f),
                            new TileMetricsUtil.Tile(8, 1204, 615764.56f, 1773402.0f),
                            new TileMetricsUtil.Tile(4, 2205, 607054.1f, 1748316.0f),
                            new TileMetricsUtil.Tile(8, 1205, 626597.2f, 1804600.0f),
                            new TileMetricsUtil.Tile(1, 1306, 695878.8f, 2004131.0f),
                            new TileMetricsUtil.Tile(8, 1206, 625105.9f, 1800305.0f),
                            new TileMetricsUtil.Tile(1, 1305, 688254.8f, 1982174.0f),
                            new TileMetricsUtil.Tile(8, 1207, 629282.6f, 1812334.0f),
                            new TileMetricsUtil.Tile(1, 1308, 724458.3f, 2086440.0f),
                            new TileMetricsUtil.Tile(8, 1208, 518821.84f, 1494207.0f),
                            new TileMetricsUtil.Tile(1, 1307, 711070.1f, 2047882.0f),
                            new TileMetricsUtil.Tile(4, 2201, 549562.1f, 1582739.0f),
                            new TileMetricsUtil.Tile(8, 1209, 643954.1f, 1854588.0f),
                            new TileMetricsUtil.Tile(1, 1309, 737026.4f, 2122636.0f),
                            new TileMetricsUtil.Tile(5, 1215, 696313.5f, 2005383.0f),
                            new TileMetricsUtil.Tile(5, 1112, 629804.8f, 1813838.0f),
                            new TileMetricsUtil.Tile(5, 1216, 722283.3f, 2080176.0f),
                            new TileMetricsUtil.Tile(5, 1113, 646438.9f, 1861744.0f),
                            new TileMetricsUtil.Tile(5, 1213, 649733.6f, 1871233.0f),
                            new TileMetricsUtil.Tile(5, 1110, 600022.9f, 1728066.0f),
                            new TileMetricsUtil.Tile(5, 1214, 664704.8f, 1914350.0f),
                            new TileMetricsUtil.Tile(5, 1111, 613195.1f, 1766002.0f),
                            new TileMetricsUtil.Tile(5, 1211, 617793.75f, 1779246.0f),
                            new TileMetricsUtil.Tile(5, 1116, 696982.6f, 2007310.0f),
                            new TileMetricsUtil.Tile(5, 1212, 633925.0f, 1825704.0f),
                            new TileMetricsUtil.Tile(5, 1114, 662869.06f, 1909063.0f),
                            new TileMetricsUtil.Tile(5, 1210, 606142.0f, 1745689.0f),
                            new TileMetricsUtil.Tile(5, 1115, 682282.94f, 1964975.0f),
                            new TileMetricsUtil.Tile(5, 1202, 502318.72f, 1446678.0f),
                            new TileMetricsUtil.Tile(5, 1203, 509032.28f, 1466013.0f),
                            new TileMetricsUtil.Tile(5, 1204, 524129.5f, 1509493.0f),
                            new TileMetricsUtil.Tile(5, 1101, 464122.56f, 1336673.0f),
                            new TileMetricsUtil.Tile(5, 1205, 537867.0f, 1549057.0f),
                            new TileMetricsUtil.Tile(5, 1102, 475782.97f, 1370255.0f),
                            new TileMetricsUtil.Tile(5, 1206, 554514.56f, 1597002.0f),
                            new TileMetricsUtil.Tile(5, 1207, 554870.44f, 1598027.0f),
                            new TileMetricsUtil.Tile(5, 1208, 534909.0f, 1540538.0f),
                            new TileMetricsUtil.Tile(5, 1209, 423839.9f, 1220659.0f),
                            new TileMetricsUtil.Tile(5, 1107, 533332.25f, 1535997.0f),
                            new TileMetricsUtil.Tile(5, 1108, 526180.2f, 1515399.0f),
                            new TileMetricsUtil.Tile(5, 1109, 434699.62f, 1251935.0f),
                            new TileMetricsUtil.Tile(5, 1103, 490662.8f, 1413109.0f),
                            new TileMetricsUtil.Tile(5, 1104, 505400.7f, 1455554.0f),
                            new TileMetricsUtil.Tile(5, 1105, 517630.2f, 1490775.0f),
                            new TileMetricsUtil.Tile(5, 1201, 487380.2f, 1403655.0f),
                            new TileMetricsUtil.Tile(5, 1106, 535730.2f, 1542903.0f),
                            new TileMetricsUtil.Tile(5, 2210, 608615.94f, 1752814.0f),
                            new TileMetricsUtil.Tile(5, 2211, 619886.06f, 1785272.0f),
                            new TileMetricsUtil.Tile(5, 2212, 639725.3f, 1842409.0f),
                            new TileMetricsUtil.Tile(5, 2213, 657432.25f, 1893405.0f),
                            new TileMetricsUtil.Tile(5, 2214, 676718.0f, 1948948.0f),
                            new TileMetricsUtil.Tile(4, 2302, 567359.7f, 1633996.0f),
                            new TileMetricsUtil.Tile(5, 2215, 710085.06f, 2045045.0f),
                            new TileMetricsUtil.Tile(4, 2301, 559488.2f, 1611326.0f),
                            new TileMetricsUtil.Tile(5, 2216, 743956.94f, 2142596.0f),
                            new TileMetricsUtil.Tile(4, 2304, 597910.4f, 1721982.0f),
                            new TileMetricsUtil.Tile(4, 2303, 589819.75f, 1698681.0f),
                            new TileMetricsUtil.Tile(4, 2306, 631276.7f, 1818077.0f),
                            new TileMetricsUtil.Tile(4, 2305, 610759.4f, 1758987.0f),
                            new TileMetricsUtil.Tile(4, 2308, 661488.5f, 1905087.0f),
                            new TileMetricsUtil.Tile(4, 2307, 642497.56f, 1850393.0f),
                            new TileMetricsUtil.Tile(8, 2115, 832733.6f, 2398273.0f),
                            new TileMetricsUtil.Tile(4, 2309, 668314.9f, 1924747.0f),
                            new TileMetricsUtil.Tile(8, 2114, 803602.4f, 2314375.0f),
                            new TileMetricsUtil.Tile(8, 2116, 867472.2f, 2498320.0f),
                            new TileMetricsUtil.Tile(8, 2111, 726911.44f, 2093505.0f),
                            new TileMetricsUtil.Tile(8, 2110, 702797.56f, 2024057.0f),
                            new TileMetricsUtil.Tile(8, 2113, 774282.25f, 2229933.0f),
                            new TileMetricsUtil.Tile(8, 2112, 747976.0f, 2154171.0f),
                            new TileMetricsUtil.Tile(4, 2311, 685694.06f, 1974799.0f),
                            new TileMetricsUtil.Tile(4, 2310, 573007.6f, 1650262.0f),
                            new TileMetricsUtil.Tile(4, 2315, 802572.56f, 2311409.0f),
                            new TileMetricsUtil.Tile(4, 2314, 776761.44f, 2237073.0f),
                            new TileMetricsUtil.Tile(4, 2313, 741337.1f, 2135051.0f),
                            new TileMetricsUtil.Tile(4, 2312, 717835.06f, 2067365.0f),
                            new TileMetricsUtil.Tile(4, 2316, 846926.4f, 2439148.0f),
                            new TileMetricsUtil.Tile(8, 2106, 635906.56f, 1831411.0f),
                            new TileMetricsUtil.Tile(8, 2105, 614185.75f, 1768855.0f),
                            new TileMetricsUtil.Tile(8, 2104, 602665.94f, 1735678.0f),
                            new TileMetricsUtil.Tile(8, 2103, 586181.25f, 1688202.0f),
                            new TileMetricsUtil.Tile(8, 2102, 574919.75f, 1655769.0f),
                            new TileMetricsUtil.Tile(8, 2101, 567252.44f, 1633687.0f),
                            new TileMetricsUtil.Tile(8, 2109, 673661.06f, 1940144.0f),
                            new TileMetricsUtil.Tile(8, 2108, 566904.8f, 1632686.0f),
                            new TileMetricsUtil.Tile(8, 2107, 649836.06f, 1871528.0f),
                            new TileMetricsUtil.Tile(1, 2115, 923581.94f, 2659916.0f),
                            new TileMetricsUtil.Tile(1, 2116, 934428.8f, 2691155.0f),
                            new TileMetricsUtil.Tile(6, 2116, 772013.2f, 2223398.0f),
                            new TileMetricsUtil.Tile(1, 2111, 708667.3f, 2040962.0f),
                            new TileMetricsUtil.Tile(1, 2112, 844992.3f, 2433578.0f),
                            new TileMetricsUtil.Tile(1, 2113, 874324.6f, 2518055.0f),
                            new TileMetricsUtil.Tile(1, 2114, 901182.94f, 2595407.0f),
                            new TileMetricsUtil.Tile(6, 2110, 600281.25f, 1728810.0f),
                            new TileMetricsUtil.Tile(1, 2210, 754377.75f, 2172608.0f),
                            new TileMetricsUtil.Tile(6, 2111, 548103.8f, 1578539.0f),
                            new TileMetricsUtil.Tile(1, 2211, 659499.6f, 1899359.0f),
                            new TileMetricsUtil.Tile(1, 2212, 808203.8f, 2327627.0f),
                            new TileMetricsUtil.Tile(1, 2110, 795863.9f, 2292088.0f),
                            new TileMetricsUtil.Tile(1, 2213, 831343.0f, 2394268.0f),
                            new TileMetricsUtil.Tile(6, 2114, 716921.5f, 2064734.0f),
                            new TileMetricsUtil.Tile(2, 1307, 715829.1f, 2061588.0f),
                            new TileMetricsUtil.Tile(1, 2214, 858290.56f, 2471877.0f),
                            new TileMetricsUtil.Tile(6, 2115, 742764.2f, 2139161.0f),
                            new TileMetricsUtil.Tile(2, 1306, 701070.1f, 2019082.0f),
                            new TileMetricsUtil.Tile(1, 2215, 885655.9f, 2550689.0f),
                            new TileMetricsUtil.Tile(6, 2112, 605255.56f, 1743136.0f),
                            new TileMetricsUtil.Tile(2, 1309, 743669.75f, 2141769.0f),
                            new TileMetricsUtil.Tile(1, 2216, 914737.8f, 2634445.0f),
                            new TileMetricsUtil.Tile(6, 2113, 686910.06f, 1978301.0f),
                            new TileMetricsUtil.Tile(2, 1308, 729707.94f, 2101559.0f),
                            new TileMetricsUtil.Tile(2, 1302, 641778.8f, 1848323.0f),
                            new TileMetricsUtil.Tile(2, 1303, 643553.44f, 1853434.0f),
                            new TileMetricsUtil.Tile(2, 1304, 566862.1f, 1632563.0f),
                            new TileMetricsUtil.Tile(2, 1305, 678740.25f, 1954772.0f),
                            new TileMetricsUtil.Tile(2, 1301, 645812.5f, 1859940.0f),
                            new TileMetricsUtil.Tile(6, 2105, 531359.0f, 1530314.0f),
                            new TileMetricsUtil.Tile(6, 2106, 545602.06f, 1571334.0f),
                            new TileMetricsUtil.Tile(5, 2209, 422861.78f, 1217842.0f),
                            new TileMetricsUtil.Tile(6, 2107, 563268.4f, 1622213.0f),
                            new TileMetricsUtil.Tile(5, 2208, 527286.06f, 1518584.0f),
                            new TileMetricsUtil.Tile(6, 2108, 576791.3f, 1661159.0f),
                            new TileMetricsUtil.Tile(5, 2207, 554846.2f, 1597957.0f),
                            new TileMetricsUtil.Tile(6, 2109, 584765.25f, 1684124.0f),
                            new TileMetricsUtil.Tile(5, 2301, 497327.06f, 1432302.0f),
                            new TileMetricsUtil.Tile(5, 2206, 552916.6f, 1592400.0f),
                            new TileMetricsUtil.Tile(5, 2205, 538032.6f, 1549534.0f),
                            new TileMetricsUtil.Tile(5, 2204, 523163.2f, 1506710.0f),
                            new TileMetricsUtil.Tile(5, 2203, 509249.62f, 1466639.0f),
                            new TileMetricsUtil.Tile(5, 2305, 544915.94f, 1569358.0f),
                            new TileMetricsUtil.Tile(5, 2202, 503321.84f, 1449567.0f),
                            new TileMetricsUtil.Tile(5, 2304, 530174.6f, 1526903.0f),
                            new TileMetricsUtil.Tile(5, 2201, 490223.94f, 1411845.0f),
                            new TileMetricsUtil.Tile(5, 2303, 518052.06f, 1491990.0f),
                            new TileMetricsUtil.Tile(5, 2302, 507330.53f, 1461112.0f),
                            new TileMetricsUtil.Tile(6, 2101, 479912.47f, 1382148.0f),
                            new TileMetricsUtil.Tile(5, 2309, 467297.22f, 1345816.0f),
                            new TileMetricsUtil.Tile(6, 2102, 489415.25f, 1409516.0f),
                            new TileMetricsUtil.Tile(5, 2308, 549378.1f, 1582209.0f),
                            new TileMetricsUtil.Tile(6, 2103, 503457.28f, 1449957.0f),
                            new TileMetricsUtil.Tile(5, 2307, 558159.4f, 1607499.0f),
                            new TileMetricsUtil.Tile(6, 2104, 519150.3f, 1495153.0f),
                            new TileMetricsUtil.Tile(5, 2306, 557682.25f, 1606125.0f),
                            new TileMetricsUtil.Tile(5, 2106, 542138.5f, 1561359.0f),
                            new TileMetricsUtil.Tile(5, 2107, 539329.1f, 1553268.0f),
                            new TileMetricsUtil.Tile(5, 2104, 511818.72f, 1474038.0f),
                            new TileMetricsUtil.Tile(5, 2105, 525148.56f, 1512428.0f),
                            new TileMetricsUtil.Tile(5, 2108, 529528.8f, 1525043.0f),
                            new TileMetricsUtil.Tile(5, 2109, 436176.38f, 1256188.0f),
                            new TileMetricsUtil.Tile(5, 2102, 489248.6f, 1409036.0f),
                            new TileMetricsUtil.Tile(5, 2103, 498902.4f, 1436839.0f),
                            new TileMetricsUtil.Tile(5, 2101, 478850.7f, 1379090.0f),
                            new TileMetricsUtil.Tile(5, 2310, 614297.56f, 1769177.0f),
                            new TileMetricsUtil.Tile(1, 1108, 732500.3f, 2109601.0f),
                            new TileMetricsUtil.Tile(3, 1314, 761796.5f, 2193974.0f),
                            new TileMetricsUtil.Tile(1, 1107, 723057.6f, 2082406.0f),
                            new TileMetricsUtil.Tile(3, 1315, 779274.6f, 2244311.0f),
                            new TileMetricsUtil.Tile(5, 2312, 654979.1f, 1886340.0f),
                            new TileMetricsUtil.Tile(3, 1316, 805652.06f, 2320278.0f),
                            new TileMetricsUtil.Tile(5, 2311, 630389.9f, 1815523.0f),
                            new TileMetricsUtil.Tile(1, 1109, 737068.0f, 2122756.0f),
                            new TileMetricsUtil.Tile(3, 1310, 698116.3f, 2010575.0f),
                            new TileMetricsUtil.Tile(3, 1311, 705486.75f, 2031802.0f),
                            new TileMetricsUtil.Tile(3, 1312, 722473.94f, 2080725.0f),
                            new TileMetricsUtil.Tile(3, 1210, 691075.3f, 1990297.0f),
                            new TileMetricsUtil.Tile(3, 1313, 743791.3f, 2142119.0f),
                            new TileMetricsUtil.Tile(3, 1211, 669740.25f, 1928852.0f),
                            new TileMetricsUtil.Tile(3, 1212, 683764.56f, 1969242.0f),
                            new TileMetricsUtil.Tile(2, 2301, 644702.75f, 1856744.0f),
                            new TileMetricsUtil.Tile(3, 1213, 745240.6f, 2146293.0f),
                            new TileMetricsUtil.Tile(2, 2302, 640053.8f, 1843355.0f),
                            new TileMetricsUtil.Tile(3, 1214, 763265.25f, 2198204.0f),
                            new TileMetricsUtil.Tile(5, 2314, 707079.1f, 2036388.0f),
                            new TileMetricsUtil.Tile(2, 2303, 637679.1f, 1836516.0f),
                            new TileMetricsUtil.Tile(3, 1215, 784061.75f, 2258098.0f),
                            new TileMetricsUtil.Tile(5, 2313, 685909.0f, 1975418.0f),
                            new TileMetricsUtil.Tile(2, 2304, 558232.6f, 1607710.0f),
                            new TileMetricsUtil.Tile(3, 1216, 817701.7f, 2354981.0f),
                            new TileMetricsUtil.Tile(5, 2316, 756095.44f, 2177555.0f),
                            new TileMetricsUtil.Tile(2, 2305, 672488.2f, 1936766.0f),
                            new TileMetricsUtil.Tile(5, 2315, 726102.75f, 2091176.0f),
                            new TileMetricsUtil.Tile(2, 2306, 696259.7f, 2005228.0f),
                            new TileMetricsUtil.Tile(2, 2308, 724660.06f, 2087021.0f),
                            new TileMetricsUtil.Tile(2, 2307, 709184.0f, 2042450.0f),
                            new TileMetricsUtil.Tile(2, 2309, 738271.9f, 2126223.0f),
                            new TileMetricsUtil.Tile(5, 2115, 719669.75f, 2072649.0f),
                            new TileMetricsUtil.Tile(5, 2116, 747281.94f, 2152172.0f),
                            new TileMetricsUtil.Tile(5, 2111, 622326.7f, 1792301.0f),
                            new TileMetricsUtil.Tile(5, 2112, 642663.9f, 1850872.0f),
                            new TileMetricsUtil.Tile(5, 2113, 666584.4f, 1919763.0f),
                            new TileMetricsUtil.Tile(1, 1101, 638016.3f, 1837487.0f),
                            new TileMetricsUtil.Tile(5, 2114, 690995.8f, 1990068.0f),
                            new TileMetricsUtil.Tile(1, 1102, 628837.8f, 1811053.0f),
                            new TileMetricsUtil.Tile(1, 1103, 541017.3f, 1558130.0f),
                            new TileMetricsUtil.Tile(1, 1104, 684565.25f, 1971548.0f),
                            new TileMetricsUtil.Tile(1, 1105, 694928.44f, 2001394.0f),
                            new TileMetricsUtil.Tile(5, 2110, 603435.75f, 1737895.0f),
                            new TileMetricsUtil.Tile(1, 1106, 709664.56f, 2043834.0f),
                            new TileMetricsUtil.Tile(1, 2102, 644288.5f, 1855551.0f),
                            new TileMetricsUtil.Tile(3, 1305, 637067.3f, 1834754.0f),
                            new TileMetricsUtil.Tile(1, 2103, 558346.5f, 1608038.0f),
                            new TileMetricsUtil.Tile(3, 1306, 650456.25f, 1873314.0f),
                            new TileMetricsUtil.Tile(3, 1303, 608183.7f, 1751569.0f),
                            new TileMetricsUtil.Tile(1, 2101, 651085.75f, 1875127.0f),
                            new TileMetricsUtil.Tile(3, 1304, 621200.3f, 1789057.0f),
                            new TileMetricsUtil.Tile(1, 2106, 737575.7f, 2124218.0f),
                            new TileMetricsUtil.Tile(3, 1301, 589746.9f, 1698471.0f),
                            new TileMetricsUtil.Tile(1, 2107, 749932.6f, 2159806.0f),
                            new TileMetricsUtil.Tile(3, 1302, 595741.6f, 1715736.0f),
                            new TileMetricsUtil.Tile(1, 2104, 709154.1f, 2042364.0f),
                            new TileMetricsUtil.Tile(1, 2105, 722732.94f, 2081471.0f),
                            new TileMetricsUtil.Tile(2, 2312, 791178.8f, 2278595.0f),
                            new TileMetricsUtil.Tile(3, 1202, 576400.0f, 1660032.0f),
                            new TileMetricsUtil.Tile(2, 2313, 823073.56f, 2370452.0f),
                            new TileMetricsUtil.Tile(3, 1203, 590679.1f, 1701156.0f),
                            new TileMetricsUtil.Tile(2, 2310, 754680.2f, 2173479.0f),
                            new TileMetricsUtil.Tile(2, 2311, 775202.75f, 2232584.0f),
                            new TileMetricsUtil.Tile(3, 1201, 565834.0f, 1629602.0f),
                            new TileMetricsUtil.Tile(2, 2316, 901895.44f, 2597459.0f),
                            new TileMetricsUtil.Tile(3, 1206, 629970.44f, 1814315.0f),
                            new TileMetricsUtil.Tile(3, 1207, 644813.2f, 1857062.0f),
                            new TileMetricsUtil.Tile(2, 2314, 850203.06f, 2448585.0f),
                            new TileMetricsUtil.Tile(3, 1204, 604347.56f, 1740521.0f),
                            new TileMetricsUtil.Tile(2, 2315, 878050.3f, 2528785.0f),
                            new TileMetricsUtil.Tile(3, 1205, 617782.94f, 1779215.0f),
                            new TileMetricsUtil.Tile(3, 1209, 672435.75f, 1936615.0f),
                            new TileMetricsUtil.Tile(3, 1208, 658543.75f, 1896606.0f),
                            new TileMetricsUtil.Tile(1, 1112, 804387.1f, 2316635.0f),
                            new TileMetricsUtil.Tile(1, 1113, 812433.3f, 2339808.0f),
                            new TileMetricsUtil.Tile(1, 1110, 759368.75f, 2186982.0f),
                            new TileMetricsUtil.Tile(1, 2109, 772350.0f, 2224368.0f),
                            new TileMetricsUtil.Tile(1, 1111, 679406.56f, 1956691.0f),
                            new TileMetricsUtil.Tile(1, 2108, 763281.25f, 2198250.0f),
                            new TileMetricsUtil.Tile(1, 1116, 877692.7f, 2527755.0f),
                            new TileMetricsUtil.Tile(3, 1309, 680868.4f, 1960901.0f),
                            new TileMetricsUtil.Tile(1, 1114, 825847.2f, 2378440.0f),
                            new TileMetricsUtil.Tile(3, 1308, 670088.5f, 1929855.0f),
                            new TileMetricsUtil.Tile(1, 1115, 840287.1f, 2420027.0f),
                            new TileMetricsUtil.Tile(3, 1307, 659739.2f, 1900049.0f),
                            new TileMetricsUtil.Tile(3, 1103, 592339.56f, 1705938.0f),
                            new TileMetricsUtil.Tile(3, 1104, 607652.75f, 1750040.0f),
                            new TileMetricsUtil.Tile(3, 1101, 564907.25f, 1626933.0f),
                            new TileMetricsUtil.Tile(3, 1102, 578869.75f, 1667145.0f),
                            new TileMetricsUtil.Tile(3, 1107, 651271.1f, 1875661.0f),
                            new TileMetricsUtil.Tile(3, 1108, 669996.1f, 1929589.0f),
                            new TileMetricsUtil.Tile(3, 1105, 622082.25f, 1791597.0f),
                            new TileMetricsUtil.Tile(3, 1106, 634596.9f, 1827639.0f),
                            new TileMetricsUtil.Tile(3, 1109, 683126.7f, 1967405.0f),
                            new TileMetricsUtil.Tile(2, 1206, 724352.75f, 2086136.0f),
                            new TileMetricsUtil.Tile(3, 1112, 714129.8f, 2056694.0f),
                            new TileMetricsUtil.Tile(2, 1205, 698064.2f, 2010425.0f),
                            new TileMetricsUtil.Tile(3, 1113, 774392.0f, 2230249.0f),
                            new TileMetricsUtil.Tile(2, 1204, 526577.44f, 1516543.0f),
                            new TileMetricsUtil.Tile(3, 1114, 792397.9f, 2282106.0f),
                            new TileMetricsUtil.Tile(2, 1203, 657213.5f, 1892775.0f),
                            new TileMetricsUtil.Tile(3, 1115, 802558.3f, 2311368.0f),
                            new TileMetricsUtil.Tile(2, 1202, 667077.06f, 1921182.0f),
                            new TileMetricsUtil.Tile(3, 1116, 839689.56f, 2418306.0f),
                            new TileMetricsUtil.Tile(2, 1201, 667060.75f, 1921135.0f),
                            new TileMetricsUtil.Tile(3, 1110, 704187.8f, 2028061.0f),
                            new TileMetricsUtil.Tile(3, 1111, 694706.94f, 2000756.0f),
                            new TileMetricsUtil.Tile(2, 1209, 768146.1f, 2212261.0f),
                            new TileMetricsUtil.Tile(2, 1207, 738194.75f, 2126001.0f),
                            new TileMetricsUtil.Tile(2, 1208, 753541.6f, 2170200.0f),
                            new TileMetricsUtil.Tile(6, 2307, 611084.4f, 1759923.0f),
                            new TileMetricsUtil.Tile(6, 1110, 595422.2f, 1714816.0f),
                            new TileMetricsUtil.Tile(2, 1211, 801727.06f, 2308974.0f),
                            new TileMetricsUtil.Tile(6, 2308, 623703.8f, 1796267.0f),
                            new TileMetricsUtil.Tile(2, 1210, 782715.25f, 2254220.0f),
                            new TileMetricsUtil.Tile(6, 2309, 626620.1f, 1804666.0f),
                            new TileMetricsUtil.Tile(2, 1213, 840329.8f, 2420150.0f),
                            new TileMetricsUtil.Tile(2, 1212, 817961.44f, 2355729.0f),
                            new TileMetricsUtil.Tile(6, 2303, 564128.1f, 1624689.0f),
                            new TileMetricsUtil.Tile(6, 1114, 677938.9f, 1952464.0f),
                            new TileMetricsUtil.Tile(2, 1215, 870524.94f, 2507112.0f),
                            new TileMetricsUtil.Tile(6, 2304, 574673.25f, 1655059.0f),
                            new TileMetricsUtil.Tile(6, 1113, 662806.25f, 1908882.0f),
                            new TileMetricsUtil.Tile(2, 1214, 853374.94f, 2457720.0f),
                            new TileMetricsUtil.Tile(6, 2305, 588415.6f, 1694637.0f),
                            new TileMetricsUtil.Tile(6, 1112, 594454.1f, 1712028.0f),
                            new TileMetricsUtil.Tile(6, 2306, 598005.56f, 1722256.0f),
                            new TileMetricsUtil.Tile(6, 1111, 539406.94f, 1553492.0f),
                            new TileMetricsUtil.Tile(2, 1216, 876697.56f, 2524889.0f),
                            new TileMetricsUtil.Tile(2, 1314, 824424.94f, 2374344.0f),
                            new TileMetricsUtil.Tile(1, 2207, 706444.06f, 2034559.0f),
                            new TileMetricsUtil.Tile(2, 1313, 804652.4f, 2317399.0f),
                            new TileMetricsUtil.Tile(1, 2208, 720043.0f, 2073724.0f),
                            new TileMetricsUtil.Tile(6, 2301, 546032.25f, 1572573.0f),
                            new TileMetricsUtil.Tile(6, 1116, 706471.9f, 2034639.0f),
                            new TileMetricsUtil.Tile(2, 1316, 860230.5f, 2477464.0f),
                            new TileMetricsUtil.Tile(1, 2209, 729175.3f, 2100025.0f),
                            new TileMetricsUtil.Tile(6, 2302, 553479.1f, 1594020.0f),
                            new TileMetricsUtil.Tile(6, 1115, 691797.9f, 1992378.0f),
                            new TileMetricsUtil.Tile(2, 1315, 846660.75f, 2438383.0f),
                            new TileMetricsUtil.Tile(2, 1310, 756808.3f, 2179608.0f),
                            new TileMetricsUtil.Tile(2, 1312, 788834.7f, 2271844.0f),
                            new TileMetricsUtil.Tile(2, 1311, 774573.94f, 2230773.0f),
                            new TileMetricsUtil.Tile(2, 1115, 853926.0f, 2459307.0f),
                            new TileMetricsUtil.Tile(2, 1116, 858920.8f, 2473692.0f),
                            new TileMetricsUtil.Tile(1, 2202, 611577.44f, 1761343.0f),
                            new TileMetricsUtil.Tile(1, 2201, 639297.2f, 1841176.0f),
                            new TileMetricsUtil.Tile(2, 1111, 803672.9f, 2314578.0f),
                            new TileMetricsUtil.Tile(1, 2204, 665189.9f, 1915747.0f),
                            new TileMetricsUtil.Tile(2, 1112, 816662.44f, 2351988.0f),
                            new TileMetricsUtil.Tile(1, 2203, 487072.2f, 1402768.0f),
                            new TileMetricsUtil.Tile(2, 1113, 832510.4f, 2397630.0f),
                            new TileMetricsUtil.Tile(1, 2206, 690076.4f, 1987420.0f),
                            new TileMetricsUtil.Tile(2, 1114, 842916.3f, 2427599.0f),
                            new TileMetricsUtil.Tile(1, 2205, 676509.4f, 1948347.0f),
                            new TileMetricsUtil.Tile(2, 1110, 786951.4f, 2266420.0f),
                            new TileMetricsUtil.Tile(6, 1314, 726556.94f, 2092484.0f),
                            new TileMetricsUtil.Tile(6, 1313, 705262.5f, 2031156.0f),
                            new TileMetricsUtil.Tile(6, 1316, 762970.8f, 2197356.0f),
                            new TileMetricsUtil.Tile(6, 1315, 745886.06f, 2148152.0f),
                            new TileMetricsUtil.Tile(6, 2316, 784323.25f, 2258851.0f),
                            new TileMetricsUtil.Tile(6, 2314, 745560.06f, 2147213.0f),
                            new TileMetricsUtil.Tile(6, 2315, 764134.7f, 2200708.0f),
                            new TileMetricsUtil.Tile(6, 2312, 646565.6f, 1862109.0f),
                            new TileMetricsUtil.Tile(6, 2313, 717774.25f, 2067190.0f),
                            new TileMetricsUtil.Tile(6, 2310, 650334.4f, 1872963.0f),
                            new TileMetricsUtil.Tile(6, 2311, 607506.25f, 1749618.0f),
                            new TileMetricsUtil.Tile(6, 1310, 652161.44f, 1878225.0f),
                            new TileMetricsUtil.Tile(6, 1312, 643798.94f, 1854141.0f),
                            new TileMetricsUtil.Tile(6, 1311, 608755.2f, 1753215.0f),
                            new TileMetricsUtil.Tile(6, 1301, 543110.4f, 1564158.0f),
                            new TileMetricsUtil.Tile(6, 1308, 628543.75f, 1810206.0f),
                            new TileMetricsUtil.Tile(6, 1309, 632060.75f, 1820335.0f),
                            new TileMetricsUtil.Tile(6, 1306, 601763.9f, 1733080.0f),
                            new TileMetricsUtil.Tile(6, 1307, 612346.9f, 1763559.0f),
                            new TileMetricsUtil.Tile(6, 1304, 577054.5f, 1661917.0f),
                            new TileMetricsUtil.Tile(6, 1305, 589227.44f, 1696975.0f),
                            new TileMetricsUtil.Tile(6, 1302, 552162.1f, 1590227.0f),
                            new TileMetricsUtil.Tile(6, 1303, 564858.7f, 1626793.0f),
                            new TileMetricsUtil.Tile(2, 2112, 831845.1f, 2395714.0f),
                            new TileMetricsUtil.Tile(2, 2113, 869011.06f, 2502752.0f),
                            new TileMetricsUtil.Tile(2, 2114, 893550.3f, 2573425.0f),
                            new TileMetricsUtil.Tile(2, 2210, 781965.6f, 2252061.0f),
                            new TileMetricsUtil.Tile(2, 2115, 915250.7f, 2635922.0f),
                            new TileMetricsUtil.Tile(2, 2116, 935106.2f, 2693106.0f),
                            new TileMetricsUtil.Tile(2, 2215, 916326.4f, 2639020.0f),
                            new TileMetricsUtil.Tile(2, 2216, 934624.25f, 2691718.0f),
                            new TileMetricsUtil.Tile(2, 2211, 801881.2f, 2309418.0f),
                            new TileMetricsUtil.Tile(2, 2212, 831820.1f, 2395642.0f),
                            new TileMetricsUtil.Tile(2, 2213, 865767.3f, 2493410.0f),
                            new TileMetricsUtil.Tile(2, 2110, 771794.75f, 2222769.0f),
                            new TileMetricsUtil.Tile(2, 2214, 889628.06f, 2562129.0f),
                            new TileMetricsUtil.Tile(2, 2111, 804690.25f, 2317508.0f),
                            new TileMetricsUtil.Tile(2, 1108, 755969.44f, 2177192.0f),
                            new TileMetricsUtil.Tile(2, 1109, 773586.75f, 2227930.0f),
                            new TileMetricsUtil.Tile(2, 2209, 764854.1f, 2202780.0f),
                            new TileMetricsUtil.Tile(3, 2308, 665623.56f, 1916996.0f),
                            new TileMetricsUtil.Tile(2, 2208, 748762.5f, 2156436.0f),
                            new TileMetricsUtil.Tile(3, 2309, 677830.2f, 1952151.0f),
                            new TileMetricsUtil.Tile(1, 2312, 795999.25f, 2292478.0f),
                            new TileMetricsUtil.Tile(1, 2311, 768010.06f, 2211869.0f),
                            new TileMetricsUtil.Tile(1, 2310, 756285.75f, 2178103.0f),
                            new TileMetricsUtil.Tile(2, 2109, 716002.75f, 2062088.0f),
                            new TileMetricsUtil.Tile(1, 2316, 925979.8f, 2666822.0f),
                            new TileMetricsUtil.Tile(1, 2315, 896454.1f, 2581788.0f),
                            new TileMetricsUtil.Tile(1, 2314, 864106.56f, 2488627.0f),
                            new TileMetricsUtil.Tile(1, 2313, 819590.94f, 2360422.0f),
                            new TileMetricsUtil.Tile(2, 2103, 401740.97f, 1157014.0f),
                            new TileMetricsUtil.Tile(2, 2104, 395323.25f, 1138531.0f),
                            new TileMetricsUtil.Tile(2, 2101, 468776.03f, 1350075.0f),
                            new TileMetricsUtil.Tile(2, 2102, 351150.7f, 1011314.0f),
                            new TileMetricsUtil.Tile(2, 2107, 622077.4f, 1791583.0f),
                            new TileMetricsUtil.Tile(2, 2108, 616934.7f, 1776772.0f),
                            new TileMetricsUtil.Tile(2, 2105, 608495.8f, 1752468.0f),
                            new TileMetricsUtil.Tile(2, 2106, 654560.4f, 1885134.0f),
                            new TileMetricsUtil.Tile(2, 2206, 721626.7f, 2078285.0f),
                            new TileMetricsUtil.Tile(2, 1103, 658908.6f, 1897657.0f),
                            new TileMetricsUtil.Tile(3, 2307, 654492.7f, 1884939.0f),
                            new TileMetricsUtil.Tile(2, 2207, 732366.6f, 2109216.0f),
                            new TileMetricsUtil.Tile(2, 1102, 656570.8f, 1890924.0f),
                            new TileMetricsUtil.Tile(3, 2306, 645191.3f, 1858151.0f),
                            new TileMetricsUtil.Tile(2, 2204, 522755.2f, 1505535.0f),
                            new TileMetricsUtil.Tile(2, 1101, 655590.94f, 1888102.0f),
                            new TileMetricsUtil.Tile(3, 2305, 629939.56f, 1814226.0f),
                            new TileMetricsUtil.Tile(2, 2205, 695380.2f, 2002695.0f),
                            new TileMetricsUtil.Tile(3, 2304, 615780.2f, 1773447.0f),
                            new TileMetricsUtil.Tile(2, 2202, 651867.3f, 1877378.0f),
                            new TileMetricsUtil.Tile(2, 1107, 741657.25f, 2135973.0f),
                            new TileMetricsUtil.Tile(3, 2303, 603444.06f, 1737919.0f),
                            new TileMetricsUtil.Tile(2, 2203, 640879.8f, 1845734.0f),
                            new TileMetricsUtil.Tile(2, 1106, 727610.4f, 2095518.0f),
                            new TileMetricsUtil.Tile(3, 2302, 594963.9f, 1713496.0f),
                            new TileMetricsUtil.Tile(2, 1105, 705420.8f, 2031612.0f),
                            new TileMetricsUtil.Tile(3, 2301, 590563.9f, 1700824.0f),
                            new TileMetricsUtil.Tile(2, 2201, 658020.1f, 1895098.0f),
                            new TileMetricsUtil.Tile(2, 1104, 553236.8f, 1593322.0f),
                            new TileMetricsUtil.Tile(5, 1304, 527790.25f, 1520036.0f),
                            new TileMetricsUtil.Tile(1, 2303, 571845.1f, 1646914.0f),
                            new TileMetricsUtil.Tile(5, 1303, 514778.1f, 1482561.0f),
                            new TileMetricsUtil.Tile(1, 2302, 634523.25f, 1827427.0f),
                            new TileMetricsUtil.Tile(5, 1302, 503301.72f, 1449509.0f),
                            new TileMetricsUtil.Tile(1, 2305, 687812.1f, 1980899.0f),
                            new TileMetricsUtil.Tile(5, 1301, 489341.66f, 1409304.0f),
                            new TileMetricsUtil.Tile(1, 2304, 678138.5f, 1953039.0f),
                            new TileMetricsUtil.Tile(5, 1308, 551351.7f, 1587893.0f),
                            new TileMetricsUtil.Tile(5, 1307, 555872.56f, 1600913.0f),
                            new TileMetricsUtil.Tile(5, 1306, 554025.0f, 1595592.0f),
                            new TileMetricsUtil.Tile(1, 2301, 645645.44f, 1859459.0f),
                            new TileMetricsUtil.Tile(5, 1305, 542228.8f, 1561619.0f),
                            new TileMetricsUtil.Tile(5, 1309, 468386.78f, 1348954.0f),
                            new TileMetricsUtil.Tile(1, 2306, 694729.1f, 2000820.0f),
                            new TileMetricsUtil.Tile(1, 2307, 712121.9f, 2050911.0f),
                            new TileMetricsUtil.Tile(1, 2308, 727114.9f, 2094091.0f),
                            new TileMetricsUtil.Tile(1, 2309, 740285.75f, 2132023.0f),
                            new TileMetricsUtil.Tile(5, 1311, 632258.3f, 1820904.0f),
                            new TileMetricsUtil.Tile(5, 1310, 618028.44f, 1779922.0f),
                            new TileMetricsUtil.Tile(5, 1313, 673544.44f, 1939808.0f),
                            new TileMetricsUtil.Tile(5, 1312, 649723.25f, 1871203.0f),
                            new TileMetricsUtil.Tile(5, 1315, 710485.06f, 2046197.0f),
                            new TileMetricsUtil.Tile(5, 1314, 692534.4f, 1994499.0f),
                            new TileMetricsUtil.Tile(5, 1316, 736742.3f, 2121818.0f),
                            new TileMetricsUtil.Tile(7, 1109, 659062.8f, 1898101.0f),
                            new TileMetricsUtil.Tile(4, 2102, 575167.0f, 1656481.0f),
                            new TileMetricsUtil.Tile(4, 2101, 556701.7f, 1603301.0f),
                            new TileMetricsUtil.Tile(7, 1105, 607005.56f, 1748176.0f),
                            new TileMetricsUtil.Tile(7, 1106, 622574.6f, 1793015.0f),
                            new TileMetricsUtil.Tile(7, 1107, 634426.7f, 1827149.0f),
                            new TileMetricsUtil.Tile(7, 1108, 648441.3f, 1867511.0f),
                            new TileMetricsUtil.Tile(7, 1101, 514711.78f, 1482370.0f),
                            new TileMetricsUtil.Tile(7, 1102, 562830.9f, 1620953.0f),
                            new TileMetricsUtil.Tile(4, 2109, 652550.7f, 1879346.0f),
                            new TileMetricsUtil.Tile(7, 1103, 580733.7f, 1672513.0f),
                            new TileMetricsUtil.Tile(4, 2108, 641942.3f, 1848794.0f),
                            new TileMetricsUtil.Tile(7, 1104, 593361.8f, 1708882.0f),
                            new TileMetricsUtil.Tile(4, 2107, 633418.0f, 1824244.0f),
                            new TileMetricsUtil.Tile(4, 2106, 628767.3f, 1810850.0f),
                            new TileMetricsUtil.Tile(4, 2105, 608104.8f, 1751342.0f),
                            new TileMetricsUtil.Tile(4, 2104, 590379.8f, 1700294.0f),
                            new TileMetricsUtil.Tile(4, 2103, 582709.7f, 1678204.0f),
                            new TileMetricsUtil.Tile(7, 1302, 610862.5f, 1759284.0f),
                            new TileMetricsUtil.Tile(7, 1301, 562763.9f, 1620760.0f),
                            new TileMetricsUtil.Tile(7, 1308, 692898.56f, 1995548.0f),
                            new TileMetricsUtil.Tile(7, 1307, 679246.1f, 1956229.0f),
                            new TileMetricsUtil.Tile(7, 1309, 704346.9f, 2028519.0f),
                            new TileMetricsUtil.Tile(7, 1304, 640838.9f, 1845616.0f),
                            new TileMetricsUtil.Tile(7, 1303, 626787.1f, 1805147.0f),
                            new TileMetricsUtil.Tile(7, 1306, 667224.25f, 1921606.0f),
                            new TileMetricsUtil.Tile(7, 1305, 654089.56f, 1883778.0f),
                            new TileMetricsUtil.Tile(8, 1114, 757451.0f, 2181459.0f),
                            new TileMetricsUtil.Tile(4, 2111, 680774.25f, 1960630.0f),
                            new TileMetricsUtil.Tile(8, 1216, 805801.0f, 2320707.0f),
                            new TileMetricsUtil.Tile(8, 1113, 740480.2f, 2132583.0f),
                            new TileMetricsUtil.Tile(4, 2110, 540845.8f, 1557636.0f),
                            new TileMetricsUtil.Tile(8, 1116, 795597.2f, 2291320.0f),
                            new TileMetricsUtil.Tile(4, 2113, 750105.9f, 2160305.0f),
                            new TileMetricsUtil.Tile(8, 1115, 771454.1f, 2221788.0f),
                            new TileMetricsUtil.Tile(4, 2112, 722734.0f, 2081474.0f),
                            new TileMetricsUtil.Tile(8, 1110, 696113.9f, 2004808.0f),
                            new TileMetricsUtil.Tile(8, 1112, 724169.44f, 2085608.0f),
                            new TileMetricsUtil.Tile(7, 1116, 759220.1f, 2186554.0f),
                            new TileMetricsUtil.Tile(8, 1111, 709771.5f, 2044142.0f),
                            new TileMetricsUtil.Tile(7, 1114, 676538.2f, 1948430.0f),
                            new TileMetricsUtil.Tile(7, 1115, 742581.94f, 2138636.0f),
                            new TileMetricsUtil.Tile(8, 1211, 695630.9f, 2003417.0f),
                            new TileMetricsUtil.Tile(7, 1112, 682896.9f, 1966743.0f),
                            new TileMetricsUtil.Tile(8, 1210, 678184.7f, 1953172.0f),
                            new TileMetricsUtil.Tile(7, 1113, 671356.56f, 1933507.0f),
                            new TileMetricsUtil.Tile(8, 1213, 738898.25f, 2128027.0f),
                            new TileMetricsUtil.Tile(7, 1110, 666704.8f, 1920110.0f),
                            new TileMetricsUtil.Tile(4, 2115, 807847.56f, 2326601.0f),
                            new TileMetricsUtil.Tile(8, 1212, 719052.06f, 2070870.0f),
                            new TileMetricsUtil.Tile(7, 1111, 674252.06f, 1941846.0f),
                            new TileMetricsUtil.Tile(4, 2114, 778309.7f, 2241532.0f),
                            new TileMetricsUtil.Tile(8, 1215, 777625.3f, 2239561.0f),
                            new TileMetricsUtil.Tile(8, 1214, 753288.2f, 2169470.0f),
                            new TileMetricsUtil.Tile(4, 2116, 844673.56f, 2432660.0f),
                            new TileMetricsUtil.Tile(7, 1313, 753995.8f, 2171508.0f),
                            new TileMetricsUtil.Tile(7, 1312, 746043.75f, 2148606.0f),
                            new TileMetricsUtil.Tile(7, 1311, 730011.44f, 2102433.0f),
                            new TileMetricsUtil.Tile(7, 1310, 711337.8f, 2048653.0f),
                            new TileMetricsUtil.Tile(7, 1316, 816043.0f, 2350204.0f),
                            new TileMetricsUtil.Tile(7, 1315, 802815.94f, 2312110.0f),
                            new TileMetricsUtil.Tile(7, 1314, 771105.5f, 2220784.0f),
                            new TileMetricsUtil.Tile(8, 1101, 560968.06f, 1615588.0f),
                            new TileMetricsUtil.Tile(3, 2313, 763862.5f, 2199924.0f),
                            new TileMetricsUtil.Tile(3, 2314, 791914.56f, 2280714.0f),
                            new TileMetricsUtil.Tile(3, 2311, 706188.9f, 2033824.0f),
                            new TileMetricsUtil.Tile(3, 2312, 726022.9f, 2090946.0f),
                            new TileMetricsUtil.Tile(8, 1105, 614783.6f, 1770577.0f),
                            new TileMetricsUtil.Tile(7, 2301, 567577.75f, 1634624.0f),
                            new TileMetricsUtil.Tile(8, 1104, 603324.6f, 1737575.0f),
                            new TileMetricsUtil.Tile(8, 1103, 587471.2f, 1691917.0f),
                            new TileMetricsUtil.Tile(7, 2303, 628137.5f, 1809036.0f),
                            new TileMetricsUtil.Tile(3, 2315, 817940.94f, 2355670.0f),
                            new TileMetricsUtil.Tile(8, 1102, 573460.4f, 1651566.0f),
                            new TileMetricsUtil.Tile(7, 2302, 615363.5f, 1772247.0f),
                            new TileMetricsUtil.Tile(3, 2316, 859486.44f, 2475321.0f),
                            new TileMetricsUtil.Tile(8, 1109, 669675.7f, 1928666.0f),
                            new TileMetricsUtil.Tile(7, 2305, 655222.2f, 1887040.0f),
                            new TileMetricsUtil.Tile(8, 1108, 567472.2f, 1634320.0f),
                            new TileMetricsUtil.Tile(7, 2304, 638620.1f, 1839226.0f),
                            new TileMetricsUtil.Tile(8, 1107, 647659.7f, 1865260.0f),
                            new TileMetricsUtil.Tile(7, 2307, 683385.4f, 1968150.0f),
                            new TileMetricsUtil.Tile(8, 1106, 633415.25f, 1824236.0f),
                            new TileMetricsUtil.Tile(7, 2306, 672001.0f, 1935363.0f),
                            new TileMetricsUtil.Tile(7, 2309, 711445.44f, 2048963.0f),
                            new TileMetricsUtil.Tile(7, 2308, 696809.0f, 2006810.0f),
                            new TileMetricsUtil.Tile(3, 2310, 694548.94f, 2000301.0f),
                            new TileMetricsUtil.Tile(7, 2106, 624401.7f, 1798277.0f),
                            new TileMetricsUtil.Tile(7, 2107, 635721.1f, 1830877.0f),
                            new TileMetricsUtil.Tile(7, 2108, 651519.75f, 1876377.0f),
                            new TileMetricsUtil.Tile(7, 2109, 665493.4f, 1916621.0f),
                            new TileMetricsUtil.Tile(3, 2112, 705862.8f, 2032885.0f),
                            new TileMetricsUtil.Tile(3, 2111, 686178.1f, 1976193.0f),
                            new TileMetricsUtil.Tile(3, 2110, 695652.75f, 2003480.0f),
                            new TileMetricsUtil.Tile(3, 2116, 857244.06f, 2468863.0f),
                            new TileMetricsUtil.Tile(3, 2115, 824751.0f, 2375283.0f),
                            new TileMetricsUtil.Tile(3, 2114, 805906.2f, 2321010.0f),
                            new TileMetricsUtil.Tile(7, 2101, 523016.3f, 1506287.0f),
                            new TileMetricsUtil.Tile(3, 2113, 779751.7f, 2245685.0f),
                            new TileMetricsUtil.Tile(7, 2102, 568420.44f, 1637051.0f),
                            new TileMetricsUtil.Tile(7, 2103, 580344.06f, 1671391.0f),
                            new TileMetricsUtil.Tile(7, 2104, 593800.7f, 1710146.0f),
                            new TileMetricsUtil.Tile(7, 2105, 608100.3f, 1751329.0f),
                            new TileMetricsUtil.Tile(8, 1307, 678750.7f, 1954802.0f),
                            new TileMetricsUtil.Tile(7, 1205, 602532.6f, 1735294.0f),
                            new TileMetricsUtil.Tile(8, 1306, 664292.0f, 1913161.0f),
                            new TileMetricsUtil.Tile(7, 1204, 589775.3f, 1698553.0f),
                            new TileMetricsUtil.Tile(8, 1305, 667551.4f, 1922548.0f),
                            new TileMetricsUtil.Tile(7, 1207, 630500.3f, 1815841.0f),
                            new TileMetricsUtil.Tile(8, 1304, 656542.7f, 1890843.0f),
                            new TileMetricsUtil.Tile(7, 1206, 621207.6f, 1789078.0f),
                            new TileMetricsUtil.Tile(7, 1209, 656298.25f, 1890139.0f),
                            new TileMetricsUtil.Tile(7, 1208, 644718.75f, 1856790.0f),
                            new TileMetricsUtil.Tile(8, 1309, 706734.4f, 2035395.0f),
                            new TileMetricsUtil.Tile(8, 1308, 640287.8f, 1844029.0f),
                            new TileMetricsUtil.Tile(8, 1303, 645088.2f, 1857854.0f),
                            new TileMetricsUtil.Tile(7, 1201, 523716.66f, 1508304.0f),
                            new TileMetricsUtil.Tile(8, 1302, 634223.94f, 1826565.0f),
                            new TileMetricsUtil.Tile(8, 1301, 628762.8f, 1810837.0f),
                            new TileMetricsUtil.Tile(7, 1203, 579497.9f, 1668954.0f),
                            new TileMetricsUtil.Tile(7, 1202, 566370.1f, 1631146.0f),
                            new TileMetricsUtil.Tile(7, 2210, 659881.25f, 1900458.0f),
                            new TileMetricsUtil.Tile(7, 2211, 681405.9f, 1962449.0f),
                            new TileMetricsUtil.Tile(3, 2209, 665386.75f, 1916314.0f),
                            new TileMetricsUtil.Tile(7, 1216, 798989.56f, 2301090.0f),
                            new TileMetricsUtil.Tile(7, 1215, 755635.06f, 2176229.0f),
                            new TileMetricsUtil.Tile(7, 2214, 700608.3f, 2017752.0f),
                            new TileMetricsUtil.Tile(7, 2215, 781395.44f, 2250419.0f),
                            new TileMetricsUtil.Tile(7, 2315, 838348.56f, 2414444.0f),
                            new TileMetricsUtil.Tile(7, 2212, 707106.56f, 2036467.0f),
                            new TileMetricsUtil.Tile(7, 2316, 855710.4f, 2464446.0f),
                            new TileMetricsUtil.Tile(7, 2213, 694383.6f, 1999825.0f),
                            new TileMetricsUtil.Tile(7, 2313, 781775.3f, 2251513.0f),
                            new TileMetricsUtil.Tile(7, 1210, 660261.75f, 1901554.0f),
                            new TileMetricsUtil.Tile(7, 2314, 802218.4f, 2310389.0f),
                            new TileMetricsUtil.Tile(7, 2311, 739138.2f, 2128718.0f),
                            new TileMetricsUtil.Tile(7, 2216, 830322.9f, 2391330.0f),
                            new TileMetricsUtil.Tile(7, 2312, 761812.8f, 2194021.0f),
                            new TileMetricsUtil.Tile(7, 1214, 682258.6f, 1964905.0f),
                            new TileMetricsUtil.Tile(7, 2310, 719639.56f, 2072562.0f),
                            new TileMetricsUtil.Tile(7, 1213, 679747.9f, 1957674.0f),
                            new TileMetricsUtil.Tile(7, 1212, 695909.0f, 2004218.0f),
                            new TileMetricsUtil.Tile(7, 1211, 677979.8f, 1952582.0f),
                            new TileMetricsUtil.Tile(3, 2106, 631833.6f, 1819681.0f),
                            new TileMetricsUtil.Tile(3, 2107, 645582.6f, 1859278.0f),
                            new TileMetricsUtil.Tile(6, 1108, 576558.7f, 1660489.0f),
                            new TileMetricsUtil.Tile(3, 2108, 660356.56f, 1901827.0f),
                            new TileMetricsUtil.Tile(6, 1109, 582984.7f, 1678996.0f),
                            new TileMetricsUtil.Tile(3, 2109, 674721.9f, 1943199.0f),
                            new TileMetricsUtil.Tile(6, 1201, 515860.06f, 1485677.0f),
                            new TileMetricsUtil.Tile(6, 1106, 545413.5f, 1570791.0f),
                            new TileMetricsUtil.Tile(3, 2102, 582434.4f, 1677411.0f),
                            new TileMetricsUtil.Tile(6, 1202, 523040.94f, 1506358.0f),
                            new TileMetricsUtil.Tile(6, 1107, 553355.9f, 1593665.0f),
                            new TileMetricsUtil.Tile(3, 2103, 593973.56f, 1710644.0f),
                            new TileMetricsUtil.Tile(6, 1104, 516796.84f, 1488375.0f),
                            new TileMetricsUtil.Tile(3, 2104, 607844.75f, 1750593.0f),
                            new TileMetricsUtil.Tile(6, 1105, 530281.56f, 1527211.0f),
                            new TileMetricsUtil.Tile(3, 2105, 621182.6f, 1789006.0f),
                            new TileMetricsUtil.Tile(6, 1205, 560279.5f, 1613605.0f),
                            new TileMetricsUtil.Tile(6, 1102, 483174.28f, 1391542.0f),
                            new TileMetricsUtil.Tile(3, 2201, 567232.25f, 1633629.0f),
                            new TileMetricsUtil.Tile(6, 1206, 569772.56f, 1640945.0f),
                            new TileMetricsUtil.Tile(6, 1103, 500713.88f, 1442056.0f),
                            new TileMetricsUtil.Tile(3, 2202, 578353.1f, 1665657.0f),
                            new TileMetricsUtil.Tile(6, 1203, 538762.1f, 1551635.0f),
                            new TileMetricsUtil.Tile(3, 2203, 589069.44f, 1696520.0f),
                            new TileMetricsUtil.Tile(6, 1204, 548861.8f, 1580722.0f),
                            new TileMetricsUtil.Tile(6, 1101, 470429.16f, 1354836.0f),
                            new TileMetricsUtil.Tile(3, 2101, 571115.25f, 1644812.0f),
                            new TileMetricsUtil.Tile(3, 2204, 600790.6f, 1730277.0f),
                            new TileMetricsUtil.Tile(6, 1209, 600827.44f, 1730383.0f),
                            new TileMetricsUtil.Tile(3, 2205, 611413.9f, 1760872.0f),
                            new TileMetricsUtil.Tile(3, 2206, 625997.9f, 1802874.0f),
                            new TileMetricsUtil.Tile(6, 1207, 574950.0f, 1655856.0f),
                            new TileMetricsUtil.Tile(3, 2207, 638296.5f, 1838294.0f),
                            new TileMetricsUtil.Tile(6, 1208, 597518.4f, 1720853.0f),
                            new TileMetricsUtil.Tile(3, 2208, 653278.8f, 1881443.0f),
                            new TileMetricsUtil.Tile(7, 2201, 524665.25f, 1511036.0f),
                            new TileMetricsUtil.Tile(7, 2202, 567193.75f, 1633518.0f),
                            new TileMetricsUtil.Tile(7, 2203, 581718.4f, 1675349.0f),
                            new TileMetricsUtil.Tile(7, 2204, 591475.3f, 1703449.0f),
                            new TileMetricsUtil.Tile(7, 2209, 657746.9f, 1894311.0f),
                            new TileMetricsUtil.Tile(7, 2205, 603689.56f, 1738626.0f),
                            new TileMetricsUtil.Tile(7, 2206, 621389.2f, 1789601.0f),
                            new TileMetricsUtil.Tile(7, 2207, 630440.25f, 1815668.0f),
                            new TileMetricsUtil.Tile(7, 2208, 644510.4f, 1856190.0f),
                            new TileMetricsUtil.Tile(6, 2207, 580963.2f, 1673174.0f),
                            new TileMetricsUtil.Tile(6, 1210, 612937.5f, 1765260.0f),
                            new TileMetricsUtil.Tile(3, 2210, 685847.56f, 1975241.0f),
                            new TileMetricsUtil.Tile(6, 2206, 567367.7f, 1634019.0f),
                            new TileMetricsUtil.Tile(6, 1211, 539128.8f, 1552691.0f),
                            new TileMetricsUtil.Tile(3, 2211, 662213.5f, 1907175.0f),
                            new TileMetricsUtil.Tile(6, 2205, 555303.8f, 1599275.0f),
                            new TileMetricsUtil.Tile(6, 1212, 583076.7f, 1679261.0f),
                            new TileMetricsUtil.Tile(6, 2204, 545144.44f, 1570016.0f),
                            new TileMetricsUtil.Tile(6, 1213, 671834.4f, 1934883.0f),
                            new TileMetricsUtil.Tile(6, 2209, 592840.6f, 1707381.0f),
                            new TileMetricsUtil.Tile(6, 2208, 589674.3f, 1698262.0f),
                            new TileMetricsUtil.Tile(3, 2216, 850352.06f, 2449014.0f),
                            new TileMetricsUtil.Tile(6, 2203, 534482.6f, 1539310.0f),
                            new TileMetricsUtil.Tile(6, 1214, 689705.9f, 1986353.0f),
                            new TileMetricsUtil.Tile(3, 2214, 782345.1f, 2253154.0f),
                            new TileMetricsUtil.Tile(6, 2202, 522701.72f, 1505381.0f),
                            new TileMetricsUtil.Tile(6, 1215, 708256.94f, 2039780.0f),
                            new TileMetricsUtil.Tile(3, 2215, 805199.6f, 2318975.0f),
                            new TileMetricsUtil.Tile(6, 2201, 516852.4f, 1488535.0f),
                            new TileMetricsUtil.Tile(6, 1216, 743507.6f, 2141302.0f),
                            new TileMetricsUtil.Tile(3, 2212, 688010.75f, 1981471.0f),
                            new TileMetricsUtil.Tile(3, 2213, 757998.56f, 2183036.0f),
                            new TileMetricsUtil.Tile(8, 1316, 831816.6f, 2395632.0f),
                            new TileMetricsUtil.Tile(8, 1315, 814369.75f, 2345385.0f),
                            new TileMetricsUtil.Tile(8, 1312, 765417.0f, 2204401.0f),
                            new TileMetricsUtil.Tile(8, 1311, 747768.75f, 2153574.0f),
                            new TileMetricsUtil.Tile(8, 1314, 796930.2f, 2295159.0f),
                            new TileMetricsUtil.Tile(8, 1313, 783144.06f, 2255455.0f),
                            new TileMetricsUtil.Tile(8, 1310, 727851.0f, 2096211.0f),
                            new TileMetricsUtil.Tile(6, 2216, 759651.0f, 2187795.0f),
                            new TileMetricsUtil.Tile(6, 2215, 718353.8f, 2068859.0f),
                            new TileMetricsUtil.Tile(6, 2210, 602608.3f, 1735512.0f),
                            new TileMetricsUtil.Tile(6, 2212, 580998.94f, 1673277.0f),
                            new TileMetricsUtil.Tile(6, 2211, 530460.4f, 1527726.0f),
                            new TileMetricsUtil.Tile(6, 2214, 697545.1f, 2008930.0f),
                            new TileMetricsUtil.Tile(6, 2213, 676289.56f, 1947714.0f)
                    )
            ),
            new Testcase(new File(TEST_DATA_DIRECTORY, "HiSeq2500TileMetricsOut.bin"),
                    HashableTile.from(
                            new TileMetricsUtil.Tile(2, 1206, 653662.75f, 1891585.0f),
                            new TileMetricsUtil.Tile(2, 1205, 639157.0f, 1849608.0f),
                            new TileMetricsUtil.Tile(2, 1204, 624875.56f, 1808280.0f),
                            new TileMetricsUtil.Tile(2, 1203, 609763.4f, 1764548.0f),
                            new TileMetricsUtil.Tile(2, 1202, 597868.4f, 1730126.0f),
                            new TileMetricsUtil.Tile(2, 1201, 587080.25f, 1698907.0f),
                            new TileMetricsUtil.Tile(2, 1209, 695481.8f, 2012602.0f),
                            new TileMetricsUtil.Tile(2, 1207, 667445.2f, 1931469.0f),
                            new TileMetricsUtil.Tile(2, 1208, 678062.6f, 1962194.0f),
                            new TileMetricsUtil.Tile(1, 1216, 842967.25f, 2439399.0f),
                            new TileMetricsUtil.Tile(2, 1211, 731061.7f, 2115564.0f),
                            new TileMetricsUtil.Tile(1, 1215, 844388.56f, 2443512.0f),
                            new TileMetricsUtil.Tile(2, 1210, 698612.25f, 2021661.0f),
                            new TileMetricsUtil.Tile(1, 1214, 827993.3f, 2396067.0f),
                            new TileMetricsUtil.Tile(2, 1213, 775162.9f, 2243185.0f),
                            new TileMetricsUtil.Tile(1, 1213, 810556.9f, 2345609.0f),
                            new TileMetricsUtil.Tile(2, 1212, 755212.44f, 2185452.0f),
                            new TileMetricsUtil.Tile(1, 1212, 790185.2f, 2286657.0f),
                            new TileMetricsUtil.Tile(2, 1215, 819531.8f, 2371581.0f),
                            new TileMetricsUtil.Tile(1, 1211, 765546.5f, 2215357.0f),
                            new TileMetricsUtil.Tile(2, 1214, 796655.9f, 2305382.0f),
                            new TileMetricsUtil.Tile(1, 1210, 736556.9f, 2131466.0f),
                            new TileMetricsUtil.Tile(2, 1216, 832327.0f, 2408608.0f),
                            new TileMetricsUtil.Tile(1, 2207, 678786.94f, 1964290.0f),
                            new TileMetricsUtil.Tile(1, 2208, 693180.0f, 2005941.0f),
                            new TileMetricsUtil.Tile(1, 2209, 705999.4f, 2043038.0f),
                            new TileMetricsUtil.Tile(2, 1115, 825756.8f, 2389595.0f),
                            new TileMetricsUtil.Tile(2, 1116, 832026.75f, 2407739.0f),
                            new TileMetricsUtil.Tile(1, 2202, 606812.25f, 1756008.0f),
                            new TileMetricsUtil.Tile(1, 2201, 597893.6f, 1730199.0f),
                            new TileMetricsUtil.Tile(1, 2204, 635193.44f, 1838138.0f),
                            new TileMetricsUtil.Tile(2, 1111, 741915.9f, 2146974.0f),
                            new TileMetricsUtil.Tile(1, 2203, 618945.4f, 1791119.0f),
                            new TileMetricsUtil.Tile(2, 1112, 765042.7f, 2213899.0f),
                            new TileMetricsUtil.Tile(1, 2206, 665331.75f, 1925353.0f),
                            new TileMetricsUtil.Tile(2, 1113, 785456.2f, 2272972.0f),
                            new TileMetricsUtil.Tile(1, 2205, 652078.0f, 1886999.0f),
                            new TileMetricsUtil.Tile(2, 1114, 806461.25f, 2333757.0f),
                            new TileMetricsUtil.Tile(2, 1110, 706431.0f, 2044287.0f),
                            new TileMetricsUtil.Tile(1, 1203, 632849.44f, 1831355.0f),
                            new TileMetricsUtil.Tile(1, 1202, 621137.6f, 1797463.0f),
                            new TileMetricsUtil.Tile(1, 1205, 666326.25f, 1928231.0f),
                            new TileMetricsUtil.Tile(1, 1204, 650857.44f, 1883467.0f),
                            new TileMetricsUtil.Tile(1, 1201, 609630.4f, 1764163.0f),
                            new TileMetricsUtil.Tile(1, 1206, 682072.56f, 1973798.0f),
                            new TileMetricsUtil.Tile(1, 1207, 696573.44f, 2015761.0f),
                            new TileMetricsUtil.Tile(1, 1208, 710467.2f, 2055967.0f),
                            new TileMetricsUtil.Tile(1, 1209, 723957.25f, 2095005.0f),
                            new TileMetricsUtil.Tile(1, 2115, 861918.0f, 2494239.0f),
                            new TileMetricsUtil.Tile(1, 2116, 888303.5f, 2570594.0f),
                            new TileMetricsUtil.Tile(1, 2111, 747951.8f, 2164441.0f),
                            new TileMetricsUtil.Tile(1, 2112, 780144.2f, 2257600.0f),
                            new TileMetricsUtil.Tile(1, 2113, 807111.94f, 2335640.0f),
                            new TileMetricsUtil.Tile(1, 2114, 831982.1f, 2407610.0f),
                            new TileMetricsUtil.Tile(1, 2210, 719660.9f, 2082572.0f),
                            new TileMetricsUtil.Tile(1, 2211, 750860.75f, 2172859.0f),
                            new TileMetricsUtil.Tile(1, 2212, 778388.4f, 2252519.0f),
                            new TileMetricsUtil.Tile(1, 2110, 718140.75f, 2078173.0f),
                            new TileMetricsUtil.Tile(1, 2213, 800401.44f, 2316221.0f),
                            new TileMetricsUtil.Tile(1, 2214, 825330.4f, 2388361.0f),
                            new TileMetricsUtil.Tile(1, 2215, 854579.25f, 2473002.0f),
                            new TileMetricsUtil.Tile(1, 2216, 873627.06f, 2528123.0f),
                            new TileMetricsUtil.Tile(2, 2112, 755250.8f, 2185563.0f),
                            new TileMetricsUtil.Tile(2, 2113, 779303.75f, 2255168.0f),
                            new TileMetricsUtil.Tile(2, 2114, 806558.0f, 2334037.0f),
                            new TileMetricsUtil.Tile(2, 2210, 688795.8f, 1993254.0f),
                            new TileMetricsUtil.Tile(2, 2115, 837416.1f, 2423335.0f),
                            new TileMetricsUtil.Tile(2, 2116, 857135.0f, 2480398.0f),
                            new TileMetricsUtil.Tile(2, 2215, 807698.7f, 2337338.0f),
                            new TileMetricsUtil.Tile(2, 2216, 828607.06f, 2397843.0f),
                            new TileMetricsUtil.Tile(2, 2211, 713208.2f, 2063899.0f),
                            new TileMetricsUtil.Tile(2, 2212, 737697.94f, 2134768.0f),
                            new TileMetricsUtil.Tile(2, 2213, 757521.9f, 2192135.0f),
                            new TileMetricsUtil.Tile(2, 2110, 704750.5f, 2039424.0f),
                            new TileMetricsUtil.Tile(2, 2214, 781637.4f, 2261921.0f),
                            new TileMetricsUtil.Tile(2, 2111, 732328.9f, 2119231.0f),
                            new TileMetricsUtil.Tile(2, 1108, 691490.9f, 2001053.0f),
                            new TileMetricsUtil.Tile(2, 1109, 709288.1f, 2052555.0f),
                            new TileMetricsUtil.Tile(2, 2209, 679636.7f, 1966749.0f),
                            new TileMetricsUtil.Tile(2, 2208, 666553.6f, 1928889.0f),
                            new TileMetricsUtil.Tile(2, 2109, 697222.4f, 2017639.0f),
                            new TileMetricsUtil.Tile(2, 2103, 610412.0f, 1766425.0f),
                            new TileMetricsUtil.Tile(2, 2104, 624860.7f, 1808237.0f),
                            new TileMetricsUtil.Tile(2, 2101, 586815.2f, 1698140.0f),
                            new TileMetricsUtil.Tile(2, 2102, 598569.2f, 1732154.0f),
                            new TileMetricsUtil.Tile(2, 2107, 667244.44f, 1930888.0f),
                            new TileMetricsUtil.Tile(2, 2108, 680053.44f, 1967955.0f),
                            new TileMetricsUtil.Tile(2, 2105, 638144.5f, 1846678.0f),
                            new TileMetricsUtil.Tile(2, 2106, 652707.25f, 1888820.0f),
                            new TileMetricsUtil.Tile(2, 2206, 640678.2f, 1854010.0f),
                            new TileMetricsUtil.Tile(2, 1103, 619946.8f, 1794017.0f),
                            new TileMetricsUtil.Tile(2, 2207, 653583.6f, 1891356.0f),
                            new TileMetricsUtil.Tile(2, 1102, 607545.56f, 1758130.0f),
                            new TileMetricsUtil.Tile(2, 2204, 612980.25f, 1773857.0f),
                            new TileMetricsUtil.Tile(2, 1101, 594830.2f, 1721334.0f),
                            new TileMetricsUtil.Tile(2, 2205, 626773.44f, 1813772.0f),
                            new TileMetricsUtil.Tile(2, 2202, 586832.5f, 1698190.0f),
                            new TileMetricsUtil.Tile(2, 1107, 678805.6f, 1964344.0f),
                            new TileMetricsUtil.Tile(2, 2203, 599638.75f, 1735249.0f),
                            new TileMetricsUtil.Tile(2, 1106, 665179.7f, 1924913.0f),
                            new TileMetricsUtil.Tile(2, 1105, 650463.2f, 1882326.0f),
                            new TileMetricsUtil.Tile(2, 2201, 576017.75f, 1666894.0f),
                            new TileMetricsUtil.Tile(2, 1104, 635376.56f, 1838668.0f),
                            new TileMetricsUtil.Tile(1, 1108, 707248.25f, 2046652.0f),
                            new TileMetricsUtil.Tile(1, 1107, 694722.6f, 2010405.0f),
                            new TileMetricsUtil.Tile(1, 1109, 720901.8f, 2086163.0f),
                            new TileMetricsUtil.Tile(1, 1101, 604748.94f, 1750037.0f),
                            new TileMetricsUtil.Tile(1, 1102, 618156.06f, 1788835.0f),
                            new TileMetricsUtil.Tile(1, 1103, 630256.7f, 1823852.0f),
                            new TileMetricsUtil.Tile(1, 1104, 644520.2f, 1865128.0f),
                            new TileMetricsUtil.Tile(1, 1105, 663272.5f, 1919394.0f),
                            new TileMetricsUtil.Tile(1, 1106, 680005.06f, 1967815.0f),
                            new TileMetricsUtil.Tile(1, 2102, 600601.44f, 1738035.0f),
                            new TileMetricsUtil.Tile(1, 2103, 613472.7f, 1775282.0f),
                            new TileMetricsUtil.Tile(1, 2101, 589004.4f, 1704475.0f),
                            new TileMetricsUtil.Tile(1, 2106, 662705.8f, 1917754.0f),
                            new TileMetricsUtil.Tile(1, 2107, 676511.4f, 1957705.0f),
                            new TileMetricsUtil.Tile(1, 2104, 627960.75f, 1817208.0f),
                            new TileMetricsUtil.Tile(1, 2105, 646421.8f, 1870631.0f),
                            new TileMetricsUtil.Tile(1, 1112, 787963.56f, 2280228.0f),
                            new TileMetricsUtil.Tile(1, 1113, 807714.6f, 2337384.0f),
                            new TileMetricsUtil.Tile(1, 2109, 704151.3f, 2037690.0f),
                            new TileMetricsUtil.Tile(1, 1110, 732698.3f, 2120300.0f),
                            new TileMetricsUtil.Tile(1, 2108, 689368.8f, 1994912.0f),
                            new TileMetricsUtil.Tile(1, 1111, 761022.75f, 2202266.0f),
                            new TileMetricsUtil.Tile(1, 1116, 836718.1f, 2421315.0f),
                            new TileMetricsUtil.Tile(1, 1114, 818766.75f, 2369367.0f),
                            new TileMetricsUtil.Tile(1, 1115, 829341.0f, 2399967.0f)
                    )
            )
    );
}
