## This file is part of mlpy.
## DWT
   
## This code is written by Davide Albanese, <albanese@fbk.eu>.
## (C) 2009 Fondazione Bruno Kessler - Via Santa Croce 77, 38100 Trento, ITALY.

## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

import dwtcore

__all__ = ['dwt', 'idwt']

def dwt(x, wf, k):
    """
    Discrete Wavelet Tranform
    
    :Parameters:
      x : 1d ndarray float (the length is restricted to powers of two)
        data
      wf : string ('d': daubechies, 'h': haar, 'b': bspline)
         wavelet type
      k : integer
        member of the wavelet family
        
        * daubechies : k = 4, 6, ..., 20 with k even
        * haar : the only valid choice of k is k = 2
        * bspline : k = 103, 105, 202, 204, 206, 208, 301, 303, 305 307, 309
    
    :Returns:
      X : 1d ndarray float
        discrete wavelet transformed data

    Example:
    
    >>> import numpy as np
    >>> import mlpy
    >>> x = np.array([1,2,3,4,3,2,1,0])
    >>> mlpy.dwt(x=x, wf='d', k=6)
    array([ 5.65685425,  3.41458985,  0.29185347, -0.29185347, -0.28310081,
           -0.07045258,  0.28310081,  0.07045258])
    """

    return dwtcore.dwt(x, wf, k)


def idwt(X, wf, k):
    """
    Inverse Discrete Wavelet Tranform
    
    :Parameters:
      X : 1d ndarray float
        discrete wavelet transformed data
      wf : string ('d': daubechies, 'h': haar, 'b': bspline)
         wavelet type
      k : integer
        member of the wavelet family
        
        * daubechies : k = 4, 6, ..., 20 with k even
        * haar : the only valid choice of k is k = 2
        * bspline : k = 103, 105, 202, 204, 206, 208, 301, 303, 305 307, 309

    :Returns:
      x : 1d ndarray float
        data

    Example:
    
    >>> import numpy as np
    >>> import mlpy
    >>> X = np.array([ 5.65685425,  3.41458985,  0.29185347, -0.29185347, -0.28310081,
    ...               -0.07045258,  0.28310081,  0.07045258])
    >>> mlpy.idwt(X=X, wf='d', k=6)
    array([  1.00000000e+00,   2.00000000e+00,   3.00000000e+00,
             4.00000000e+00,   3.00000000e+00,   2.00000000e+00,
             1.00000000e+00,  -3.53954610e-09])
    """

    return dwtcore.idwt(X, wf, k)
