/* boardBitMask.h
 */
#ifndef _BOARD_BIT_MASK_H
#define _BOARD_BIT_MASK_H
#include "osl/square.h"
#include "osl/misc/carray.h"
#include "osl/misc/carray2d.h"
#include <iosfwd>

namespace osl
{
namespace effect
{
  /**
   * SSE2も可なのだが，あまり良い方法が見つからない
   */

#ifdef USE_XMM
    typedef int v4sf __attribute__ ((mode(V4SF))); // works
#endif
  struct BoardBitMask{
#ifdef USE_XMM
    union {
      v4sf xmm;
      CArray<unsigned long long,2> mask;
      CArray<unsigned char,16> bMask;
    };
#else
    union {
      CArray<unsigned long long,2> mask;
      CArray<unsigned char,16> bMask;
    };
#endif
    friend BoardBitMask& operator&=(BoardBitMask& lhs,BoardBitMask const& rhs);
    friend BoardBitMask& operator^=(BoardBitMask& lhs,BoardBitMask const& rhs);
    friend BoardBitMask operator^(BoardBitMask& src1,BoardBitMask const& src2);
  public:
    BoardBitMask(){}
    template<class State>
    explicit BoardBitMask(State const& st){
      clearAll();
      for(int y=1;y<=9;y++){
	for(int x=1;x<=9;x++){
	  Square position(x,y);
	  if (st.pieceAt(position).isEmpty())
	    setBit(positionToOffset(position));
	}
      }
    }
    /**
     * すべてのビットが0のものを作る
     */
    void clearAll(){mask[0]=mask[1]=0ull;}
    void setAll(){mask[0]=mask[1]=static_cast<unsigned long long>(-1ll);}
    /**
     * 0-8 11-19 22-30
     * 33-41 44-52 55-63
     * 66-74 77-85 88-96
     * でやってみる
     * 香車と角の利きをなんとかするアイデアもあったのだが
     */
    static int positionToOffset(Square pos){
      assert(pos.isOnBoard());
      const int x=pos.x();
      const int y=pos.y();
      return (x-1)*11+(y-1);
    }
    /**
     * @param offset(0-96) - bitをセットする
     */
    void setBit(int offset){
      assert(0<=offset && offset<=96);
      int index=offset>>6;
      unsigned long long tmpMask=1ull<<(offset&63);
      assert((index == 0) || (index == 1));
      mask[index]|= tmpMask;
    }
    void setBit(Square pos){
      setBit(positionToOffset(pos));
    }
    /**
     * @param offset(0-96) - bitをクリアする
     */
    void clearBit(int offset){
      assert(0<=offset && offset<=96);
      int index=offset>>6;
      unsigned long long tmpMask=1ull<<(offset&63);
      assert((index == 0) || (index == 1));
      mask[index]&= ~tmpMask;
    }
    void clearBit(Square pos){
      clearBit(positionToOffset(pos));
    }
    bool isZero() const{
      return (mask[0]|mask[1])==0ull;
    }
    BoardBitMask& operator=(BoardBitMask const& rhs){
      if (this == &rhs)
	return *this;
      
#ifdef USE_XMM
      xmm=rhs.xmm;
#else
      mask[0]=rhs.mask[0];
      mask[1]=rhs.mask[1];
#endif
      return *this;
    }

  };

		      
  inline BoardBitMask& operator^=(BoardBitMask& lhs,BoardBitMask const& rhs){
#ifdef USE_XMM
    lhs.xmm=__builtin_ia32_xorps(lhs.xmm,rhs.xmm);
#else
    lhs.mask[0]^=rhs.mask[0];
    lhs.mask[1]^=rhs.mask[1];
#endif
    return lhs;
  }

  inline BoardBitMask operator^(BoardBitMask const& lhs,BoardBitMask const& rhs){
#ifdef USE_XMM
    BoardBitMask ret=lhs;
    ret.xmm=__builtin_ia32_xorps(lhs.xmm,rhs.xmm);
    return ret;
#else
    BoardBitMask ret=lhs;
    ret.mask[0]^=rhs.mask[0];
    ret.mask[1]^=rhs.mask[1];
    return ret;
#endif
  }
  std::ostream& operator<<(std::ostream& os,BoardBitMask const& boardBitMask);

  class BoardBitMaskTable{
    CArray<BoardBitMask, Square::SIZE> maskOfSquare;
    /**
     * lanceに関しては作らなくても良いかも
     * lanceBetweenMask[from][to] が non all 0 なら黒からの利きがある
     */
    CArray2d<BoardBitMask,Square::SIZE,Square::SIZE> rookBetweenMask;
    /**
     *
     */
    CArray2d<BoardBitMask, Square::SIZE,Square::SIZE> lanceBetweenMask;
    CArray2d<BoardBitMask, Square::SIZE,Square::SIZE> bishopBetweenMask;
  private:
    void initMaskOfSquare();
    void initBetweenMask();
  public:
    BoardBitMaskTable();
    const BoardBitMask& getMask(Square pos) const{
      assert(pos.isOnBoard());
      return maskOfSquare[pos.index()];
    }
    const BoardBitMask& getRookMask(Square from,Square to) const{
      assert(from.isOnBoard() && to.isOnBoard());
      return rookBetweenMask[from.index()][to.index()];
    }
    const BoardBitMask& getBishopMask(Square from,Square to) const{
      assert(from.isOnBoard() && to.isOnBoard());
      return bishopBetweenMask[from.index()][to.index()];
    }
    const BoardBitMask& getLanceMask(Square from,Square to) const{
      assert(from.isOnBoard() && to.isOnBoard());
      return lanceBetweenMask[from.index()][to.index()];
    }
  };
#if 0
  // テーブル削除予定
  extern const BoardBitMaskTable Board_Bit_Mask_Table;
#endif
} // namespace effect
} // namespace osl
#endif // _BOARD_BIT_MASK_H
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
