/*
 * Copyright (c) 2003-2012
 * Distributed Systems Software.  All rights reserved.
 * See the file LICENSE for redistribution information.
 *
 * $Id: html.h 2621 2013-01-22 17:56:05Z brachman $
 */

#ifndef _HTML_H_
#define _HTML_H_

#include "ds.h"

/*
 * Detailed format for one cell (a td/th tag)
 */
typedef struct Html_cell_desc {
  char *id;
  char *class;
  int width;
  char *align;
  char *valign;
} Html_cell_desc;

/*
 * A formatted-to-memory cell (a td/th tag)
 */
typedef struct Html_cell {
  char *id;
  char *class;
  int width;
  char *align;
  char *valign;
  char *attrs;
  char *content;

  /* Private */
  int row;
  int column;
} Html_cell;

typedef struct Html_table {
  Ds *ds;
  /*
   * If auto_row_nclasses is non-zero, this is the prefix of the class attribute
   * value for the tr tag; if auto_row_nclasses is zero, this is the class
   * for every tr tag.
   */
  char *row_class;
  /*
   * If non-zero, the modulus for class attribute name suffixes for the
   * tr tag; e.g., <tr class="class_1" ...>, <tr class="class_2" ...>
   * This is mainly used to set different background colours for adjacent
   * rows to improve readability.
   */
  int auto_row_nclasses;
  /*
   * Prefix of the class attribute value for the td tag; e.g.,
   * <td class="column_1" ...>, <td class="column_2" ...>
   */
  char *auto_column_class;
  /*
   * If non-zero, use the value of auto_column_class as the class attribute
   * value for every td tag; e.g.,
   * <td class="el" ...>, <td class="el" ...>
   */
  int repeat_column_class;
  /* Text to insert in empty cells if no content is specified */
  char *default_cell_contents;
  /* Number of columns */
  int max_cols;
  Dsvec *cell_desc;			/* Vector of cell descriptors */
  /* Emit the th tag instead of td -- XXX fixme */
  int in_header;

  /* Private */
  int in_row;		/* Non-zero if a row is being filled. */
  int curr_col;		/* Current column being filled, starting at zero. */
  int curr_row;		/* Current row being filled, starting at zero. */
} Html_table;

#ifdef __cplusplus
extern "C" {
#endif

extern Html_table *html_table(Ds *ods, Html_table *otab);
extern void html_table_free(Html_table *tab);
extern int html_table_begin(Html_table *tab, Dsvec *attrs, int max_cols);
extern int html_table_end(Html_table *tab);
extern void html_row_begin(Html_table *tab);
extern void html_row_end(Html_table *tab);
extern void html_cell(Html_table *tab, char *content);
extern void html_cells(Html_table *tab, int ncells, ...);
extern void html_cellf(Html_table *tab, char *fmt, ...);
extern void html_cell2(Html_table *tab, char *val1, char *val2);
extern int html_cell_format(Html_table *tab, char *content, Html_cell *cell);
extern void html_cell_emit(Html_table *tab, Html_cell *cell);

extern char *html_span(char *class, char *text);

#ifdef __cplusplus
}
#endif

#endif
