# +========================================================================+
# || module: CipUX::Passwd                                                ||
# ||                                                                      ||
# || Module for cipuxpasswd skript. XML-RPC client to change own          ||
# || password.                                                            ||
# ||                                                                      ||
# || Copyright (C) 2009 by Christian Kuelker                              ||
# ||                                                                      ||
# || License: GNU General Public License - GPL version 2                  ||
# ||          or (at you opinion) any later version.                      ||
# ||                                                                      ||
# +========================================================================+
#  ID:       $Id$
#  Revision: $Revision$
#  Head URL: $HeadURL$
#  Date:     $Date$
#  Source:   $Source$

package CipUX::Passwd;

use warnings;
use strict;
use CipUX::RPC::Client;
use Class::Std;
use English qw( -no_match_vars);
use Readonly;

use base qw(CipUX);

{    # BEGIN CLASS

    use version; our $VERSION = qv('3.4.0.3');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safer

    # +======================================================================+
    # || CONST                                                              ||
    # +======================================================================+
    Readonly::Scalar my $EMPTY_STRING       => q{};
    Readonly::Scalar my $CIPUX_DEFAULT_PORT => 8001;

    # +======================================================================+
    # || OBJECT                                                             ||
    # +======================================================================+
    my %cipuxpasswd_cfg : ATTR( :get<cipuxpasswd_cfg> );
    my %name_of : ATTR( init_arg => 'name');

    # +======================================================================+
    # || GLOBAL                                                             ||
    # +======================================================================+
    my $debug = 0;
    my $url   = $EMPTY_STRING;
    my $host  = 'localhost';
    my $port  = $CIPUX_DEFAULT_PORT;
    my $ssl   = 0;

    # +======================================================================+
    # || CONSTRUCTOR                                                        ||
    # +======================================================================+
    sub BUILD {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $ident, $arg_r ) = @_;

        # add prefix for cfg, if needed
        my $pref
            = ( exists $arg_r->{pref} )
            ? $self->l( $arg_r->{pref} )
            : $EMPTY_STRING;

        # +------------------------------------------------------------------+
        # | main

        # Enable this lines, if you would like to use storage configuration
        # with cache.
        # (With cache you have to set rights for /var/cache/cipux accordingly
        #  if you would like to use this programm with other then root.
        #  Example: chown -R root:admin /var/cache/cipux &&
        #           chmod -R g+rX /var/cache/cipux )
        #$cipuxpasswd_cfg{$ident}
        #    = $self->cfg( { 'sub' => 'storage', pref => $pref } );

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +======================================================================+
    # || DESTRUCTOR                                                         ||
    # +======================================================================+
    sub DEMOLISH {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $ident ) = @_;

        # +------------------------------------------------------------------+
        # | main
        delete $cipuxpasswd_cfg{$ident};
        delete $name_of{ ident $self};

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +======================================================================+
    # || run                                                                ||
    # +======================================================================+
    sub run {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # +------------------------------------------------------------------+
        # | prepare

        # config overwrites global variables:
        $host
            = ( exists $cipuxpasswd_cfg{ ident $self}->{host} )
            ? $cipuxpasswd_cfg{ ident $self}->{host}
            : 'localhost';

        $port
            = ( exists $cipuxpasswd_cfg{ ident $self}->{port} )
            ? $cipuxpasswd_cfg{ ident $self}->{port}
            : $CIPUX_DEFAULT_PORT;

        $ssl
            = ( exists $cipuxpasswd_cfg{ ident $self}->{ssl} )
            ? $cipuxpasswd_cfg{ ident $self}->{ssl}
            : 0;

        # url  = 'https://localhost:8000/RPC2';
        # url  = 'http://localhost:8001/RPC2';
        my $proto = 'http';
        if ($ssl) {
            $proto = 'https';
        }

        $url
            = $proto . q{://}
            . $self->l($host) . q{:}
            . $self->l($port)
            . q{/RPC2};

        my $prompt = "$name_of{ ident $self}:";

        # +------------------------------------------------------------------+
        # | main

        my $rpc = CipUX::RPC::Client->new(
            {
                url     => $url,
                client  => $name_of{ ident $self},
                version => "$VERSION"
            }
        );

        if ( $rpc->rpc_ping() ) {
            $self->out("$prompt Server [$url] is available.\n");
        }
        else {
            $self->out("$prompt Server [$url] is NOT available.\n");
            return 0;
        }

        if ( $rpc->rpc_login() ) {
            $self->out("$prompt Authentication sucessful.\n");
        }
        else {
            $self->out("$prompt Login or password wrong!\n");
            return 0;
        }
        my $login = $rpc->get_login();
        $self->out("$prompt Please type two times an identical password.\n");
        my $cipux   = CipUX->new();
        my $npwmsg1 = "$prompt New password: ";
        my $p1      = $cipux->password_prompt( { prompt => $npwmsg1 } );
        my $npwmsg2 = "$prompt Re-enter new password: ";
        my $p2      = $cipux->password_prompt( { prompt => $npwmsg2 } );

        my $error = "$prompt ERROR:";
        my $todo  = ' Please try again.';

        if (    defined $p1
            and $p1 ne $EMPTY_STRING
            and $p1 eq $p2
            and $rpc->rpc_selfpasswd( { password => $p1 } ) )
        {
            $self->out("$prompt Password information changed for $login.\n");
        }
        else {
            $self->out("$prompt Set password was not sucessful.\n");
        }

        if ( not defined $p1 or $p1 eq $EMPTY_STRING ) {
            $self->out("$error First input is not given!$todo\n");
        }
        if ( not defined $p2 or $p2 eq $EMPTY_STRING ) {
            $self->out("$error Second input is not given!$todo\n");
        }

        if ( $p1 ne $p2 ) {
            $self->out("$error The two inputs are not identical!$todo\n");
        }

        if ( $rpc->rpc_logout() ) {
            $self->out("$prompt Logout sucessful.\n");
        }

        # +------------------------------------------------------------------+
        # | API
        return 1;

    }

}    # END INSIDE OUT CLASS

1;   # Magic true value required at end of module

__END__

=pod

=for stopwords cipuxpasswd RPC XML-RPC CipUX Destructor webpage Kuelker

=head1 NAME

CipUX::Passwd - XML-RPC client to change own password


=head1 VERSION

This document describes CipUX::Passwd version 3.4.0.3


=head1 SYNOPSIS

 use warnings;
 use strict;
 use CipUX::Passwd;
 use version; our $VERSION = qv('3.4.0.0');

 my $cipux = CipUX::Passwd->new( { name=>'cipuxpasswd' } );

 $cipux->run();

 exit 0;

=head1 DESCRIPTION

Implements a XML-RPC client to change once own password using the CipUX XML-RPC
server.

=head1 SUBROUTINES/METHODS

=head2 BUILD

Constructor.


=head2 DEMOLISH

Destructor.

=head2 run

Main subroutine.


=head1 DIAGNOSTICS

=over

=item C<< First input is not given! Please try again >>

=item C<< Second input is not given! Please try again >>

=item C<< The two inputs are not identical! Please try again >>

=back

=head1 CONFIGURATION AND ENVIRONMENT

CipUX::Passwd requires no configuration files or environment
variables. However if there is a configuration file it will be used.


=head1 DEPENDENCIES

        CipUX
        CipUX::RPC::Client
        Class::Std
        English
        Readonly
        version


=head1 INCOMPATIBILITIES

None reported.


=head1 BUGS AND LIMITATIONS

No bugs have been reported.


=head1 SEE ALSO

See the CipUX webpage and the manual at L<http://www.cipux.org>

See the mailing list L<http://sympa.cipworx.org/wws/info/cipux-devel>


=head1 AUTHOR

Christian Kuelker  christian.kuelker@cipworx.org

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2009, Christian Kuelker. All rights reserved.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA

=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENSE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

=cut

