/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <libburn/libburn.h>

#include "cdw_libburn.h"
#include "cdw_debug.h"
#include "cdw_task.h"


cdw_rv_t cdw_libburn_get_meta_info(cdw_disc_t *disc, const char *drive_path)
{
	int rv = burn_initialize();
	if (rv == 0) {
		cdw_vdm ("ERROR: error while initializing library\n");
		return CDW_ERROR;
	}

	struct burn_drive_info *drive_infos = (struct burn_drive_info *) NULL;

	char *path = drive_path; /* this will give compiler warnings
				    but it's better to get the warning
				    for code specific to cdw rather than
				    a warning about improper use of
				    external API */
	rv = burn_drive_scan_and_grab(&drive_infos, path, 0);
	if (rv == 0) {
		cdw_vdm ("ERROR: error while scanning: drive \"%s\" not found\n", drive_path);
		burn_finish();
		return CDW_ERROR;
	} else if (rv == -1) {
		cdw_vdm ("ERROR: error while scanning: other error\n");
		burn_finish();
		return CDW_ERROR;
	} else {
		;
	}

	/* writing speeds */
	struct burn_speed_descriptor *speed_list = (struct burn_speed_descriptor *) NULL;
	rv = burn_drive_get_speedlist(drive_infos[0].drive, &speed_list);
	if (rv != 1) {
		cdw_vdm ("ERROR: error while getting speedlist\n");
		burn_drive_info_free(drive_infos);
		burn_finish();
		return CDW_ERROR;
	}

	struct burn_speed_descriptor *head = speed_list;

	int speed_factor = 0;
	if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		speed_factor = 176; /* 176.4 k/s to be exact (k = 1000 B); source: libburn.h */
	} else if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		speed_factor = 1385; /* 1385 k/s (k = 1000 B); source: libburn.h */
	} else {
		cdw_vdm ("ERROR: invalid disc simple type %d\n", disc->cdio->simple_type);
		burn_drive_info_free(drive_infos);
		burn_finish();
		return CDW_ERROR;
	}


	int i = 0;
	do {
		disc->write_speeds.speeds[i] = head->write_speed / speed_factor;
		cdw_vdm ("INFO: speed #%d = %d -> %d\n", i, head->write_speed, disc->write_speeds.speeds[i]);
		i++;
		head = head->next;
	} while (head != (struct burn_speed_descriptor *) NULL
		 && i < CDW_DISC_N_SPEEDS_MAX);

	burn_drive_free_speedlist(&speed_list);


	/* tao/sao/dao */
	struct burn_multi_caps *caps = (struct burn_multi_caps *) NULL;
	rv = burn_disc_get_multi_caps(drive_infos[0].drive, BURN_WRITE_NONE,
				      &caps, 0); /* 0 - unused arg, submit 0 */
	if (rv != 1) {
		cdw_vdm ("ERROR: error while getting capabilities\n");
	} else {
		if (caps->might_do_tao) {
			cdw_vdm ("INFO: might do tao\n");
			disc->accepted_disc_modes |= CDW_DISC_MODE_TAO;
		}
		if (caps->might_do_sao) {
			cdw_vdm ("INFO: might do sao\n");
			disc->accepted_disc_modes |= CDW_DISC_MODE_SAO;
		}
		/*
		if (caps->might_do_raw) {
			disc->accepted_disc_modes |= CDW_DISC_MODE_RAW;
		}
		*/
	}
	burn_disc_free_multi_caps(&caps);


	int status = 0;
	off_t size = 0;
	unsigned bl_sas = 0;
	int num_formats;
	rv = burn_disc_get_formats(drive_infos[0].drive, &status, &size,
				   &bl_sas, &num_formats);
	if (rv != 1) {
		cdw_vdm ("ERROR: error while getting formats\n");
	} else {
		/*
		  empty DVD-R:
		  status = 1 (BURN_FORMAT_IS_UNFORMATTED)
		  size = 4706074624 - "the maximum achievable size of the media"
		  num_formats = 0

		  closed CD-R:
		  status = 2 (BURN_FORMAT_IS_FORMATTED)
		  size = 356722688 - "the currently formatted capacity"
		  num_formats = 0

		*/
		if (status == BURN_FORMAT_IS_UNFORMATTED) {
			cdw_vdm ("INFO: disc is not formatted, size of unformatted disc:\n");
			cdw_vdm ("INFO: %jd bytes / %f sectors / %f MB\n", size, (float) size / 2048.0, (float) size / (1024.0 * 1024.0));

			disc->libburn_info.unformatted_size = size;
		} else if (status == BURN_FORMAT_IS_FORMATTED) {
			cdw_vdm ("INFO: disc is formatted, size of readable data:\n");

			int read_capacity = 0;
			/* FIXME: this function is not available in older
			   versions of libburn */
			rv = burn_get_read_capacity(drive_infos[0].drive, &read_capacity, 0);
			if (rv != 1) {
				cdw_vdm ("ERROR: error while getting capacity\n");
			} else {
				cdw_vdm ("INFO: %d sectors / %f MB\n", read_capacity, (read_capacity * 2048.0) / (1024.0 * 1024.0));
			}
		} else { /* status == BURN_FORMAT_IS_UNKNOWN */
			cdw_vdm ("WARNING: unknown disc format\n");
		}
	}

	burn_drive_info_free(drive_infos);
	burn_finish();

	return CDW_OK;
}
