/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
   \file cdw_file_manager.c

   File implements two parts of application, that allow user to manipulate
   group of files:
   \li file selector
   \li list of selected files

   First element is a file system browser (displayed in separate window, on
   top of main ui) in which user can select files to be burned on optical
   disc. User can use Up/Down arrow keys, Home/End keys, PageUp/PageDown
   keys and Enter key to move in file system listing (just like in any file
   manager, e.g. in Midnight Commander, but with single panel).
   Files/directories are selected using Space key.

   Selected files are added to list of selected files and displayed in
   second element, that is list of selected files.

   Second element: list of selected files is a window (embedded in main
   application window) displaying list of files selected by user in
   file selector. User can use Up/Down arrow keys, Home/End keys,
   PageUp/PageDown keys to move on this list, and Delete key to deselect
   files from the list. Files selected by user will be burned to optical
   disc or written to ISO image.

   List of selected files can be accessed by other parts of application,
   e.g. to create graftpoints file.
*/

#define _BSD_SOURCE /* strdup() */

#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <stddef.h>

#include "cdw_main_window.h"
#include "gettext.h"
#include "cdw_file.h"
#include "cdw_fs.h"
#include "cdw_file_manager.h"
#include "cdw_string.h"
#include "cdw_list_display.h"
#include "cdw_ncurses.h"
#include "cdw_widgets.h"
#include "cdw_window.h"
#include "cdw_debug.h"
#include "cdw_fs_browser.h"
#include "cdw_graftpoints.h"
#include "cdw_config.h"
#include "cdw_logging.h"
#include "canonicalize.h"

extern cdw_config_t global_config;


static struct {
	long long size; /** \brief Size of currently selected files */
	double size_mb; /** \brief Size of currently selected files (in megabytes) */
	bool has_4GB_file;
	CDW_LIST_DISPLAY *display;

	bool follow_symlinks;
} selected_files;


static struct {
	/** \brief Path to file/directory that will initially be
	    displayed when file manager window is opened */
	char *initial_fullpath;

	/* \brief is this the first time during this
	   session with cdw that a file will be selected? */
	bool first_selection;
} file_selector;



static void     cdw_file_manager_internal_error_dialog(int error);

static cdw_rv_t cdw_file_selector_init(void);
static cdw_rv_t cdw_file_selector(void);
static cdw_rv_t cdw_file_selector_add_file_to_selected(cdw_fs_browser_t *fs_browser);
static void     cdw_file_selector_display_current_path(cdw_fs_browser_t *browser, WINDOW *window);

static void     cdw_selected_files_init(void);
static cdw_rv_t cdw_selected_files_clone_and_append_file(const cdw_file_t *file);
static cdw_rv_t cdw_selected_files_remove_file(size_t file_i);


enum {
	CDW_FM_E_UNEXPECTED,     /* unexpected error, generic error message */
	CDW_FM_E_NOT_ADDED,      /* file was not added to list of selected files */
	CDW_FM_E_NO_SELECTOR,    /* can't show file selector for some reason */
	CDW_FM_E_NO_SELECTED     /* can't show list of selected files */
};






void cdw_selected_files_init(void)
{
	selected_files.size_mb = 0.0;
	selected_files.size = 0;
	selected_files.has_4GB_file = false;
	selected_files.display = (CDW_LIST_DISPLAY *) NULL;

	/* The function is called during initialization of cdw, so
	   this global variable holds the most recent value. */
	selected_files.follow_symlinks = global_config.general.selected_follow_symlinks;

	return;
}





cdw_rv_t cdw_file_selector_init(void)
{
	file_selector.initial_fullpath = (char *) NULL;
	file_selector.first_selection = true;

	/* cdw_fs_get_initial_dirpath() tries to return one of
	   following: cwd, home, /tmp, or /  */
	file_selector.initial_fullpath = cdw_fs_get_initial_dirpath();
	if (file_selector.initial_fullpath == (char *) NULL) {
		/* something is clearly not right, we won't try to guess
		   what, but we will provide simple fallback value */
		file_selector.initial_fullpath = strdup("/");
	}
	if (file_selector.initial_fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create initial fullpath\n");
		return CDW_ERROR;
	}

	/* initially, after calling cdw_fs_get_initial_dirpath(),
	   file_selector.initial_fullpath points to dir, so it is OK to call
	   cdw_fs_check_existing_path() with CDW_FS_DIR */
	int rv = cdw_fs_check_existing_path(file_selector.initial_fullpath, R_OK | X_OK, CDW_FS_DIR);
	if (rv == 0) {
		cdw_sdm ("INFO: cdw file manager: initial fullpath initialized as = \"%s\"\n", file_selector.initial_fullpath);
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to validate proposed initial fullpath \"%s\"\n", file_selector.initial_fullpath);

		free(file_selector.initial_fullpath);
		file_selector.initial_fullpath = (char *) NULL;

		return CDW_ERROR;
	}

}





/**
   \brief Initialize file manager module

   Function initializes initial path that will be used by
   file selector window. Result of this is returned as return value
   of the function.

   You should initialize extern "char *base_dir_fullpath" before
   calling this function.

   \return CDW_OK on success
   \return CDW_MEM_ERROR on errors
*/
cdw_rv_t cdw_file_manager_init(void)
{
	cdw_selected_files_init();

	cdw_rv_t crv = cdw_file_selector_init();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to initialize file selector in file manager module\n");
		//cdw_file_manager_internal_error_dialog(CDW_FM_E_NO_SELECTOR);
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Clean up resources used by file manager module

   Call this function to deallocate all resources used by
   file manager module during work of the program
*/
void cdw_file_manager_clean(void)
{
	/* This 'if ()' duplicates code from
	   cdw_file_manager_delete_selected_files_view() and should be
	   removed from here, but I will leave it for now. */
	if (selected_files.display) {
		cdw_file_dealloc_files_from_list(selected_files.display->list);
		cdw_list_display_delete(&(selected_files.display));
	}

	cdw_string_delete(&file_selector.initial_fullpath);

	return;
}





/**
   \brief Return pointer to list of selected files

   Function returns pointer to head of doubly linked list
   of selected files (files selected by user to be burned to
   optical disc or written to ISO file).

   'selected_files' object should be initialized before calling
   this function.

   \return pointer to list of selected files, may be NULL if the list is empty
*/
cdw_dll_item_t *cdw_file_manager_get_list_of_selected_files(void)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"selected files\" display is null\n");
	cdw_assert (selected_files.display->list != (cdw_dll_item_t *) NULL, "ERROR: list of selected files is null\n");

	return selected_files.display->list;
}





/**
   \brief Remember current setting regarding "follow selected symbolic links" policy

   \param selected_follow_symlinks
*/
void cdw_file_manager_follow_symlinks(bool selected_follow_symlinks)
{
	selected_files.follow_symlinks = selected_follow_symlinks;

	return;
}





/**
   \brief Top level function for process of selecting files from file system

   Function displays file selector widget, in which user can select
   files/directories from file system, adds the files to list of
   selected files, and refreshes information (displayed in main app window)
   about currently selected files.

   'selected_files' display should be already initialized with
   cdw_file_manager_create_selected_files_view().

   'initial_dirpath' also needs to be initialized before calling this function.

   \return CDW_OK if some files were selected without problems
   \return CDW_NO if no files were selected (and no problems occurred)
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_file_manager_handle_adding_to_selected_files(void)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL, "ERROR: display not initialized properly\n");
	cdw_assert (file_selector.initial_fullpath != (char *)  NULL, "ERROR: initial dirpath is null\n");

	/* selector window, loop, key handling, adding files to list,
	   etc. is done by this function */
	cdw_rv_t crv = cdw_file_selector();
	cdw_main_window_wrefresh();

	cdw_rv_t retval = CDW_ERROR;
	if (crv == CDW_ERROR) {
		cdw_vdm ("ERROR: file selector returns CDW_ERROR\n");
		retval = CDW_ERROR;
	} else {
		if (selected_files.display->n_items == 0) {
			cdw_sdm ("INFO: selected_files->n_items == 0\n");
			retval = CDW_NO;
		} else {
			retval = CDW_OK;
		}
	}
	/* scroll()/refresh() even in case of errors: error may have occurred
	   when selecting 2nd or 3rd file, but the files that were selected
	   correctly must be displayed - using scroll() */
	cdw_list_display_scroll_to(selected_files.display, 0, 0, false);
	cdw_list_display_refresh(selected_files.display);

	return retval;
}





/**
   \brief Top level function for process of deselecting files from list of selected files

   Function moves keyboard focus to view with list of selected files
   and allows user to use movement keys to move on list of selected
   files to move on the list, and to use Delete key to deselect items
   from list of selected files.

   Information in 'selected files info' view in main app window is
   updated every time a file is deselected.

   Function displays error message if some problem occurred.

   \return CDW_OK on success
   \return CDW_ERROR on problems with deselecting file.
*/
cdw_rv_t cdw_file_manager_handle_deleting_from_selected_files(void)
{
	int key = 'a';
	cdw_rv_t crv = CDW_OK;

	while (selected_files.display->n_items > 0) {
		key = cdw_list_display_driver(selected_files.display);
		cdw_assert (selected_files.display->n_items > selected_files.display->current_item_ind,
			    "ERROR: file index is larger than number of files\n");
		if (key == KEY_DC) {
			crv = cdw_selected_files_remove_file(selected_files.display->current_item_ind);
			if (crv == CDW_OK) {
				crv = cdw_main_window_volume_info_view_update(-1, -1, false);
				if (crv == CDW_OK) {
					crv = CDW_OK;
				} else {
					cdw_vdm ("ERROR: failed to update files info view with fetch_data=false\n");
					crv = CDW_ERROR;
					break;
				}
			} else {
				cdw_vdm ("ERROR: failed to remove file nr %zd / %zd from 'selected files'\n",
					 selected_files.display->current_item_ind, selected_files.display->n_items);
				cdw_file_manager_internal_error_dialog(CDW_FM_E_UNEXPECTED);
				crv = CDW_ERROR;
				break;
			}
		} else if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
			/* 'Q' = quit file selector */
			break;
		} else {
			;
		}
	}
	/* FIXME: this function does not inform user when some errors occur */

	crv = cdw_main_window_volume_info_view_update(-1, -1, true);
	if (crv == CDW_OK) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to update files info view with fetch_data=false\n");
		return CDW_ERROR;
	}
}





/**
   \brief Get number of files that are currently selected by user

   Function returns number of files in 'selected_files' display.
   'selected_files' object need to be initialized before calling
   this function.

   \return number of files on 'selected files' list
*/
size_t cdw_file_manager_number_of_selected_files(void)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL, "'selected files' object not initialized\n");

	return selected_files.display->n_items;
}





/**
   \brief Display a window with file system browser

   Function displays new window with list of files in initial directory.
   User can use movement keys to move on the list, and Enter key to change
   directory - like in any file manager. User can't use Delete key to remove
   items from the list, but can use Space key to select them. Every selected
   file is added to 'selected files' list, so that later it can be burned
   to disc or written to ISO file.

   Function updates 'selected files info' area whenever new file is selected.
   It also updates number of selected files accordingly.

   User can close the window by pressing Escape key.

   The function modifies global (in file) variable initial_dirpath, so that
   it 'remembers' last directory which was visited during last call of the
   function (last time that the user selected files).

   In case of errors function returns with CDW_ERROR, but does not erase
   list of already selected files.

   \return CDW_OK when user pressed Escape key and no errors occurred during selection of files
   \return CDW_ERROR on errors in process of selecting files
*/
cdw_rv_t cdw_file_selector(void)
{
	/* the values are selected so that:
	   - there is enough space at the bottom so 'selected files size' line is visible
	   - the window is big enough to cover whole list of selected files
	     displayed in main app window, so we don't have to refresh it
	     every time a file is selected; */
	int n_lines = LINES - 6;
	int n_cols = COLS;
	int begin_y = 0;
	int begin_x = 0;
	WINDOW *window = newwin(n_lines, n_cols, begin_y, begin_x);
	if (window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create window for file system browser\n");
		return CDW_ERROR;
	}
	cdw_window_add_help(window);

	/* fs browser will occupy whole window, without leaving any margins or borders */
	cdw_fs_browser_t *fs_browser = cdw_fs_browser_new(window, file_selector.initial_fullpath);
	if (fs_browser == (cdw_fs_browser_t *) NULL) {
		free(window);
		window = (WINDOW *) NULL;

		cdw_vdm ("ERROR: failed to create file system browser\n");
		return CDW_ERROR;
	}

	if (file_selector.first_selection) {
		/* first call to file selector in this session with cdw,
		   path will be most probably a dirpath (cwd, HOME or tmp) */
		cdw_fs_browser_browse_into_dir(fs_browser, file_selector.initial_fullpath);
	} else {
		/* this is not the first call to file selector, and
		   there was some file selected previously; try to go
		   to the same file again */
		cdw_fs_browser_browse_to_file(fs_browser, file_selector.initial_fullpath);
	}
	cdw_file_selector_display_current_path(fs_browser, window);

	int key = 'a';
	cdw_rv_t retval = CDW_OK;
	while (key != CDW_KEY_ESCAPE) {
		key = cdw_fs_browser_driver(fs_browser);
		if (key == CDW_KEY_ENTER || key == KEY_BACKSPACE || key == '~') {
			cdw_file_selector_display_current_path(fs_browser, window);
		} else if (key == ' ') {
#ifndef NDEBUG
			char *fp = cdw_fs_browser_get_current_fullpath(fs_browser);
			cdw_vdm ("INFO: file manager: adding file \"%s\"\n", fp);
			free(fp);
			fp = (char *) NULL;
#endif
			cdw_rv_t crv = cdw_file_selector_add_file_to_selected(fs_browser);
			if (crv == CDW_ERROR) {
				cdw_vdm ("ERROR: failed while trying to add file\n");
				retval = CDW_ERROR;
				break;
			} else if (crv == CDW_NO) { /* item is not appendable, do nothing */
				;
			} else { /* crv == CDW_OK, file was added to list of
				    selected files; save current location so
				    that next time user opens file selector,
				    then working directory and highlighted
				    file is the same as when user previously
				    used the selector; fs browser can handle
				    situation in which previously visited dir
				    / file disappears between two calls to fs
				    browser */
				char *fullpath = cdw_fs_browser_get_current_fullpath(fs_browser);
				cdw_assert (fullpath != (char *) NULL, "ERROR: failed to get current dirpath\n");
				cdw_string_set(&file_selector.initial_fullpath, fullpath);
				free(fullpath);
				fullpath = (char *) NULL;
				cdw_vdm ("INFO: saved last file fullpath as %s\n", file_selector.initial_fullpath);
				file_selector.first_selection = false;
			}
		} else if (key == '?') {
			cdw_fs_browser_help_window(fs_browser);
			cdw_list_display_refresh(fs_browser->display);
		} else if (key == -1) {
			cdw_vdm ("ERROR: file system browser returned -1\n");
			retval = CDW_ERROR;
			break;
		} else if (key == 'q' || key == 'Q') {
			/* 'Q' = quit file selector */
			break;
		} else {
			;
		}
	}

	cdw_fs_browser_delete(&fs_browser);
	cdw_assert (fs_browser == (cdw_fs_browser_t *) NULL, "failed to properly delete fs browser\n");

	delwin(window);
	window = (WINDOW *) NULL;

	if (selected_files.display->n_items > 0) {
		/* true - upon finishing adding files recount size of
		   selected files */
		cdw_rv_t crv = cdw_main_window_volume_info_view_update(-1, -1, true);
		if (crv == CDW_OK) {
			return CDW_OK;
		} else {
			cdw_vdm ("ERROR: failed to update files info view with fetch_data=true\n");
			return CDW_ERROR;
		}
	}

	return retval;
}





/**
   \brief Create widget displayed in 'selected files' area of main app window

   Function creates widget with list of files (initially empty) and puts
   it in given \p parent window. \p parent should be initialized ncurses
   window, part of main application window. Widget created by this function
   will be 'embedded' in this window.

   \param parent - window into which new widget will be put

   \return CDW_OK on success
   \return CDW_ERROR when widget was not created properly
*/
cdw_rv_t cdw_file_manager_create_selected_files_view(WINDOW *parent)
{
	/* these are parameters for embedded selector window */
	int n_lines = getmaxy(parent);
	cdw_assert (n_lines > 2, "ERROR: parent has no more than 2 lines\n");
	int n_cols = getmaxx(parent);
	cdw_assert (n_cols > 2, "ERROR: parent has no more than 2 columns\n");

	cdw_assert (selected_files.display == (CDW_LIST_DISPLAY *) NULL,
		    "'selected_files' display is already initialized\n");

	/* 1, 1 - begin_y, begin_x */
	selected_files.display = cdw_list_display_new(parent, n_lines - 2, n_cols - 2, 1, 1, CDW_COLORS_MAIN);
	if (!selected_files.display) {
		cdw_vdm ("ERROR: failed to create 'selected files' display\n");
		cdw_file_manager_internal_error_dialog(CDW_FM_E_NO_SELECTED);

		return CDW_ERROR;
	} else {
		selected_files.display->display_item = cdw_file_display_file;
		cdw_list_display_add_return_key(selected_files.display, KEY_DC);
		selected_files.display->display_format = CDW_LIST_DISPLAY_FORMAT_LONG;

		cdw_list_display_refresh(selected_files.display);

		return CDW_OK;
	}
}





void cdw_file_manager_delete_selected_files_view(void)
{
	if (selected_files.display) {
		cdw_file_dealloc_files_from_list(selected_files.display->list);
		cdw_list_display_delete(&(selected_files.display));
	}

	return;
}





/**
   \brief Display dialog window with error message

   Function displays dialog window with one of several error messages.
   Error messages are chosen based on value of \p error. All dialog windows
   have only "OK" button.

   \param error - id of an error, accepted values are defined on top of this file
*/
void cdw_file_manager_internal_error_dialog(int error)
{
	/* gettext(): "resulting string [...] must not be modified or freed" */
	char *message = (char *) NULL;
	if (error == CDW_FM_E_NOT_ADDED) {
		/* 2TRANS: this is message in dialog window */
		message = _("File not added. Please try again.");
	} else if (error == CDW_FM_E_UNEXPECTED) {
		/* 2TRANS: this is message in dialog window */
		message = _("Unexpected error. Please close this window and open it again.");
	} else if (error == CDW_FM_E_NO_SELECTOR) {
		/* 2TRANS: this is message in dialog window. Some
		   function returned error value. No further action
		   will be performed, no other explanations provided */
		message = _("Cannot show file selector window. Some error occurred.");
	} else if (error == CDW_FM_E_NO_SELECTED) {
		/* 2TRANS: this is message in dialog window. Some
		   function returned error value. No further action
		   will be performed, no other explanations provided */
		message = _("Cannot display view with selected files. You should restart application.");
	} else {
		return;
	}

	/* 2TRANS: this is title of dialog window */
	cdw_buttons_dialog(_("File manager error"),
			   message, CDW_BUTTONS_OK, CDW_COLORS_ERROR);

	return;
}





/**
   \brief Refresh 'selected files' widget that is displayed in main application window

   Function redraws list of selected files, the list displays files from
   first to whichever still fits into widget's window.

   \return CDW_OK
*/
cdw_rv_t cdw_file_manager_regenerate_selected_files_view(void)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL,
		    "you called the function too early, selected_files is not initialized yet\n");
	cdw_list_display_scroll_to(selected_files.display, 0, 0, false);

	return CDW_OK;
}





/**
   \brief Count size (in file system) of all items on 'selected files' list

   Function calls cdw_fs_dirsize_path() for each item on list of files
   selected by user, and sums values returned. On success the sum value in
   megabytes is returned.

   TODO: if one of selected files is removed in native file system, and then
   this function is called, it detects that a file is missing and returns -1.
   The function should somehow mark files on selected files list that are
   missing, and mark them somehow.

   \return size (in megabytes) of all files on 'selected list' on success
   \return -1 on error
*/
double cdw_file_manager_calculate_selected_files_size_mb(void)
{
	cdw_fs_visitor_data_t vdata;
	vdata.size = 0;
	vdata.has_file_over_4GB = false;
	vdata.follow_symlinks = selected_files.follow_symlinks;
	selected_files.size_mb = 0.0;
	selected_files.size = 0;

	size_t i = 0;
	for (cdw_dll_item_t *f = selected_files.display->list; f != (cdw_dll_item_t *) NULL; f = f->next) {
		cdw_file_t *file = (cdw_file_t *) f->data;
		cdw_assert(file != (cdw_file_t *) NULL,
			   "ERROR: NULL file #%zd was not found on the list of selected files\n", i);
		cdw_assert(file->fullpath != (char *) NULL, "ERROR: file #%zd has no fullpath\n", i);

#ifndef NDEBUG
		fprintf(stderr, "\n\n");
#endif
		cdw_vdm ("INFO: traversing file \"%s\"\n", file->fullpath);

		cdw_rv_t crv = cdw_fs_traverse_path(file->fullpath, cdw_fs_visitor, &vdata);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to get dirsize of \"%s\" item\n", file->fullpath);
			return -1; /* error value */
		} else {
			/* this is to update size displayed next to file name
			   in "selected files" view - the size may have changed
			   after toggling "selected_follow_symlinks" flag */
			file->size = vdata.size;

			selected_files.size += vdata.size;
			vdata.size = 0;
		}
		i++;
	}

	selected_files.size_mb = (((double) selected_files.size) / 1024.0) / 1024.0;
	return selected_files.size_mb;
}





/**
   \brief Append copy of given file at the end of list of selected files

   The function does not refresh displayed information, you have to call
   cdw_list_display_scroll() to display new, updated content.

   The function will not append a file if it is invalid, unsupported, or if
   it is reference to parent dir ("..").

   The function creates a copy of \p file, and the copy is appended to
   display.

   This function was created to wrap in one function the code responsible
   for adding file to list of selected files. Therefore it is quite
   specialized (e.g. can't append ".."). TODO: Perhaps passing some predicate
   function operating on 'cdw_file_t *' as an argument would help to make
   it more generic and allow to move it to cdw_list_display.c

   \param file - file that you want to append to a display with list of selected files

   \return CDW_NO if no action was taken, but no error occurred either
   \return CDW_ERROR on errors
   \return CDW_OK if file was appended correctly
*/
cdw_rv_t cdw_selected_files_clone_and_append_file(const cdw_file_t *file)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"selected files\" display is null\n");
	cdw_assert (file != (cdw_file_t *) NULL, "ERROR: \"file\" argument is null\n");
	cdw_assert (file->fullpath != (char *) NULL, "ERROR: file has no fullpath set\n");

	if (file->invalid) {
		return CDW_NO;
	}

	if (file->is_ref_to_parent_dir) {
		/* we don't want to add whole parent directory */
		cdw_vdm ("WARNING: NOT attempting to add parent dir; BUT WHY this hasn't been caught before?\n");
		return CDW_NO;
	}

	if (file->type != CDW_FS_DIR && file->type != CDW_FS_FILE) {
		/* do nothing, unsupported file type: socket, pipe, etc. */
		return CDW_NO;
	}

	cdw_file_t *copy = cdw_file_duplicate(file);
	if (copy == (cdw_file_t *) NULL) {
		cdw_vdm ("ERROR: failed to duplicate file %ld / \"%s\"\n", (long int) file, file->fullpath);
		return CDW_ERROR;
	}

	cdw_rv_t crv = cdw_list_display_add_item(selected_files.display, (void *) copy, cdw_file_equal);

	if (crv == CDW_OK) {
		selected_files.size += copy->size; /* this is to avoid counting size of all selected files again */
		cdw_vdm ("INFO: added file with size = %lld, fullpath = \"%s\"\n",
			 copy->size, copy->fullpath);

	} else if (crv == CDW_NO) {
		/* given file was already on a list, no need to do anything */
		cdw_vdm ("INFO: not added file with path \"%s\", was already on list\n", copy->fullpath);
	} else {
		cdw_vdm ("ERROR: failed to add file \"%s\"\n", copy->fullpath);
		cdw_file_manager_internal_error_dialog(CDW_FM_E_NOT_ADDED);
		/* even if adding fails, nothing really happens, this
		   warning was all we could do */
	}

	return crv;
}





/**
   \brief Remove i-th item from "selected files" display

   Function looks up \p file_i-th item in "selected files" display, removes it
   from associated window and associated list. It also updates value of "number
   of files" variable associated with the display.

   \param file_i - index of file to be removed

   \return CDW_OK if removing was successful
   \return CDW_ERROR if it failed
*/
cdw_rv_t cdw_selected_files_remove_file(size_t file_i)
{
	cdw_assert (selected_files.display != (CDW_LIST_DISPLAY *) NULL, "ERROR: display is null\n");
	cdw_assert (selected_files.display->n_items > 0, "ERROR: you called the function for empty list of selected files\n");
	cdw_assert (selected_files.display->n_items > file_i, "ERROR: index of file is larger than number of files in \"selected files\" display\n");
	cdw_assert (selected_files.display->list != (cdw_dll_item_t *) NULL, "ERROR: list associated with the display is null\n");

	selected_files.display->current_item_ind = file_i;

	cdw_dll_item_t *item = cdw_dll_ith_item(selected_files.display->list, selected_files.display->current_item_ind);
	cdw_file_t *file = (cdw_file_t *) item->data;
	selected_files.size -= file->size;
	if (selected_files.size < 0) {
		cdw_vdm ("ERROR: size of selected files dropped to negative value %lld\n", selected_files.size);
		selected_files.size = 0;
	}
	/* WARNING: we have to free here item->data, not file */
	cdw_file_delete((cdw_file_t **) &(item->data));

	cdw_rv_t crv = cdw_list_display_remove_item(selected_files.display, selected_files.display->current_item_ind);

	if (crv == CDW_OK) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to remove item %zd from list of selected files\n", file_i);
		return CDW_ERROR;
	}
}





/**
   \brief Simple wrapper for two lines of code that appeared in two places

   Helps to avoid using the same window title in two places: window title
   ("Add files") is defined once - in this function.

   \param browser - file system browser widget to get current dirpath from
   \param window - window, in which current dirpath will be printed
*/
void cdw_file_selector_display_current_path(cdw_fs_browser_t *browser, WINDOW *window)
{
	char *dirpath = cdw_fs_browser_get_current_printable_dirpath(browser);
	cdw_assert (dirpath != (char *) NULL, "ERROR: failed to get current dirpath\n");
	/* 2TRANS: this is title of file selector window */
	cdw_window_add_strings(window, _("Add files"), dirpath);
	cdw_window_add_help(window);
	wrefresh(window);
	free(dirpath);
	dirpath = (char *) NULL;

	return;
}





cdw_rv_t cdw_file_manager_create_graftpoints_file(void)
{
	if (selected_files.display->n_items == 0){
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("No files selected"),
				   /* 2TRANS: this is message in dialog window, user
				      wants to write files or create image, but no
				      files from hdd are selected yet */
				   _("No files selected. Please use 'Add files'"),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		return CDW_NO;
	} else {
		cdw_rv_t crv = cdw_graftpoints_create_file();
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to create graftpoints file\n");
			return CDW_ERROR;
		} else {
			return CDW_OK;
		}
	}

}




void cdw_file_manager_delete_graftpoints_file(void)
{
	cdw_graftpoints_delete_file();

	return;
}





bool cdw_selected_files_file_over_4gb_present(void)
{
	return selected_files.has_4GB_file;
}





long long cdw_selected_files_get_size(void)
{
	return selected_files.size;
}





/**
   \param fs_browser

   \return CDW_NO if type of currently highlighted file is non-appendable
   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_file_selector_add_file_to_selected(cdw_fs_browser_t *fs_browser)
{
	cdw_assert (fs_browser->display->n_items > fs_browser->display->current_item_ind,
		    "ERROR: index of file is larger than number of files: n_items = %zd, item_i = %zd\n",
		    fs_browser->display->n_items, fs_browser->display->current_item_ind);

	cdw_file_t *file = cdw_fs_browser_get_current_file(fs_browser);
	cdw_assert (file != (cdw_file_t *) NULL, "ERROR: failed to get current file from fs browser\n");
	if (file->is_ref_to_parent_dir) {
		/* don't scan parent dir, don't attempt to add it */
		cdw_vdm ("INFO: NOT attempting to add parent dir\n");
		return CDW_NO; /* item non-appendable */
	} else if (file->type != CDW_FS_FILE && file->type != CDW_FS_DIR && file->type != CDW_FS_LINK) {
		cdw_vdm ("INFO: NOT attempting to add file of strange type \"%d\"\n", file->type);
		return CDW_NO; /* item non-appendable */
	} else if (cdw_list_display_is_member(selected_files.display, (void *) file, cdw_file_equal)) {
		/* file is already on list of selected files */
		cdw_vdm ("INFO: file \"%s\" already selected, not adding\n", file->fullpath);
		return CDW_NO;
	}

	cdw_fs_visitor_data_t vdata;
	vdata.size = 0;
	vdata.has_file_over_4GB = false;
	vdata.follow_symlinks = selected_files.follow_symlinks;

	cdw_rv_t crv = cdw_fs_traverse_path(file->fullpath, cdw_fs_visitor, &vdata);
	if (crv == CDW_OK) {
		file->size = vdata.size;
		crv = cdw_selected_files_clone_and_append_file(file);
		if (crv == CDW_OK) {

			/* false - don't recount size of selected files (yet),
			   size of selected files was updated in
			   cdw_selected_files_clone_and_append_file(), and
			   will be recounted again on file selector exit */
			crv = cdw_main_window_volume_info_view_update(-1, -1, false);
			/* updating files info view may disrupt part of file
			   selector window; redraw file selector */
			cdw_list_display_refresh(fs_browser->display);
			if (crv == CDW_OK) {
				return CDW_OK;
			} else {
				cdw_vdm ("ERROR: failed to update files info view with fetch_data=false\n");
				return CDW_ERROR;
			}
		} else if (crv == CDW_ERROR) {
			cdw_vdm ("ERROR: failed to append file to selected_files\n");
			return CDW_ERROR;
		} else {
			/* item not appendable (i.e. not file dir, or link) */
			return CDW_NO;
		}
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("For some reason cdw can't visit all files in selected directory (consult log file for details)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		/* 2TRANS: this is message printed into log file;
		   %s is a full path to a file */
		cdw_logging_write(_("ERROR: failed to fully scan this path: \"%s\"\n"), file->fullpath);

		return CDW_ERROR;
	}
}





/*
  \brief Look for \p path on list of selected files

  This is just a very simple implementation. It does not attempt to
  follow symbolic links that may be in selected files list.

  The \p path is canonicalized before function starts looking for it
  on list of selected files.

  \p can_mode is canonicalization mode, as seen in
  gnulib/lib/canonicalize.h. Not completely sure if this argument is
  useful at all.

  TODO: improve the function so that it covers more cases.

  \param path - path to look for on list of selected files
  \param can_mode - canonicalization mode
*/
cdw_rv_t cdw_selected_files_search(const char *path, int can_mode)
{
	char *fullpath = canonicalize_filename_mode(path, can_mode);
	if (!fullpath) {
		cdw_vdm ("ERROR: failed to canonicalize path \"%s\"\n", path);
		return CDW_ERROR;
	} else {
		cdw_vdm ("INFO: canonicalized: \"%s\"\n", fullpath);
	}


	size_t n = strlen(fullpath);

	size_t i = 0;
	for (cdw_dll_item_t *f = selected_files.display->list; f; f = f->next) {
		cdw_file_t *file = (cdw_file_t *) f->data;
		cdw_assert(file, "ERROR: NULL file #%zd was not found on the list of selected files\n", i);
		cdw_assert(file->fullpath, "ERROR: file #%zd has no fullpath\n", i);

		size_t sel = strlen(file->fullpath);
		int cmp = 0;
		if (sel >= n) {
			cmp = strncmp(fullpath, file->fullpath, n);
		} else {
			cmp = strncmp(fullpath, file->fullpath, n);
		}

		if (!cmp) {
			free(fullpath);
			fullpath = (char *) NULL;

			return CDW_OK;
		}

		i++;
	}

	free(fullpath);
	fullpath = (char *) NULL;

	return CDW_NO;
}
